#
# oo.tcl --- object oriented tcl extensions
#
# Copyright 1996, SATO Mitsuhide <mit-sato@scc-kk.co.jp>
#
# Comments, suggestions, and bug fixes are welcome.
# Please report to me <mit-sato@scc-kk.co.jp>.
#
# Support tcl version: tcl7.4 or later
#
# History
#   19.Jun.96 : sample implimentation.
#   22.Jun.96 : sample implimentation second edition.
#   24.Jun.96 : version 0.0-alpha-0 -- all implimentations are complete.
#   27.Jun.96 : version 0.0-alpha-1 -- method search optimized.
#			add method ":+", ":-", "??", ":??", "@dup" and
#			modify method ":=", "?", ":?"
#   28.Jun.96 : version 0.0-alpha-2 -- self -> $self
#   17.Jul.96 : version 0.0-alpha-3 -- :+ method optimize
#    2.Aug.96 : version 0.0-alpha-4 -- add method =^ ?^ :=^ :?^ @array @[in]vars
#   28.Aug.96 : version 0.0-alpha-5 -- add method ?instance ?null ?array ?var ?method
#			add @array option unset
#   30.Aug.96 : version 0.0-beta-1  -- auto load function.
#                       add @incr method
#   13.Sep.96 : version 0.0-beta-2  -- destructor bug fix
#   25.Sep.96 : version 0.0-beta-3  -- add method @symbol
#    8.Oct.96 : version 0.0-beta-4  -- change @symbol method syntax
#   10.Dec.96 : version 0.0         -- change my e-mail address.
#                                      First release version 0.0
#   11.Dec.96 : version 0.1         -- add method + -
#

#------------------------------------------------------------------------------
# <<< USAGE >>>
#
# 0. for use in tcl program
#
#	source oo.tcl
#
# 1. class define
# 
#	class CLASSNAME {
#		? {inherit SUPERCLASS} ?
#		? {var VAR1 VAR2 ...} ?
#	}
# 
#	method CLASSNAME new {} {<NEW_OBJECT_SCRIPT>}		 ; # constructor
#	method CLASSNAME delete {} {<DELETE_OBJECT_SCRIPT>}	 ; # destructor
#	method CLASSNAME METHOD1 {? ARG1 ARG2 ... ?} {<SCRIPT1>}
#	method CLASSNAME METHOD2 {? ARG1 ARG2 ... ?} {<SCRIPT2>}
#			:
#			:
#	defmethod CLASSNAME		  ; # make method table
# 
# 2. create object
#
# 2.1 create temporary object
#
#	new CLASSNAME OBJCTNAME
#
# 2.2 create instance object
#
#	$OBJCTNAME := VARNAME [new CLASSNAME]
#	$OBJCTNAME :+ VARNAME [new CLASSNAME]
#	$OBJCTNAME :=^ VARNAME INDEX [new CLASSNAME]
#
#  or
#       set var [new CLASSNAME]
#       $OBJECTNAME := VARNAME $var
#       $OBJECTNAME :+ VARNAME $var
#       $OBJECTNAME :=^ VARNAME INDEX $var
#
#   *** DO NOT EXECUTE THIS ***
#       delete $var
# 
# 3. delete object
# 
#	delete $OBJECTNAME
# 
# 4. method call
# 
# 4.1 user defined method
# 
#	$OBJNAME METHOD ? ARG1 ARG2 ... ?
#	$OBJNAME SUPERCLASS:METHOD ? ARG1 ARG2 ... ?
# 
# 4.2 generic method
# 
#	$OBJNAME = {VAR1 VAR2 ...} {VALUE1 VALUE2 ...}	; # set variable
#	$OBJNAME =^ VAR INDEX VALUE			; # set index variable
#       $OBJNAME + VAR VALUE1 ? VALUE2 ... ?            ; # add value to variable list
#       $OBJNAME - VAR VALUE1 ? VALUE2 ... ?            ; # delete value from variable list
#	$OBJNAME ?? {VAR1 VAR2 ...}			; # get variable
#	$OBJNAME ? VAR					; # get variable
#	$OBJNAME ?^ VAR INDEX				; # get index variable
#	$OBJNAME := {VAR1 VAR2 ...} {OBJ1 OBJ2 ...}	; # set instance variable
#	$OBJNAME :=^ VAR INDEX OBJ			; # set index instance variable
#	$OBJNAME :?? {VAR1 VAR2 ...}			; # get instance variable
#	$OBJNAME :? VAR					; # get instance variable
#	$OBJNAME :?^ VAR INDEX				; # get index instance variable
#	$OBJNAME :+ VAR OBJ1 ? OBJ2 ... ?		; # add instance variable
#	$OBJNAME :- VAR OBJ1 ? OBJ2 ... ?		; # delete instance variable
#	$OBJNAME <- ARRAY	; # set array to instance variable (without instance)
#	$OBJNAME -> ARRAY	; # set instance variable to array (without instance)
#	$OBJNAME @type		; # get instance type
#	$OBJNAME @vtype VAR	; # get instance variable type
#	$OBJNAME @array opt VAR ? INDEX ? ; # get all index names
#	$OBJNAME @super		; # get super class
#	$OBJNAME @vars		; # get instance variable all list
#	$OBJNAME @ivars		; # get instance variable instance list
#	$OBJNAME @nvars		; # get instance variable non instance list
#	$OBJNAME @vals		; # get instance variable value list
#	$OBJNAME @methods	; # get method list
#	$OBJNAME @inherit	; # get inherit tree
#	$OBJNAME @dup		; # duplicate instance
#       $OBJNAME @incr VAR ?inc?         ; # incriment
#       $OBJNAME @symbol VAR ? INDEX ?   ; # make instance variable symbol
#	$OBJNAME ?instance VAR ? INDEX ? ; # if VAR is instance variable return true
#	$OBJNAME ?null VAR		 ; # if VAR is null return true
#	$OBJNAME ?array VAR		 ; # if VAR is array return true
#	$OBJNAME ?var VAR		 ; # if VAR exists return true
#	$OBJNAME ?method METHOD		 ; # if METHOD exists return true
#
# 5. only useful class method:
#    '$self' is instance name itself.
#
#	$self METHOD ? ARG1 ARG2 ... ?		   ; # send message myself
#	$self SUPERCLASS:METHOD ? ARG1 ARG2 ... ?  ; # send message myself
#

#------------------------------------------------------------------------------
# <<< VARIABLE DESCRIPTION >>>
#
# current objects indicate variable
#
set __curObject ""	   ; # current object name indicated
if {! [info exist __objectCount]} {
    set __objectCount 0	   ; # object uniq indicate counter
}

#
# class indicate array variable
#
#  __className($class)		      --- exists class
#  __classVarList($class)	      --- class var array
#  __classMethodList($class)	      --- class method list
#  __classTree($class)		      --- class inherit tree
#  __classInherit($class)	      --- class inherit
#  __classMethod__${class}($method)   --- class method for method call
#
#
# object indicate array variable
#
#  __objectClass($obj)		      --- object class name,
#					  and indicated object alive
#  __objectVar($obj:$var)	      --- object variable
#  __objectVarType($obj:$var)	      --- instance variable type, 
#					  if not exists non object variable
#  __objectVarArrayType($obj:$var,$index)  --- array instance variable type,
#					       if not exists non object variable
#  __objectRef($obj)		      --- object reference count
#  __objectSymbol($obj)		      --- object symbol
#
# other variable
set __genericMethods \
  {= =^ + - ?? ? ?^ := :=^ :?? :? :?^ :+ :- <- -> \
  @type @vtype @array @super @vars @ivars @nvars @vals @methods @inherit @dup \
  @incr @symbol \
  ?instance ?null ?array ?var ?method}

#------------------------------------------------------------------------------
# <<< SOURCE >>>

#
# class: new class define command
#
proc class {name args} {
    global __className __classTree __classVarList __classMethodList \
	    __classInherit errorInfo

    if [info exists __className($name)] {
	return -code error "already defined class \"$name\""
    }
    set __className($name) $name
    set __classTree($name) {}
    set __classVarList($name) {}
    set __classMethodList($name) {}
    set __classInherit($name) {}
    foreach f [lindex $args 0] {
	if [catch {eval "__class_[lindex $f 0] $name [lrange $f 1 end]"}] {
	    __unsetClass $name
	    set err [lindex [split $errorInfo "\n"] 0]
	    return -code error \
		    "$err
illegal class define, class=\"$name\", indicate=\"[lindex $f 0]\""
	}
    }
    if [catch {__classNameCheck $name}] {
	__unsetClass $name
	return -code error [lindex [split $errorInfo "\n"] 0]
    }
    __defmethod $name
}

#
# __unsetClass: delete class information when illegal class defined
#
proc __unsetClass {name} {
    global __className __classTree __classVarList __classMethodList \
	    __classInherit

    unset __className($name)
    unset __classTree($name)
    unset __classVarList($name)
    unset __classMethodList($name)
    unset __classInherit($name)
}

#
# __class_inherit: class inherit process
#
proc __class_inherit {name super} {
    global __className __classTree __classInherit

    if {! [info exists __className($super)]} {
	set result [__classAutoLoad $super]
	if {$result == 0} {
	    return -code error \
		    "not defined super class, class=\"$name\", super=\"$super\""
	}
    }

    set __classInherit($name) $super
    set __classTree($name) $__classTree($super)
    lappend __classTree($name) $super
}

#
# __class_var: class var process
#
proc __class_var {name args} {
    global __classVarList

    set __classVarList($name) $args
}

#
# __classNameCheck: class variable duplicate check
#
proc __classNameCheck {name} {
    global __classVarList __classTree

    set namel {}
    foreach f $__classTree($name) {
	append namel "$__classVarList($f) "
    }
    foreach f $__classVarList($name) {
	if {[lsearch $namel $f] != -1} {
	    return -code error "already exist instance variable \"$f\""
	}
    }
}

#
# method: new method define command
#
proc method {class method arg body} {
    global __className __classMethodList

    if {! [info exists __className($class)]} {
	return -code error "not defined class \"$class\""
    }
    proc __$class:$method $arg "upvar 1 __curObject self;\n$body"
    if {[lsearch $__classMethodList($class) $method] == -1} {
	lappend __classMethodList($class) $method
	return "new"
    }
    return
}

#
# defmethod: create method table command
#
proc defmethod {name} {
    global __className __classTree

    if {! [info exists __className($name)]} {
	return -code error "not defined class \"$class\""
    }
    __defmethod $name
    foreach class [array names __className] {
	if {[lsearch $__classTree($class) $name] != -1} {__defmethod $class}
    }
    return
}

#
# __defmethod: create method table
#
proc __defmethod {name} {
    global __classTree __classMethodList __classMethod__${name} \
	    __classInherit __genericMethods

    catch {unset __classMethod__${name}}
    set super $__classInherit($name)
    global __classMethod__${super}
    if {$super == ""} {
	foreach f $__genericMethods {
	    set __classMethod__${name}($f) __$f
	}
    } {
	array set __classMethod__${name} [array get __classMethod__${super}]
    }
    foreach f $__classMethodList($name) {
	set __classMethod__${name}($f) __$name:$f
	set __classMethod__${name}($name:$f) __$name:$f
    }
}

#
# new: new instance command
#
proc new {class {obj {}}} {
    global __className __objectClass __objectCount __objectRef __objectSymbol
    
    if {! [info exists __className($class)]} {
	set result [__classAutoLoad $class]
	if {$result == 0} {
	    return -code error "not defined class \"$class\""
	}
    }
    incr __objectCount
    if {$obj != ""} {upvar $obj o}
    set o __o$__objectCount
    set __objectClass($o) $class
    proc $o {args} \
	    "global __curObject errorInfo __classMethod__$class;\
	    set prev \$__curObject;\
	    set __curObject $o ; set ret \"\";\
	    if \[catch {set func \$__classMethod__${class}(\[lindex \$args 0\])}] {\
		set __curObject \$prev;\
		return -code error \"no such method \\\"\[lindex \$args 0\]\\\"\";\
	    };\
	    set sts \[catch {set ret \[eval \"\$func \[lrange \$args 1 end\]\"\]}\];\
	    set __curObject \$prev;\
	    if {\$sts} {return -code error \$errorInfo};\
	    return \$ret;"
    if {$obj == ""} {set __objectRef($o) 0} {set __objectRef($o) 1}
    eval "__new $o"
    if {$obj != ""} {
	set __objectSymbol($o) $obj
	uplevel 1 "\
		trace variable $obj u \"__autoDelete $o\";\
		trace variable $obj w \"__illegalSet $o\";\
		"
    }

    if {$obj == ""} {return $o}
    return
}

#
# __classAutoLoad: auto load class.tcl file
#
proc __classAutoLoad {class} {
    global auto_path errorInfo __className

    foreach dir "$auto_path ." {
	set file $dir/${class}.tcl
	if {[file readable $file] && [file isfile $file]} {
	    set result [catch {source $file}]
	    if $result {
		puts stderr $errorInfo
		return 0
	    } {
		if {! [info exists __className($class)]} {
		    puts stderr "not defined class $class in file \"$file\""
		    return 0
		}
		return 1
	    }
	}
    }
    return 0
}

#
# __new: execute constructor
#
proc __new {obj} {
    global __objectClass __classTree __classVarList __objectVar __objectVarType \
	    __curObject __classMethodList errorInfo

    set prev $__curObject
    set __curObject $obj
    set class $__objectClass($obj)
    foreach f "$__classTree($class) $class" {
	foreach g $__classVarList($f) {
	    catch {unset __objectVarType($obj:$g)}
	    set __objectVar($obj:$g) ""
	}
    }
    foreach f "$__classTree($class) $class" {
	if {[lsearch $__classMethodList($f) new] != -1} {
	    if [catch {__$f:new}] {
		puts stderr $errorInfo
	    }
	}
    }
    set __curObject $prev
}

#
# delete: delete instance
#
proc delete {obj} {
    global __objectClass __objectRef __objectSymbol

    if {! [info exists __objectClass($obj)]} {
	return -code error "no such object \"$obj\""
    }
    incr __objectRef($obj) -1
    if {$__objectRef($obj) <= 0} {
	eval "__delete $obj"
	rename $obj ""
	catch {unset __objectClass($obj)}
	catch {unset __objectRef($obj)}
	if [info exists __objectSymbol($obj)] {
	    uplevel 1 "
		trace vdelete $__objectSymbol($obj) u \"__autoDelete $obj\";\
		trace vdelete $__objectSymbol($obj) w \"__illegalSet $obj\";\
		"
	    unset __objectSymbol($obj)
	}
    }
    return
}

#
# __delete: execute destructor
#
proc __delete {obj} {
    global __objectClass __classTree __classVarList __objectVar __objectVarType \
	    __curObject __classMethodList __objectVarArrayType errorInfo

    set prev $__curObject
    set __curObject $obj
    set class $__objectClass($obj)
    foreach f "$class [__reverse $__classTree($class)]" {
	if {[lsearch $__classMethodList($f) delete] != -1} {
	    if [catch {__$f:delete}] {
		puts stderr $errorInfo
	    }
	}
	foreach g $__classVarList($f) {
	    if [info exists __objectVarType($obj:$g)] {
		foreach o $__objectVar($obj:$g) {
		    catch {delete $o}
		}
		unset __objectVarType($obj:$g)
	    }
	    unset __objectVar($obj:$g)
	    foreach h [array names __objectVar $obj:$g,*] {
		catch {delete $__objectVar($h)}
		catch {unset __objectVarArrayType($h)}
		unset __objectVar($h)
	    }
	}
    }
    set __curObject $prev
}

#
# __autoDelete: auto delete execute when variable unset
#
proc __autoDelete {obj name1 name2 mode} {
    trace vdelete $name1 u "__autoDelete $obj"
    trace vdelete $name1 w "__illegalSet $obj"
    delete $obj
}

#
# __illegalSet: caution to set instance variable
#
proc __illegalSet {obj name1 name2 mode} {
    upvar $name1 o
    set o $obj
    return -code error "illegal set to instance \"$name1\""
}

#
# __reverse: reverse list
#
proc __reverse {list} {
    set ret {}
    foreach f $list {
	set ret "[list $f] $ret"
    }
    return $ret
}

#
# <<< GENERIC METHODS >>>
#

proc __= {args} {
    global __curObject __objectVar __objectVarType
    if {[llength $args] != 2} {
	return -code error "wrong # args: should be \"\$object = {var ...} {val ...}\""
    }
    set val [lindex $args 1]
    set i 0
    foreach f [lindex $args 0] {
	if {! [info exists __objectVar($__curObject:$f)]} {
	    return -code error "no such instance variable \"$f\""
	}
	if [info exists __objectVarType($__curObject:$f)] {
	    return -code error "\"$f\" is instance, do not replace"
	}
	set __objectVar($__curObject:$f) [lindex $val $i]
	incr i
    }
}

proc __=^ {args} {
    global __curObject __objectVar __objectVarArrayType
    if {[llength $args] != 3} {
	return -code error "wrong # args: should be \"\$object =^ var index val\""
    }
    set var [lindex $args 0]
    if {! [info exists __objectVar($__curObject:$var)]} {
	    return -code error "no such instance variable \"$var\""
    }
    set index [lindex $args 1]
    if [info exists __objectVarArrayType($__curObject:$var,$index)] {
	return -code error "\"${var}($index)\" is instance, do not replace"
    }
    set __objectVar($__curObject:$var,$index) [lindex $args 2]
    return ""
}

proc __+ {args} {
    global __curObject  __objectVar __objectVarType
    if {[llength $args] < 2} {
	return -code error "wrong # args: should be \"\$object + var val ? val ... ?\""
    }
    set var [lindex $args 0]
    if {! [info exists __objectVar($__curObject:$var)] } {
	return -code error "no such instance variable \"$var\""
    }
    if {[info exists __objectVarType($__curObject:$var)]} {
	return -code error "\"$var\" is instance, do not replace"
    }
    foreach f [lrange $args 1 end] {
	lappend __objectVar($__curObject:$var) $f
    }
}

proc __- {args} {
    global __curObject __objectVar __objectVarType
    if {[llength $args] < 2} {
	return -code error "wrong # args: should be \"\$object - var val ? val ... ?\""
    }
    set var [lindex $args 0]
    if {[info exists __objectVarType($__curObject:$var)]} {
	return -code error "\"$var\" is instance, do not replace"
    }
    foreach f [lrange $args 1 end] {
	set i [lsearch $__objectVar($__curObject:$var) $f]
	if {-1 == $i} continue
	set __objectVar($__curObject:$var) \
		[lreplace $__objectVar($__curObject:$var) $i $i]
    }
}

proc __?? {args} {
    global __curObject __objectVar __objectVarType
    if {[llength $args] != 1} {
	return -code error "wrong # args: should be \"\$object ?? {var ...}\""
    }
    set ret {}
    foreach f [lindex $args 0] {
	if {! [info exists __objectVar($__curObject:$f)]} {
	    return -code error "no such instance variable \"$f\""
	}
	if [info exists __objectVarType($__curObject:$f)] {
	    return -code error "\"$f\" is instance, do not reference"
	}
	lappend ret $__objectVar($__curObject:$f)
    }
    return $ret
}

proc __? {args} {
    global __curObject __objectVar __objectVarType
    if {[llength $args] != 1} {
	return -code error "wrong # args: should be \"\$object ? var\""
    }
    if {! [info exists __objectVar($__curObject:$args)]} {
	return -code error "no such instance variable \"$args\""
    }
    if [info exists __objectVarType($__curObject:$args)] {
	return -code error "\"$args\" is instance, do not reference"
    }
    return $__objectVar($__curObject:$args)
}

proc __?^ {args} {
    global __curObject __objectVar __objectVarArrayType
    if {[llength $args] != 2} {
	return -code error "wrong # args: should be \"\$object ?^ var index\""
    }
    set var [lindex $args 0]
    if {! [info exists __objectVar($__curObject:$var)]} {
	    return -code error "no such instance variable \"$var\""
    }
    set index [lindex $args 1]
    if [info exists __objectVarArrayType($__curObject:$var,$index)] {
	return -code error "\"${var}($index)\" is instance, do not reference"
    }
    if [info exists __objectVar($__curObject:$var,$index)] {
	return $__objectVar($__curObject:$var,$index)
    } {
	return -code error "no such element in array \"${var}($index)\""
    }
}

proc __:= {args} {
    global __curObject __objectClass __objectVar __objectVarType __objectRef
    if {[llength $args] != 2} {
	return -code error "wrong # args: should be \"\$object := {var ...} {obj ...}\""
    }
    set val [lindex $args 1]
    set i 0
    foreach f [lindex $args 0] {
	if {! [info exists __objectVar($__curObject:$f)]} {
	    return -code error "no such instance variable \"$f\""
	}
	set obj [lindex $val $i]
	foreach o $obj {
	    if {! [info exists __objectClass($o)]} {
		return -code error "\"$o\" is not instance"
	    }
	}
	set saveObj $__objectVar($__curObject:$f)
	set __objectVar($__curObject:$f) {}
	set __objectVarType($__curObject:$f) {}
	foreach o $obj {
	    incr __objectRef($o)
	    lappend __objectVar($__curObject:$f) $o
	    lappend __objectVarType($__curObject:$f) $__objectClass($o)
	}
	foreach o $saveObj {
	    catch {delete $o}
	}
	if {$__objectVarType($__curObject:$f) == ""} {
	    unset __objectVarType($__curObject:$f)
	}
	incr i
    }
}

proc __:=^ {args} {
    global __curObject __objectClass __objectVar __objectVarArrayType __objectRef
    if {[llength $args] != 3} {
	return -code error "wrong # args: should be \"\$object :=^ var index obj\""
    }
    set var [lindex $args 0]
    if {! [info exists __objectVar($__curObject:$var)]} {
	return -code error "no such instance variable \"${var}($index)\""
    }
    set val [lindex $args 2]
    if {($val != "") && (! [info exists __objectClass($val)])} {
	return -code error "\"$val\" is not instance"
    }
    set index [lindex $args 1]
    if [info exists __objectVarArrayType($__curObject:$var,$index)] {
	catch {delete $__objectVar($__curObject:$var,$index)}
    }
    if {$val == ""} {
	unset __objectVar($__curObject:$var,$index)
	unset __objectVarArrayType($__curObject:$var,$index)
	return
    }
    incr __objectRef($val)
    set __objectVar($__curObject:$var,$index) $val
    set __objectVarArrayType($__curObject:$var,$index) $__objectClass($val)
    return ""
}

proc __:?? {args} {
    global __curObject __objectVar __objectVarType
    if {[llength $args] != 1} {
	return -code error "wrong # args: should be \"\$object :?? {var ...}\""
    }
    set ret {}
    foreach f [lindex $args 0] {
	if {! [info exists __objectVarType($__curObject:$f)]} {
	    return -code error "no such instance variable \"$f\""
	}
	lappend ret $__objectVar($__curObject:$f)
    }
    return $ret
}

proc __:? {args} {
    global __curObject __objectVar __objectVarType
    if {[llength $args] != 1} {
	return -code error "wrong # args: should be \"\$object :? var\""
    }
    if {! [info exists __objectVarType($__curObject:$args)]} {
	return -code error "no such instance variable \"$args\""
    }
    return $__objectVar($__curObject:$args)
}

proc __:?^ {args} {
    global __curObject __objectVar __objectVarArrayType __objectRef
    if {[llength $args] != 2} {
	return -code error "wrong # args: should be \"\$object :?^ var index\""
    }
    set var [lindex $args 0]
    set index [lindex $args 1]
    if {! [info exists __objectVarArrayType($__curObject:$var,$index)]} {
	return -code error "no such instance variable \"${var}($index)\""
    }
    return $__objectVar($__curObject:$var,$index)
}

proc __:+ {args} {
    global __curObject __objectClass __objectVar __objectVarType __objectRef
    if {[llength $args] < 2} {
	return -code error "wrong # args: should be \"\$object :+ var obj ? obj ... ?\""
    }
    set var [lindex $args 0]
    if {! [info exists __objectVar($__curObject:$var)] } {
	return -code error "no such instance variable \"$var\""
    }
    if {(![info exists __objectVarType($__curObject:$var)]) && \
	    ($__objectVar($__curObject:$var) != "") } {
	return -code error "no such instance variable \"$var\""
    }
    if {! [info exists __objectVarType($__curObject:$var)]} {
	set __objectVarType($__curObject:$var) {}
    }
    foreach f [lrange $args 1 end] {
	if {! [info exists __objectClass($f)]} {
	    return -code error "\"$f\" is not instance"
	}
	incr __objectRef($f)
	lappend __objectVar($__curObject:$var) $f
	lappend __objectVarType($__curObject:$var) $__objectClass($f)
    }
}

proc __:- {args} {
    global __curObject __objectVar __objectVarType
    if {[llength $args] < 2} {
	return -code error "wrong # args: should be \"\$object :- var obj ? obj ... ?\""
    }
    set var [lindex $args 0]
    if {! [info exists __objectVarType($__curObject:$var)]} {
	return -code error "no such instance variable \"$var\""
    }
    foreach f [lrange $args 1 end] {
	set i [lsearch $__objectVar($__curObject:$var) $f]
	if {-1 == $i} continue
	catch {delete $f}
	set __objectVar($__curObject:$var) \
		[lreplace $__objectVar($__curObject:$var) $i $i]
	set __objectVarType($__curObject:$var) \
		[lreplace $__objectVarType($__curObject:$var) $i $i]
    }
    if {[llength $__objectVar($__curObject:$var)] == 0} {
	unset __objectVarType($__curObject:$var)
    }
}

proc __<- {args} {
    global __curObject __classVarList __objectVar __objectVarType
    if {[llength $args] != 1} {
	return -code error "wrong # args: should be \"\$object <- array\""
    }
    upvar 2 $args a
    foreach var [__@vars] {
	if [info exists __objectVarType($__curObject:$var)] continue
	if {! [info exists __objectVar($__curObject:$var)]} continue
	set __objectVar($__curObject:$var) $a($var)
    }
}

proc __-> {args} {
    global __curObject __classVarList __objectVar __objectVarType
    if {[llength $args] != 1} {
	return -code error "wrong # args: should be \"\$object -> array\""
    }
    upvar 2 $args a
    foreach var [__@vars] {
	if [info exists __objectVarType($__curObject:$var)] continue
	set a($var) $__objectVar($__curObject:$var)
    }
}

proc __@type {args} {
    global __curObject __objectClass
    if {[llength $args] != 0} {
	return -code error "wrong # args: should be \"\$object @type\""
    }
    return $__objectClass($__curObject)
}

proc __@vtype {args} {
    global __curObject __objectVarType
    if {[llength $args] != 1} {
	return -code error "wrong # args: should be \"\$object @vtype\""
    }
    if [info exists __objectVarType($__curObject:$args)] {
	return $__objectVarType($__curObject:$args)
    }
    return
}

proc __@array {args} {
    if {([llength $args] < 2) || ([llength $args] > 3)} {
	return -code error "wrong # args: should be \"\$object @array opt var ?index?\""
    }
    set opt [lindex $args 0]
    set var [lindex $args 1]
    set index [lindex $args 2]
    if [catch {set ret [__@array_$opt $var $index]}] {
	return -code error \
		"bad option \"$opt\": should be exists, names, size, type or unset."
    }
    return $ret
}

proc __@array_exists {var index} {
    global __curObject __objectVar
    if [info exists __objectVar($__curObject:$var,$index)] {
	return 1
    } {
	return 0
    }
}

proc __@array_names {var index} {
    global __curObject __objectVar
    set ret {}
    foreach f [array names __objectVar $__curObject:$var,*] {
	lappend ret [join [lrange [split $f ,] 1 end] ,]
    }
    return $ret
}

proc __@array_size {var index} {
    global __curObject __objectVar
    return [llength [array names __objectVar $__curObject:$var,*]]
}

proc __@array_type {var index} {
    global __curObject __objectVarArrayType
    if [info exists __objectVarArrayType($__curObject:$var,$index)] {
	return $__objectVarArrayType($__curObject:$var,$index)
    }
    return ""
}

proc __@array_unset {var index} {
    global __curObject __objectVar __objectVarArrayType
    if [info exists __objectVarArrayType($__curObject:$var,$index)] {
	catch {delete $__objectVar($__curObject:$var,$index)}
    }
    catch {unset __objectVar($__curObject:$var,$index)}
    catch {unset __objectVarArrayType($__curObject:$var,$index)}
    return ""
}

proc __@super {args} {
    global __curObject __objectClass __classTree
    if {[llength $args] != 0} {
	return -code error "wrong # args: should be \"\$object @super\""
    }
    return [lindex $__classTree($__objectClass($__curObject)) end]
}

proc __@vars {args} {
    global __curObject __objectClass __classTree __classVarList
    if {[llength $args] != 0} {
	return -code error "wrong # args: should be \"\$object @vars\""
    }
    set class $__objectClass($__curObject)
    set ret $__classVarList($class)
    foreach f $__classTree($class) {
	set ret "$ret $__classVarList($f)"
    }
    return $ret
}

proc __@ivars {args} {
    global __curObject __classTree __classVarList __objectVarType
    if {[llength $args] != 0} {
	return -code error "wrong # args: should be \"\$object @ivars\""
    }
    set ret {}
    foreach f [__@vars] {
	if [info exists __objectVarType($__curObject:$f)] {
	    lappend ret $f
	}
    }
    return $ret
}

proc __@nvars {args} {
    global __curObject __classTree __classVarList __objectVarType
    if {[llength $args] != 0} {
	return -code error "wrong # args: should be \"\$object @nvars\""
    }
    set ret {}
    foreach f [__@vars] {
	if ![info exists __objectVarType($__curObject:$f)] {
	    lappend ret $f
	}
    }
    return $ret
}

proc __@vals {args} {
    global __curObject __objectVar
    if {[llength $args] != 0} {
	return -code error "wrong # args: should be \"\$object @vals\""
    }
    set ret {}
    foreach f [__@vars] {
	lappend ret $__objectVar($__curObject:$f)
    }
    return $ret
}

proc __@methods {args} {
    global __curObject __objectClass
    if {[llength $args] != 0} {
	return -code error "wrong # args: should be \"\$object @methods\""
    }
    set class $__objectClass($__curObject)
    global __classMethod__${class}
    return [array names __classMethod__${class}]
}

proc __@inherit {args} {
    global __curObject __objectClass __classTree
    if {[llength $args] != 0} {
	return -code error "wrong # args: should be \"\$object @inherit\""
    }
    return $__classTree($__objectClass($__curObject))
}

proc __@dup {args} {
    global __curObject __classTree __objectRef
    if {[llength $args] != 0} {
	return -code error "wrong # args: should be \"\$object @dup\""
    }
    incr __objectRef($__curObject)
    return $__curObject
}

proc __@incr {args} {
    global __curObject __objectVar __objectVarType
    if {([llength $args] != 1) && ([llength $args] != 2)} {
	return -code error "wrong # args: should be \"\$object @incr ?incriment?\""
    }
    set var [lindex $args 0]
    if [info exists __objectVarType($__curObject:$var)] {
	return -code error "\"$var\" is instance, do not incriment"
    }
    set inc 1
    if {[llength $args] == 2} {
	set inc [lindex $args 1]
    }
    set v [expr $__objectVar($__curObject:$var) + $inc]
    set __objectVar($__curObject:$var) $v
    return $v
}

proc __@symbol {args} {
    global __curObject
    if {! (([llength $args] == 1) || ([llength $args] == 2))} {
	return -code error "wrong # args: should be \"\$object @symbol var ?index?\""
    }
    if {[llength $args] == 1} {
	return "__objectVar($__curObject:$args)"
    }
    return "__objectVar($__curObject:[lindex $args 0],[lindex $args 1])"
}

proc __?instance {args} {
    global __curObject __objectVarType __objectVarArrayType
    if {[llength $args] == 1} {
	set var [lindex $args 0]
	if [info exists __objectVarType($__curObject:$var)] {return 1}
	return 0
    }
    if {[llength $args] == 2} {
	set var [lindex $args 0]
	set index [lindex $args 1]
	if [info exists __objectVarArrayType($__curObject:$var,$index)] {return 1}
	return 0
    }
    return -code error "wrong # args: should be \"\$object ?instance var ?index?\""
}

proc __?null {args} {
    global __curObject __objectVar __objectVarType
    if {[llength $args] != 1} {
	return -code error "wrong # args: should be \"\$object ?null var\""
    }
    if {! [info exists __objectVar($__curObject:$args)]} {return 0}
    if [info exists __objectVarType($__curObject:$args)] {return 0}
    if {$__objectVar($__curObject:$args) == ""} {return 1}
    return 0
}

proc __?array {args} {
    global __curObject __objectVar
    if {[llength $args] != 1} {
	return -code error "wrong # args: should be \"\$object ?array var\""
    }
    if {[array names __objectVar $__curObject:$args,*] == ""} {return 0}
    return 1
}

proc __?var {args} {
    global __curObject __objectVar
    if {[llength $args] != 1} {
	return -code error "wrong # args: should be \"\$object ?var var\""
    }
    if [info exists __objectVar($__curObject:$args)] {return 1}
    return 0
}

proc __?method {args} {
    global __curObject __objectClass
    if {[llength $args] != 1} {
	return -code error "wrong # args: should be \"\$object ?method method\""
    }
    set class $__objectClass($__curObject)
    global __classMethod__${class}
    if [info exists __classMethod__${class}($args)] {return 1}
    return 0
}

#
# <<< END >>>
#------------------------------------------------------------------------------
