/*
 * file: ~/ptcl/ptcl.c
 *
 *	 persistent Tcl variables
 *
 * /-----------------------------------------\
 * | Copyright (c) 1997 by Torsten Rottmann  |
 * | 31162 Bad Salzdetfurth, Griesbergstr.11 |
 * | Germany, Phone: (+49)05063/5710	     |
 * | eMail: trott@rottmann.hi.shuttle.de     |
 * \_________________________________________/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABLITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * TRott: Sat Oct 11 18:54:00 CEST 1997
 *
 * $Id: ptcl.c,v 1.4 1997/10/19 15:30:42 trott Exp trott $
 *
 */

#include <ctype.h>
#include <gdbm.h>
#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include "tcl.h"

#if TCL_MAJOR_VERSION<8
#error bad version of Tcl: TCL_VERSION<8.0
#endif

#define PTCL_FRESH	0	/* neither reads nor writes: fresh varName */
#define PTCL_READ	1	/* a read was the first access to it */
#define PTCL_WRITE	2	/* user has modified the var: it's dirty */
#define PTCL_UNSET	3	/* var was in use and has been unset */

#define DEBUG	0		/* set to 1, if you want to watch things */

extern gdbm_error gdbm_errno;

static Tcl_HashTable pvars;		/* table of names of persistent vars */
static GDBM_FILE pf=NULL;		/* persistent storage... */
static Tcl_DString pfNameDS;		/* ...and the name of its file */
static char *pfName;
static Tcl_HashEntry *ptcl_entervar(Tcl_Interp *interp, char *name, int traces);

static datum
makekey(char *name1, char *name2)
{
    static char keyname[255];
    
    if (name2)
    {
    	datum key;

	sprintf(keyname,"%s(%s)",name1,name2);
    	key.dptr=keyname;
    	key.dsize=strlen(name1)+strlen(name2)+2;
    	
    	return key;
    }
    else
    {
	datum key={name1,strlen(name1)};

	return key;
     }
}

static char *
ptcl_trace(ClientData clientData, Tcl_Interp *interp,
	   char *name1, char *name2, int flags)
{
    Tcl_HashEntry *e=(Tcl_HashEntry*)clientData;
    int stat;

#if DEBUG
    printf("ptcl_trace: %s %s %08x %d\n",
    	   name1,name2,flags,(int)Tcl_GetHashValue(e));
    fflush(stdout);
#endif

    assert(e!=NULL);
    
    /* a persistent array is traced -> enter its component */
    if (strcmp(Tcl_GetHashKey(&pvars,e),name1)==0 && name2!=NULL)
    {
        char n[255];
        
        sprintf(n,"%s(%s)",name1,name2);
	e=ptcl_entervar(interp,n,0);
    }
    
    stat=(int)Tcl_GetHashValue(e);

    if (flags&TCL_TRACE_READS && stat==PTCL_FRESH)
    {
	/* try to open the "persistent storage": it does'nt matter
	   if it exists or not: This simply means that we run the
	   application the first time and don't have anything
	   stored yet
	*/
	if (pf==NULL)
	    pf=gdbm_open(pfName,1024,GDBM_READER,0666,0);

	if (pf!=NULL)	/* yes, we have something from a previous run */
	{
	    datum v=gdbm_fetch(pf,makekey(name1,name2));
        
	    if (v.dptr)
	    {
    		*(v.dptr+v.dsize)='\0';
		Tcl_SetVar2(interp,name1,name2,v.dptr,TCL_GLOBAL_ONLY);
		free(v.dptr);
	    }
	}
	Tcl_SetHashValue(e,(ClientData)PTCL_READ);
	Tcl_UntraceVar2(interp,name1,name2,
	   	TCL_GLOBAL_ONLY|TCL_TRACE_READS,ptcl_trace,(ClientData)e);
	Tcl_TraceVar2(interp,name1,name2,TCL_GLOBAL_ONLY|TCL_TRACE_UNSETS,
    		ptcl_trace,(ClientData)e);
    }
    else if (flags&TCL_TRACE_WRITES)
    {
	Tcl_UntraceVar2(interp,name1,name2,
	   	TCL_GLOBAL_ONLY|TCL_TRACE_WRITES,ptcl_trace,(ClientData)e);
	if (stat==PTCL_FRESH)
	    Tcl_UntraceVar2(interp,name1,name2,TCL_GLOBAL_ONLY|TCL_TRACE_READS,
			ptcl_trace,(ClientData)e);
	Tcl_TraceVar2(interp,name1,name2,TCL_GLOBAL_ONLY|TCL_TRACE_UNSETS,
    		ptcl_trace,(ClientData)e);			
	Tcl_SetHashValue(e,(ClientData)PTCL_WRITE);
    }
    else if (flags&TCL_TRACE_UNSETS)
    {
	Tcl_SetHashValue(e,(ClientData)PTCL_UNSET);
	/* since the variable will be unset, the traces are automatically
	   deleted */
    }

    return NULL;
}

static Tcl_HashEntry *
ptcl_entervar(Tcl_Interp *interp, char *name, int traces)
{
    int isnew;
    Tcl_HashEntry *e=Tcl_CreateHashEntry(&pvars,name,&isnew);

    if (isnew)
    {
	if (Tcl_GetVar(interp,name,TCL_GLOBAL_ONLY)!=NULL)
	{
	    Tcl_SetHashValue(e,(ClientData)PTCL_WRITE);
	    if (traces)
	    if (Tcl_TraceVar(interp,name,TCL_GLOBAL_ONLY|TCL_TRACE_UNSETS,
    		ptcl_trace,(ClientData)e)!=TCL_OK) return NULL;
    	}
	else
	{
	    Tcl_SetHashValue(e,(ClientData)PTCL_FRESH);
	    if (traces)
	    {
	      if (Tcl_TraceVar(interp,name,TCL_GLOBAL_ONLY|TCL_TRACE_READS,
    		ptcl_trace,(ClientData)e)!=TCL_OK) return NULL;
	      if (Tcl_TraceVar(interp,name,TCL_GLOBAL_ONLY|TCL_TRACE_WRITES,
    		ptcl_trace,(ClientData)e)!=TCL_OK) return NULL;
    	    }
	}
    }
    return e;
}

/* the "persistent" command:
 */
static int
ptcl_persistent(ClientData clientData, Tcl_Interp *interp,
	        int objc, Tcl_Obj *CONST objv[])
{
    /* argument check */
    if (objc<2)
    {
	Tcl_WrongNumArgs(interp,1,objv,"varName ?varName ...?");
	return TCL_ERROR;
    }

    while (objc>1)
        if (ptcl_entervar(interp,Tcl_GetStringFromObj(objv[--objc],NULL),1)
        	==NULL) return TCL_ERROR;
    
    return TCL_OK;
}

static void
ptcl_exithandler(ClientData clientData)
{
    Tcl_Interp *interp=(Tcl_Interp *)clientData;
    Tcl_HashSearch s;
    Tcl_HashEntry *e;

#if DEBUG
    printf("dumping to %s: ",pfName);
#endif

    if (pf) gdbm_close(pf);
    pf=gdbm_open(pfName,1024,GDBM_WRCREAT|GDBM_FAST,0666,0);
    if (pf==NULL)
    {
        if (gdbm_errno==GDBM_CANT_BE_WRITER)
        	return;	/* this is not the last instance to exit */
	Tcl_AppendResult(interp,"can't create dbmfile: ",
	    		 gdbm_strerror(gdbm_errno),NULL);
	Tcl_BackgroundError(interp);
	Tcl_DoOneEvent(TCL_IDLE_EVENTS);	/* report error immediatly */

	return;
    }

    e=Tcl_FirstHashEntry(&pvars,&s);
    if (e!=NULL)
    {
	do 
	{
	    switch ((int)Tcl_GetHashValue(e))
	    {
	       case PTCL_WRITE:	/* dirty? */
	       {
		char *vn=Tcl_GetHashKey(&pvars,e);
		char *vc=Tcl_GetVar(interp,vn,TCL_GLOBAL_ONLY);
		datum dn={vn,strlen(vn)};
		datum dc={vc,strlen(vc)};
#if DEBUG
		printf("storing %s\n",vn);
#endif
	    	gdbm_store(pf,dn,dc,GDBM_REPLACE);
	       }
	       break;

	       case PTCL_UNSET:
	       {
		char *vn=Tcl_GetHashKey(&pvars,e);
		datum dn={vn,strlen(vn)};
#if DEBUG
		printf("deleting %s\n",vn);
#endif
	    	gdbm_delete(pf,dn);
	       }
	       break;
	    }

	}
	while ((e=Tcl_NextHashEntry(&s))!=NULL);
    }

    if (pf) gdbm_close(pf);
    Tcl_DeleteHashTable(&pvars);
    Tcl_DStringFree(&pfNameDS);
}

/* return a RCS revision id and only the id number */
static char *
revisionid(void)
{
#if defined(__GNUC__)
    /* GNU C allows a string to be writable, if it is declared
       as an unbounded array of chars...
     */
    static char f[]="$Revision: 1.4 $";
    char *b,*t=f;
    
    while (*t!='\0' && !isdigit(*t)) t++;
    for (b=t; *b!='\0' && !isspace(*b); b++);
#else
    /* ...and others don't.
       Since the revisionstring is allocated in the text(=code) segment
       and therefore not writable, it is extracted at runtime.
       A poor solution, but only needed once at package inclusion time...
     */
    char *b,*f="$Revision: 1.4 $";
    static char t[10];
    
    while (*f!='\0' && !isdigit(*f)) f++;
    for (b=t; *f!='\0' && !isspace(*f); *b++ = *f++);
#endif
    *b='\0';
    
    return t;
}

int
Ptcl_Init(Tcl_Interp *interp)
{
    if (interp==NULL) return TCL_ERROR;

    /* make shure we have the same major Tcl version
       as we have compiled with: */
    if (Tcl_PkgRequire(interp,"Tcl",TCL_VERSION,0)==NULL)
    	return TCL_ERROR;

    /* choose a name for the persistent storage file: */
    if (Tcl_Eval(interp,"format ~/.[file root [file tail $argv0]].pst")!=TCL_OK)
    	return TCL_ERROR;
    if ((pfName=Tcl_TranslateFileName(interp,interp->result,&pfNameDS))==NULL)
    	return TCL_ERROR;

    /* create a hash table for the names of our persistent vars */
    Tcl_InitHashTable(&pvars,TCL_STRING_KEYS);

    Tcl_CreateObjCommand(interp,"persistent",ptcl_persistent,NULL,NULL);

    /* The exithandler stores our variables and makes them
       really "persistent": */
    Tcl_CreateExitHandler((Tcl_ExitProc *)ptcl_exithandler,(ClientData)interp);

    return Tcl_PkgProvide(interp,"ptcl",revisionid());
}
