/* 
 * assocData.c --
 *
 *	Implements association of Data with interpreters for Tcl 7.4
 *	and earlier, where this facility is not part of the core.
 *
 * Copyright (c) 1995 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#ifndef lint
static char sccsid[] = "%Z% %M% %I% %E% %U%";
#endif

#include	<tcl.h>
#include	<tclPort.h>

#if ((TCL_MAJOR_VERSION < 7) || (TCL_MINOR_VERSION < 5))

/*
 * Data type for storing data with an interpreter:
 */
typedef struct AssocData {
    Tcl_InterpDeleteProc *proc;		/* Proc called when deleting. */
    ClientData clientData;		/* Value to pass to proc. */
} AssocData;

/*
 *----------------------------------------------------------------------
 * Static global data to store the associations for all interpreters. We
 * use one global hash table which is keyed by TCL_ONE_WORD_KEYS for which
 * we use the address of an interpreter. The value of each hash entry is
 * a hash table of data associated with the specific interpreter.
 *----------------------------------------------------------------------
 */
static Tcl_HashTable *globalAssocDataPtr = (Tcl_HashTable *) NULL;


/*
 *----------------------------------------------------------------------
 *
 * Tcl_DeleteAssociatedData --
 *
 *	Helper function called when an interpreter is deleted to delete
 *	all of its associated data.
 *
 * Results:
 *	NONE.
 *
 * Side effects:
 *	Deletes the data associated with a specific interpreter.
 *
 *----------------------------------------------------------------------
 */

static void
Tcl_DeleteAssociatedData(clientData, interp)
    ClientData clientData;
    Tcl_Interp *interp;
{
    Tcl_HashEntry *hPtr = (Tcl_HashEntry *) clientData;
    Tcl_HashTable *hTbl;
    Tcl_HashSearch hSearch;
    Tcl_HashEntry *hEnt;
    AssocData *d;

    hTbl = (Tcl_HashTable *) Tcl_GetHashValue(hPtr);
    for (hEnt = Tcl_FirstHashEntry(hTbl, &hSearch);
         hEnt != NULL;
         hEnt = Tcl_FirstHashEntry(hTbl, &hSearch)) {
        d = Tcl_GetHashValue(hEnt);
        if (d->proc != NULL)
            (*d->proc)(d->clientData, interp);
        ckfree(d);
        Tcl_DeleteHashEntry(hEnt);
    }
    Tcl_DeleteHashTable(hTbl);
    ckfree(hTbl);

    Tcl_DeleteHashEntry(hPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_SetAssocData --
 *
 *	Creates an association between an interpreter and a client
 *	specified piece of data under a string key. If the key is already
 *	present then it overwrites existing data. A delete function
 *	can also be specified (or given as NULL); when the interpreter
 *	is destroyed the delete function will be called.
 *
 * Results:
 *	TCL_OK if the association was set (and maybe created), TCL_ERROR else.
 *
 * Side effects:
 *	Creates the association.
 *
 *----------------------------------------------------------------------
 */

int
Tcl_SetAssocData(interp, name, proc, clientData)
    Tcl_Interp *interp;		/* Interpreter to associate with. */
    char *name;			/* Name for association. */
    Tcl_InterpDeleteProc *proc;	/* Proc to call when interpreter is
                                 * about to be deleted. */
    ClientData clientData;	/* One-word value to pass to proc. */
{
    Tcl_HashEntry *hPtr;
    Tcl_HashTable *interpAssocDataPtr;
    AssocData *d;
    int new;

    if (globalAssocDataPtr == (Tcl_HashTable *) NULL) {
        globalAssocDataPtr = (Tcl_HashTable *) ckalloc(sizeof(Tcl_HashTable));
        Tcl_InitHashTable(globalAssocDataPtr, TCL_ONE_WORD_KEYS);
    }
    hPtr = Tcl_FindHashEntry(globalAssocDataPtr, (char *) interp);
    if (hPtr == (Tcl_HashEntry *) NULL) {
        interpAssocDataPtr = (Tcl_HashTable *) ckalloc(sizeof(Tcl_HashTable));
        Tcl_InitHashTable(interpAssocDataPtr, TCL_STRING_KEYS);
        hPtr = Tcl_CreateHashEntry(globalAssocDataPtr, (char *) interp, &new);
        Tcl_SetHashValue(hPtr, interpAssocDataPtr);
        Tcl_CallWhenDeleted(interp, Tcl_DeleteAssociatedData, hPtr);
    }
    interpAssocDataPtr = (Tcl_HashTable *) Tcl_GetHashValue(hPtr);
    hPtr = Tcl_CreateHashEntry(interpAssocDataPtr, name, &new);
    if (new == 0) {
        d = (AssocData *) Tcl_GetHashValue(hPtr);
    } else {
        d = (AssocData *) ckalloc(sizeof(AssocData));
    }
    d->proc = proc;
    d->clientData = clientData;
    Tcl_SetHashValue(hPtr, d);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_GetAssocData --
 *
 *	Returns the client data associated with this name in the
 *	specified interpreter.
 *
 * Results:
 *	The client data in the AssocData record denoted by the named
 *	association, or NULL.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

ClientData
Tcl_GetAssocData(interp, name, procPtr)
    Tcl_Interp *interp;
    char *name;
    Tcl_InterpDeleteProc **procPtr;
{
    Tcl_HashEntry *hPtr;
    Tcl_HashTable *hTbl;
    AssocData *d;

    if (globalAssocDataPtr == (Tcl_HashTable *) NULL)
        return NULL;
    hPtr = Tcl_FindHashEntry(globalAssocDataPtr, (char *) interp);
    if (hPtr == (Tcl_HashEntry *) NULL)
        return NULL;
    hTbl = (Tcl_HashTable *) Tcl_GetHashValue(hPtr);
    hPtr = Tcl_FindHashEntry(hTbl, name);
    if (hPtr == (Tcl_HashEntry *) NULL)
        return NULL;
    d = (AssocData *) Tcl_GetHashValue(hPtr);
    if (procPtr != (Tcl_InterpDeleteProc **) NULL)
        *procPtr = d->proc;

    return d->clientData;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_DeleteAssocData --
 *
 *	Deletes a named association of user specified data with the
 *	specified interpreter.
 *
 * Results:
 *	TCL_OK if the association was deleted, TCL_ERROR else.
 *
 * Side effects:
 *	The association is deleted.
 *
 *----------------------------------------------------------------------
 */

int
Tcl_DeleteAssocData(interp, name)
    Tcl_Interp *interp;			/* Interpreter to associate with. */
    char *name;				/* Name of association. */
{
    Tcl_HashEntry *hPtr;
    Tcl_HashTable *hTbl;
    AssocData *d;

    if (globalAssocDataPtr == (Tcl_HashTable *) NULL) {
        return TCL_OK;
    }
    hPtr = Tcl_FindHashEntry(globalAssocDataPtr, (char *) interp);
    if (hPtr == (Tcl_HashEntry *) NULL) {
        return TCL_OK;
    }
    hTbl = (Tcl_HashTable *) Tcl_GetHashValue(hPtr);
    hPtr = Tcl_FindHashEntry(hTbl, name);
    if (hPtr == (Tcl_HashEntry *) NULL) {
        return TCL_OK;
    }
    d = (AssocData *) Tcl_GetHashValue(hPtr);

    ckfree(d);
    Tcl_DeleteHashEntry(hPtr);

    return TCL_OK;
}

#endif		/* ENTIRE FILE!!! */

