/* $Id: stooop.c,v 1.82 1997/08/25 21:36:22 jfontain Exp $ */

/*
    # how to create a dynamically loadable library on different platforms (please send me commands for your platform if not listed):
    # TCLSRC is the Tcl source directory path name, TCLSRC=/usr/sources/tcl8.0 for example

    # SunOS
    cc -O2 -pic -I$TCLSRC/generic -c stooop.c                                                                  # with stock compiler
    gcc -O2 -fPIC -Wall -I$TCLSRC/generic -c stooop.c                                                                     # with gcc
    ld -assert pure-text -o libstooop3.0.so stooop.o

    # AT&T SVR4.0
    cc -O -KPIC -Xa -v -I$TCLSRC/generic -c stooop.c                                                           # with stock compiler
    gcc -O2 -fPIC -Wall -I$TCLSRC/generic -c stooop.c                                                                     # with gcc
    ld -G -o libstooop3.0.so -h libstooop3.0.so stooop.o

    # solaris 2.5.1 (thanks to Larry W. Virden <lvirden@cas.org>)
    cc -fast -xO5 -xdepend -KPIC -Xa -v -I$TCLSRC/generic -c stooop.c
    ld -G -o libstooop3.0.so -h libstooop3.0.so stooop.o

    # linux
    cc -shared -o libstooop3.0.so -O2 -fPIC -Wall -I$TCLSRC/generic stooop.c

    # AIX 4.1
    gcc -O2 -fPIC -Wall -I$TCLSRC/generic -c stooop.c                                                                     # with gcc
    # use ldAix command from Tcl 7.5 distribution
    $TCLSRC/unix/ldAix /bin/ld -bhalt:4 -bM:SRE -bE:lib.exp -H512 -T512 -o libstooop3.0.so stooop.o -L/usr/local/lib -ltcl7.5 -lc

    # HP-UX 9.0x (thanks to Roldano Cattoni <cattoni@irst.itc.it>)
    cc -O -Ae +z -I$TCLSRC/generic -c stooop.c                                                                 # with stock compiler
    gcc -O2 -fPIC -Wall -I$TCLSRC/generic -c stooop.c                                                                     # with gcc
    ld -b stooop.o -o libstooop3.0.sl 

    # (the Tcl generic source code directory is needed for tclInt.h, its location may be different on your machine)

    # to use it in your Tcl code, just load it before or after sourcing stooop.tcl, as in:
    load ./libstooop3.0.so
*/

#include <tcl.h>
#include <tclInt.h>

static struct Tcl_HashTable identifiers;
static struct Tcl_HashTable classes;
static int newId;

static unsigned addIdentifier(char *class, int identifier)          /* returns a boolean, true if identifier was added to classes */
{                                            /* associative array, which should always be the case since the identifier is unique */
    struct Tcl_HashEntry *entry;
    char *stored;
    int created;

    entry = Tcl_CreateHashEntry(&classes, class, &created);                                    /* first eventually register class */
    if(created)                                                                                                      /* new class */
        Tcl_SetHashValue(entry, stored = strdup(class));   /* new entry never freed for there should not be a huge number of them */
    else
        stored = (char *)Tcl_GetHashValue(entry);

    Tcl_SetHashValue(Tcl_CreateHashEntry(&identifiers, (ClientData)identifier, &created), stored);     /* now register identifier */
    return created;
}

static int copyObject(Tcl_Interp *interpreter, int from)                                                     /* object identifier */
{
    struct Tcl_HashEntry *entry;
    int i;
    char *class;
    int to;                                           /* use local variable for identifier because new can be invoked recursively */
    Tcl_Obj *object;

    entry = Tcl_FindHashEntry(&identifiers, (ClientData)from);                                               /* find object class */
    if(entry == 0){
        object = Tcl_NewStringObj("invalid object identifier", -1);
        Tcl_ListObjAppendElement(interpreter, object, Tcl_NewIntObj(from));
        Tcl_SetObjResult(interpreter, object);
        return TCL_ERROR;
    }
    class = (char *)Tcl_GetHashValue(entry);
    to = ++newId;                                                                                           /* set new identifier */

    object = Tcl_NewObj();                                                         /* build class copy constructor procedure call */
    Tcl_AppendStringsToObj(object, class, "::_copy", 0);
    Tcl_ListObjAppendElement(interpreter, object, Tcl_NewIntObj(to));                                      /* with new identifier */
    Tcl_ListObjAppendElement(interpreter, object, Tcl_NewIntObj(from));                                 /* and sibling identifier */
    i = Tcl_EvalObj(interpreter, object);
    Tcl_DecrRefCount(object);                                                                                 /* no longer needed */
    if(i != TCL_OK)
        return TCL_ERROR;
    if(!addIdentifier(class, to)){
        object = Tcl_NewStringObj("fatal error: could not register identifier for new object of class ", -1);
        Tcl_AppendStringsToObj(object, class, 0);
        Tcl_SetObjResult(interpreter, object);
        return TCL_ERROR;
    }
    Tcl_SetObjResult(interpreter, Tcl_NewIntObj(to));
    return TCL_OK;
}

static int new(ClientData clientData, Tcl_Interp *interpreter, int numberOfArguments, Tcl_Obj * CONST arguments[])
{
    int identifier;                                   /* use local variable for identifier because new can be invoked recursively */
    int i;
    char *class;
    Tcl_Obj *object;

    if(numberOfArguments < 2){
        Tcl_WrongNumArgs(interpreter, 1, arguments, "class|identifier ?arg arg ...?");
        return TCL_ERROR;
    }

    if(Tcl_GetIntFromObj(interpreter, arguments[1], &identifier) == TCL_OK)
        return copyObject(interpreter, identifier);    /* first argument is an object identifier therefore this is an object copy */

    identifier = ++newId;                                                     /* set new identifier (arrange for 0 to be invalid) */

    class = Tcl_GetStringFromObj(arguments[1], 0);
    object = Tcl_NewObj();
    Tcl_AppendStringsToObj(object, "::", class, "::", class, 0);                                        /* call class constructor */
    Tcl_ListObjAppendElement(interpreter, object, Tcl_NewIntObj(identifier));                              /* with new identifier */
    for(i = 2; i < numberOfArguments; i++)                                                           /* and constructor arguments */
        Tcl_ListObjAppendElement(interpreter, object, arguments[i]);
    i = Tcl_EvalObj(interpreter, object);
    Tcl_DecrRefCount(object);                                                                                 /* no longer needed */
    if(i != TCL_OK)
        return TCL_ERROR;
    if(!addIdentifier(class, identifier)){
        object = Tcl_NewStringObj("fatal error: could not register identifier for new object of class ", -1);
        Tcl_AppendStringsToObj(object, class, 0);
        Tcl_SetObjResult(interpreter, object);
        return TCL_ERROR;
    }
    Tcl_SetObjResult(interpreter, Tcl_NewIntObj(identifier));
    return TCL_OK;
}

static int classOf(ClientData clientData, Tcl_Interp *interpreter, int numberOfArguments, Tcl_Obj * CONST arguments[])
{
    struct Tcl_HashEntry *entry;
    int identifier;
    Tcl_Obj *object;

    if(numberOfArguments != 2){
        Tcl_WrongNumArgs(interpreter, 1, arguments, "identifier");
        return TCL_ERROR;
    }
    if(Tcl_GetIntFromObj(interpreter, arguments[1], &identifier) != TCL_OK){
        object = Tcl_NewStringObj("invalid object identifier", -1);
        Tcl_ListObjAppendElement(interpreter, object, arguments[1]);
        Tcl_SetObjResult(interpreter, object);
        return TCL_ERROR;
    }
    entry = Tcl_FindHashEntry(&identifiers, (ClientData)identifier);
    if(entry == 0){
        object = Tcl_NewStringObj("invalid object identifier", -1);
        Tcl_ListObjAppendElement(interpreter, object, Tcl_NewIntObj(identifier));
        Tcl_SetObjResult(interpreter, object);
        return TCL_ERROR;
    }
    Tcl_SetObjResult(interpreter, Tcl_NewStringObj(Tcl_GetHashValue(entry), -1));            /* core takes care of freeing object */
    return TCL_OK;
}

static int deleteObject(ClientData clientData, Tcl_Interp *interpreter, int numberOfArguments, Tcl_Obj * CONST arguments[])
{                                                                        /* warning: for internal use only, no arguments checking */
    char *class;
    Tcl_Obj *identifier;
    Tcl_Obj *object;
    Tcl_HashEntry *entry;
    Tcl_HashSearch search;
    Var *array;
    char *name;
    char *prefix;
    char *n;
    int length;
    int i;

    class = Tcl_GetStringFromObj(arguments[1], 0);

    object = Tcl_NewObj();
    Tcl_AppendStringsToObj(object, "::", class, "::~", class, 0);                                        /* call class destructor */
    Tcl_ListObjAppendElement(interpreter, object, arguments[2]);                                               /* with identifier */
    i = Tcl_EvalObj(interpreter, object);
    Tcl_DecrRefCount(object);                                                                                 /* no longer needed */
    if(i != TCL_OK)
        return TCL_ERROR;

    entry = Tcl_FindHashEntry(&((Interp *)interpreter)->globalNsPtr->childTable, class);                  /* find class namespace */
    if(entry == 0){
        object = Tcl_NewStringObj("fatal error: could not find namespace of class ", -1);
        Tcl_AppendStringsToObj(object, class, 0);
        Tcl_SetObjResult(interpreter, object);
        return TCL_ERROR;
    }
    entry = Tcl_FindHashEntry(&((Namespace *)Tcl_GetHashValue(entry))->varTable, "");                 /* data array name is empty */
    if(entry == 0)                                                                                  /* if variable does not exist */
        return TCL_OK;                                                                                           /* nothing to do */

    array = (Var *)Tcl_GetHashValue(entry);
    if(array->value.tablePtr == 0)                                                                     /* if array does not exist */
        return TCL_OK;                                                                                           /* nothing to do */

    object = Tcl_NewObj();                                                       /* class data array name must be fully qualified */
    Tcl_AppendStringsToObj(object, "::", class, "::", 0);
    name = Tcl_GetStringFromObj(object, 0);

    identifier = Tcl_DuplicateObj(arguments[2]);
    Tcl_AppendStringsToObj(identifier, ",", 0);                             /* build array index prefix with identifier and comma */
    prefix = Tcl_GetStringFromObj(identifier, &length);

    for(entry = Tcl_FirstHashEntry(array->value.tablePtr, &search); entry != 0; entry = Tcl_NextHashEntry(&search)){
        n = Tcl_GetHashKey(array->value.tablePtr, entry);
        if(strncmp(n, prefix, length) == 0)                                                      /* unset all object data members */
             Tcl_UnsetVar2(interpreter, name, n, TCL_NAMESPACE_ONLY);
    }

    Tcl_DecrRefCount(object);
    Tcl_DecrRefCount(identifier);                                                                                 /* free objects */

    return TCL_OK;                                                                                              /* return nothing */
}

static int delete(ClientData clientData, Tcl_Interp *interpreter, int numberOfArguments, Tcl_Obj * CONST arguments[])
{
    unsigned u;
    int identifier;
    struct Tcl_HashEntry *entry;
    Tcl_Obj *objects[3];
    Tcl_Obj *object;
    int i;

    i = TCL_OK;
    objects[0] = Tcl_NewStringObj("_delete", 7);                                        /* invoke the internal delete Tcl command */
    objects[1] = Tcl_NewObj();
    for(u = 1; u < numberOfArguments; u++){
        if(Tcl_GetIntFromObj(interpreter, arguments[u], &identifier) != TCL_OK){
            object = Tcl_NewStringObj("invalid object identifier", -1);
            Tcl_ListObjAppendElement(interpreter, object, arguments[u]);
            Tcl_SetObjResult(interpreter, object);
            i = TCL_ERROR;
            break;
        }
        entry = Tcl_FindHashEntry(&identifiers, (ClientData)identifier);
        if(entry == 0){
            object = Tcl_NewStringObj("invalid object identifier", -1);
            Tcl_ListObjAppendElement(interpreter, object, Tcl_NewIntObj(identifier));
            Tcl_SetObjResult(interpreter, object);
            i = TCL_ERROR;
            break;
        }
        Tcl_SetStringObj(objects[1], Tcl_GetHashValue(entry), -1);
        /* remove identifier from class associative array before actually deleting the object, which by the destructor invocation */
        /* may result in recursive calls of this fonction (may happen when a binding on a destroy event deletes the object) */
        Tcl_DeleteHashEntry(entry);
        objects[2] = arguments[u];
        if(deleteObject(clientData, interpreter, 3, objects) != TCL_OK){
            i = TCL_ERROR;
            break;
        }
    }
    Tcl_DecrRefCount(objects[0]);                                                                 /* do not free actual arguments */
    Tcl_DecrRefCount(objects[1]);
    return i;
}

static int copy(ClientData clientData, Tcl_Interp *interpreter, int numberOfArguments, Tcl_Obj * CONST arguments[])
{                                                                        /* warning: for internal use only, no arguments checking */
    char *class;
    Tcl_Obj *object;
    Tcl_Obj *from;
    Tcl_Obj *to;
    Tcl_Obj *fromKey;
    Tcl_Obj *toKey;
    Tcl_HashEntry *entry;
    Tcl_HashSearch search;
    Var *array;
    char *n;
    char *fromPrefix;
    char *toPrefix;
    int fromLength;
    int toLength;


    class = Tcl_GetStringFromObj(arguments[1], 0);

    entry = Tcl_FindHashEntry(&((Interp *)interpreter)->globalNsPtr->childTable, class);                  /* find class namespace */
    if(entry == 0){
        object = Tcl_NewStringObj("fatal error: could not find namespace of class ", -1);
        Tcl_AppendStringsToObj(object, class, 0);
        Tcl_SetObjResult(interpreter, object);
        return TCL_ERROR;
    }
    entry = Tcl_FindHashEntry(&((Namespace *)Tcl_GetHashValue(entry))->varTable, "");                 /* data array name is empty */
    if(entry == 0)                                                                                  /* if variable does not exist */
        return TCL_OK;                                                                                           /* nothing to do */

    array = (Var *)Tcl_GetHashValue(entry);
    if(array->value.tablePtr == 0)                                                                     /* if array does not exist */
        return TCL_OK;                                                                                           /* nothing to do */

    from = Tcl_DuplicateObj(arguments[2]);
    to = Tcl_DuplicateObj(arguments[3]);

    object = Tcl_NewObj();                                                       /* class data array name must be fully qualified */
    Tcl_AppendStringsToObj(object, "::", class, "::", 0);

    Tcl_AppendStringsToObj(from, ",", 0);                            /* build array index prefix with source identifier and comma */
    fromPrefix = Tcl_GetStringFromObj(from, &fromLength);
    Tcl_AppendStringsToObj(to, ",", 0);                                                         /* build target identifier prefix */
    toPrefix = Tcl_GetStringFromObj(to, &toLength);

    fromKey = Tcl_NewObj();
    toKey = Tcl_NewObj();

    for(entry = Tcl_FirstHashEntry(array->value.tablePtr, &search); entry != 0; entry = Tcl_NextHashEntry(&search)){
        n = Tcl_GetHashKey(array->value.tablePtr, entry);
        if(strncmp(n, fromPrefix, fromLength) != 0)
            continue;
        Tcl_SetStringObj(toKey, toPrefix, toLength);
        Tcl_AppendStringsToObj(toKey, n + fromLength, 0);
        Tcl_SetStringObj(fromKey, n, -1);
        Tcl_ObjSetVar2(
            interpreter, object, toKey, Tcl_ObjGetVar2(interpreter, object, fromKey, TCL_NAMESPACE_ONLY), TCL_NAMESPACE_ONLY
        );
    }

    Tcl_DecrRefCount(object);                                                                                     /* free objects */
    Tcl_DecrRefCount(from);
    Tcl_DecrRefCount(to);
    Tcl_DecrRefCount(fromKey);
    Tcl_DecrRefCount(toKey);

    return TCL_OK;                                                                                              /* return nothing */
}

int Stooop_Init(interpreter)
    Tcl_Interp *interpreter;
{
    Tcl_Namespace *namespace;
    int i;

    Tcl_InitHashTable(&identifiers, TCL_ONE_WORD_KEYS);
    Tcl_InitHashTable(&classes, TCL_STRING_KEYS);
    newId = 0;

    
    if((namespace = Tcl_FindNamespace(interpreter, "::stooop", 0, 0)) == 0)         /* if the stooop namespace does not yet exist */
        namespace = Tcl_CreateNamespace(interpreter, "::stooop", 0, 0);    /* create it as a direct child of the global namespace */
    if(namespace == 0){
        Tcl_SetObjResult(interpreter, Tcl_NewStringObj("fatal error: could not create stooop namespace", -1));
        return TCL_ERROR;
    }

    Tcl_CreateObjCommand(interpreter, "::stooop::new", new, 0, 0);
    Tcl_CreateObjCommand(interpreter, "::stooop::classof", classOf, 0, 0);
    Tcl_CreateObjCommand(interpreter, "::stooop::delete", delete, 0, 0);
    Tcl_CreateObjCommand(interpreter, "::stooop::_delete", deleteObject, 0, 0);
    Tcl_CreateObjCommand(interpreter, "::stooop::copy", copy, 0, 0);

    i = Tcl_Export(interpreter, namespace, "new", 1);                                                   /* export the new command */
    if(i != TCL_OK) return i;
    i = Tcl_Export(interpreter, namespace, "classof", 0);                                           /* export the classof command */
    if(i != TCL_OK) return i;
    i = Tcl_Export(interpreter, namespace, "delete", 0);                                             /* export the delete command */
    if(i != TCL_OK) return i;

    return Tcl_PkgProvide(interpreter, "stooop", "3.0");
}

int Stooop_SafeInit(interpreter)
    Tcl_Interp *interpreter;
{
    return Stooop_Init(interpreter);                                                                  /* stooop is safe by nature */
}
