
/* 
 * svipcLink.c --
 *
 *	This file implements linked variables (a C variable that is
 *	tied to a Tcl variable).  The idea of linked variables was
 *	first suggested by Andreas Stolcke and this implementation is
 *	based heavily on a prototype implementation provided by
 *	him.
 *
 * Copyright (c) 1993 The Regents of the University of California.
 * Copyright (c) 1994 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * SCCS: @(#) tclLink.c 1.12 96/02/15 11:50:26
 */

#ifndef lint
static char rcsid[] = "$Header: /users/jaws/joek/SVipc/RCS/svipcLink.c,v 1.3 1996/05/10 18:30:01 joek Exp $ SPRITE (Berkeley)";
#endif /* not lint */

#include "tclInt.h"

#include "svipc.h"

/*
 * For each linked variable there is a data structure of the following
 * type, which describes the link and is the clientData for the trace
 * set on the Tcl variable.
 */

typedef struct Link {
    Tcl_Interp *interp;		/* Interpreter containing Tcl variable. */
    char *varName;		/* Name of variable (must be global).  This
				 * is needed during trace callbacks, since
				 * the actual variable may be aliased at
				 * that time via upvar. */
    char *addr;			/* Location of C variable. */
    int type;			/* Type of link (TCL_LINK_INT, etc.). */
    int elements;		/* Number of elements in group (array). */
    int writable;		/* Zero means Tcl variable is read-only. */
    union {
        char *s;
	int i;
	double d;
    } lastValue;		/* Last known value of C variable;  used to
				 * avoid string conversions. */
} Link;

/*
 * Forward references to procedures defined later in this file:
 */

static char *	LinkTraceProc _ANSI_ARGS_((ClientData clientData,
					   Tcl_Interp *interp, char *name1,
					   char *name2, int flags));
static char *	StringValue _ANSI_ARGS_((Link *linkPtr, char *buffer));

/*
 *----------------------------------------------------------------------
 *
 * Svipc_LinkVar --
 *
 *	Link a C variable to a Tcl variable so that changes to either
 *	one causes the other to change.
 *
 * Results:
 *	The return value is TCL_OK if everything went well or TCL_ERROR
 *	if an error occurred (interp->result is also set after errors).
 *
 * Side effects:
 *	The value at *addr is linked to the Tcl variable "varName",
 *	using "type" to convert between string values for Tcl and
 *	binary values for *addr.
 *
 *----------------------------------------------------------------------
 */

int
Svipc_LinkVar(interp, varName, addr, type, elements)
    Tcl_Interp *interp;		/* Interpreter in which varName exists. */
    char *varName;		/* Name of a global variable in interp. */
    char *addr;			/* Address of a C variable to be linked
				 * to varName. */
    int type;			/* Type of C variable: TCL_LINK_INT, etc. */
    int elements;		/* Number of elements in array. */
{
    int code;
    Link *linkPtr;
    char *buffer = ckalloc(elements * TCL_DOUBLE_SPACE);

    linkPtr = (Link *) ckalloc(sizeof(Link));
    linkPtr->interp = interp;
    linkPtr->varName = (char *) ckalloc((unsigned) (strlen(varName) + 1));
    strcpy(linkPtr->varName, varName);
    linkPtr->addr = addr;
    linkPtr->type = type & ~TCL_LINK_READ_ONLY;
    if (type == SVIPC_LINK_STATICSTR) {
	linkPtr->lastValue.s = ckalloc(elements + 1);
	linkPtr->lastValue.s[elements] = 0;
    }
    linkPtr->elements = linkPtr->type == TCL_LINK_STRING ? 1 : elements;
    linkPtr->writable = (type & TCL_LINK_READ_ONLY) == 0;
    if (Tcl_SetVar(interp, varName, StringValue(linkPtr, buffer),
	    TCL_GLOBAL_ONLY|TCL_LEAVE_ERR_MSG) == NULL) {
	ckfree(linkPtr->varName);
	ckfree((char *) linkPtr);
	ckfree(buffer);
	return TCL_ERROR;
    }
    ckfree(buffer);
    code = Tcl_TraceVar(interp, varName, TCL_GLOBAL_ONLY|TCL_TRACE_READS
	    |TCL_TRACE_WRITES|TCL_TRACE_UNSETS, LinkTraceProc,
	    (ClientData) linkPtr);
    if (code != TCL_OK) {
	ckfree(linkPtr->varName);
	ckfree((char *) linkPtr);
    }
    return code;
}

/*
 *----------------------------------------------------------------------
 *
 * Svipc_UnlinkVar --
 *
 *	Destroy the link between a Tcl variable and a C variable.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	If "varName" was previously linked to a C variable, the link
 *	is broken to make the variable independent.  If there was no
 *	previous link for "varName" then nothing happens.
 *
 *----------------------------------------------------------------------
 */

void
Svipc_UnlinkVar(interp, varName)
    Tcl_Interp *interp;		/* Interpreter containing variable to unlink. */
    char *varName;		/* Global variable in interp to unlink. */
{
    Link *linkPtr;

    linkPtr = (Link *) Tcl_VarTraceInfo(interp, varName, TCL_GLOBAL_ONLY,
	    LinkTraceProc, (ClientData) NULL);
    if (linkPtr == NULL) {
	return;
    }
    Tcl_UntraceVar(interp, varName,
	    TCL_GLOBAL_ONLY|TCL_TRACE_READS|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
	    LinkTraceProc, (ClientData) linkPtr);
    if (linkPtr->type == SVIPC_LINK_STATICSTR)
	ckfree(linkPtr->lastValue.s);
    ckfree(linkPtr->varName);
    ckfree((char *) linkPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_UpdateLinkedVar --
 *
 *	This procedure is invoked after a linked variable has been
 *	changed by C code.  It updates the Tcl variable so that
 *	traces on the variable will trigger.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The Tcl variable "varName" is updated from its C value,
 *	causing traces on the variable to trigger.
 *
 *----------------------------------------------------------------------
 */

void
Svipc_UpdateLinkedVar(interp, varName)
    Tcl_Interp *interp;		/* Interpreter containing variable. */
    char *varName;		/* Name of global variable that is linked. */
{
    Link *linkPtr;
    char *buffer;

    linkPtr = (Link *) Tcl_VarTraceInfo(interp, varName, TCL_GLOBAL_ONLY,
	    LinkTraceProc, (ClientData) NULL);
    if (linkPtr == NULL) {
	return;
    }
    buffer = (char *) ckalloc(TCL_DOUBLE_SPACE * linkPtr->elements);
    Tcl_SetVar(interp, linkPtr->varName, StringValue(linkPtr, buffer),
	    TCL_GLOBAL_ONLY);
    ckfree(buffer);
}

/*
 *----------------------------------------------------------------------
 *
 * LinkTraceProc --
 *
 *	This procedure is invoked when a linked Tcl variable is read,
 *	written, or unset from Tcl.  It's responsible for keeping the
 *	C variable in sync with the Tcl variable.
 *
 * Results:
 *	If all goes well, NULL is returned; otherwise an error message
 *	is returned.
 *
 * Side effects:
 *	The C variable may be updated to make it consistent with the
 *	Tcl variable, or the Tcl variable may be overwritten to reject
 *	a modification.
 *
 *----------------------------------------------------------------------
 */

static char *
LinkTraceProc(clientData, interp, name1, name2, flags)
    ClientData clientData;	/* Contains information about the link. */
    Tcl_Interp *interp;		/* Interpreter containing Tcl variable. */
    char *name1;		/* First part of variable name. */
    char *name2;		/* Second part of variable name. */
    int flags;			/* Miscellaneous additional information. */
{
    Link *linkPtr = (Link *) clientData;
    int changed;
    char *buffer = (char *) ckalloc(TCL_DOUBLE_SPACE * linkPtr->elements);
    char *value, **pp;
    Tcl_DString savedResult;
    int argc;
    char **argv;
    char **pvalue;
    int i;

    /*
     * If the variable is being unset, then just re-create it (with a
     * trace) unless the whole interpreter is going away.
     */

    if (flags & TCL_TRACE_UNSETS) {
	if (flags & TCL_INTERP_DESTROYED) {
	    if (linkPtr->type == SVIPC_LINK_STATICSTR)
		ckfree(linkPtr->lastValue.s);
	    ckfree(linkPtr->varName);
	    ckfree((char *) linkPtr);
	} else if (flags & TCL_TRACE_DESTROYED) {
	    Tcl_SetVar(interp, linkPtr->varName, StringValue(linkPtr, buffer),
		    TCL_GLOBAL_ONLY);
	    Tcl_TraceVar(interp, linkPtr->varName, TCL_GLOBAL_ONLY
		    |TCL_TRACE_READS|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		    LinkTraceProc, (ClientData) linkPtr);
	}
	ckfree(buffer);
	return NULL;
    }

    /*
     * For read accesses, update the Tcl variable if the C variable
     * has changed since the last time we updated the Tcl variable.
     */

    if (flags & TCL_TRACE_READS) {
	if (linkPtr->elements > 1) {
	    changed = 1;
	}
	else {
	    switch (linkPtr->type) {
	        case TCL_LINK_INT:
	        case TCL_LINK_BOOLEAN:
		    changed = *(int *)(linkPtr->addr) != linkPtr->lastValue.i;
		    break;
		case TCL_LINK_DOUBLE:
		    changed = *(double *)(linkPtr->addr) != linkPtr->lastValue.d;
		    break;
		case TCL_LINK_STRING:
		    changed = 1;
		    break;
		case SVIPC_LINK_CHAR:
		    changed = *linkPtr->addr != linkPtr->lastValue.i;
		    break;
		case SVIPC_LINK_UCHAR:
		    changed = *(unsigned char *)(linkPtr->addr) != linkPtr->lastValue.i;
		    break;
		case SVIPC_LINK_SHORT:
		    changed = *(short *)(linkPtr->addr) != linkPtr->lastValue.i;
		    break;
		case SVIPC_LINK_USHORT:
		    changed = *(unsigned short *)(linkPtr->addr) != linkPtr->lastValue.i;
		    break;
		case SVIPC_LINK_STATICSTR:
		    changed = 1;
		    break;
		default:
		    ckfree(buffer);
		    return "internal error: bad linked variable type";
	    }
	}
	if (changed) {
	    Tcl_SetVar(interp, linkPtr->varName, StringValue(linkPtr, buffer),
		    TCL_GLOBAL_ONLY);
	}
	ckfree(buffer);
	return NULL;
    }

    /*
     * For writes, first make sure that the variable is writable.  Then
     * convert the Tcl value to C if possible.  If the variable isn't
     * writable or can't be converted, then restore the varaible's old
     * value and return an error.  Another tricky thing: we have to save
     * and restore the interpreter's result, since the variable access
     * could occur when the result has been partially set.
     */

    if (!linkPtr->writable) {
	Tcl_SetVar(interp, linkPtr->varName, StringValue(linkPtr, buffer),
		    TCL_GLOBAL_ONLY);
	ckfree(buffer);
	return "linked variable is read-only";
    }
    value = Tcl_GetVar(interp, linkPtr->varName, TCL_GLOBAL_ONLY);
    if (value == NULL) {
	/*
	 * This shouldn't ever happen.
	 */
	ckfree(buffer);
	return "internal error: linked variable couldn't be read";
    }
    if (linkPtr->elements == 1 ||
	linkPtr->type == TCL_LINK_STRING ||
	linkPtr->type == SVIPC_LINK_STATICSTR) {
	argc = 1;
	argv = (char **)ckalloc(sizeof(char *) * 2);
	argv[0] = value;
	argv[1] = 0;
    }
    else {
	if (Tcl_SplitList (interp, value, &argc, &argv) != TCL_OK) {
	    ckfree(buffer);
	    return "bad list of elements";
	}
	if (argc != linkPtr->elements) {
	    ckfree((char *)argv);
	    ckfree(buffer);
	    return "wrong number of elements in list";
	}
    }
    pvalue = argv;
    Tcl_DStringInit(&savedResult);
    Tcl_DStringAppend(&savedResult, interp->result, -1);
    Tcl_ResetResult(interp);
    for (i = 0; i < argc; i++, pvalue++) {
	switch (linkPtr->type) {
	    case TCL_LINK_INT:
	        if (Tcl_GetInt(interp, *pvalue, &linkPtr->lastValue.i)
		    != TCL_OK) {
		    Tcl_DStringResult(interp, &savedResult);
		    Tcl_SetVar(interp, linkPtr->varName,
				StringValue(linkPtr, buffer), TCL_GLOBAL_ONLY);
		    ckfree((char *)argv);
		    ckfree(buffer);
		    return "variable must have integer value";
		}
		((int *)linkPtr->addr)[i] = linkPtr->lastValue.i;
		break;
	    case TCL_LINK_DOUBLE:
		if (Tcl_GetDouble(interp, *pvalue, &linkPtr->lastValue.d)
		    != TCL_OK) {
		    Tcl_DStringResult(interp, &savedResult);
		    Tcl_SetVar(interp, linkPtr->varName,
				StringValue(linkPtr, buffer), TCL_GLOBAL_ONLY);
		    ckfree((char *)argv);
		    ckfree(buffer);
		    return "variable must have real value";
		}
		((double *)linkPtr->addr)[i] = linkPtr->lastValue.d;
		break;
	    case TCL_LINK_BOOLEAN:
		if (Tcl_GetBoolean(interp, *pvalue, &linkPtr->lastValue.i)
		    != TCL_OK) {
		    Tcl_DStringResult(interp, &savedResult);
		    Tcl_SetVar(interp, linkPtr->varName,
				StringValue(linkPtr, buffer), TCL_GLOBAL_ONLY);
		    ckfree((char *)argv);
		    ckfree(buffer);
		    return "variable must have boolean value";
		}
		((int *)linkPtr->addr)[i] = linkPtr->lastValue.i;
		break;
	    case TCL_LINK_STRING:
		pp = (char **)(linkPtr->addr);
		if (*pp != NULL) {
		    ckfree(*pp);
		}
		*pp = ckalloc((unsigned) (strlen(*pvalue) + 1));
		strcpy(*pp, *pvalue);
		break;
	    case SVIPC_LINK_CHAR:
		if (Tcl_GetInt(interp, *pvalue, &linkPtr->lastValue.i)
		    != TCL_OK) {
		    Tcl_DStringResult(interp, &savedResult);
		    Tcl_SetVar(interp, linkPtr->varName,
				StringValue(linkPtr, buffer), TCL_GLOBAL_ONLY);
		    ckfree((char *)argv);
		    ckfree(buffer);
		    return "variable must have integer value";
		}
		linkPtr->addr[i] = linkPtr->lastValue.i;
		break;
	    case SVIPC_LINK_UCHAR:
		if (Tcl_GetInt(interp, *pvalue, &linkPtr->lastValue.i)
		    != TCL_OK) {
		    Tcl_DStringResult(interp, &savedResult);
		    Tcl_SetVar(interp, linkPtr->varName,
				StringValue(linkPtr, buffer), TCL_GLOBAL_ONLY);
		    ckfree((char *)argv);
		    ckfree(buffer);
		    return "variable must have integer value";
		}
		((unsigned char *)linkPtr->addr)[i] = linkPtr->lastValue.i;
		break;
	    case SVIPC_LINK_SHORT:
		if (Tcl_GetInt(interp, *pvalue, &linkPtr->lastValue.i)
		    != TCL_OK) {
		    Tcl_DStringResult(interp, &savedResult);
		    Tcl_SetVar(interp, linkPtr->varName,
				StringValue(linkPtr, buffer), TCL_GLOBAL_ONLY);
		    ckfree((char *)argv);
		    ckfree(buffer);
		    return "variable must have integer value";
		}
		((short *)linkPtr->addr)[i] = linkPtr->lastValue.i;
		break;
	    case SVIPC_LINK_USHORT:
		if (Tcl_GetInt(interp, *pvalue, &linkPtr->lastValue.i)
		    != TCL_OK) {
		    Tcl_DStringResult(interp, &savedResult);
		    Tcl_SetVar(interp, linkPtr->varName,
				StringValue(linkPtr, buffer), TCL_GLOBAL_ONLY);
		    ckfree((char *)argv);
		    ckfree(buffer);
		    return "variable must have integer value";
		}
		((unsigned short *)linkPtr->addr)[i] = linkPtr->lastValue.i;
		break;
	    case SVIPC_LINK_STATICSTR:
		strncpy (linkPtr->lastValue.s, *pvalue, linkPtr->elements);
		memcpy (*(char **)(linkPtr->addr), linkPtr->lastValue.s,
			linkPtr->elements);
		break;
	    default:
		ckfree((char *)argv);
		ckfree(buffer);
		return "internal error: bad linked variable type";
	}
    }
    ckfree((char *)argv);
    ckfree(buffer);
    Tcl_DStringResult(interp, &savedResult);
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * StringValue --
 *
 *	Collects the values of all of the elements of the array into
 *	a list and returns the value.  N.B. Strings (TCL_LINK_STRING
 *	and SVIPC_LINK_STATICSTR) have simple values.
 *
 * Results:
 *	The return value is a pointer to a string that represents
 *	the value of the C variable given by linkPtr.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static char *
StringValue(linkPtr, buffer)
    Link *linkPtr;		/* Structure describing linked variable. */
    char *buffer;		/* Small buffer to use for converting
				 * values.  Must have TCL_DOUBLE_SPACE
				 * bytes or more. */
{
    char *p;
    int i;
    Tcl_DString dstr;

    if (linkPtr->type != TCL_LINK_STRING &&
	linkPtr->type != SVIPC_LINK_STATICSTR) {
	Tcl_DStringInit(&dstr);
    }
    for (i = 0; i < linkPtr->elements; i++) {
	switch (linkPtr->type) {
	    case TCL_LINK_INT:
	        linkPtr->lastValue.i = ((int *)linkPtr->addr)[i];
		sprintf(buffer, "%d", linkPtr->lastValue.i);
		p = buffer;
		break;
	    case TCL_LINK_DOUBLE:
	        linkPtr->lastValue.d = ((double *)linkPtr->addr)[i];
		Tcl_PrintDouble(linkPtr->interp, linkPtr->lastValue.d, buffer);
		p = buffer;
		break;
	    case TCL_LINK_BOOLEAN:
	        linkPtr->lastValue.i = ((int *)linkPtr->addr)[i];
		if (linkPtr->lastValue.i != 0) {
		    p = "1";
		}
		else {
		    p = "0";
		}
		break;
	    case TCL_LINK_STRING:
		p = ((char **)linkPtr->addr)[i];
		if (p == NULL) {
		    p = "NULL";
		}
		return p;
	    case SVIPC_LINK_CHAR:
		linkPtr->lastValue.i = linkPtr->addr[i];
		sprintf(buffer, "%d", linkPtr->lastValue.i);
		p = buffer;
		break;
	    case SVIPC_LINK_UCHAR:
		linkPtr->lastValue.i = ((unsigned char *)linkPtr->addr)[i];
		sprintf(buffer, "%d", linkPtr->lastValue.i);
		p = buffer;
		break;
	    case SVIPC_LINK_SHORT:
		linkPtr->lastValue.i = ((short *)linkPtr->addr)[i];
		sprintf(buffer, "%d", linkPtr->lastValue.i);
		p = buffer;
		break;
	    case SVIPC_LINK_USHORT:
		linkPtr->lastValue.i = ((unsigned short *)linkPtr->addr)[i];
		sprintf(buffer, "%d", linkPtr->lastValue.i);
		p = buffer;
		break;
	    case SVIPC_LINK_STATICSTR:
		return (char *) memcpy(linkPtr->lastValue.s,
				       *((char **)linkPtr->addr),
				       linkPtr->elements);
	    default:
		Tcl_DStringFree(&dstr);
		return "??";
	}
	(void) Tcl_DStringAppendElement(&dstr, buffer);
    }
    (void) strcpy(buffer, Tcl_DStringValue(&dstr));
    Tcl_DStringFree(&dstr);
    return buffer;
}

/*
 *----------------------------------------------------------------------
 *
 * Svipc_LinkCmd --
 *
 *	Command line interface to Svipc_LinkVar.
 *		link varName addr type ?elements?
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	Links a Tcl variable to a memory location.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
Svipc_LinkCmd(dummy, interp, argc, argv)
    ClientData dummy;			/* Not used. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    int addr;
    int type = 0;
    int elements = 1;
    int varName = 1;

    if (argc < 4 || 6 < argc) {
	Tcl_AppendResult(interp, "usage: ", argv[0],
			 " ?-rdonly? varName addr type ?elements?",
			 (char *) NULL);
	return TCL_ERROR;
    }
    if (strncmp ("-rdonly", argv[1], strlen(argv[1])) == 0) {
	type = TCL_LINK_READ_ONLY;
	varName++;
    }
    if (Tcl_GetInt (interp, argv[varName+1], &addr) != TCL_OK) {
	return TCL_ERROR;
    }
    if (strcmp ("int", argv[varName+2]) == 0) {
	type |= TCL_LINK_INT;
    } else if (strcmp ("boolean", argv[varName+2]) == 0) {
	type |= TCL_LINK_BOOLEAN;
    } else if (strcmp ("double", argv[varName+2]) == 0) {
	type |= TCL_LINK_DOUBLE;
    } else if (strcmp ("string", argv[varName+2]) == 0) {
	type |= TCL_LINK_STRING;
    } else if (strcmp ("char", argv[varName+2]) == 0) {
	type |= SVIPC_LINK_CHAR;
    } else if (strcmp ("uchar", argv[varName+2]) == 0) {
	type |= SVIPC_LINK_UCHAR;
    } else if (strcmp ("short", argv[varName+2]) == 0) {
	type |= SVIPC_LINK_SHORT;
    } else if (strcmp ("ushort", argv[varName+2]) == 0) {
	type |= SVIPC_LINK_USHORT;
    } else if (strcmp ("staticstr", argv[varName+2]) == 0) {
	type |= SVIPC_LINK_STATICSTR;
    } else {
	Tcl_AppendResult(interp, "invalid link type \"", argv[varName+2],
			 "\"", (char *)0);
	return TCL_ERROR;
    }
    if (argc == varName+3 && Tcl_GetInt (interp, argv[varName+3],
					 &elements) != TCL_OK) {
	return TCL_ERROR;
    }

#if SVIPC_CHECK_ALIGN
    switch (type) {
	case TCL_LINK_INT:
	case TCL_LINK_BOOLEAN:
	    if (((long)addr) & SVIPC_LONG_ALIGN) {
		Tcl_AppendResult (interp, "unaligned long address", (char *)0);
		return TCL_ERROR;
	    }
	    break;
	case TCL_LINK_DOUBLE:
	    if (((long)addr) & SVIPC_DOUBLE_ALIGN) {
		Tcl_AppendResult (interp, "unaligned double address",
				  (char *)0);
		return TCL_ERROR;
	    }
	    break;
	case SVIPC_LINK_SHORT:
	case SVIPC_LINK_USHORT:
	    if (((long)addr) & SVIPC_SHORT_ALIGN) {
		Tcl_AppendResult (interp, "unaligned short address",
				  (char *)0);
		return TCL_ERROR;
	    }
	    break;
    }
#endif /* SVIPC_CHECK_ALIGN */

    if (Svipc_LinkVar (interp, argv[varName], (char *)addr, type,
		       elements) != TCL_OK) {
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Svipc_UnlinkCmd --
 *
 *	Command line interface to Svipc_UnlinkVar.
 *		unlink varName
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	Unlinks a Tcl variable from a memory location.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
Svipc_UnlinkCmd(dummy, interp, argc, argv)
    ClientData dummy;			/* Not used. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    if (argc != 1) {
	Tcl_AppendResult(interp, "usage: ", argv[0], " varName",
			 (char *) NULL);
	return TCL_ERROR;
    }

    Svipc_UnlinkVar (interp, argv[1]);
    return TCL_OK;
}

/*
 * Local Variables:
 * c-auto-newline: nil
 * c-indent-level: 4
 * c-continued-statement-offset: 4
 * c-brace-offset: -4
 * c-argdecl-indent: 4
 * c-label-offset: -4
 * End:
 */
