/************************************************************************
 *									*
 *	Copyright 1992 by Motorola Mobile Data Division,		*
 *			  Bothell, WA					*
 *									*
 *	Motorola hereby grants permission to use, copy, modify and	*
 *	distribute  this software and its documentation for any		*
 *	purpose and without fee, provided that you retain this		*
 *	copyright notice in all copies.  Motorola makes no		*
 *	representations about the suitability of this software for any	*
 *	purpose.  Motorola provides this software ``as is'' without	*
 *	express or implied warranty.					*
 *									*
 ************************************************************************/
#ifndef lint
static char RCSid[] = "$Header: /users/jaws/joek/SVipc/RCS/svipcShm.c,v 1.7 1996/05/10 18:31:23 joek Exp $";
#endif /* lint */

/*
 * File: SVipc/svipcShm.c
 * Facility: Tcl C Routines
 * Author: Joe Kelsey
 * Description:
 *	Shared memory access for Tcl.
 *
 * Global Functions:
 *	Svipc_ShmatCmd
 *	Svipc_ShmdtCmd
 *	Svipc_ShmgetCmd
 *	Svipc_ShmgetcharCmd
 *	Svipc_ShmgetlongCmd
 *	Svipc_ShmgetshortCmd
 *	Svipc_ShmgetstrCmd
 *	Svipc_ShmgetstrnCmd
 *	Svipc_ShmreadCmd
 *	Svipc_ShmrmidCmd
 *	Svipc_ShmsetCmd
 *	Svipc_ShmsetcharCmd
 *	Svipc_ShmsetlongCmd
 *	Svipc_ShmsetshortCmd
 *	Svipc_ShmsetstrCmd
 *	Svipc_ShmsetstrnCmd
 *	Svipc_ShmstatCmd
 *	Svipc_ShmwriteCmd
 *
 * $Log: svipcShm.c,v $
 * Revision 1.7  1996/05/10 18:31:23  joek
 * Remove prototype for shmat, as HP-UX seems especially recalcitrant
 * about accepting it.
 *
 * Revision 1.6  1993/10/14 23:04:27  kelsey
 * Make our prototype for shmat agree with the system header file
 * on SVR4.
 *
 * Revision 1.5  1993/10/01  23:07:38  kelsey
 * Fix all of the AppendElement calls to add trailing 0 parameter.
 *
 * Revision 1.4  1993/08/18  21:34:41  kelsey
 * Use Svipc_ prefix instead of Tcl_.
 * Update for Tcl 7.0.
 *
 * Revision 1.3  1993/04/09  01:32:35  kelsey
 * Make keyword arguments abbreviatable.
 * Make stat return structured result even if variable specified.
 *
 * Revision 1.2  1993/04/02  22:19:14  kelsey
 * Change copyright message.
 * Add read and write commands.
 *
 * Revision 1.1  1992/06/02  14:09:34  kelsey
 * Initial revision
 *
 */

#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include <tclInt.h>

#include "svipc.h"


/*
 * Function: Svipc_ShmatCmd
 * Description:
 *	Attach to the shared memory segment.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		Number of command arguments (3).
 * char **	argv		Vector of command arguments.
 *				[ shmat shmid shmflg ]
 *
 * Return type: int
 * Side Effects:
 *	Sets interp->result.
 */
/* ARGSUSED */
int
Svipc_ShmatCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char ** argv;

/* Detail Design:

 */

{
  int shmid;
  char * shmaddr;
  int shmflg = 0;

  if (argc < 2 || 3 < argc)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0], " shmid ?rdonly?",
			(char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], &shmid) != TCL_OK)
    {
      return TCL_ERROR;
    }

  if (argc == 3)
    {
      if (strncmp ("rdonly", argv[2], strlen (argv[2])))
	{
	  (void) strcpy (interp->result, "shmflg must be \"rdonly\".");
	  return TCL_ERROR;
	}
      else
	{
	  shmflg = SHM_RDONLY;
	}
    }

  if ((int)(shmaddr = (char *) shmat (shmid, (char *)0, shmflg)) == -1)
    {
      Tcl_AppendResult (interp, "cannot attach shared memory segment ",
			argv[1], ": ", Tcl_PosixError (interp), (char *)0);
      return TCL_ERROR;
    }

  (void) sprintf (interp->result, "0x%08x", shmaddr);
  return TCL_OK;
}

/*
 * Function: Svipc_ShmdtCmd
 * Description:
 *	Detach from the shared memory segment.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		Number of command arguments (2).
 * char **	argv		Vector of command arguments.
 *				[ shmdt shmaddr ]
 *
 * Return type: int
 * Side Effects:
 *	Sets interp->result.
 *	Removes the shared memory segment mapping.
 */
/* ARGSUSED */
int
Svipc_ShmdtCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char ** argv;

/* Detail Design:

 */

{
  char *shmaddr;

  if (argc != 2)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0], " shmaddr", (char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], (int *)&shmaddr) != TCL_OK)
    {
      return TCL_ERROR;
    }

  if (shmdt (shmaddr) == -1)
    {
      Tcl_AppendResult (interp, "cannot detach shared memory address ",
			argv[1], ": ", Tcl_PosixError (interp), (char *)0);
      return TCL_ERROR;
    }
  return TCL_OK;
}

/*
 * Function: Svipc_ShmgetCmd
 * Description:
 *	Acquire a shared memory segment id.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		Number of command arguments (3-6).
 * char **	argv		Vector of command arguments.
 *				[ shmget key segsize mode create excl ]
 *
 * Return type: int
 * Side Effects:
 *	Sets interp->result.
 *	Creates the shared memory segment.
 */
/* ARGSUSED */
int
Svipc_ShmgetCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char ** argv;

/* Detail Design:

 */

{
  key_t key;
  int size;
  int shmflg;
  int shmid;

  if (argc < 3 || 6 < argc)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0],
			" key size ?mode? ?create? ?excl?", (char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], (int *)&key) != TCL_OK)
    {
      if (strncmp (argv[1], "private", strlen (argv[1])))
	{
	  return TCL_ERROR;
	}

      key = IPC_PRIVATE;
    }

  if (Tcl_GetInt (interp, argv[2], &size) != TCL_OK)
    {
      return TCL_ERROR;
    }
  if (argc > 3)
    {
      char ** pargv = argv + 3;

      if (Tcl_GetInt (interp, *pargv, &shmflg) != TCL_OK)
	{
	  shmflg = 0666;
	}
      else
	{
	  pargv++;
	}

      while (*pargv)
	{
	  int plen = strlen (*pargv);

	  if (strncmp (*pargv, "create", plen) == 0)
	    {
	      shmflg |= IPC_CREAT;
	    }
	  else if (strncmp (*pargv, "excl", plen) == 0)
	    {
	      shmflg |= IPC_EXCL;
	    }
	  else
	    {
	      (void) strcpy (interp->result,
			     "flag argument must be \"create\" or \"excl\".");
	      return TCL_ERROR;
	    }
	  pargv++;
	}
    }
  else
    {
      shmflg = 0666;
    }

  if ((shmid = shmget (key, size, shmflg)) == -1)
    {
      int i;
      char flgstr[20];

      (void) sprintf (flgstr, "0%o", shmflg);
      Tcl_AppendResult (interp, "cannot get shared memory segment id for \"",
			argv[1], " ", argv[2], " ", flgstr, (char *)0);
      for (i = 3; i < argc; i++)
	{
	  Tcl_AppendResult (interp, i == 3 ? " (" : " ", argv[i], (char *)0);
	}
      Tcl_AppendResult (interp, argc == 3 ? "\": " : ")\": ",
			Tcl_PosixError (interp), (char *)0);
      return TCL_ERROR;
    }

  (void) sprintf (interp->result, "%d", shmid);
  return TCL_OK;
}

/*
 * Function: Svipc_ShmgetcharCmd
 * Description:
 *	Return the value of the char at the given offset from the shared 
 *	memory base.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		number of command arguments.
 * char **	argv		vector of command arguments.
 *				[ shmaddr nitems ?varName? ]
 *
 * Return type: int
 * Side Effects:
 */

int
Svipc_ShmgetcharCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char ** argv;

/* Detail Design:

 */

{
  char * shmaddr;
  int nitems;
  int i;
  int flag = 0;

  if (argc < 3 || 4 < argc)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0],
			" shmaddr nitems ?varName?", (char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], (int *)&shmaddr) != TCL_OK)
    {
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[2], &nitems) != TCL_OK)
    {
      return TCL_ERROR;
    }

  for (i = 0; i < nitems; i++)
    {
      char r_str[20];

      (void) sprintf (r_str, "%d", shmaddr[i]);
      if (argc == 4)
	{
	  (void) Tcl_SetVar (interp, argv[3], r_str, flag);
	  if (!flag)
	    {
	      flag = TCL_APPEND_VALUE|TCL_LIST_ELEMENT;
	    }
	}
      Tcl_AppendElement (interp, r_str);
    }
  return TCL_OK;
}

/*
 * Function: Svipc_ShmgetlongCmd
 * Description:
 *	Return the value of the long at the given offset from the shared 
 *	memory base.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		number of command arguments.
 * char **	argv		vector of command arguments.
 *				[ shmaddr nitems ?varName? ]
 *
 * Return type: int
 * Side Effects:
 */

int
Svipc_ShmgetlongCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char ** argv;

/* Detail Design:

 */

{
  long * shmaddr;
  int nitems;
  int i;
  int flag = 0;

  if (argc < 3 || 4 < argc)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0],
			" shmaddr nitems ?varName?", (char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], (int *)&shmaddr) != TCL_OK)
    {
      return TCL_ERROR;
    }

#if SVIPC_CHECK_ALIGN
  if (((long)shmaddr)&SVIPC_LONG_ALIGN)
    {
      Tcl_AppendResult (interp, "unaligned address", (char *)0);
      return TCL_ERROR;
    }
#endif /* SVIPC_CHECK_ALIGN */

  if (Tcl_GetInt (interp, argv[2], &nitems) != TCL_OK)
    {
      return TCL_ERROR;
    }

  for (i = 0; i < nitems; i++)
    {
      char r_str[20];

      (void) sprintf (r_str, "%d", shmaddr[i]);
      if (argc == 4)
	{
	  (void) Tcl_SetVar (interp, argv[3], r_str, flag);
	  if (!flag)
	    {
	      flag = TCL_APPEND_VALUE|TCL_LIST_ELEMENT;
	    }
	}
      Tcl_AppendElement (interp, r_str);
    }
  return TCL_OK;
}

/*
 * Function: Svipc_ShmgetshortCmd
 * Description:
 *	Return the value of the short at the given offset from the shared 
 *	memory base.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		number of command arguments.
 * char **	argv		vector of command arguments.
 *				[ shmaddr nitems ?varName? ]
 *
 * Return type: int
 * Side Effects:
 */

int
Svipc_ShmgetshortCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char ** argv;

/* Detail Design:

 */

{
  short * shmaddr;
  int nitems;
  int i;
  int flag = 0;

  if (argc < 3 || 4 < argc)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0],
			" shmaddr nitems ?varName?", (char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], (int *)&shmaddr) != TCL_OK)
    {
      return TCL_ERROR;
    }

#if SVIPC_CHECK_ALIGN
  if (((long)shmaddr)&SVIPC_SHORT_ALIGN)
    {
      Tcl_AppendResult (interp, "unaligned address", (char *)0);
      return TCL_ERROR;
    }
#endif /* SVIPC_CHECK_ALIGN */

  if (Tcl_GetInt (interp, argv[2], &nitems) != TCL_OK)
    {
      return TCL_ERROR;
    }

  for (i = 0; i < nitems; i++)
    {
      char r_str[20];

      (void) sprintf (r_str, "%d", shmaddr[i]);
      if (argc == 4)
	{
	  (void) Tcl_SetVar (interp, argv[3], r_str, flag);
	  if (!flag)
	    {
	      flag = TCL_APPEND_VALUE|TCL_LIST_ELEMENT;
	    }
	}
      Tcl_AppendElement (interp, r_str);
    }
  return TCL_OK;
}

/*
 * Function: Svipc_ShmgetstrCmd
 * Description:
 *	Return the value of the string at the given offset from the shared 
 *	memory base.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		number of command arguments.
 * char **	argv		vector of command arguments.
 *				[ shmaddr ?varName? ]
 *
 * Return type: int
 * Side Effects:
 */

int
Svipc_ShmgetstrCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char ** argv;

/* Detail Design:

 */

{
  char * shmaddr;

  if (argc < 2 || 3 < argc)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0],
			" shmaddr ?varName?", (char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], (int *)&shmaddr) != TCL_OK)
    {
      return TCL_ERROR;
    }

  if (argc == 3)
    {
      (void) Tcl_SetVar (interp, argv[2], shmaddr, (int)0);
    }
  Tcl_SetResult (interp, shmaddr, TCL_VOLATILE);
  return TCL_OK;
}

/*
 * Function: Svipc_ShmgetstrnCmd
 * Description:
 *	Return the value of the string at the given offset from the shared 
 *	memory base.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		number of command arguments.
 * char **	argv		vector of command arguments.
 *				[ shmaddr length ?varName? ]
 *
 * Return type: int
 * Side Effects:
 */

int
Svipc_ShmgetstrnCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char ** argv;

/* Detail Design:

 */

{
  char * shmaddr;
  int length;
  char * buffer;

  if (argc < 3 || 4 < argc)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0],
			" shmaddr length ?varName?", (char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], (int *)&shmaddr) != TCL_OK)
    {
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[2], &length) != TCL_OK)
    {
      return TCL_ERROR;
    }

  buffer = (char *)ckalloc (length + 1);
  (void) strncpy (buffer, shmaddr, length);
  buffer[length] = 0;

  if (argc == 4)
    {
      (void) Tcl_SetVar (interp, argv[3], buffer, (int)0);
      ckfree (buffer);
    }
  Tcl_SetResult (interp, buffer, TCL_DYNAMIC);
  return TCL_OK;
}

/*
 * Function: Svipc_ShmreadCmd
 * Description:
 *	Copy a block of memory.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		number of command arguments.
 * char **	argv		vector of command arguments.
 *				[ shmaddr fileId ?numBytes? ]
 *
 * Return type: int
 * Side Effects:
 */

int
Svipc_ShmreadCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char ** argv;

/* Detail Design:

 */

{
  char *shmaddr;
  FILE *filePtr;
  int numBytes = 1<<30;
  int numRead = 0;

  if (argc != 3 && argc != 4)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0],
			"shmaddr fileId ?numBytes?", (char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], (int *)&shmaddr) != TCL_OK)
    {
      return TCL_ERROR;
    }

  if (Tcl_GetOpenFile(interp, argv[2], 0, 1, &filePtr) != TCL_OK)
    {
      return TCL_ERROR;
    }

  if (argc > 3)
    {
      if (Tcl_GetInt (interp, argv[3], &numBytes) != TCL_OK)
	{
	  return TCL_ERROR;
	}
    }

  while (numBytes > 0)
    {
#define READ_BUF_SIZE 4096
      int count = READ_BUF_SIZE;

      if (numBytes < READ_BUF_SIZE)
	{
	  count = numBytes;
	}
      count = fread (shmaddr, 1, count, filePtr);
      if (ferror (filePtr))
	{
	  Tcl_AppendResult (interp, "error reading \"", argv[1], "\": ",
			    Tcl_PosixError (interp), (char *)0);
	  clearerr (filePtr);
	  return TCL_ERROR;
	}
      if (count == 0)
	{
	  break;
	}
      shmaddr += count;
      numBytes -= count;
      numRead += count;
    }
  (void) sprintf (interp->result, "%d", numRead);
  return TCL_OK;
}

/*
 * Function: Svipc_ShmrmidCmd
 * Description:
 *	Remove the shared memory segment from the system.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		Number of command arguments.
 * char **	argv		Vector of command arguments.
 *				[ shmrmid shmid ]
 *
 * Return type: int
 * Side Effects:
 *	Sets interp->result.
 *	Removes the shared memory segment.
 */
/* ARGSUSED */
int
Svipc_ShmrmidCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char ** argv;

/* Detail Design:

 */

{
  int shmid;

  if (argc != 2)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0], " shmid", (char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], &shmid) != TCL_OK)
    {
      return TCL_ERROR;
    }

  if (shmctl (shmid, IPC_RMID, (struct shmid_ds *)0) == -1)
    {
      Tcl_AppendResult (interp, "cannot remove shared memory segment id ",
			argv[1], ": ", Tcl_PosixError (interp), (char *)0);
      return TCL_ERROR;
    }
  return TCL_OK;
}

/*
 * Function: Svipc_ShmsetCmd
 * Description:
 *	Perform the IPC_SET function on the shmid.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		Number of command arguments.
 * char **	argv		Vector of command arguments.
 *				[ shmset shmid shm_perm ]
 *
 * Return type: int
 * Side Effects:
 *	Sets interp->result.
 *	Modifies the shared memory segment permissions.
 */
/* ARGSUSED */
int
Svipc_ShmsetCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char ** argv;

/* Detail Design:

 */

{
  int shmid;
  int temp;
  int largc;
  char **largv;
  struct shmid_ds shmid_ds;

  if (argc != 3)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0], " shmid shm_perm",
			(char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], &shmid) != TCL_OK)
    {
      return TCL_ERROR;
    }

  if (Tcl_SplitList (interp, argv[2], &largc, &largv) != TCL_OK)
    {
      return TCL_ERROR;
    }

  if (largc < 3)
    {
      (void) strcpy (interp->result,
		     "shm_perm list needs 3 arguments: uid gid mode.");
      ckfree ((char *)largv);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, largv[0], &temp) != TCL_OK)
    {
      ckfree ((char *)largv);
      return TCL_ERROR;
    }
  shmid_ds.shm_perm.uid = temp;

  if (Tcl_GetInt (interp, largv[1], &temp) != TCL_OK)
    {
      ckfree ((char *)largv);
      return TCL_ERROR;
    }
  shmid_ds.shm_perm.gid = temp;

  if (Tcl_GetInt (interp, largv[2], &temp) != TCL_OK)
    {
      ckfree ((char *)largv);
      return TCL_ERROR;
    }
  shmid_ds.shm_perm.mode = temp;

  ckfree ((char *)largv);

  if (shmctl (shmid, IPC_SET, &shmid_ds) == -1)
    {
      Tcl_AppendResult (interp,
			"cannot set permissions for shared memory segment id ",
			argv[1], " {", argv[2], "}: ", Tcl_PosixError (interp),
			(char *)0);
      return TCL_ERROR;
    }
  return TCL_OK;
}

/*
 * Function: Svipc_ShmsetcharCmd
 * Description:
 *	Change the value of the char at the given offset from the shared 
 *	memory base.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		number of command arguments.
 * char **	argv		vector of command arguments.
 *				[ shmaddr list ]
 *
 * Return type: int
 * Side Effects:
 */

int
Svipc_ShmsetcharCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char ** argv;

/* Detail Design:

 */

{
  char * shmaddr;

  if (argc < 2)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0],
			" shmaddr ?byte ...?", (char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], (int *)&shmaddr) != TCL_OK)
    {
      return TCL_ERROR;
    }

  if (argc > 2)
    {
      argv += 2;
      
      while (*argv)
	{
	  char rstr[20];
	  int temp;
	  
	  if (Tcl_GetInt (interp, *argv++, &temp) != TCL_OK)
	    {
	      return TCL_ERROR;
	    }
	  
	  *shmaddr = temp;
	  (void) sprintf (rstr, "%d", *shmaddr++);
	  Tcl_AppendElement (interp, rstr);
	}
    }
  else
    {
      (void) sprintf (interp->result, "%d", *shmaddr);
    }
  return TCL_OK;
}

/*
 * Function: Svipc_ShmsetlongCmd
 * Description:
 *	Change the value of the long at the given offset from the shared 
 *	memory base.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		number of command arguments.
 * char **	argv		vector of command arguments.
 *				[ shmaddr list ]
 *
 * Return type: int
 * Side Effects:
 */

int
Svipc_ShmsetlongCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char ** argv;

/* Detail Design:

 */

{
  long * shmaddr;

  if (argc < 2)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0],
			" shmaddr ?long ...?", (char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], (int *)&shmaddr) != TCL_OK)
    {
      return TCL_ERROR;
    }

#if SVIPC_CHECK_ALIGN
  if (((long)shmaddr)&SVIPC_LONG_ALIGN)
    {
      Tcl_AppendResult (interp, "unaligned address", (char *)0);
      return TCL_ERROR;
    }
#endif /* SVIPC_CHECK_ALIGN */

  if (argc > 2)
    {
      argv += 2;
      
      while (*argv)
	{
	  char rstr[20];

	  if (Tcl_GetInt (interp, *argv++, (int *)shmaddr) != TCL_OK)
	    {
	      return TCL_ERROR;
	    }
	  (void) sprintf (rstr, "%d", *shmaddr++);
	  Tcl_AppendElement (interp, rstr);
	}
    }
  else
    {
      (void) sprintf (interp->result, "%d", *shmaddr);
    }
  return TCL_OK;
}

/*
 * Function: Svipc_ShmsetshortCmd
 * Description:
 *	Change the value of the short at the given offset from the shared 
 *	memory base.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		number of command arguments.
 * char **	argv		vector of command arguments.
 *				[ shmaddr list ]
 *
 * Return type: int
 * Side Effects:
 */

int
Svipc_ShmsetshortCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char ** argv;

/* Detail Design:

 */

{
  short * shmaddr;

  if (argc < 2)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0],
			" shmaddr ?short ...?", (char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], (int *)&shmaddr) != TCL_OK)
    {
      return TCL_ERROR;
    }

#if SVIPC_CHECK_ALIGN
  if (((long)shmaddr)&SVIPC_SHORT_ALIGN)
    {
      Tcl_AppendResult (interp, "unaligned address", (char *)0);
      return TCL_ERROR;
    }
#endif /* SVIPC_CHECK_ALIGN */

  if (argc > 2)
    {
      argv += 2;
      
      while (*argv)
	{
	  char rstr[20];
	  int temp;
	  
	  if (Tcl_GetInt (interp, *argv++, &temp) != TCL_OK)
	    {
	      return TCL_ERROR;
	    }
	  *shmaddr = temp;
	  (void) sprintf (rstr, "%d", *shmaddr++);
	  Tcl_AppendElement (interp, rstr);
	}
    }
  else
    {
      (void) sprintf (interp->result, "%d", *shmaddr);
    }
  return TCL_OK;
}

/*
 * Function: Svipc_ShmsetstrCmd
 * Description:
 *	Change the value of the string at the given offset from the shared 
 *	memory base.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		number of command arguments.
 * char **	argv		vector of command arguments.
 *				[ shmaddr string ]
 *
 * Return type: int
 * Side Effects:
 */

int
Svipc_ShmsetstrCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char ** argv;

/* Detail Design:

 */

{
  char * shmaddr;

  if (argc != 2 && argc != 3)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0],
			" shmaddr ?string?", (char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], (int *)&shmaddr) != TCL_OK)
    {
      return TCL_ERROR;
    }

  if (argc == 3)
    {
      (void) strcpy (shmaddr, argv[2]);
    }
  Tcl_SetResult (interp, shmaddr, TCL_VOLATILE);
  return TCL_OK;
}

/*
 * Function: Svipc_ShmsetstrnCmd
 * Description:
 *	Change the value of the string at the given offset from the shared 
 *	memory base.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		number of command arguments.
 * char **	argv		vector of command arguments.
 *				[ shmaddr length string ]
 *
 * Return type: int
 * Side Effects:
 */

int
Svipc_ShmsetstrnCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char ** argv;

/* Detail Design:

 */

{
  char * shmaddr;
  int length;

  if (argc != 3 && argc != 4)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0],
			" shmaddr length ?string?", (char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], (int *)&shmaddr) != TCL_OK)
    {
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[2], &length) != TCL_OK)
    {
      return TCL_ERROR;
    }

  if (argc == 4)
    {
      (void) strncpy (shmaddr, argv[3], length);
    }
  if (length < TCL_RESULT_SIZE)
    {
      (void) strncpy (interp->result, shmaddr, length);
      interp->result[length] = 0;
    }
  else
    {
      char *buffer = (char *)ckalloc (length + 1);

      (void) strncpy (buffer, shmaddr, length);
      buffer[length] = 0;
      Tcl_SetResult (interp, buffer, TCL_DYNAMIC);
    }
  return TCL_OK;
}

/*
 * Function: Svipc_ShmstatCmd
 * Description:
 *	Return a vector of information about a shared memory segment.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		Number of command arguments.
 * char **	argv		Vector of command arguments.
 *				[ shmstat shmid ]
 *
 * Return type: int
 * Side Effects:
 *	Sets interp->result.
 */
/* ARGSUSED */
int
Svipc_ShmstatCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char ** argv;
/* Detail Design:

 */

{
  int shmid;
  struct shmid_ds shm_data;
  int i;
  char rstr[7][20];
  char * margv[7];
  char * shm_list;

  if (argc < 2 || 3 < argc)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0], " shmid ?varName?",
			(char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], &shmid) != TCL_OK)
    {
      return TCL_ERROR;
    }

  if (shmctl (shmid, IPC_STAT, &shm_data) == -1)
    {
      Tcl_AppendResult (interp, "cannot stat shared memory segment id ",
			argv[1], ": ", Tcl_PosixError (interp), (char *)0);
      return TCL_ERROR;
    }

  (void) sprintf (rstr[0], "%d", shm_data.shm_perm.cuid);
  (void) sprintf (rstr[1], "%d", shm_data.shm_perm.cgid);
  (void) sprintf (rstr[2], "%d", shm_data.shm_perm.uid);
  (void) sprintf (rstr[3], "%d", shm_data.shm_perm.gid);
  (void) sprintf (rstr[4], "%07o", shm_data.shm_perm.mode);
  (void) sprintf (rstr[5], "%d", shm_data.shm_perm.seq);
  (void) sprintf (rstr[6], "%d", shm_data.shm_perm.key);

  margv[0] = rstr[0];
  margv[1] = rstr[1];
  margv[2] = rstr[2];
  margv[3] = rstr[3];
  margv[4] = rstr[4];
  margv[5] = rstr[5];
  margv[6] = rstr[6];
  
  shm_list = Tcl_Merge (7, margv);
  Tcl_AppendElement (interp, shm_list);
  ckfree (shm_list);
  if (argc == 3)
    {
      (void) Tcl_SetVar2 (interp, argv[2], "cuid", rstr[0], (int)0);
      (void) Tcl_SetVar2 (interp, argv[2], "cgid", rstr[1], (int)0);
      (void) Tcl_SetVar2 (interp, argv[2], "uid", rstr[2], (int)0);
      (void) Tcl_SetVar2 (interp, argv[2], "gid", rstr[3], (int)0);
      (void) Tcl_SetVar2 (interp, argv[2], "mode", rstr[4], (int)0);
      (void) Tcl_SetVar2 (interp, argv[2], "seq", rstr[5], (int)0);
      (void) Tcl_SetVar2 (interp, argv[2], "key", rstr[6], (int)0);
    }

  (void) sprintf (rstr[0], "%d", shm_data.shm_segsz);
  Tcl_AppendElement (interp, rstr[0]);
  if (argc == 3)
    {
      (void) Tcl_SetVar2 (interp, argv[2], "segsz", rstr[0], (int)0);
    }

  (void) sprintf (rstr[0], "%d", shm_data.shm_cpid);
  Tcl_AppendElement (interp, rstr[0]);
  if (argc == 3)
    {
      (void) Tcl_SetVar2 (interp, argv[2], "cpid", rstr[0], (int)0);
    }

  (void) sprintf (rstr[0], "%d", shm_data.shm_lpid);
  Tcl_AppendElement (interp, rstr[0]);
  if (argc == 3)
    {
      (void) Tcl_SetVar2 (interp, argv[2], "lpid", rstr[0], (int)0);
    }

  (void) sprintf (rstr[0], "%d", shm_data.shm_nattch);
  Tcl_AppendElement (interp, rstr[0]);
  if (argc == 3)
    {
      (void) Tcl_SetVar2 (interp, argv[2], "nattch", rstr[0], (int)0);
    }

  (void) sprintf (rstr[0], "%d", shm_data.shm_atime);
  Tcl_AppendElement (interp, rstr[0]);
  if (argc == 3)
    {
      (void) Tcl_SetVar2 (interp, argv[2], "atime", rstr[0], (int)0);
    }

  (void) sprintf (rstr[0], "%d", shm_data.shm_dtime);
  Tcl_AppendElement (interp, rstr[0]);
  if (argc == 3)
    {
      (void) Tcl_SetVar2 (interp, argv[2], "dtime", rstr[0], (int)0);
    }

  (void) sprintf (rstr[0], "%d", shm_data.shm_ctime);
  Tcl_AppendElement (interp, rstr[0]);
  if (argc == 3)
    {
      (void) Tcl_SetVar2 (interp, argv[2], "ctime", rstr[0], (int)0);
    }
  return TCL_OK;
}

/*
 * Function: Svipc_ShmwriteCmd
 * Description:
 *	Copy a block of memory.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		number of command arguments.
 * char **	argv		vector of command arguments.
 *				[ shmaddr fileId numBytes ]
 *
 * Return type: int
 * Side Effects:
 */

int
Svipc_ShmwriteCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char ** argv;

/* Detail Design:

 */

{
  char *shmaddr;
  FILE *filePtr;
  int numBytes;
  int numWrote = 0;

  if (argc != 4)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0],
			"shmaddr fileId numBytes", (char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], (int *)&shmaddr) != TCL_OK)
    {
      return TCL_ERROR;
    }

  if (Tcl_GetOpenFile(interp, argv[2], 1, 1, &filePtr) != TCL_OK)
    {
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[3], &numBytes) != TCL_OK)
    {
      return TCL_ERROR;
    }

  while (numBytes > 0)
    {
#define WRITE_BUF_SIZE 4096
      int count = WRITE_BUF_SIZE;

      if (numBytes < WRITE_BUF_SIZE)
	{
	  count = numBytes;
	}
      count = fwrite (shmaddr, 1, count, filePtr);
      if (ferror (filePtr))
	{
	  Tcl_AppendResult (interp, "error writing \"", argv[1], "\": ",
			    Tcl_PosixError (interp), (char *)0);
	  clearerr (filePtr);
	  return TCL_ERROR;
	}
      if (count == 0)
	{
	  break;
	}
      shmaddr += count;
      numBytes -= count;
      numWrote += count;
    }
  (void) sprintf (interp->result, "%d", numWrote);
  return TCL_OK;
}
