#
# File:		svipcStruct.tcl
#
# Description:
#	Provide access to structures in shared memory.
#
# This module implements access to C structures using the linkvar command
# provided by the Svipc package.  Linkvar allows a program to create a 
# connection between a memory address (presumably in a shared memory segment)
# and a Tcl array representing a structure.
#
# First, declare the structure using a C-like syntax.  Given the C struct:
#
# struct linkedList {
#   int next;			// Offset to next list entry
#   int prev;			// Offset to previous list entry
#   char contents[512];		// Payload
# }
#
# you can declare a Tcl equivalent:
#
# struct linkedList {
#   int next;			# Offset to next list entry
#   int prev;			# Offset to previous list entry
#   staticstr contents 512;	# staticstr treats the area as a character
#				# string.  char or uchar returns byte values.
# }
#
# The struct command sets up the global array linkedList with members next,
# prev, and contents whose initial values represent the offset from the 
# beginning of the structure.  It also adds linkedList.next, linkedList.prev
# and linkedList.contents to each of the global arrays sizeof, typeof,
# numberof, and offsetof to track the various attributes of the structure.
#
# The command linkStruct uses these various attribute arrays to associate
# members of an array with offsets from a given base address.  This
# association causes read references to the array members to return a
# representation of the actual memory contents and write references to
# update the associated memory area.
#
# The predefined types are those accepted by the linkvar command and consist
# of the basic Tcl types (int, boolean, and double, excluding dynamic string)
# enhanced with short, ushort, char, uchar, and staticstr.  The short and
# char types treat the referenced addresses as integral types, and the
# staticstr type treats the address as an ASCII string.
#
# After defining a struct, you can then use the struct name anywhere you
# would use a basic type.  The type definition process will expand single
# structure references in place.  To reference structured arrays, use
# linkStruct with a non-zero index to access the individual array elements.
#
# $Header: /home2/src/SVipc/RCS/svipcStruct.tcl,v 1.1 1996/06/25 00:37:09 joe Exp $
#

package provide Struct 1.0

# Initialize basic type sizes.

set sizeof(int) 4
set sizeof(boolean) 4
set sizeof(double) 8
set sizeof(char) 1
set sizeof(uchar) 1
set sizeof(short) 2
set sizeof(ushort) 2
set sizeof(staticstr) 1

# Basic types.  These correspond to the types recognized by linkvar.

set BasicTypes {int boolean double char uchar short ushort staticstr}

# Global type-definition context.  If this is non-null, then we are
# in the middle of a ``struct'' type definition, otherwise this is a
# simple type definition (such as int, char, etc.)

set Context {}

# The type-definition procedure body.  We use this body to define procs
# for all of the BasicTypes and for any ``struct'' type that occurs.

set TypeBody { # name length
  global BasicTypes Context numberof offsetof sizeof typeof
  set type [lindex [info level 0] 0]

  if [info exists sizeof($Context$name)] {
    error "Redeclaration of $Context$name."
  }

  set struct [string trimright $Context .]
  if {$struct == {}} {
    set offset 0
  } {
    upvar #0 $struct this-struct
    set offset $sizeof($struct)
    set this-struct($name) $offset
  }
  set offsetof($Context$name) $offset
  set typeof($Context$name) $type
  set numberof($Context$name) $length
  set sizeof($Context$name) [expr $sizeof($type)*$length]

  # Insert names for all structure members, except for structure arrays.
  # To access structure arrays, use linkStruct on a local variable to
  # walk through the array.
  if {[lsearch $BasicTypes $type] == -1 && $length == 1} {
    global $type
    foreach elt [array names $type] {
      # Set correct offset for this structure member.
      set sizeof($struct) [expr $offset+$offsetof($type.$elt)]
      $typeof($type.$elt) $name.$elt $numberof($type.$elt)
    }
  }

  if {$struct != {}} {
    set sizeof($struct) [expr $offset+$sizeof($Context$name)]
  }
  return $name
}

# Define procs for all basic types.

foreach i $BasicTypes {
  proc $i {name {length 1}} $TypeBody
}

# Define the struct type.  This sets up the Context and then evals the
# body, which is assumed to be a sequence of type definitions.  If no errors
# occur during this evaluation, then we define a new proc with the given name
# using TypeBody.

proc struct {name body} {
  global Context sizeof

  if {$Context != {}} {
    error "Cannot define a struct within a struct."
  }

  if [info exists sizeof($name)] {
    error "Redeclaration of $Context$name."
  } {
    set sizeof($name) 0
  }

  set Context $name.
  uplevel $body
  set Context {}
  proc $name {name {length 1}} $TypeBody
  return [info procs $name]
}

# Go through all of the members of the given struct and attach them to
# appropriate offsets from the given base address.

proc linkStruct {struct name base {index 0} {rdonly {}}} {
  global BasicTypes baseof numberof offsetof sizeof typeof
  global $struct

  # Perform array element access, if necessary
  if $index {incr base [expr $sizeof($struct)*$index]}

  foreach elt [array names $struct] {
    if {[lsearch $BasicTypes $typeof($struct.$elt)] == -1} {
      # Use char for structure instances.  Each member gets a separate entry.
      set type char
      set number $sizeof($struct.$elt)
    } {
      set type $typeof($struct.$elt)
      set number $numberof($struct.$elt)
    }
    uplevel linkvar $rdonly ${name}($elt) \
	[expr $base+$offsetof($struct.$elt)] $type $number
  }
}
