/*****************************************************************************
 * Assignment #2
 *
 * Dave Beazley
 * CS522
 * 1/24/95
 *
 * Solves the steady state heat equation
 *
 *           Lap(u) = 0     (Lap = Laplacian operator)
 *
 * On regions composed of rectangular regions using a
 * finite difference scheme.
 *
 * The finite difference scheme is based on the following
 * 5-point approximation :
 *
 *   1/h^2[u(i+1,j) + u(i-1,j) + u(i,j+1) + u(i,j-1) - 4u(i,j)] = 0     (1)
 *
 * Boundary conditions may be specified as Dirichlet or Neumann.
 *
 * Implementation details :
 *
 * 1.  Grid structure
 *     A rectangular grid with spacing h is set up on the unit
 *     square (0,M) x (0,M).   All problems must be scaled to
 *     fit into this region although some parts of the grid
 *     may be undefined (as is the case in this assignment).
 *
 *     A grid spacing of h is specified by the user with the requirement
 *     that n = 1/h is an integer.
 *
 *     This divides the unit square into an array of (n+1)x(n+1) points
 *     including the boundaries. 
 *
 *     In order to handle Neumann boundary conditions easily, a layer of
 *     "ficticous" points is also allocated.    Thus, internally, we
 *     store a (n+3)x(n+3) array of data.
 *
 *               ________________________
 *              | ______________________ |<- Ficticous points
 *              | |                    | |
 *              | |                    | |
 *              | |                    | |
 *              | |                    | |
 *              | |                    | |
 *              | |   Interior points  | |
 *              | |                    | |
 *              | |                    | |
 *              | |                    | |
 *              | |     <-  N+1 ->     | |
 *              | |____________________| |
 *              |________________________|
 *                      <-  N+3 ->       
 *     
 *      With this grid structure, it is important to keep in mind that
 *      the point (0,0) is mapped to the location (1,1) in the array.
 *      Thus, the point (i*h, j*h) corresponds to location ((i+1)*h, (j+1)*h)).
 *
 * 2.   Data Structures
 *      With each grid point, we store the following data :
 *
 *             double     u;     Value of function u(x,y)
 *             int     type;     Type of grid point
 *                               -1 = undefined
 *                                0 = interior point
 *                                1 = Dirichlet boundary point
 *                                2 = Neumann boundary point
 *             int     Bx;        Specifies which type of x-boundary
 *                                -1 = Left, +1 = Right
 *             int     By;        Specifies which type of y-boundary
 *                                -1 = Bottom, +1 = Top
 *             double  f;         Value of function specifying the
 *                                boundary condition (may be
 *                                dirichlet or neumann).
 * 
 * 3.   Solution strategy
 *      The Gauss-Seidel method is used to solve the resulting
 *      linear system of equations.   This algorithm has the advantage
 *      that it is extremely memory efficient, easy to implement and
 *      can be easily mapped onto the data structures.
 *
 *      Using equation (1) above, we iterate by solving for u(i,j)
 *      as follows:
 *
 *           u(i,j) = 1/(4*h^2)[u(i+1,j) + u(i-1,j) + u(i,j+1) + u(i,j-1)]   (2)
 *
 *      On each iteration, we calculate new values u(i,j) for all of the grid
 *      points.  The iteration stops when the difference between successive
 *      iterations drops below the value specified by TOLERANCE.
 *
 *      For each grid point, the following rules are followed :
 *          1.  if u(i,j) undefined, do nothing.
 *          2.  if u(i,j) is an interior point, evaluate equation (2).
 *          3.  if u(i,j) is a Dirichlet boundary point, simply set its
 *              value to the appropriate boundary value.
 *          4.  if u(i,j) is a Neumann boundary point, calculate the
 *              proper value of the appropriate ficticious point.  Then
 *              use equation (2) to evaluate.
 *              
 *      Ficticious points are calculated as follows :
 * 
 *      LEFT/RIGHT :   u(i + Bx, j) = 2*h + u(i - Bx, j)
 *      TOP/BOTTOM :   u(i, j + By) = 2*h + u(i, j - By)
 *
 *      Where Bx, By are defined as in part 2 above.  f is a function value
 *      specifying the boundary condition. 
 *
 *      Note: On each iteration, a temporary copy of the points is made
 *
 * Ok, enough blathering.  Now onto some code.....
 ***********************************************************************************/

#include <math.h>
#include <stdio.h>

/* Size of region */

#define   M                2.0

/* Tolerance of Gauss-Seidel iteration */

/* Grid Point Structure */

typedef struct {
    double   u;
    int      type;
    int      Bx;
    int      By;
    double   f;
  } GridPoint;

/* Global data structures */

GridPoint  **Grid;         /* Grid points           */
int        n;              /* Number of grid points */
double     h;              /* Grid spacing          */

/***********************************************************************
 * Utility function for getting values
 ***********************************************************************/

double get_u(int i, int j) {

  return (Grid[i+1][j+1].u);

}

/***********************************************************************
 * main program
 ***********************************************************************/

void
solve(int npoints, double tolerance) {

  int        i, j, numiter;
  double     error = 9999999999.0;
  double     u, fp, e, twoh;
  GridPoint  *gp, *gpl, *gpr, *gpt, *gpb;  /* Pointers used in G-Seidel */

  void   initial_condition();

  printf("Finite Difference PDE Solver\n");
  printf("Dave Beazley\n");
  printf("CS522, W95.  1/24/95\n\n");

  /* Calculate number of grid points */

  n = npoints+1;
  h = (double) M/n;
  
  /* Allocate memory */

  if (Grid) {
    free(Grid[0]);
    free(Grid);
  }
  Grid = (GridPoint **)   malloc((n+2)*sizeof(GridPoint *));
  Grid[0] = (GridPoint *) malloc((n+2)*(n+2)*sizeof(GridPoint));
  for (i = 0; i < n+2; i++) {
    Grid[i] = Grid[0] + (n+2)*i;
  }

  /* Clear all of the grid points */

  for (i = 0; i < n+2; i++) 
    for (j = 0; j < n+2; j++) {
      Grid[i][j].u = 0;
      Grid[i][j].type = -1;
      Grid[i][j].Bx = 0;
      Grid[i][j].By = 0;
      Grid[i][j].f = 0;
    }

  initial_condition();

  printf("Solving using Gauss-Seidel Iteration...\n");

  /* Now solve using Gauss-Seidel method */

  twoh = 2*h;              /* Constant used in calculation */

  numiter = 0;
  while (error > tolerance) {

    error = 0.0;

    /* Calculate a new set of grid points */

    /* Set up some pointers */

    for (i = 0; i < n; i++) {
      gp = &Grid[i+1][1];     /* Center point */
      gpl = &Grid[i][1];      /* Left point   */
      gpr = &Grid[i+2][1];    /* Right point  */
      gpt = &Grid[i+1][0];    /* Top point    */
      gpb = &Grid[i+1][2];    /* Bottom point */

      for (j = 0; j < n; j++) {
	if (gp->type == 0) {
	  u = (gpr->u + gpl->u + gpt->u + gpb->u)*0.25;
	} else if (gp->type == 2) {
	  
	  /* Point is a Neumann boundary point.
           * Calculate appropriate ficticious point value first
           *
           * Note that the proper choice of Bx or By simplifies the
           * calculation by eliminating a few conditionals.
           */

          if (gp->Bx != 0) 
	    Grid[i+1+gp->Bx][j+1].u = twoh*gp->f + Grid[i+1-gp->Bx][j+1].u;
	  if (gp->By != 0)
	    Grid[i+1][j+1+gp->By].u = twoh*gp->f + Grid[i+1][j+1-gp->By].u;

	  /* Calculate value of boundary */
	  u = (gpr->u + gpl->u + gpt->u + gpb->u)*0.25;
	} else {
	  u = gp->u;   /* Do nothing. Just copy the point */
	}
	e = fabs(gp->u - u);
	if (e > error) error = e;
	gp->u = u;
	gp++;
	gpl++;
	gpr++;
	gpt++;
	gpb++;
      }
    }
    if ((numiter % 200) == 0) 
       printf("%d error = %g\n", numiter, error); 
    numiter++;
  }

  printf("Solution in %d iterations.\n", numiter);

}

/***************************************************************************
 * void initial_condition(void)
 *
 * Sets up an initial condition.
 * This is done by specifying the following :
 *     1.  All of the interior points
 *     2.  Boundary points.
 *     3.  Type of boundary
 *     4.  Boundary condition functions.
 *
 * For the assignment, the following problem is solved
 *
 *                          Insulated
 *                         ____________  (2,2)
 *                        |     D      |
 *                        |            |
 *                        |            |
 *                        |            |
 *                        |E           |
 *                        |            |
 *            Insulated   |            |
 *            ____________|            |
 *           |            (1,1)       C|  Bdy = 0
 *           |                         |
 *           |                         |
 *   Bdy = 1 |A                        |
 *           |                         |
 *           |                         |
 *           | (0,0)     B             |
 *           |_________________________|
 *                      Insulated 
 *
 *
 * The boundary points along (1,y) correspond to grid points ((n-1)/2, j).
 ***************************************************************************/
 
#define    INITIAL_INTERIOR  0.5

void initial_condition()
{

  int   i, j;
  double f_A(), f_C(), f_insulated();


  /* Define interior points in region (0,2) x (0,1) */

  for (i = 0; i < n; i++) 
    for (j = 0; j < (n-1)/2; j++) {
      Grid[i+1][j+1].u = INITIAL_INTERIOR;
      Grid[i+1][j+1].type = 0;    /* Interior point */
    }

  /* Define interior points in region (1,2) x (1,2) */

      
  for (i = (n-1)/2; i < n; i++)
    for (j = (n-1)/2; j < n; j++) {
      Grid[i+1][j+1].u = INITIAL_INTERIOR;
      Grid[i+1][j+1].type = 0;    /* interior point */
    }

  /* Define boundary along (0,0) - (0,1) */

  for (j = 0; j <= (n-1)/2; j++) {
    Grid[1][j+1].u = 1.0;            /* Boundary = 1*/
    Grid[1][j+1].type = 1;           /* Dirichlet */
    Grid[1][j+1].f = 1.0;            /* Boundary function */
  }

  /* Define boundary along (0,0) - (2,0) */

  for (i = 1; i < n - 1; i++) {
    Grid[i+1][1].u = 0;              
    Grid[i+1][1].type = 2;           /* Neumann boundary */
    Grid[i+1][1].f = 0;              /* Insulated boundary */
    Grid[i+1][1].Bx = 0;
    Grid[i+1][1].By = -1;            /* Bottom boundary */
  }

  /* Define boundary along (2,0) - (2,2) */

  for (j = 0; j < n; j++) {
    Grid[n][j+1].u = 0;              /* T = 0 */
    Grid[n][j+1].type = 1; 
    Grid[n][j+1].f = 0.0;            /* T = 0 */
  }

  /* Define boundary along (1,2) - (2,2)  */

  for (i = (n-1)/2; i < n - 1; i++) {
    Grid[i+1][n].u = 0;               /* (1,2) - (2,2) boundary */
    Grid[i+1][n].type = 2;
    Grid[i+1][n].f = 0.0;             /* Insulated boundary */
    Grid[i+1][n].Bx = 0;
    Grid[i+1][n].By = 1;              /* Top boundary */
  }
  
  /* Define boundary along (1,1) - (1,2) */

  for (j = (n-1)/2; j < n; j++) {
     Grid[(n-1)/2 + 1][j+1].u = 0;     
     Grid[(n-1)/2 + 1][j+1].type = 2;
     Grid[(n-1)/2 + 1][j+1].f = 0;     /* Insulated */
     Grid[(n-1)/2 + 1][j+1].Bx = -1;   /* Left boundary */
     Grid[(n-1)/2 + 1][j+1].By = 0;  
   }

  /* Define boundary along (0,1) - (1,1) */

  for (i = 1; i < (n-1)/2; i++) {
    Grid[i+1][(n-1)/2 + 1].u = 0;
    Grid[i+1][(n-1)/2 + 1].type = 2;
    Grid[i+1][(n-1)/2 + 1].f = 0;      /* Insulated */
    Grid[i+1][(n-1)/2 + 1].Bx = 0;
    Grid[i+1][(n-1)/2 + 1].By = 1;     /* Top boundary */

  }

  /* Have some strangeness with the point (1,1).
   * Under the 5-point formula, this can be treated as
   * an interior point.
   */

  Grid[(n-1)/2 + 1][(n-1)/2 + 1].type = 0;


  /* Better check this out for sure, uncomment to print
   * out type of each grid point. */

/*  printf("Grid point types.\n");

  for (j = n+1; j >= 0; j--) {
    for (i = 0; i < n+2; i++) 
      printf("%2d ", Grid[i][j].type);
    printf("\n");
  }

  printf("\n");
*/

}



  
