
/**********************************************************************
 * Simplified Wrapper and Interface Generator  (SWIG)
 * 
 * Dave Beazley
 * 
 * Theoretical Division (T-11)           Department of Computer Science
 * Los Alamos National Laboratory        University of Utah
 * Los Alamos, New Mexico  87545         Salt Lake City, Utah  84112
 * beazley@lanl.gov                      beazley@cs.utah.edu
 *
 * Copyright (c) 1995-1996
 * The Regents of the University of California and the University of Utah
 * All Rights Reserved
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that 
 * (1) The above copyright notice and the following two paragraphs
 * appear in all copies of the source code and (2) redistributions
 * including binaries reproduces these notices in the supporting
 * documentation.   Substantial modifications to this software may be
 * copyrighted by their authors and need not follow the licensing terms
 * described here, provided that the new terms are clearly indicated in
 * all files where they apply.
 * 
 * IN NO EVENT SHALL THE AUTHOR, THE UNIVERSITY OF CALIFORNIA, THE 
 * UNIVERSITY OF UTAH OR DISTRIBUTORS OF THIS SOFTWARE BE LIABLE TO ANY
 * PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
 * DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION,
 * EVEN IF THE AUTHORS OR ANY OF THE ABOVE PARTIES HAVE BEEN ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE AUTHOR, THE UNIVERSITY OF CALIFORNIA, AND THE UNIVERSITY OF UTAH
 * SPECIFICALLY DISCLAIM ANY WARRANTIES,INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND 
 * THE AUTHORS AND DISTRIBUTORS HAVE NO OBLIGATION TO PROVIDE MAINTENANCE,
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 **************************************************************************/

#include "internal.h"

#include <limits.h>

// ------------------------------------------------------------------------
// $Header: /home/beazley/SWIG/SWIG1.1b5/SWIG/RCS/typemap.cxx,v 1.6 1997/03/12 05:01:14 beazley Exp $
//
// typemap.cxx
//
// This file provides universal support for typemaps.   Typemaps are created
// using the following SWIG command in an interface file:
//
//       %typemap(lang,operation) type { code }        Make a new typemap
//       %typemap(lang,operation) type;                Clears any previous typemap
//
// lang is an identifier indicating the target language.  The typemap will
// simply be ignored if its for a different language.  The code is the
// corresponding C code for the mapping.  An example typemap might look
// like this :
//
//       %typemap(tcl,get) double {
//              $target = atof($source);
//       }
//       %typemap(tcl,set) double {
//              sprintf($target,"%0.17f",$source);
//       }
//
// The variables $target and $source should be used in any type-mappings.
// Additional local variables can be created, but this should be done 
// by enclosing the entire code fragment in an extra set of braces.
//
// The C++ API to the type-mapper is as follows :
//
// void typemap_register(char *op, char *lang, DataType *type, char *pname, String &getcode)
// char *typemap_lookup(char *op, char *lang, DataType *type, char *pname, char *source, char *target);
// void typemap_clear(char *op, char *lang, DataType *type, char *pname);
//
// The lookup functions return a character string corresponding to the type-mapping
// code or NULL if none exists.   The string return will have the source and target
// strings substituted for the strings "$source" and "$target" in the type-mapping code.
//
// (2/19/97) This module has been extended somewhat to provide generic mappings
// of other parts of the code--most notably exceptions.
//
// void fragment_register(char *op, char *lang, String &code)
// char fragment_lookup(char *op, char *lang, int age);
// char fragment_clear(char *op, char *lang);
//
// $Log: typemap.cxx,v $
// Revision 1.6  1997/03/12 05:01:14  beazley
// Removed compiler warning.
//
// Revision 1.5  1997/03/08 19:18:09  beazley
// Added better support for typemaps involving arrays.
//
// Revision 1.4  1997/02/19 23:00:47  beazley
// Added support for generic code fragments.  These are used to
// support exceptions.
//
// Revision 1.3  1997/02/16 20:15:29  beazley
// Fixed major bug in typemap scoping
//
// Revision 1.2  1997/01/08 01:24:51  beazley
// Pre 1.1b3 checkin
//
// Revision 1.1  1997/01/06 17:08:17  beazley
// Initial revision
//
// ------------------------------------------------------------------------

// Structure for holding a typemap

struct TypeMap {
  char       *lang;
  DataType   *type;
  String      code;
  int         first;
  int         last;
  TypeMap     *next;
  TypeMap(char *l, DataType *t, String &c) {
    lang = copy_string(l);
    type = new DataType(t);
    code << c;
    first = type_id;
    last = INT_MAX;
    next = 0;
  }
  TypeMap(char *l, DataType *t, char *c) {
    lang = copy_string(l);
    type = new DataType(t);
    code << c;
    first = type_id;
    last = INT_MAX;
    next = 0;
  }
  TypeMap(char *l, char *c) {
    lang = copy_string(l);
    type = 0;
    code << c;
    first = type_id;
    last = INT_MAX;
    next = 0;
  }
};

// Hash tables for storing type-mappings

static Hash  typemap_hash;

// ------------------------------------------------------------------------
// char *typemap_string(char *lang, DataType *type, char *pname, char *ary, char *suffix)
//
// Produces a character string corresponding to a lang, datatype, and
// method.   This string is used as the key for our typemap hash table.
// ------------------------------------------------------------------------

static char *typemap_string(char *lang, DataType *type, char *pname, char *ary, char *suffix) {
  static String str;

  int old_status;
  old_status = type->status;
  type->status = 0;
  str = "";

  if (ary)
    str << lang << type->print_type() << pname << ary << suffix;
  else
    str << lang << type->print_type() << pname << suffix;

  type->status = old_status;
  return str;
}

// ------------------------------------------------------------------------
// void typemap_register(char *op, char *lang, DataType *type, char *pname, char *getcode)
//
// Register a new mapping with the type-mapper.
// ------------------------------------------------------------------------

void typemap_register(char *op, char *lang, DataType *type, char *pname, char *getcode) {
  
  char     *key;
  TypeMap  *tm,*tm_old;
  char     temp[256];

  tm = new TypeMap(lang,type,getcode);
  key = typemap_string(lang,type,pname,type->arraystr, op);

  // Get any previous setting of the typemap

  tm_old = (TypeMap *) typemap_hash.lookup(key);

  if (tm_old) {

    // Perform a chaining operation 
    sprintf(temp,"$%s",op);
    tm->code.replace(temp,tm_old->code);

    // If found, we need to attach the old version to the new one

    tm->next = tm_old;
    tm_old->last = type_id;

    // Remove the old one from the hash
  
    typemap_hash.remove(key);
  }

  // Add new typemap to the hash table
  typemap_hash.add(key,(void *) tm);
}
// ------------------------------------------------------------------------
// static TypeMap *typemap_search(char *key, int id) 
//
// An internal function for searching for a particular typemap given
// a key value and datatype id.
//
// Basically this checks the hash table and then checks the id against
// first and last values, looking for a match.   This is to properly
// handle scoping problems.
// ------------------------------------------------------------------------

TypeMap *typemap_search(char *key, int id) {
  
  TypeMap *tm;

  tm = (TypeMap *) typemap_hash.lookup(key);
  while (tm) {
    if ((id >= tm->first) && (id < tm->last)) return tm;
    else tm = tm->next;
  }
  return tm;
}

// ------------------------------------------------------------------------
// TypeMap *typemap_search_array(char *op, char *lang, DataType *type, char *pname, String &str)
//
// Performs a typemap lookup on an array type.  This is abit complicated
// because we need to look for ANY tags specifying that any array dimension
// is valid.   The resulting code will be placed in str with dimension variables
// substituted.
// ------------------------------------------------------------------------

TypeMap *typemap_search_array(char *op, char *lang, DataType *type, char *pname, String &str) {
  char      *origarr = type->arraystr;
  char      *key;
  int       ndim,i,j,k,n;
  TypeMap   *tm;
  char      temp[10];

  if (!type->arraystr) return 0;

  // First check to see if exactly this array has been mapped

  key = typemap_string(lang,type,pname,type->arraystr,op);
  tm = typemap_search(key,type->id);

  // Check for unnamed array of specific dimensions
  if (!tm) {
    key = typemap_string(lang,type,"",type->arraystr,op);
    tm = typemap_search(key,type->id);
  } 

  if (!tm) {
    // We're going to go search for matches with the ANY tag
    String  tempastr;
    ndim = type->array_dimensions();             // Get number of dimensions
    j = (1 << ndim) - 1;                         // Status bits
    for (i = 0; i < (1 << ndim); i++) {
      // Form an array string
      tempastr = "";
      k = j;
      for (n = 0; n < ndim; n++) {
	if (k & 1) {
	  tempastr << "[" << type->get_dimension(n) << "]";
	} else {
	  tempastr << "[ANY]";
	}
	k = k >> 1;
      }
      //      printf("checking (%s) : %s\n",origarr,tempastr.get());
      type->arraystr = tempastr.get();
      key = typemap_string(lang,type,pname,type->arraystr,op);
      tm = typemap_search(key,type->id);
      if (!tm) {
	key = typemap_string(lang,type,"",type->arraystr,op);
	tm = typemap_search(key,type->id);
      }
      type->arraystr = origarr;
      if (tm) break;
      j--;
    }
  }      
	
  if (tm) {
    str << tm->code;
    ndim = type->array_dimensions();
    for (i = 0; i < ndim; i++) {
      sprintf(temp,"$dim%d",i);
      str.replace(temp,type->get_dimension(i));
    }
  }
  return tm;
}

// ------------------------------------------------------------------------
// char *typemap_lookup(char *op, char *lang, DataType *type, char *pname, char *source, char *target)
//
// Looks up a "get" function in the type-map and returns a character string
// containing the appropriate translation code.
//
// op       is string code for type of mapping
// lang     is the target language string
// type     is the datatype
// pname    is an optional parameter name
// source   is a string with the source variable
// target   is a string containing the target value
//
// Returns NULL if no mapping is found.
//
// Typemaps follow a few rules regarding naming and C pointers by checking
// declarations in this order.
//
//         1.   type name []         - A named array (most specific)
//         2.   type name            - Named argument
//         3.   type []              - Type with array
//         4.   type                 - Ordinary type
// 
// Array checking is only made if the datatype actally has an array specifier      
// 
// Array checking uses a special token "ANY" that indicates that any
// dimension will match.  Since we are passed a real datatype here, we
// need to hack this a special case.
//
// Array dimensions are substituted into the variables $dim1, $dim2,...,$dim9
// ------------------------------------------------------------------------

char *typemap_lookup(char *op, char *lang, DataType *type, char *pname, char *source, char *target) {
  static String str;
  char *key = 0;
  TypeMap *tm = 0;

  if (!lang) {
    return 0;
  }

  // First check for named array
  str = "";
  tm = typemap_search_array(op,lang,type,pname,str);

  // Check for named argument
  if (!tm) {
    key = typemap_string(lang,type,pname,0,op);
    tm = typemap_search(key,type->id);
    if (tm)
      str << tm->code;
  }

  // Check for unnamed type
  if (!tm) {
    key = typemap_string(lang,type,"",0,op);
    tm = typemap_search(key,type->id);
    if (tm)
      str << tm->code;
  }
  if (!tm) return 0;
  
  // Now perform character replacements

  str.replace("$source",source);
  str.replace("$target",target);
  str.replace("$type", type->print_type());
  str.replace("$mangle",type->print_mangle());

  // Return character string

  return str;
}

// ----------------------------------------------------------------------------
// char *typemap_check(char *op, char *lang, DataType *type, char *pname)
//
// Checks to see if there is a typemap.  Returns typemap string if found, NULL
// if not.
// ----------------------------------------------------------------------------

char *typemap_check(char *op, char *lang, DataType *type, char *pname) {
  static String str;
  char *key = 0;
  TypeMap *tm = 0;

  if (!lang) {
    return 0;
  }

  // First check for named array

  if (type->arraystr) {
    key = typemap_string(lang,type,pname,type->arraystr,op);
    tm = typemap_search(key,type->id);
  }

  // Check for named argument
  if (!tm) {
    key = typemap_string(lang,type,pname,0,op);
    tm = typemap_search(key,type->id);
  }

  // Check for unnamed array
  if ((!tm) && (type->arraystr)) {
    key = typemap_string(lang,type,"",type->arraystr,op);
    tm = typemap_search(key,type->id);
  } 

  // Check for unname type
  if (!tm) {
    key = typemap_string(lang,type,"",0,op);
    tm = typemap_search(key,type->id);
  }
  if (!tm) return 0;
  
  str = "";
  str << tm->code;

  // Return character string

  return str;
}


// ------------------------------------------------------------------------
// void typemap_clear(char *op, char *lang, DataType *type, char *pname)
//
// Clears any previous typemap
// ------------------------------------------------------------------------

void typemap_clear(char *op, char *lang, DataType *type, char *pname) {
  
  char     *key;
  TypeMap  *tm;

  key = typemap_string(lang,type,pname,type->arraystr,op);

  // Look for any previous version, simply set the last id if
  // applicable.
  
  tm = (TypeMap *) typemap_hash.lookup(key);
  if (tm) {
    if (tm->last > type_id) tm->last = type_id;
  }
}

// ------------------------------------------------------------------------
// void typemap_copy(char *op, char *lang, DataType *stype, char *sname,
//                   DataType *ttype, char *tname)
//
// Copies the code associate with a typemap
// ------------------------------------------------------------------------

void typemap_copy(char *op, char *lang, DataType *stype, char *sname,
		      DataType *ttype, char *tname) {
  
  char     *key;
  TypeMap  *tm;

  // Try to locate a previous typemap

  key = typemap_string(lang,stype,sname,stype->arraystr,op);
  tm = typemap_search(key,stype->id);
  if (!tm) return;
  typemap_register(op,lang,ttype,tname,tm->code);
}

// ------------------------------------------------------------------------
// char *fragment_string(char *op, char *lang)
//
// Produces a character string corresponding to a language and method
// This string is used as the key for our typemap hash table.
// ------------------------------------------------------------------------

static char *fragment_string(char *op, char *lang) {
  static String str;

  str = "";

  str << "fragment:" << lang << op;
  return str;
}

// ------------------------------------------------------------------------
// void fragment_register(char *op, char *lang, char *code)
//
// Register a code fragment with the type-mapper.
// ------------------------------------------------------------------------

void fragment_register(char *op, char *lang, char *code) {
  
  char     *key;
  TypeMap  *tm,*tm_old;
  char      temp[256];
  
  tm = new TypeMap(lang,code);
  key = fragment_string(op,lang);

  // Get any previous setting of the typemap

  tm_old = (TypeMap *) typemap_hash.lookup(key);
  if (tm_old) {
    // If found, we need to attach the old version to the new one

    // Perform a chaining operation 

    sprintf(temp,"$%s",op);
    tm->code.replace(temp,tm_old->code);

    tm->next = tm_old;
    tm_old->last = type_id;

    // Remove the old one from the hash
  
    typemap_hash.remove(key);
  }
  // Add new typemap to the hash table
  typemap_hash.add(key,(void *) tm);
}


// ------------------------------------------------------------------------
// char *fragment_lookup(char *op, char *lang, int age)
//
// op       is string code for type of mapping
// lang     is the target language string
// age      is age of fragment.
//
// Returns NULL if no mapping is found.
//
// ------------------------------------------------------------------------

char *fragment_lookup(char *op, char *lang, int age) {
  static String str;
  char *key = 0;
  TypeMap *tm = 0;

  if (!lang) {
    return 0;
  }

  str = "";
  key = fragment_string(op,lang);
  tm = typemap_search(key,age);

  if (!tm) return 0;

  str << tm->code;
  return str;
}

// ------------------------------------------------------------------------
// void fragment_clear(char *op, char *lang)
//
// Clears any previous fragment definition
// ------------------------------------------------------------------------

void fragment_clear(char *op, char *lang) {
  
  char     *key;
  TypeMap  *tm;

  key = fragment_string(op,lang);

  // Look for any previous version, simply set the last id if
  // applicable.
  
  tm = (TypeMap *) typemap_hash.lookup(key);
  if (tm) {
    if (tm->last > type_id) tm->last = type_id;
  }
}
