;;; -*- Mode: LISP; Package: TCLTK -*-
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                      Usage & Distribution Agreement                      ;;
;;                            Tk/Lisp Interface                             ;;
;;                                                                          ;;
;; This file is part of the Tk/Lisp Interface (TLI) developed by the        ;;
;; Lockheed Artificial Intelligence Center.  Use, duplication,              ;;
;; distribution, and disclosure of the TLI is subject to the following      ;;
;; restrictions and limitations.                                            ;;
;;                                                                          ;;
;; 1. Lockheed makes no warranty regarding the merchantability or fitness   ;;
;;    of the TLI for any particular purpose.                                ;;
;;                                                                          ;;
;; 2. Lockheed assumes no responsibility for the TLI and assumes no         ;;
;;    obligation to support or maintain the TLI.                            ;;
;;                                                                          ;;
;; 3. The TLI can be duplicated and redistributed without restriction,      ;;
;;    provided that this Usage & Distribution Agreement header is left      ;;
;;    intact and unaltered in the source code.                              ;;
;;                                                                          ;;
;; 4. The TLI can be modified and/or extended without restriction, provided ;;
;;    that all modifications and their authors are clearly noted in a       ;;
;;    header preceding this header in the source code files.                ;;
;;                                                                          ;;
;;    Lockheed asks that the authors of any such modifications or           ;;
;;    extensions to the TLI share them with the Lockheed AI Center in the   ;;
;;    same open manner as the Lockheed AI Center is sharing the TLI.        ;;
;;                                                                          ;;
;; Please send comments about or enhancements to the TLI to Dan Kuokka      ;;
;; (kuokka@aic.lockheed.com)  or Larry Harada (harada@aic.lockheed.com).    ;;
;;                                                                          ;;
;; For more information about the Lockheed AI Center, see our               ;;
;; World-Wide Web page @ http://www.rdd.lmsc.lockheed.com.                  ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;-----------------------------------------------------------------------------
;;; Sample file: 
;;;
;;; A verbose description ...
;;;-----------------------------------------------------------------------------
(eval-when (:load-toplevel :compile-toplevel :execute)
  (unless (find-package :tcltk)
    (defpackage :tcltk))
  (in-package :tcltk))

;;;-----------------------------------------------------------------------------
;;; Globals
;;;-----------------------------------------------------------------------------
(defparameter *interp*      0)   ;; Tcl/Tk interpreter
(defparameter *main-window* 0)   ;; Main window.

;;;-----------------------------------------------------------------------------
;;; A demo
;;;-----------------------------------------------------------------------------
(defun demo ()
  ;;---------------------------------------
  ;; Create/start up the Tcl/Tk interpreter.
  ;; This macro starts the tk event loop as a separate Lisp process.
  ;; It doesn't block, I can use the blocking version to prevent it
  ;; from crunching extra cpu time - but it also blocks the Lisp 
  ;; interpreter (I'll have to figure that out sometime).
  ;;
  ;; The main thing needed here is the binding for the interpreter *interp*.
  (tk-init-session *interp* *main-window* " Demo ")

  ;;---------------------------------------
  ;; Load default look & feel 
  ;; This is optional - The Tk option file is like an .Xdefaults file.
  ;; (You can actually put the the contents of the options file the .Xdefaults
  ;; for your Tk app ... here I load it explicitly).
  (Tkcmd *interp* 'option 'readfile "sample.opt")
  
  ;;---------------------------------------
  ;; Create some buttons.
  (Tkcmd *interp* 'label  ".title" :text "Sample")
  (Tkcmd *interp* 'button ".hello" :text "Hello" :command "say-hi" :width 30)
  (Tkcmd *interp* 'button '.bye    :text "Bye"   :command 'say-bye :width 30)

  ;;---------------------------------------
  ;; Add them to a parent frame (the parent is denoted by ".")
  (Tkcmd *interp* 'pack 'append "." 
	  ".title" '(top fillx)
	  ".hello" '(top fillx) 
	  ".bye"   '(top fillx))

  ;;---------------------------------------
  ;; Register callbacks.
  ;; Note: These were given as the button commands above.
  ;;                             LispName  TclName 
  (register-tcl-command *interp* 'say-hi  "say-hi")
  (register-tcl-command *interp* 'say-bye "say-bye")
  )


;;;-----------------------------------------------------------
;;; Tcl commands. You can put arbitrary Lisp code in them.
(def-tcl-command say-hi ()
  (format t "Hello World!~%")
  (listener))

(def-tcl-command say-bye ()
  (format t "Good Bye World!~%")
  (shutdown))

;;; Shutdown the event loop and destroy all windows. 
(defun shutdown ()
  (Tkcmd *interp* 'destroy ".")
  (setq *tk-loop-exit* t))  ;; a kludge


;;;-----------------------------------------------------------------------------
;;; Pseudo Lisp listener - not too reliable.
;;;-----------------------------------------------------------------------------

;;;-----------------------------------------------------------------------------
;;; Two forms for "lisp-eval". The first is general purpose - given a string -
;;; and can be used anywhere in Tcl.
;;;
;;; The second form is specific for this demo.
;;;-----------------------------------------------------------------------------
(def-tcl-command eval-lisp-from-tcl (string)  
  ;; Minimal protection against a lisp break crashing the event loop.
  (ignore-errors
   (tkset-interp-result 
    *interp*
    (prin1-to-string (eval (read-from-string string))))))

(def-tcl-command process-lisp ()
  (let ((string (Tkcmd *interp* ".send.txt" 'get 0.0 'end)))
    (do ((ret (multiple-value-list (read-from-string string nil 'end-of-string))
	      (multiple-value-list 
		  (read-from-string string nil 'end-of-string :start (second ret)))))
	((eq (first ret) 'end-of-string))

      (Tkcmd *interp* ".receive.txt" 'insert 'end 
	     (format nil "\"~a\""
		     (prin1-to-string (ignore-errors (eval (first ret))))))
      (Tkcmd *interp* ".receive.txt" 'insert 'end "\"\\n\""))))

(def-tcl-command shutdown-listener ()
  (Tkcmd *interp* 'destroy ".send" ".receive"))

;;;-----------------------------------------------------------------------------
;;; listener creates the top level listener window
;;;-----------------------------------------------------------------------------
(defun listener ()
  (let ((sendframe    (make-pane ".send"    "Expression" "Eval" "process_lisp"))
	(receiveframe (make-pane ".receive" "Result"     "Quit" "shutdown_listener")))

    ;; Key bindings 
    (Tkcmd *interp* 'bind ".send.txt" "<Control-x>" 'process_lisp)
    (Tkcmd *interp* 'bind ".send.txt" "<Control-c>" '(.send.txt delete 0.0 end))
    (Tkcmd *interp* 'bind ".receive.txt" "<Control-c>" '(.receive.txt delete 0.0 end))

    (Tkcmd *interp* 'pack 'append "."
	   receiveframe '(bottom fillx filly padx 10)
	   sendframe    '(bottom fillx filly padx 10))

    (register-tcl-command *interp* 'shutdown-listener "shutdown_listener")
    (register-tcl-command *interp* 'process-lisp      "process_lisp")))

;;;-----------------------------------------------------------------------------
;;; make-pane creates listener sub-panels
;;;-----------------------------------------------------------------------------
(defun make-pane (root title b1-title b1-command)
  (let* ((frame (Tkcmd *interp* 'frame root :relief 'raised :borderwidth 2))
	 (label (Tkcmd *interp* 'label 
		       (format nil "~a.contents" frame)
		       :text title))
	 (text  (Tkcmd *interp* 'text 
		       (format nil "~a.txt" frame)
		       :yscrollcommand (format nil "\"~a.scroll set\"" root)
		       :wrap 'word
		       :relief 'sunken
		       :height 15
		       :borderwidth 2))
	 (scrollbar (Tkcmd *interp* 'scrollbar 
			   (format nil "~a.scroll" frame)
			   :relief 'sunken
			   :command (format nil "\"~a yview\"" text)))
	 (button1 (Tkcmd *interp* 'button (format nil "~a.b1" frame)
			 :text b1-title
			 :command b1-command))
	 (button2 (Tkcmd *interp* 'button (format nil "~a.b2" frame)
			 :text "Clear"
			 :command (format nil "\"~a delete 0.0 end\"" text))))

    (Tkcmd *interp* 'pack 'append frame 
	   label   '(top)
	   scrollbar '(right filly)
	   text    '(top)
	   button1 '(bottom padx 5 pady 7 fillx right expand)
	   button2 '(bottom padx 5 pady 7 fillx left expand))
    
    frame))
  
;;; End of file.       
