/* -*- c -*-
 * transformInt.h - internal definitions
 *
 * Copyright (c) 1995 Andreas Kupries (aku@kisters.de)
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL I BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL,
 * INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS
 * SOFTWARE AND ITS DOCUMENTATION, EVEN IF I HAVE BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * I SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND
 * I HAVE NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 * ENHANCEMENTS, OR MODIFICATIONS.
 *
 * CVS: $Id: transformInt.h,v 1.3.2.1 1996/12/03 21:43:49 aku Exp $
 */

#ifndef TRF_INT_H
#define TRF_INT_H

#include <errno.h>
#include <string.h>
#include <assert.h>
#include "transform.h"

/* make internal procedure of tcl available */
EXTERN void
panic _ANSI_ARGS_ ((CONST char* format, ...));


/*
 * A structure of the type below is created and maintained
 * for every registered transformer (and every interpreter).
 */

typedef struct _Trf_RegistryEntry_ {
  Trf_TypeDefinition* trfType;    /* reference to transformer specification */
  Tcl_ChannelType*    transType;  /* reference to derived channel type specification */
  Tcl_Command         trfCommand; /* command associated to the transformer */
  Tcl_Interp*         interp;     /* interpreter the command is registered in */
} Trf_RegistryEntry;


/*
 * Procedures to access the registry of transformers for a specified
 * interpreter. The registry is a hashtable mapping from transformer
 * names to structures of type 'Trf_RegistryEntry' (see above).
 */

EXTERN Tcl_HashTable* 
TrfGetRegistry  _ANSI_ARGS_ ((Tcl_Interp* interp));

EXTERN Tcl_HashTable*
TrfPeekForRegistry _ANSI_ARGS_ ((Tcl_Interp* interp));

EXTERN int
Trf_Unregister _ANSI_ARGS_ ((Tcl_Interp*        interp,
			     Trf_RegistryEntry* entry));


/*
 * Procedures used by 3->4 encoders (uu, base64).
 */

EXTERN void TrfSplit3to4 _ANSI_ARGS_ ((CONST char* in, char* out, int length));
EXTERN void TrfMerge4to3 _ANSI_ARGS_ ((CONST char* in, char* out));

EXTERN void TrfApplyEncoding   _ANSI_ARGS_ ((char* buf, int length, CONST char* map));
EXTERN int  TrfReverseEncoding _ANSI_ARGS_ ((char* buf, int length, CONST char* reverseMap,
					     int padChar, int* hasPadding));

/*
 * Definition of option information for message digests and accessor
 * to set of vectors processing these.
 */


typedef struct _TrfMDOptionBlock {
  int         behaviour; /* IMMEDIATE vs. ATTACH, got from checkProc */
  int         mode;      /* what to to with the generated hashvalue */
  Tcl_Channel readDest;  /* channel to write the hash of read    data to (mode = TRF_WRITE_HASH) */
  Tcl_Channel writeDest; /* channel to write the hash of written data to (mode = TRF_WRITE_HASH) */
  char*       matchFlag; /* Name of global variable to write the matchresult into (TRF_ABSORB_HASH)
			  */
  Tcl_Interp* mfInterp;  /* Interpreter containing the variable named in 'matchFlag' */
} TrfMDOptionBlock;

#define TRF_IMMEDIATE (1)
#define TRF_ATTACH    (2)

#define TRF_ABSORB_HASH (1)
#define TRF_WRITE_HASH  (2)

EXTERN Trf_OptionVectors*
TrfMDOptions _ANSI_ARGS_ ((void));



/*
 * Definition of option information for blockciphers and accessor
 * to set of vectors processing these.
 */

typedef struct _TrfBlockcipherOptionBlock {
  int   direction;      /* encryption, decryption (TRF_ENCRYPT / TRF_DECRYPT) */
  int   operation_mode; /* ECB, CBC, CFB, OFB */
  int   shift_width;    /* shift per operation for feedback modes.
			 * given in bytes. */
  int   key_length;     /* length of key (required for ciphers with
			 * variable keysize) */
  VOID* key;            /* key data */
  VOID* iv;             /* initialization vector for stream modes. */

  /* ---- derived information ----
   *
   * Area used for communication between vectors
   *
   *	Trf_TypeDefinition.encoder.create,
   *	Trf_TypeDefinition.decoder.create
   *
   * to avoid duplicate execution of complex and/or common operations.
   */

  int eks_length;
  int dks_length;

  VOID* encrypt_keyschedule; /* expansion of key into subkeys for encryption */
  VOID* decrypt_keyschedule; /* expansion of key into subkeys for decryption */

} TrfBlockcipherOptionBlock;

#define TRF_ECB_MODE (1)
#define TRF_CBC_MODE (2)
#define TRF_CFB_MODE (3)
#define TRF_OFB_MODE (4)

EXTERN Trf_OptionVectors*
TrfBlockcipherOptions _ANSI_ARGS_ ((void));


/*
 * Definition of option information for ciphers and accessor
 * to set of vectors processing these.
 */

typedef struct _TrfCipherOptionBlock {
  int   direction;      /* encryption, decryption */
  int   key_length;     /* length of key (required for ciphers with
			 * variable keysize) */
  VOID* key;            /* key data */

  /* ---- derived information ----
   *
   * Area used for communication between vectors
   *
   *	Trf_TypeDefinition.encoder.create,
   *	Trf_TypeDefinition.decoder.create
   *
   * to avoid duplicate execution of complex and/or common operations.
   */

  int eks_length;
  int dks_length;

  VOID* encrypt_keyschedule; /* expansion of key into subkeys for encryption */
  VOID* decrypt_keyschedule; /* expansion of key into subkeys for decryption */

} TrfCipherOptionBlock;


EXTERN Trf_OptionVectors*
TrfCipherOptions _ANSI_ARGS_ ((void));


/*
 * Definition of option information for ZIP compressor
 * + accessor to set of vectors processing them
 */

typedef struct _TrfZipOptionBlock {
  int mode;   /* compressor mode: compress/decompress */
  int level;  /* compression level (1..9, -1 = default) */
} TrfZipOptionBlock;

EXTERN Trf_OptionVectors*
TrfZIPOptions _ANSI_ARGS_ ((void));

#define TRF_COMPRESS   (1)
#define TRF_DECOMPRESS (2)

#define TRF_MIN_LEVEL      (1)
#define TRF_MAX_LEVEL      (9)
#define TRF_DEFAULT_LEVEL (-1)

#define TRF_MIN_LEVEL_STR "1"
#define TRF_MAX_LEVEL_STR "9"

/*
 * Internal initialization procedures for all transformers implemented here.
 */


EXTERN int TrfInit_Bin      _ANSI_ARGS_ ((Tcl_Interp* interp));
EXTERN int TrfInit_Oct      _ANSI_ARGS_ ((Tcl_Interp* interp));
EXTERN int TrfInit_Hex      _ANSI_ARGS_ ((Tcl_Interp* interp));
EXTERN int TrfInit_UU       _ANSI_ARGS_ ((Tcl_Interp* interp));
EXTERN int TrfInit_B64      _ANSI_ARGS_ ((Tcl_Interp* interp));
EXTERN int TrfInit_Ascii85  _ANSI_ARGS_ ((Tcl_Interp* interp));

EXTERN int TrfInit_CRC      _ANSI_ARGS_ ((Tcl_Interp* interp));
EXTERN int TrfInit_MD5      _ANSI_ARGS_ ((Tcl_Interp* interp));
EXTERN int TrfInit_HAVAL    _ANSI_ARGS_ ((Tcl_Interp* interp));
EXTERN int TrfInit_SHA      _ANSI_ARGS_ ((Tcl_Interp* interp));
EXTERN int TrfInit_ADLER    _ANSI_ARGS_ ((Tcl_Interp* interp));
EXTERN int TrfInit_CRC_ZLIB _ANSI_ARGS_ ((Tcl_Interp* interp));

EXTERN int TrfInit_IDEA     _ANSI_ARGS_ ((Tcl_Interp* interp));
EXTERN int TrfInit_BLOWFISH _ANSI_ARGS_ ((Tcl_Interp* interp));
EXTERN int TrfInit_DES      _ANSI_ARGS_ ((Tcl_Interp* interp));
EXTERN int TrfInit_RC4      _ANSI_ARGS_ ((Tcl_Interp* interp));

EXTERN int TrfInit_RS_ECC   _ANSI_ARGS_ ((Tcl_Interp* interp));
EXTERN int TrfInit_ZIP      _ANSI_ARGS_ ((Tcl_Interp* interp));

#endif /* TRF_INT_H */
