.\"----------------------------------------------------------------------------
.\" Much of this manual was copied from the SIPP manual page which is:
.\" Copyright Jonas Yngvesson, Inge Wallin
.\" This program and documentation may be distributed freely under
.\" the terms of GNU GENERAL PUBLIC LICENSE.
.\" It is provided "as is" without express or implied warranty.
.\"----------------------------------------------------------------------------
.\" Parts of this manual that were not taken from the SIPP manual page are:
.\" Copyright 1992-1995 Mark Diekhans
.\" Permission to use, copy, modify, and distribute this software and its
.\" documentation for any purpose and without fee is hereby granted, provided
.\" that the above copyright notice appear in all copies.  Mark Diekhans makes
.\" no representations about the suitability of this software for any purpose.
.\" It is provided "as is" without express or implied warranty.
.\"----------------------------------------------------------------------------
.\" $Id: tsipp.n,v 5.22 1995/09/16 01:44:54 markd Exp $
.\"----------------------------------------------------------------------------
.\"
.TH "tsipp" 1 "" "SIPP Tcl Interface"
.ad b
.SH NAME
tsipp, tksipp - 3D image specification and rendering toolkit for use with Tcl
and Tk.
'
.SH "INTRODUCTION"
'\"@help: tsipp/Intro/Intro
'\"@brief: Introduction to TSIPP.
.PP
\fBTSIPP\fR (\fBTSIPP\fR) is a 3D image specification and rendering toolkit
for use with \fBTcl\fR and \fBTk\fR.
It provides a Tcl command interface to SIPP, the SImple Polygon Processor
library.  This provides a command interface for creating 3-dimensional scenes
and rendering them using a scanline z-buffer algorithm.  The Tcl interpretive
programming language is used to provide a powerful, yet simple interface to
this library.  The scene may be written to either a PPM format file, as
defined by the PBMPlus toolkit or a RLE file, as defined by the Utah Raster
Toolkit.
.PP
An interface to render to a Tk photo image is also
provided.  Events such as button presses may take place while rendering is
in progress.
'
'\"@endhelp
'
.SH "ON-LINE HELP"
.PP
Online help is available on TSIPP using the Extended Tcl help facility.
This can be accessed in two ways.
The command
.sp
.nf
.ft CW
   \fBtsipphelp\fR
.ft R
.fi
.sp
is a graphical interface to the help pages for \fBTSIPP\fR, \fBTcl\fR and
\fBTk\fR.
.sp
You may also access the help pages with the \fBTclX\fR command \fBhelp\fR,
used when running the \fBtsipp\fR or \fBtksipp\fR interpreters interactively.
This gives line oriented help on \fBTSIPP\fR, \fBTcl\fR and \fBTk\fR.
.sp
See the Extended Tcl manual page for details of the help facility
'
.sp 2.SH INVOCATION
'\"@help: tsipp/Intro/running
'\"@brief: Running tsipp and tksipp
.PP
TSIPP follows the calling conventions of the Tcl shell provided with
Extended Tcl.  The command line is:
.sp
.nf
.ft CW
    \fBtsipp\fR [-q] [[-f] \fIscript\fB]|[-c \fIcommand\fB] [\fIargs\fR]
.ft R
.fi
.sp
See the TclX manual page for details of the arguments and flags.
.PP
A version of the Tk wish shell containing TSIPP
may be invoked with:
.sp
.nf
.ft CW
    \fBtksipp\fR [\fIfilename\fR] [\fIoptions\fR]
.ft R
.fi
.sp
See the Tk wish manual page for details of the options.
'\"@endhelp
'
.sp 2
.SH "DATA TYPES AND RESOURCES"
'\"@help: tsipp/Intro/data-struct
'\"@brief: Data structures used by TSIPP.
.PP
Various data types are defined by Tcl SIPP.  They are all built as Tcl
lists, so standard list operations can be used to construct and access
them.
The "tcl_precision" variable is used when formatting all floating
point numbers.  The value of this variable is set to the Tcl defined
maximum less one when TSIPP is initialized.  In many cases, the precision
can be reduce for a slight increase in performance without altering the
generation of images.
The following data structures are used by TSIPP:
.IP
o \fIcolor\fR - A list containing the red, green and blue values for the
surface.  Each being a number between 0.0 and 1.0.
.IP
o \fIopacity\fR - Describes the opacity of a surface. 0.0 means totally
transparent, 1.0 means totally opaque.  May be a list of red, green and blue
opacity values or a single value to be used for all color channels.
.IP
o \fIvertex\fR - A vertex is defined by a Tcl list of 3 floating point numbers
in the form: {x\ y\ z}.
.IP
o \fIvertex/texture pair\fR - A vertex and corresponding texture coordinates
are defined by a Tcl list of two list, each having 3 floating point numbers in
the form: {{x\ y\ z}\ {u\ v\ w}}.
.IP
o \fIvector\fR - A vector is defined by a Tcl list of 3 floating point numbers
in the form: {x\ y\ z}.
.IP
o \fIboolean\fR - Boolean arguments may be specified as `0' or `1', `true' or
`false', `t' or `f' (upper or lower case).  Only `0' or `1' are valid
in Tcl expressions and commands that return booleans only return `0' or `1'.
.IP
o \fIangle\fR - An angle may be expressed in degrees or radians.  If the
number is prefixed with an "R", then it is taken as radians, if it is prefixed
with "D", then it is taken as degrees. (e.g. 1.2, R1.2, D10.5).  If no prefix
is specified, some commands assume degrees, some assume radians.  See the
documentation of the specific command.
.IP
o \fImatrix\fR - A 4x4 transformation matrix is represented as a list of 4
elements, each element being a row.  Each of the row elements is a list of 4
floating point numbers, representing the columns.  The values in column 4 must
be 0, 0, 0, 1.  For example:
.sp
.nf
.ft CW
        {{1.2  1.4  0.1  0}
         {2.1  2.3  1.6  0}
         {2.4  6.1  2.6  0}
         {2.9  1.1  3.6  1}}
.ft R
.fi
.PP
Be careful, many commands take lists of the above data types, which are 
themselves lists.
'
'\"@endhelp
'
'\"@help: tsipp/Intro/handles
'\"@brief: Handles used to reference SIPP resources.
.PP
Various resources are created during the definition of a scene, these resources
are referenced in the Tcl code with handles.  A handle is similar to a file
id number returned by the \fBopen\fR system call.  It is an abstract entity
that provides a reference to a resource that is internally managed.   The
following handles for the following resources are provided:
.sp
.IP
o \fBcamera\fR - A camera which specifies a point of view.
The handle "\fBSTDCAMERA\fR" is predefined camera located at (0.0, 0.0 10.0),
looking at the origin, with the world y-axis as the up direction
and a focal factor of 0.25.
.IP
o \fBlight\fR - A light source or spot light.
.IP
o \fBsurface\fR - Surfaces that are constructed from polygons that have been
pushed on the polygon stack.
.IP
o \fBshader\fR - A shader defines a shading algorithm and surface attributes
required by the shader.  This handle can be used to define the way surfaces
will be shaded, including their color and attributes.
.IP
o \fBobject\fR - Objects defined in the SIPP internal database. A special
handle `\fBWORLD\fR' is used to specify the world object.
.IP
o \fBpbmfile\fR - A PBMPlus Portable Pixel Map (PPM) or Portable Bit Map (PBM)
file.
.IP
o \fBrlefile\fR - Utah Raster Toolkit RLE files.
.IP
o \fBpixmap\fR - Memory resident pixel or bit map.
.PP
When a unreference operation is performed on a handle, a reference count is
decremented, the resource is not actually deleted unless the reference count
goes to zero.  The handle itself is deleted.  Thus an object is not released
until its is handle is unreferenced and it is no longer a subobject of any
object.  This behavior applies to objects, surfaces, shaders and pixmaps.
'\"@endhelp
'
.sp 2
.SH "IMAGE STORAGE AND DISPLAY"
'\"@help: tsipp/Intro/images
'\"@brief: Image storage and display.
.PP
Images can be rendered to RLE and PBMPlus files.
They can also be held in pixmaps and displayed using the a Tk photo image.
Collectively they are referred to as \fIimage store\fR.
Image store is referenced by handles in various commands.
Two attributes affect operations done on image storage, if they can be read,
written or both and their scanning direction (top to botton, botton to top, or
random).
.PP
Scanning direction determines if image store is accessed from top to bottom
scan line, from bottom to top or randomly.
File objects can only be accessed in one direction.
.PP
The following are the supported image store types and their attributes:
.sp
.RS 5
o RLE files - read/write, bottom to top access.
.sp
o PBMPlus files - readonly, top to bottom access.
.sp
o Tk photo images - read/write, random access.
.sp
o pixmaps - read/write, random access.
.RE
'\"@endhelp
'
.sp 2
.SH "ERRORS"
'\"@help: tsipp/Intro/errors
'\"@brief: Error handling.
.PP
Errors in TSIPP are handled via the standard Tcl error mechanism.
This allows the programmer to trap errors with \fBcatch\fR or receive
a procedure stack tracing of where the error occurred.  Often a error is
caused by an invalid specification of an object or surface. These errors
are generally not caught until rendering, manifesting themselves as
a floating point exception or a math function domain errors.  Probably
the best solution when one of these errors occurs is to remove objects
from the scene or surfaces from objects until the error disappears.
'\"@endhelp
'
.sp 2
.SH "POLYGON AND SURFACE CREATION COMMANDS"
.PP
These commands provide for the definitions of polygons and the creation of
surfaces from the polygons.  There is a polygon stack for defining surfaces.
The surfaces are referenced by surface handles.
'
'\"@help: tsipp/surface/PolygonPush
'\"@brief: Create a polygon and push it onto the polygon stack.
.TP
\fBSippPolygonPush \fR[\fB-flags\fR] \fIvertexList\fR
.br
Create a polygon from a list of vertices and push it onto the polygon stack.
The following flags may be specified:
.sp
.RS 10
o \fB-norm\fR - User normal vectors are supplied.
.sp
o \fB-nonorm\fR - User normal vectors are not supplied. This is the default.
.sp
o \fB-tex\fR - Texture coordinates are supplied.  The behavior of texture
coordinates depends on the shader.
.sp
o \fB-notex\fR - Texture coordinates are not supplied.  This is the default.
.sp
o \fB-clock\fR - Vertices are in the list in \fIclockwise\fR order when looking
at the front face of the polygon.
.sp
o \fB-counter\fR - Vertices are in the list in \fIcounterclockwise\fR order
when looking at the front face of the polygon.  This is the default.
.RE
.\" RS required so we don't lose .TP indentation.
.RS 7
.sp
The vertices must be pushed on the vertex stack in either
\fIcounterclockwise\fR (the default) or \fIclockwise\fR order when looking at
the "front" face of the polygon.  Otherwise the front of the surface will be
defined in the wrong direction.  
.sp
If neither \fB-tex\fR or \fB-norm\fR are specified, then the list
contains just vertices, each element being in the form "x\ y\ z".
.sp
If \fB-norm\fR is specified, then the list
contains both vertices and normal vectors, each element being in the form
"{x\ y\ z}\ {nx\ ny\ nx}".
.sp
If \fB-tex\fR is specified, then the list
contains both vertices and texture coordinates, each element being in the form
"{x\ y\ z}\ {u\ v\ w}".
.sp
If both \fB-tex\fR and \fB-norm\fR are specified, then the list
contains both vertices, normal vectors and texture coordinates, each
element being in the form
"{x\ y\ z}\ {nx\ ny\ nz}\ {u\ v\ w}".
.sp
If a vertex in the polygon is already defined in a previous polygon
that belongs to the same surface, the same vertex will be referenced,
i.e. vertices shared between polygons are only stored once, but they
must be repeated when defining the polygons.
.RE
'\"@endhelp
'
'\"@help: tsipp/surface/PolygonClear
'\"@brief: Clear the polygon stack.
.TP
\fBSippPolygonClear\fR
.br
Create the polygon stack.  Normally used for cleaning up when an error occurs
defining and pushing polygons.  Not used during a normal surface creation.
If the stack is empty, this is a no-op.
'\"@endhelp
'
'\"@help: tsipp/surface/SurfaceCreate
'\"@brief: Create a surface from the polygons on the polygon stack.
.TP
\fBSippSurfaceCreate \fIshaderhandle\fR
.br
Create a surface from the polygons on the polygon stack. A handle
to the newly-created surface is returned. The polygon stack is empty
afterwards. \fIshaderhandle\fR is a handle to the shader, which specifies
a shading algorithm and surface description. \fBSippObjectSurfaceCreate\fR
can also be used to create surfaces and added them to an object without
setting up a surface handle.
See the section \fBSHADER COMMANDS\fR.
'\"@endhelp
'
'\"@help: tsipp/surface/SurfaceUnref
'\"@brief: Remove a handle's reference to a surface and delete the handle.
.TP
\fBSippSurfaceUnref \fIsurfacelist\fR|\fBALL\fR
.br
Delete the handles in the list \fIsurfacelist\fR which are associated with
surfaces.  This lowers the reference count on the surface.
The data associated with the surfaces will not be deleted until the
objects that reference them are deleted.
If \fBALL\fR is specified, then all surface handles are unreferenced.
'\"@endhelp
'
'\"@help: tsipp/surface/SurfaceSetShader
'\"@brief: Set the shader on specified surfaces.
.TP
\fBSippSurfaceSetShader \fIsurfacelist shaderhandle\fR
.br
Set the all of the surfaces in the list \fIsurfacelist\fR to be shaded with
the the shader \fIshaderhandle\fR.
'\"@endhelp
'
.sp 2
.SH "SHADER COMMANDS"
.P
These commands creates a surface shading description, which consists of a
shading algorithm specification and surface attributes.  They returns a 
shader handle that can be passed to other commands.
'
'\"@help: tsipp/shaders/ShaderBasic
'\"@brief: Define a basic shader.
.TP
\fBSippShaderBasic \fIambient specular c3 color \fR[\fIopacity\fR]
.br
Create a basic shader, which is a modified and simplified version of Blinn's
shading model.
.sp
\fIambient\fR is a value between 0.0 and 1.0 which determines how much of the
base color of a surface that is visible when it is not illuminated by any
light source.
.sp
\fIspecular\fR is a value between 0.0 and 1.0 which is the fraction of
color specularly reflected.
.sp
\fIc3\fR is the shininess of the surface where 0.0 is the most dull and 1.0
is the most shiny.
.sp
\fIcolor\fR is a list containing the red, green and blue values for the
surface.  Each being a number between 0.0 and 1.0.
.sp
\fIopacity\fR describes the opacity of a surface. 0.0 means totally
transparent, 1.0 means totally opaque.  May be a list of red, green and blue
opacity values or a single value to be used for all color channels.  Optional,
default is 1.0.
'\"@endhelp
'
'\"@help: tsipp/shaders/ShaderPhong
'\"@brief: Define a Phong shader.
.TP
\fBSippShaderPhong \fIambient diffuse specular spec_exp color \fR[\fIopacity\fR]
.br
Create a shader that implements the Phong illumination model.
.sp
\fIambient\fR is a value between 0.0 and 1.0 which determines how much of the
base color of a surface that is visible when it is not illuminated by any
light source.
.sp
\fIdiffuse\fR is a value between 0.0 and 1.0 specifying how much
light that is reflected diffusely from the surface.
.sp
\fIspecular\fR is a value between 0.0 and 1.0 which is the fraction of
color specularly reflected.
.sp
\fIspec_exp\fR is the exponent in the specular highlight calculation.
It specifies how "shiny" the surface is. Useful values are about 1 to
200, where 1 is a rather dull surface and 200 is a very shiny one.
.sp
\fIcolor\fR is a list containing the red, green and blue values for the
surface.  Each being a number between 0.0 and 1.0.
.sp
\fIopacity\fR describes the opacity of a surface. 0.0 means totally
transparent, 1.0 means totally opaque.  May be a list of red, green and blue
opacity values or a single value to be used for all color channels.  Optional,
default is 1.0.
'\"@endhelp
'
'\"@help: tsipp/shaders/ShaderStrauss
'\"@brief: Define a Strauss intuitive shader.
.TP
\fBSippShaderStrauss \fIambient smoothness metalness color \fR[\fIopacity\fR]
.br
NOTE: This shader is currently has a serious bug.  As smoothness and metalness
get larger, the object gets darker.  This will be fixed in a future release.
.br
This is an implementation of a shader described by Paul
Strauss in IEEE CG&A Nov. 1990. In his article he explains that most shading
models in use today, i.e. Phong, Cook-Torrance, are difficult to use for
non-experts, and this for several reasons. The parameters and their effect on
a surface are non-intuitive and/or complicated. The shading model Strauss
designed has parameters that is easy to grasp and have a reasonably
deterministic effect on a surface, but yet produces very realistic results.
.sp
\fIambient\fR is a value between 0 and 1 which determines how much of the base
color of a surface that is visible when it is not illuminated by any
light source.
.sp
\fIsmoothness\fR is a value between 0 and 1 that describes how smooth the
surface is. This parameter controls both diffuse and specular reflections. 0
means a dull surface while 1 means a very smooth and shiny one.
.sp
\fImetalness\fR is also a value between 0 and 1. It describes how metallic the
material is. It controls among other things how much of the surface color
should be mixed into the specular reflections at different angles. 0 means a
non-metal while 1 means a very metallic surface.
.sp
\fIcolor\fR is a list containing the red, green and blue values for the
surface.  Each being a number between 0.0 and 1.0.
.sp
\fIopacity\fR describes the opacity of a surface. 0.0 means totally
transparent, 1.0 means totally opaque.  May be a list of red, green and blue
opacity values or a single value to be used for all color channels.  Optional,
default is 1.0.
'\"@endhelp
'
'\"@help: tsipp/shaders/ShaderWood
'\"@brief: Define a simulated wood shader.
.TP
.nf
\fBSippShaderWood \fIambient specular c3 scale basecolor ringcolor \fR[\fIopacity\fR]
.fi
.br
This shader creates a simulated wood texture on a surface.
It uses two colors, one as the base (often lighter) color of the wood
and one as the color of the (often darker) rings in it.
The rings is put into the base color about the x-axis and are then distorted 
using \fInoise()\fR and \fIturbulence()\fR. A similar pattern is repeated at
regular intervals to create an illusion of logs or boards.
.sp
\fIambient\fR is a value between 0.0 and 1.0 which determines how much of the
base color of a surface that is visible when it is not illuminated by any
light source.
.sp
\fIspecular\fR is a value between 0.0 and 1.0 which is the fraction of
color specularly reflected.
.sp
\fIc3\fR is the shininess of the surface where 0.0 is the most dull and 1.0
is the most shiny.
.sp
\fIscale\fR describes how much the texture coordinates should be scaled
before applying the wood texture.  The larger the scaling, the smaller
the area of the wood texture associated with a given area of the surface.
A large number results in a very fine pattern, a small number yields
a coarse pattern.  You will have to experiment some to get this right.
.sp
\fIbasecolor\fR is a list containing the red, green and blue values for the
wood base.  Each being a number between 0.0 and 1.0.
.sp
\fIringcolor\fR is a list containing the red, green and blue values for the
wood rings.
.sp
\fIopacity\fR describes the opacity of a surface. 0.0 means totally
transparent, 1.0 means totally opaque.  May be a list of red, green and blue
opacity values or a single value to be used for all color channels.  Optional,
default is 1.0.
'\"@endhelp
'
'\"@help: tsipp/shaders/ShaderMarble
'\"@brief: Define a simulated marble shader.
.TP
.nf
\fBSippShaderMarble \fIambient specular c3 scale basecolor stripcolor \fR[\fIopacity\fR]
.fi
.br
This shader creates a simulated marble texture on a surface.
It uses two colors, one as the base material and one as the
interspersed material. The interspersed material is put into the
base material in strips that are distorted using \fInoise()\fR and
\fIturbulence()\fR. 
.sp
\fIambient\fR is a value between 0.0 and 1.0 which determines how much of the
base color of a surface that is visible when it is not illuminated by any
light source.
.sp
\fIspecular\fR is a value between 0.0 and 1.0 which is the fraction of
color specularly reflected.
.sp
\fIc3\fR is the shininess of the surface where 0.0 is the most dull and 1.0
is the most shiny.
.sp
\fIscale\fR describes how much the texture coordinates should be scaled
before applying the marble texture.  The larger the scaling, the smaller
the area of the marble texture associated with a given area of the surface.
A large number results in a very fine pattern, a small number yields
a coarse pattern.
.sp
\fIbasecolor\fR is a list containing the red, green and blue values for the
base material.  Each being a number between 0.0 and 1.0.
.sp
\fIstripcolor\fR is a list containing the red, green and blue values for the
interspersed material.
.sp
\fIopacity\fR describes the opacity of a surface. 0.0 means totally
transparent, 1.0 means totally opaque.  May be a list of red, green and blue
opacity values or a single value to be used for all color channels.  Optional,
default is 1.0.
'\"@endhelp
'
'\"@help: tsipp/shaders/ShaderGranite
'\"@brief: Define a simulated granite shader.
.TP
.nf
\fBSippShaderGranite \fIambient specular c3 scale color1 color2 \fR[\fIopacity\fR]
.fi
.br
This shader is very similar to marble shader in that it also mixes two colors
using \fInoise()\fR and \fIturbulence()\fR. The difference is in how the
mixing is done. The two colors are mixed without treating them separately in
any way.
.sp
\fIambient\fR is a value between 0.0 and 1.0 which determines how much of the
base color of a surface that is visible when it is not illuminated by any
light source.
.sp
\fIspecular\fR is a value between 0.0 and 1.0 which is the fraction of
color specularly reflected.
.sp
\fIc3\fR is the shininess of the surface where 0.0 is the most dull and 1.0
is the most shiny.
.sp
\fIscale\fR describes how much the texture coordinates should be scaled
before applying the granite texture.  The larger the scaling, the smaller
the area of the granite texture associated with a given area of the surface.
A large number results in a very fine pattern, a small number yields
a coarse pattern.
.sp
\fIcolor1\fR and \fIcolor2\fR is a list containing the red, green and blue
values for the base material.  Each being a number between 0.0 and 1.0.
.sp
\fIopacity\fR describes the opacity of a surface. 0.0 means totally
transparent, 1.0 means totally opaque.  May be a list of red, green and blue
opacity values or a single value to be used for all color channels.  Optional,
default is 1.0.
'\"@endhelp
'
'\"@help: tsipp/shaders/ShaderBozo
'\"@brief: Define a shader that uses noise() to chose colors.
.TP
.nf
\fBSippShaderBozo \fIambient specular c3 scale colorlist\fR [\fIopacity\fR]
.fi
.br
This shader uses \fInoise()\fR to chose a color from a fixed set.  The range
of possible return value from \fInoise()\fR are divided into parts of equal
size and each part is assigned a color. The size of the parts are dependent on
the number of colors.
.sp
\fIambient\fR is a value between 0.0 and 1.0 which determines how much of the
base color of a surface that is visible when it is not illuminated by any
light source.
.sp
\fIspecular\fR is a value between 0.0 and 1.0 which is the fraction of
color specularly reflected.
.sp
\fIc3\fR is the shininess of the surface where 0.0 is the most dull and 1.0
is the most shiny.
.sp
\fIscale\fR describes how much the texture coordinates should be scaled before
applying the texture.  The larger the scaling, the smaller the area of the
texture associated with a given area of the surface.  A large number results
in a very fine pattern, a small number yields a coarse pattern.
.sp
\fIcolorlist\fR is a list contain one or more colors.  Each color is a list
containing the red, green and blue values for the surface.  Each being a
number between 0.0 and 1.0.
.sp
\fIopacity\fR describes the opacity of a surface. 0.0 means totally
transparent, 1.0 means totally opaque.  May be a list of red, green and blue
opacity values or a single value to be used for all color channels.  Optional,
default is 1.0.
'\"@endhelp
'
'\"@help: tsipp/shaders/ShaderBumpy
'\"@brief: Define a shader that adds bumps and holes to other shaders.
.TP
.nf
\fBSippShaderBumpy \fIshaderhandle scale\fR [\fBBUMPS\fR] [\fBHOLES\fR]
.fi
.br
This shader is a function that perturbates the normal of a
surface using \fInoise()\fR. Any other shader can be used to do the final
shading calculations.
.sp
\fIshaderhandle\fR is a handle to another shader and surface specification.
.sp
\fIscale\fR describes how much the texture coordinates should be scaled before
applying the texture.  The larger the scaling, the smaller the area of the
texture associated with a given area of the surface.  A large number results
in a very fine pattern, a small number yields a coarse pattern.
.sp
\fBBUMPS\fR indicates that only bumps standing out from the surface are
visible, the rest of the surface is smooth.
.sp
\fBHOLES\fR indicates that only bumps going into the surface are visible,
the rest of the surface is smooth.  If neither \fBHOLES\fR or \fBBUMPS\fR
or specified, then both are assumed to be on.
'\"@endhelp
'
'\"@help: tsipp/shaders/ShaderPlanet
'\"@brief: Define a shader that creates a simulated planet surface.
.TP
.nf
\fBSippShaderPlanet \fIambient specular c3 \fR[\fIopacity\fR]
.fi
.br
This shader is a somewhat specialized shader that produces a texture
that resembles a planet surface. The planet is of the Tellus type with a
mixture of oceans and continents. Some of the surface is covered by
semi-transparent clouds which enhances the effect greatly. On the other hand,
no polar caps are provided and this decreases the realism.
.sp
The texture is 3-dimensional, so it is possible to create cube planets or even
planets with cut-out parts that still have surfaces that resemble the earth
surface. The texture is not scalable, and is designed to be used with texture
coordinates in the range -1.0 to 1.0, e.g. a unit sphere. Of course the world
coordinates need not have the same order of magnitude.
.sp
\fIambient\fR is a value between 0.0 and 1.0 which determines how much of the
base color of a surface that is visible when it is not illuminated by any
light source.
.sp
\fIspecular\fR is a value between 0.0 and 1.0 which is the fraction of
color specularly reflected.
.sp
\fIc3\fR is the shininess of the surface where 0.0 is the most dull and 1.0
is the most shiny.
.sp
\fIopacity\fR describes the opacity of a surface. 0.0 means totally
transparent, 1.0 means totally opaque.  May be a list of red, green and blue
opacity values or a single value to be used for all color channels.  Optional,
default is 1.0.
'\"@endhelp
'
'\"@help: tsipp/shaders/ShaderUnref
'\"@brief: Remove a handle's reference to a shader and delete the handle.
.TP
\fBSippShaderUnref \fIshaderlist\fR|\fBALL\fR
.br
Delete the handles in the list \fIshaderlist\fR which are associated with
shaders.  This lowers the reference count on the shaders.
The data associated with the shaders will not be deleted until the
surfaces that reference them are deleted.
If \fBALL\fR is specified, then all shaders handles are unreferenced.
'\"@endhelp
'
.sp 2
.SH "OBJECT MANIPULATION COMMANDS"
.PP
These commands are provided for creating and manipulating objects.  An 
object is referenced by an object handle.
'\"@help: tsipp/objects/ObjectCreate
'\"@brief: Create an empty object.
.TP
\fBSippObjectCreate\fR
.br
Create an empty object, i.e. an object with no surfaces or subobjects
in it. The transformation matrix in the new object will be a identity
matrix initially.  A handle to the object is returned which is passed to
the other object commands to specify this object. Object must be added as a
subobject of the `\fBworld\fR' object or some other object to be become part
of the scene.
'\"@endhelp
'
'\"@help: tsipp/objects/ObjectUnref
'\"@brief: Remove a handle's reference to an object and delete the handle.
.TP
\fBSippObjectUnref \fIobjectlist\fR|\fBALL\fR
.br
Delete the handles in the list \fIobjectlist\fR which are associated with
objects.  This lowers the reference count on the objects.
The data associated with the object will not be deleted until the
objects that reference them are deleted.
.sp
If \fBALL\fR is specified, then all object handles are deleted.  The only
objects that will be left are those that are subobjects of \fBWORLD\fR.  To
delete those objects, decrement their reference counts with the command:
.sp
.nf
.ft CW
    SippObjectSubSubobj WORLD ALL
.ft R
.fi
'\"@endhelp
'
'\"@help: tsipp/objects/ObjectInstance
'\"@brief: Create a new instance of a previously defined object.
.TP
\fBSippObjectInstance \fR[\fI-flags\fR] \fIobjecthandle\fR
.br
Create a new instance of a previously defined object referenced by
\fIobjecthandle\fR. The lists of surfaces and subobjects in the object are not
copied, but a new reference with its own transformation matrix is created.
.sp
If the object is changed, i.e. if one of its subobjects or surfaces are
transformed, one is deleted or added, the change will also be seen in the
copy.
.sp
The following optional flags may be specified:
.sp
.RS 10
o \fB-cleartransf\fR - Set the object's transformation matrix to the identity
matrix.  Default.
.sp
o \fB-keeptransf\fR - Don't reset the object's transformation matrix
.RE
.\" RS required so we don't lose .TP indentation.
.RS 7
.sp
Clearing the transformation matrix has the affect of undoing all rotate, move
and scale operations on the object.  This repositions the object to the world
center, which often makes moving the object to another location easier.  The
transformations on subobjects are not cleared, leaving the structure of
composite objects intact.
.RE
'\"@endhelp
'
'\"@help: tsipp/objects/ObjectDup
'\"@brief: Copy recursively an object and its subobjects.
.TP
\fBSippObjectDup \fR[\fI-flags\fR] \fIobjecthandle\fR
.br
Copy recursively an object and its subobjects. The surfaces in the object tree
are not copied, only new references to them are made.
.sp
The following optional flags may be specified:
.sp
.RS 10
o \fB-cleartransf\fR - Set the object's transformation matrix to the identity
matrix.  Default.
.sp
o \fB-keeptransf\fR - Don't reset the object's transformation matrix
.RE
.\" RS required so we don't lose .TP indentation.
.RS 7
.sp
Clearing the transformation matrix has the affect of undoing all rotate, move
and scale operations on the object.  This repositions the object to the world
center, which often makes moving the object to another location easier.  The
transformations on subobjects are not cleared, leaving the structure of
composite objects intact.
.RE
'\"@endhelp
'
'\"@help: tsipp/objects/ObjectDeepDup
'\"@brief: Copy the entire tree of an object.
.TP
\fBSippObjectDeepDup \fR[\fI-flags\fR] \fIobjecthandle\fR
.br
Copy the entire tree of an object, including subobjects and all surfaces,
polygons and vertices. This is a costly operation if the object is complex.
.sp
The following optional flags may be specified:
.sp
.RS 10
o \fB-cleartransf\fR - Set the object's transformation matrix to the identity
matrix.  Default.
.sp
o \fB-keeptransf\fR - Don't reset the object's transformation matrix
.RE
.\" RS required so we don't lose .TP indentation.
.RS 7
.sp
Clearing the transformation matrix has the affect of undoing all rotate, move
and scale operations on the object.  This repositions the object to the world
center, which often makes moving the object to another location easier.  The
transformations on subobjects are not cleared, leaving the structure of
composite objects intact.
.RE
'\"@endhelp
'
'\"@help: tsipp/objects/ObjectAddSurface
'\"@brief: Add surfaces to an object.
.TP
\fBSippObjectAddSurface \fIobjecthandle surfacelist\fR
.br
Add the surfaces in the list \fIsurfacelist\fR to the specified object.
'\"@endhelp
'
'\"@help: tsipp/objects/ObjectSurfaceCreate
'\"@brief: Create a surface from the polygon stack and add it to an object
.TP
\fBSippObjectSurfaceCreate \fIobjectHandle shaderhandle\fR
.br
Create a surface from the polygons on the polygon stack.  The polygon is added
to the object specified by \fIobjectHandle\fR, no surface handle is returned.
The polygon stack is empty afterwards. \fIShaderhandle\fR is a handle to the
shader, which specifies a shading algorithm and surface description.
See the section \fBSHADER COMMANDS\fR.
'\"@endhelp
'
'\"@help: tsipp/objects/ObjectSubSurface
'\"@brief: Subtract surfaces from an object.
.TP
\fBSippObjectSubSurface \fIobjecthandle surfacelist\fR
.br
Remove (subtract) the surfaces in the list \fIsurfacelist\fR from the
specified object.
'\"@endhelp
'
'\"@help: tsipp/objects/ObjectAddSubobj
'\"@brief: Add subobjects to an specified object.
.TP
\fBSippObjectAddSubobj \fR[\fI-flag\fR] \fIobjecthandle subobjlist\fR
.br
Add the objects in the list \fIsubobjlist\fR to the specified object.
One of the following optional flags may be specified:
.sp
.RS 10
o \fB-check\fR - Validate that adding the subobjects will not cause the
object tree to become circular.  This is the default.
.sp
o \fB-nocheck\fR - Don't check for building a circular object tree. This
option is faster, but will cause a core dump when rendering or deleting
objects if a circular tree has been built.
.RE
'\"@endhelp
'
'\"@help: tsipp/objects/ObjectSubSubobj
'\"@brief: Subtract subobjects from a object.
.TP
\fBSippObjectSubSubobj \fIobjecthandle subobjlist\fR|\fBALL\fR
.br
Remove (subtract) the objects in the list \fIsubobjlist\fR from the specified
object.  If \fBALL\fR is specified, all subobjects of the specified are
deleted.
'\"@endhelp
'
'\"@help: tsipp/objects/ObjectGetTransf
'\"@brief: Return the transformation matrix for an object.
.TP
\fBSippObjectGetTransf \fIobjecthandle\fR
.br
Return the transformation matrix currently stored in the object specified 
object.
'\"@endhelp
'
'\"@help: tsipp/objects/ObjectSetTransf
'\"@brief: Set the transformation matrix of an object.
.TP
\fBSippObjectSetTransf \fIobject matrix\fR
.br
Set the transformation matrix of the specified object to \fImatrix\fR.
'\"@endhelp
'
'\"@help: tsipp/objects/ObjectClearTransf
'\"@brief: Set the transformation matrix of an object to the unit matrix.
.TP
\fBSippObjectClearTransf \fIobject\fR
.br
Set the transformation matrix of the specified object to the unit matrix.
'\"@endhelp
'
'\"@help: tsipp/objects/ObjectTransform
'\"@brief: Post multiply a matrix into the transformation matrix of an object.
.TP
\fBSippObjectTransform \fIobject matrix\fR
.br
Post multiply specified matrix into the transformation matrix of \fIobject\fR.
'\"@endhelp
'
'\"@help: tsipp/objects/ObjectRotateX
'\"@brief: Rotate an object about the X axis.
.TP
\fBSippObjectRotateX \fIobject angle\fR
.br
Rotate the \fIobject\fR \fIangle\fR radians or degrees about the X axis.
If \fIangle\fR is prefixed with "R", then it is in radians.  If it
is prefixed with "D", then it is in degrees.  If no prefix is supplied, then
radians is assumed.
'\"@endhelp
'
'\"@help: tsipp/objects/ObjectRotateY
'\"@brief: Rotate an object about the Y axis.
.TP
\fBSippObjectRotateY \fIobject angle\fR
.br
Rotate the \fIobject\fR \fIangle\fR radians or degrees about the Y axis.
If \fIangle\fR is prefixed with "R", then it is in radians.  If it
is prefixed with "D", then it is in degrees.  If no prefix is supplied, then
radians is assumed.
'\"@endhelp
'
'\"@help: tsipp/objects/ObjectRotateZ
'\"@brief: Rotate an object about the Z axis.
.TP
\fBSippObjectRotateZ \fIobject angle\fR
.br
Rotate the \fIobject\fR \fIangle\fR radians or degrees about the Z axis.
If \fIangle\fR is prefixed with "R", then it is in radians.  If it
is prefixed with "D", then it is in degrees.  If no prefix is supplied, then
radians is assumed.
'\"@endhelp
'
'\"@help: tsipp/objects/ObjectRotate
'\"@brief: Rotate an object about an arbitrary line.
.TP
\fBSippObjectRotate \fIobjecthandle point vector angle\fR
.br
Rotate the \fIobject\fR \fIangle\fR radians or degrees about the 
line given by \fIpoint\fR and \fIvector\fR starting in
that point.
If \fIangle\fR is prefixed with "R", then it is in radians.  If it
is prefixed with "D", then it is in degrees.  If no prefix is supplied, then
radians is assumed.
'\"@endhelp
'
'\"@help: tsipp/objects/ObjectScale
'\"@brief: Scale an object.
.TP
\fBSippObjectScale \fIobjecthandle factor\fR|{\fIxfactor yfactor zfactor\fR}
.br
Scale the specified object with the scaling factors \fIxfactor,\
yfactor\fR and \fIzfactor\fR in the main directions respectively.  If only 
\fIfactor\fR is specified, it is used as the scaling factors in all directions.
'\"@endhelp
'
'\"@help: tsipp/objects/ObjectMove
'\"@brief: Move (translate) an object.
.TP
\fBSippObjectMove \fIobjecthandle {\fIxdist ydist zdist\fR}
.br
Move (translate) the specified  \fIxdist, \fIydist\fR and \fIzdist\fR in the
three main directions, respectively.
'\"@endhelp
'
.sp 2
.SH "PRIMITIVE OBJECT CREATION COMMANDS"
.PP
These commands create primitive objects of specific shapes.
'
'
'\"@help: tsipp/primitives/Texture-Map
'\"@brief: Texture mapping algorithms for primitives.
.P
Each object primitive which can be created in SIPP has an argument
that describes what kind of texture coordinates should be assigned to
the surface of the object. This parameter can have one of the following
predefined values:
'
.sp
.RS 10
o \fBNATURAL\fR - This value tell SIPP to use a two dimensional mapping which
is "natural" for this particular object. It might be one of the other
available mappings or it might be something unique for the object.  The
description of the functions for creating the individual objects specifies how
this mapping is done.
.sp
o \fBCYLINDRICAL\fR - A two dimensional mapping. The coordinates are assigned
as if the object were projected on a cylinder surrounding the object and
centered on the z-axis object. The coordinates are mapped so that `x' goes
from 0 to 1 around the base of the cylinder and `y' goes from 0 to 1 from
bottom to top on it.
.sp
o \fBSPHERICAL\fR - Same as `CYLINDRICAL', but the object are projected on a
sphere surrounding it instead.
.sp
o \fBWORLD\fR - A three dimensional mapping. The texture coordinates are the
same three dimensional coordinates as the world coordinates of the object at
creation time.
.RE
'\"@endhelp
'
'\"@help: tsipp/primitives/Torus
'\"@brief: Create a torus object.
.TP
.nf
\fBSippTorus \fIbigradius smallradius radialres tuberes shaderhandle\fR [\fItexture\fR]
.fi
.br
Create a torus centered about the origin and with the z-axis pointing up
through the ring.  The surface is shaded with the algorithm and surface
parameters specified by \fIshaderhandle\fR.  \fIbigradius\fR is the radius of
the ring and \fIsmallradius\fR is the radius of the "tube" itself.
\fIradialres\fR is the number of polygons that will be created radially around
the ring and \fItuberes\fR is the number of polygons that will be created
around the tube.
.sp
\fITexture\fR is the texture mapping algorithm to use for the object.
The \fBNATURAL\fR texture mapping is a two dimensional mapping with the
\fIx\fR coordinate going around the \fIsmall\fR circle and the \fIy\fR
coordinate going around the \fIlarge\fR circle.
Optional, \fBWORLD\fR is the default.
'\"@:.sp
'\"@:See the help file tsipp/primitives/Texture-Map for more details.
.sp
An object handle that references the new object is returned.
.br
'\"@endhelp
'
'\"@help: tsipp/primitives/Cone
'\"@brief: Create a cone object.
.TP
.nf
\fBSippCone \fIbottomradius topradius length resolution shaderhandle\fR [\fItexture\fR]
.fi
.br
Create a cone centered about the origin and with the z-axis along the cones
main axis.  The surface is shaded with the algorithm and surface parameters
specified by \fIshaderhandle\fR.  \fIbottomradius\fR and \fItopradius\fR
determines the radius of the cone in its bottom and top, if both of these are
non zero a truncated cone is created. \fIlength\fR defines the length of the
cone.  \fIresolution\fR is the number of polygons that will be created
radially around the rim.
.sp
\fITexture\fR is the texture mapping algorithm to use for the object.
The \fBNATURAL\fR texture mapping is \fBCYLINDRICAL\fR.
Optional, \fBWORLD\fR is the default.
'\"@:.sp
'\"@:See the help file tsipp/primitives/Texture-Map for more details.
.sp
An object handle that references the new object is returned.
'\"@endhelp
'
'\"@help: tsipp/primitives/Cylinder
'\"@brief: Create a cylinder object.
.TP
\fBSippCylinder \fIradius length resolution shaderhandle\fR [\fItexture\fR]
.br
Create a cylinder centered about the origin and with the z-axis along the
cylinders main axis.  The surface is shaded with the algorithm and surface
parameters specified by \fIshaderhandle\fR.  \fIradius\fR and \fIlength\fR
defines the size of the cylinder. \fIresolution\fR is the number of polygons
that will be created radially around the rim.
.sp
\fITexture\fR is the texture mapping algorithm to use for the object.
The \fBNATURAL\fR texture mapping is \fBCYLINDRICAL\fR.
Optional, \fBWORLD\fR is the default.
'\"@:.sp
'\"@:See the help file tsipp/primitives/Texture-Map for more details.
.sp
An object handle that references the new object is returned.
'\"@endhelp
'
'\"@help: tsipp/primitives/Ellipsoid
'\"@brief: Create a ellipsoid object.
.TP
.nf
\fBSippEllipsoid {\fIxradius yradius zradius\fR} \fIresolution\fR \fIshaderhandle\fR [\fItexture\fR]
.fi
.br
Create a ellipsoid centered about the origin.
The surface is shaded with the algorithm and surface parameters specified
by \fIshaderhandle\fR.
\fIxradius\fR, \fIyradius\fR and \fIzradius\fR
defines the size of the ellipsoid. \fIresolution\fR
is the number of polygons that will be created around it's "equator".
.sp
\fITexture\fR is the texture mapping algorithm to use for the object.
The \fBNATURAL\fR texture mapping is \fBSPHERICAL\fR.
Optional, \fBWORLD\fR is the default.
'\"@:.sp
'\"@:See the help file tsipp/primitives/Texture-Map for more details.
.sp
An object handle that references the new object is returned.
'\"@endhelp
'
'\"@help: tsipp/primitives/Sphere
'\"@brief: Create a sphere object.
.TP
\fBSippSphere \fIradius resolution shaderhandle\fR [\fItexture\fR]
.br
Create a sphere centered about the origin.
The surface is shaded with the algorithm and surface parameters specified
by \fIshaderhandle\fR.
\fIradius\fR defines the size of the sphere. \fIresolution\fR
is the number of polygons that will be created around it's "equator".
.sp
\fITexture\fR is the texture mapping algorithm to use for the object.
The \fBNATURAL\fR texture mapping is \fBSPHERICAL\fR.
Optional, \fBWORLD\fR is the default.
'\"@:.sp
'\"@:See the help file tsipp/primitives/Texture-Map for more details.
.sp
An object handle that references the new object is returned.
'\"@endhelp
'
'\"@help: tsipp/primitives/Prism
'\"@brief: Create a prism object.
.TP
\fBSippPrism \fI2dpointlist length shaderhandle\fR [\fItexture\fR]
.br
Creates a prism defined by the polygon in \fI2dpointlist\fR.
The surface is shaded with the algorithm and surface parameters specified
by \fIshaderhandle\fR.
\fI2dpointlist\fR is a list of lists of x and y pairs.
The prism will
ascend and descend equally far (\fIlength\fR / 2.0) from the x-y plane
along the z axis.
.sp
\fITexture\fR is the texture mapping algorithm to use for the object.
The \fBNATURAL\fR texture mapping is similar to \fBCYLINDRICAL\fR but the
\fIx\fR coordinate is not taken from projection on a cylinder but is evenly
distributed around the perimeter. An odd thing in all the 2D mappings
(all except \fBWORLD\fR) for the prism is that the top face will have
texture coordinates (0.0, 1.0) while the bottom will get (0.0, 0.0).
Optional, \fBWORLD\fR is the default.
'\"@:.sp
'\"@:See the help file tsipp/primitives/Texture-Map for more details.
.sp
An object handle that references the new object is returned.
'\"@endhelp
'
'\"@help: tsipp/primitives/Block
'\"@brief: Create a block object.
.TP
.nf
\fBSippBlock {\fIxsize ysize zsize\fR} \fIshaderhandle\fR [\fItexture\fR]
.fi
.br
Creates a square block with the size defined by \fIxsize\fR, \fIzsize\fR and
\fIzsize\fR.  The surface is shaded with the algorithm and surface parameters
specified by \fIshaderhandle\fR.  The block is centered about the origin.
.sp
\fITexture\fR is the texture mapping algorithm to use for the object.
The \fBNATURAL\fR texture mapping is similar to \fBCYLINDRICAL\fR but the
\fIx\fR coordinate is not taken from projection on a cylinder but is evenly
distributed around the perimeter. An odd thing in all the 2D mappings
(all except \fBWORLD\fR) for the block is that the top face will have
texture coordinates (0.0, 1.0) while the bottom will get (0.0, 0.0).
Optional, \fBWORLD\fR is the default.
'\"@:.sp
'\"@:See the help file tsipp/primitives/Texture-Map for more details.
.sp
An object handle that references the new object is returned.
'\"@endhelp
'
'\"@help: tsipp/primitives/Cube
'\"@brief: Create a cube object.
.TP
.nf
\fBSippCube \fIsize shaderhandle\fR [\fItexture\fR]
.fi
.br
Create a cube with the side of length \fIside\fR.  The surface is shaded with
the algorithm and surface parameters specified by \fIshaderhandle\fR.  The
cube is centered about the origin. \fIsurface\fR is the surface description
used by \fIshader()\fR which is the shading function used when shading the
cube.
.sp
\fITexture\fR is the texture mapping algorithm to use for the object.
The \fBNATURAL\fR texture mapping is similar to \fBCYLINDRICAL\fR but the
`x' coordinate is not taken from projection on a cylinder but is evenly
distributed around the perimeter. An odd thing in all the 2D mappings
(all except \fBWORLD\fR) for the cube is that the top face will have
texture coordinates (0.0, 1.0) while the bottom will get (0.0, 0.0).
Optional, \fBWORLD\fR is the default.
'\"@:.sp
'\"@:See the help file tsipp/primitives/Texture-Map for more details.
.sp
An object handle that references the new object is returned.
'\"@endhelp
'
'\"@help: tsipp/primitives/Teapot
'\"@brief: Define a Utah Teapot object.
.TP
.nf
\fBSippTeapot \fIresolution shaderhandle\fR [\fItexture\fR]
.fi
.br
This function creates a model of the famous "Utah Teapot" as a TSIPP
object. The model is built as a combination of four subobjects: the
body, the lid, the handle and the spout. These subobjects are also
available as separate primitive objects.
.sp
The body and the lid are created as Bezier rotation curves while the
handle and the spout are created as sets of four Bezier patches each.
Texture coordinates and resolution are assigned in the same way as for
these primitives.
.sp
\fIResolution\fR is the number of polygons to tessellate the surface into.
Each rotational surface will be tessellated into
`\fIresolution\fR x 4*\fIresolution\fR' polygons,
`\fIresolution\fR' vertices along the curve and `4*\fIresolution\fR' vertices
around the perimeter.
The teapot body is built from three rotational surfaces and the lid from two.
.sp
Patches will be tessellated into `\fIresolution\fR x \fIresolution\fR'
polygons. Both the handle and the spout are built from four patches.
.sp
The surfaces are shaded with the algorithm and surface parameters
specified by \fIshaderhandle\fR.
.sp
\fITexture\fR is the texture mapping algorithm to use for the object.
Only \fBNATURAL\fR or \fBWORLD\fR maybe specified.
Optional, \fBWORLD\fR is the default.
'\"@:.sp
'\"@:See the help file tsipp/primitives/Texture-Map for more details.
.sp
The following commands are available for defining subobject of a teapot.
The arguments are the same as for the \fBSippTeapot\fR command.
.sp
.RS 10
.nf
o \fBSippTeapotBody\fR - Teapot body object
o \fBSippTeapotLid\fR - Teapot lib object
o \fBSippTeapotHandle\fR - Teapot handle object
o \fBSippTeapotSpout\fR - Teapot spout object
.fi
.RE
'\"@endhelp
'
'\"@help: tsipp/primitives/VW
'\"@brief: Define an Evans & Sutherland VW object or its surfaces.
.TP
\fBSippVW \fIshader\fR
.br
This Tcl procedure defines an Evans & Sutherland Volkswagen object with
\fIshader\fR as the shader for its surfaces.  It returns an object handle.
This is really only the body of a VW, no other parts are defined.  Also only
the outside polygons are defined, but its a fun object with a special place in
the history of computer graphics.
.br
The VW has larger dimensions than the other pre-defined objects.  The default
orientation in world space is the front of the car facing down the Y axis and
the roof of the car facing positive Z.
.br
It is also possible to independently define the individual surfaces that make
up the VW.  This is useful if you wish to use different shaders for different
surfaces.  Tcl procedures in the form:
.sp
    \fBSippVW\fIxxxx side shader\fR
.sp
are used to define each surface.  Each call defines the surface for one side of
the car.  Each procedure must be called twice to product the entire body.
\fISide\fR indicates which side of the care to create the surface for, it is
one of \fBLEFT\fR of \fBRIGHT\fR. \fIShader\fR is the shader to use for the
surface.  A surface handle is returned.  The following procedures are
available to define the surfaces:
.sp
.RS 10
.nf
o \fBSippVWBackFender\fR - Back fender.
o \fBSippVWBack\fR - Back (trunk).
o \fBSippVWBackSide\fR - Back side panel.
o \fBSippVWFrontDoor\fR - Front door.
o \fBSippVWFrontFender\fR - Front fender.
o \fBSippVWFrontSide\fR - Front side panel.
o \fBSippVWHood\fR - Hood.
o \fBSippVWFrontPanel\fR - Front (below windshield) panel.
o \fBSippVWRunningBoard\fR - Running board.
o \fBSippVWRoof\fR - Roof.
.fi
.RE
'\"@endhelp
'
'\"@help: tsipp/primitives/BezierCurve
'\"@brief: Define an object using a Bezier curve rotated about the z-axis.
.TP
.nf
\fBSippBezierCurve \fIresolution vertexlist curvelist shaderhandle\fR [\fItexture\fR]
.fi
Define an object using a Bezier curve rotated about the z-axis.
\fIResolution\fR is the number of polygons to tessellate the surface into.
\fIVertexlist\fR is a list of vertices of the curve.  Each element of the list
is a list contain the X, Y and Z vertices.
.sp
\fICurvelist\fR a list of indices into \fIvertexlist\fR that defined the curve.
Each element of the list contains a list of 4 indices. These indices are zero
based. For example:
.sp
.nf
.ft CW
    {{0 1 2 3}
     {4 4 5 7}
     {6 7 8 9}}
.ft R
.fi
.sp
\fIshaderhandle\fRis handle to a previously created surface shading
specification.
.sp
\fITexture\fR is the texture mapping algorithm to use for the object.
The texture coordinates are a bit special for these surfaces.
\fBSPHERICAL\fR and \fBCYLINDRICAL\fR mappings are not applicable, and
\fBNATURAL\fR mapping will apply to the piece of surface created by each
Bezier curve separately. The \fBNATURAL\fR mapping uses the curve parameter
u along each curve as \fIx\fR coordinate and goes from 0 to 1 around the
perimeter of the rotational surface on the other axis
Optional, \fBWORLD\fR is the default.
'\"@:.sp
'\"@:See the help file tsipp/primitives/Texture-Map for more details.
.sp
An object handle is returned.
'\"@endhelp
'
'\"@help: tsipp/primitives/BezierPatch
'\"@brief: Define an object using a Bezier patch.
.TP
.nf
\fBSippBezierPatch \fIresolution vertexlist patchlist shaderhandle\fR [\fItexture\fR]
.fi
Define an object using a Bezier patch.
\fIresolution\fR is the  number of polygons to tessellate the surface into.
\fIVertexlist\fR is a list of vertices of the curve.  Each element of the list
is a list contain the X, Y and Z vertices.
.sp
\fIPatchlist\fR is list of indices into \fIvertexlist\fR that defined the
patch.  Each element of the list contains a list 4 lists.  Each of these lists
contain 4 indices.  The indices are zero based.  For example:
.sp
.nf
.ft CW
    {{ 0  1  2  3} {35 36 37 38} {28 29 30 31} {21 22 23 24}}
    {{21 22 23 24} {14 15 16 17} { 7  8  9 10} { 0  1  2  3}}
    {{ 3  4  5  6} {38 39 40 41} {31 32 33 34} {24 25 26 27}}
    {{ 6  5  4  3} {13 12 11 10} {20 19 18 17} {27 26 25 24}}}
.ft R
.fi
.sp
\fIShaderhandle\fR is handle to a previously created surface shading
specification.
.sp
\fITexture\fR is the texture mapping algorithm to use for the object.
'\"@:.sp
'\"@:See the help file tsipp/primitives/Texture-Map for more details.
The texture coordinates are a bit special for the Bezier patches.
\fBCYLINDRICAL\fR and \fBSPHERICAL\fR coordinates are not applicable, if they
are specified, SIPP will use \fBNATURAL\fR anyway. The \fBNATURAL\fR mapping is
a two dimensional mapping using the surface parameters u and v, see
figure below. Note that these parameters range from 0 to 1 within each
patch!
Optional, \fBWORLD\fR is the default.
.sp
An object handle is returned.
'\"@endhelp
'
'\"@help: tsipp/primitives/BezierFile
'\"@brief: Create an object for a Bezier curve or patch specified in a data file.
.TP
\fBSippBezierFile \fIfileid resolution shaderhandle\fR [\fItexture\fR]
.br
Read a file specified by \fIfileid\fR, containing descriptions of a set of
Bezier patches or Bezier curves. The surface is shaded with the algorithm and
surface parameters specified by \fIshaderhandle\fR.  See the \fBsipp
primitives\fR manual page for a description of this file format.
The \fB SippBezierCurve\fR and \fBSippBezierPatch\fR commands are the preferred
method of creating Bezier surfaces, this procedure is provided for
compatibility with existing SIPP data files.
.sp
If the file contains descriptions of patches, these patches will be tesselated
into \fIresolution\fR by \fIresolution\fR polygons and installed in a
\fBsipp\fR object as one surface.
.sp
If the file contains descriptions of curves, a rotational surface will be
created by rotating these curves about the y-axis. The resulting surface will
be tesselated into polygons and installed in a \fBsipp\fR object as one
surface.
.sp
\fITexture\fR is the texture mapping algorithm to use for the object.
The texture coordinates are a bit special for the Bezier patches.
\fBCYLINDRICAL\fR and \fBSPHERICAL\fR coordinates are not applicable, if they
are specified, SIPP will use \fBNATURAL\fR anyway. The \fBNATURAL\fR mapping is
a two dimensional mapping using the surface parameters u and v, see
figure below. Note that these parameters range from 0 to 1 within each
patch!
Optional, \fBWORLD\fR is the default.
'\"@:.sp
'\"@:See the help file tsipp/primitives/Texture-Map for more details.
.sp
An object handle that references the new object is returned.
'\"@endhelp
'
.sp 2
.SH "LIGHT SOURCE AND VIEWING COMMANDS"
These commands specify the light sources and the viewing parameters.
'
'\"@help: tsipp/lights/LightSourceCreate
'\"@brief: Create a new lightsource in the scene.
.TP
\fBSippLightSourceCreate \fR{\fIx y z\fR} \fIcolor \fR[\fItype]\fR
.br
Create a new lightsource in the scene.
If a directional lightsource is created these \fR{\fIx\ y\ z\fR} specifies a
vector in the direction that the light emanates
the lightsource.  If it is a point lightsource they specify
the exact location of it. \fIColor\fR specifies the color and intensity of 
the light.  \fIType\fR is either `\fBDIRECTION\fR' or `\fBPOINT\fR', if it
is omitted, `\fBDIRECTION\fR' is assumed.  A light
handle is returned by the command.
'\"@endhelp
'
'\"@help: tsipp/lights/SpotLightCreate
'\"@brief: Create a new spotlight in the scene.
.TP
.nf
\fBSippSpotLightCreate \fIposition point opening color type shadow\fR
.fi
.br
Create a new spotlight in the scene.  \fIPosition\fR is the coordinates of the
spotlight.  \fIPoint\fR is a point at which the spotlight is pointing. It is
in the middle of the lightcone. 
\fIOpening\fR is the opening angle of the lightcone. The cone
defined will be completely lit, a soft edged lightcone will start to
blend out outside this angle.  If the number is
prefixed with an "R", then it is taken as radians, if it is prefixed with "D",
then it is taken as radians.  If not prefix, then degrees is assumed.
\fIColor\fR is The color of the emitted light.
\fIType\fR is the type of spotlight to create. Should be either "SHARP" or
"SOFT".  \fIShadow\fR is a boolean that indicates if light from the
spotlight will be able to cast shadows.  Whether shadows actually are cast or
not depend on which value set by the \fBSippShadows\fR command.  A light
handle is returned by the command.
'\"@endhelp
'
'\"@help: tsipp/lights/LightDestruct
'\"@brief: Delete lightsources or spotlights.
.TP
\fBSippLightDestruct \fIlightlist\fR|\fBALL\fR
.br
Delete lightsources or spotlights. \fILightlist\fR is a list of light handles
to destruct.  If \fBALL\fR is specified, then all lights are deleted.
'\"@endhelp
'
'\"@help: tsipp/lights/LightSourcePut
'\"@brief: Modify the direction or position of a lightsource.
.TP
\fBSippLightSourcePut \fIlight \fR{\fIx y z\fR}
.br
This command is used to modify the direction, or position, of a
lightsource. If the light is a directional light, then \fIx, y, z\fR
are a vector in the direction that the light emanates.
If the light is a point lightsource, then \fIx, y, z\fR are the coordinates
of the new location of a light.  
'\"@endhelp
'
'\"@help: tsipp/lights/SpotLightPos
'\"@brief: Modify the position of a spotlight.
.TP
\fBSippSpotLightPos \fIlight \fR{\fIx y z\fR}
.br
Modify the position of a spotlight. \fIX, y, z\fR are the coordinates
of the new location of a light.  
'\"@endhelp
'
'\"@help: tsipp/lights/SpotLightAt
'\"@brief: Modify the position a spotlight is pointing at.
.TP
\fBSippSpotLightAt \fIlight \fR{\fIx y z\fR}
.br
Modify the position a spotlight is pointing at. \fIX, y, z\fR are the
coordinates of the  point that the light should point at.
'\"@endhelp
'
'\"@help: tsipp/lights/SpotLightOpening
'\"@brief: Modify the opening angle of the light cone of a spotlight.
.TP
\fBSippSpotLightOpening \fIlight opening\fR
.br
Modify the opening angle of the lightcone of a spotlight. 
\fIOpening\fR is the opening angle of the lightcone. The cone
defined will be completely lit, a soft edged lightcone will start to
blend out outside this angle.  If the number is
prefixed with an "R", then it is taken as radians, if it is prefixed with "D",
then it is taken as radians.  If not prefix, then degrees is assumed.
'\"@endhelp
'
'\"@help: tsipp/lights/SpotLightShadows
'\"@brief: Turn shadow casting on or off for a spotlight. 
.TP
\fBSippSpotLightShadows \fIlight flag\fR
.br
Turn shadow casting on or off for a specific spotlight. 
\fIShadow\fR is a boolean that indicates if light from the
spotlight will be able to cast shadows.
'\"@endhelp
'
'\"@help: tsipp/lights/LightColor
'\"@brief: Change the color of a lightsource or a spotlight.
.TP
\fBSippLightColor \fIlight color\fR
.br
Change the color of the emitted light from a lightsource or a spotlight.
fIColor\fR is the new color of the light.
'\"@endhelp
'
'\"@help: tsipp/lights/LightActive
'\"@brief: Turn a lightsource or a spotlight on or off.
.TP
\fBSippLightActive\fI light flag\fR
.br
Turn a lightsource or a spotlight on or off. \fIFlag\fR is boolean indicating
if a light should be active.
'\"@endhelp
'
'\"@help: tsipp/view/CameraCreate
'\"@brief: Create a new camera.
.TP
\fBSippCameraCreate \fR[\fIposition\fR] [\fIpoint\fR] [\fIupvector\fR] [\fIfocal\fR]
.br
Create a new camera. \fIPositon\fR is the point to world coordinates of the
camera. \fIPoint\fR is a world corrdinates point that he camera is to look at.
\fIUpvector\fR is a vector defining the up direction.  The only constraint on
this vector is that it must not be parallel to the viewing direction.
.sp
\fIFocal\fR the focal ratio of the "camera". This is the ratio between the
distance from the viewpoint to the screen and half the screen height.  A large
focal factor will result in a wide angle view while a small factor will give a
telescopic effect.
.sp
.nf
.ft CW
                                screen
                                |
                                | s
    viewpoint                   |
        *-----------------------|
                    d           |
                                |
                                |

        focal = s / d
.ft R
.fi
.sp
If any of the parameters is omitted or has a empty value, then that parameter
is defaulted.  A camera handle is returned.  If a camera is not created, the
default camera, \fBSTDCAMERA\fR, is used.
'\"@endhelp
'
'\"@help: tsipp/view/CameraDestruct
'\"@brief: Delete cameras.
.TP
\fBSippCameraDestruct \fIcameralist\fR|\fBALL\fR
.br
Delete one or more cameras.  \fICameralist\fR is a list of camera handles.  If
\fBALL\fR is specified, then all cameras are deleted (except for
\fBSTDCAMERA\fR).  If the current camera is deleted, \fBSTDCAMERA\fR becomes
the current camera.
\fBSTDCAMERA\fR may not be deleted.  
'\"@endhelp
'
'\"@help: tsipp/view/CameraParams
'\"@brief: Alter the parameters of a camera.
.TP
.nf
\fBSippCameraParams \fIcamera\fR \fR[\fIposition\fR] [\fIpoint\fR] [\fIupvector\fR] [\fIfocal\fR]
.fi
.br
Alter the parameters of a camera.  \fICamera\fR is the camera handle, all
other arguments are the same as for \fBSippCameraCreate\fR. If any of the
parameters is omitted or has a empty value, then that parameter is unchanged.
'\"@endhelp
'
'\"@help: tsipp/view/CameraUse
'\"@brief: Specify the camera to use as the current viewpoint.
.TP
\fBSippCameraUse \fIcamera\fR
Specify the camera to use as the current viewpoint.
'\"@endhelp
'
.sp 2
.SH "SHADOWS"
.PP
'\"@help: tsipp/shadows/Intro
'\"@brief: Introduction to SIPP shadow generation
.PP
SIPP creates shadows with a technique called depth maps.  A detailed
description of this technique can be found in the article Rendering
Antialiased Shadows with Depth Maps by Reeves, Salesin and Cook in the
Proceedings of SIGGRAPH 1987.
.PP
In principle, a depth map is generated for each light that should
cast shadows. The depth map is simply an image of the scene, as seen
from the light, but instead of a color we store the depth (Z-buffer
value) in each "pixel". The finished map will contain the distance to
the object closest to the light in each point.
.PP
When the scene is rendered we transform each point we are shading
into depth map coordinates and if it is further away from the light
than the value stored in the corresponding point in the depth map, the
point is in shadow. The actual implementation is of course a bit more
complicated with some sampling and filtering but we won't go into that.
.PP
The reason we describe this algorithm at all is that it is easier to
understand how to get good looking shadows and why shadows sometimes
look weird if one have an understanding of the underlying process.
.PP
First of all: The shadows are generated by sampling in the depth
maps.  Sampling usually means we are in danger of aliasing and this is
very true in our case. SIPP automatically fits the depth map for a
spotlight so that it covers all area lit by the spotlight's light cone.
If this area is large and the depth map
resolution is low, the shadows will get very jagged.
.PP
Also, if we have a large surface that is close to perpendicular to
the depth map plane, the depth map "pixels" will be projected as long
stripes on that surface, so even if the depth map resolution is high, a
shadow cast on such a surface will suffer from aliasing (be jagged).
.PP
So, if the edges of a shadow look weird, try increasing the size of
the depth map (the depth map size is set with `SippShadows').
If they still look weird, or you run out of memory,
try changing the position of the lightsource that generate the shadow.
After some tweaking it is usually possible to get fairly decent shadows.
.PP
There are two ways to generate depth maps in SIPP, either automatically
for each new rendering, or explicitly on a command which will then keep
the depth maps around until they are explicitly deleted.
'\"@:.sp
'\"@:See the help file tsipp/primitives/Texture-Map for more details.
'\"@endhelp
'
'\"@help: tsipp/shadows/Shadows
'\"@brief: Globally enable or disable shadows.
.TP
\fBSippShadows \fIflag\fR [\fIsize\fR]
.br
This command globally enables or disables the casting of shadows by objects.
If the boolean \fIflag\fR is true, then shadows will be cast by lights that
have shadows casting enabled.  \fISize\fR defines the size of the depth map
used to render the shadows.  As a rule of thumb one could say that the depth
maps should be at least as large as the image itself but this may vary from
case to case.  \fISize\fR is omitted when \fIflag\fR is false.
'\"@endhelp
'
'\"@help: tsipp/shadows/ShadowMapsCreate
'\"@brief: Explicitly create the depth maps for shadow generation.
.TP
\fBSippShadowMapsCreate \fIsize\fR
.br
   The argument \fIsize\fR determines the size of the generated depth maps,
they will be (\fIsize\fR x \fIsize\fR) "pixels". When a rendering is performed
after this function has been called, the generated depth maps are used
to create the shadows in the scene. The depth maps are not deleted
automatically afterwards. This is very useful if a static scene is
rendered several times, perhaps from different viewpoints, since the
time to generate depth maps is only spent once.
.sp
Another usage is to generate the depth maps before some of the
objects are installed in the scene, if it is certain that they will
never cast a shadow on any other object (a floor being a typical
example). The depth map generation will the be considerably faster
since fewer objects need to be rendered.  When the remaining objects
are installed, the rendering can be started and shadows is still cast
on them but not from them.
'\"@endhelp
'
'\"@help: tsipp/shadows/ShadowMapsDestruct
'\"@brief: Explicitly delete the depth maps for shadow generation.
.TP
\fBSippShadowMapsDestruct\fR
.br
Deletes the depth maps when they are not needed anymore, or the
scene has changed too much to use the same depth maps.
'\"@endhelp
'
.sp 2
.SH "RENDERING AND COPYING"
.PP
'\"@help: tsipp/output/Render
'\"@brief: Render a scene to one or more output objects.
.TP
.nf
\fBSippRender \fR[\fI-flags\fR] \fIoutputlist xsize ysize\fR [\fImode\fR] [\fIoversample\fR]
.fi
.br
This command does the rendering of a scene to one or more image stores.
The ability to render to multiple image stores allows an application to
render to a Tk photo image and save the image in a file at the same time, for
example.
.sp
The scanning direction determines which types of image stores may be rendered
together.
Output objects may be written from top to
bottom or from bottom to top. File objects are written in only one direction
or another, while memory resident objects can be written in either direction.
Image stores that are written in opposing directions may not be rendered at the
same time.  Pixmaps may be used as a buffer
to store images that can then be saved to other image stores with conflicting
scan-directions.
.sp
The image is created with the pixel dimensions \fIxsize\fR by \fIysize\fR.
\fIoutputlist\fR is a list of output target handles.
The image is written in the Portable Pixel Map (PPM) format unless \fImode\fR
is \fBLINE\fR, in which case it is written in the Portable Bit Map (PBM)
format.  fIMode\fR decides in which mode the image should be rendered and
should be one of \fBPHONG\fR, \fBGOURAUD\fR, \fBFLAT\fR or \fBLINE\fR.
The integer value \fIoversampling\fR controls how much oversampling should be
done for anti-aliasing. Each pixel will be the average of a \fIoversampling\fR
X \fIoversampling\fR matrix of subpixels.  \fIOversampling\fR is ignored if
mode is \fBLINE\fR.
.sp
The following optional flags may be specified:
.sp
.RS 10
o \fB-odd\fR - Render old scanlines, useful for creating interlaced movies.
(Not allowed for \fBLINE\fR mode).
.sp
o \fB-even\fR - Render even scanlines. (Not allowed for \fBLINE\fR
mode).
.sp
o \fB-both\fR - Render all scan lines.  This is the default, only one of
\fB-odd\fR, \fB-even\fR or \fB-both\fR may be specified.
.sp
o \fB-update \fIperiod\fR - Specified the updated period when this command is
used in a Tk application. This is the frequency at which Tk events, such as
button presses, are handled.  The approximate unit is the time it takes to
render one pixel.  Zero disables updates.  If not specified, period is the
size of one scanline.  The processing of pending signals will also occur at
this interval, which may be useful in non-Tk based applications.
.sp
o \fB-clear\fR - Causes images rendered to Tk photo images to be cleared
before rendering the image.   This is the default.
.sp
o \fB-noclear\fR - Causes images rendered to a Tk photo image to not be cleared
before rendering the image.  The contents of the previously rendered image
will remain until a scanline is overwriten by the new image.  This is most
useful for overwriting a \fBLINE\fR rendered image with fully rendered image.
Only one of \fB-clear\fR or \fB-noclear\fR may be specified.
.sp
o \fB-either\fR - Specifies that the image to be rendered either top down or
bottom up.  If no file image store overrides this specification, then
top-down is used.   This is the default.
.sp
o \fB-topdown\fR - Causes the image to be rendered from the top down.  It is
useful for specifying if scanlines appear in the a Tk photo image starting at
the top.  This may not be specified with a bottom-up image store.
.sp
o \fB-bottomup\fR - Causes the image to be rendered from the bottom up.  It
is useful for specifying if scanlines appear in a Tk photo image starting at
the bottom.  This may not be specified with a bottom-up image store.  Only
one of \fB-topdown\fR and \fB-bottomup\fR may be specified.
.RE
'\"@endhelp
'
'\"@help: tsipp/output/Copy
'\"@brief: Copy an image.
.TP
\fBSippCopy \fR[\fIflags\fR] \fIinputHandle outputHandle\fR
.br
This command copies a single image from one image store to 
another.   If the image store is capable of storing multiple images,
only one image is copied.  The input and output images can't have conflicting
scanning directions (i.e. can't copy an RLE file to a PBMPlus file).  Pixmaps
can be used to buffer copies between image stores with conflicting scanning
directions.
.sp
When copying between image stores that support image comments (RLE files and
pixmaps), comments from the input image are merged into the
output image comments. Comments in the output image override a comment by
the same name from the input image.  In RLE files, all comments are reset
after a writing the image has completed.  New comments can then be added
and another image copied to the RLE file.  With pixmaps, comments
are not reset after storing an image in it.  If a single pixmap
instance is reused, the comments should probably be cleared with
\fBSippCommentDelete\fR before adding new comments and storing a new image.
.sp
The following optional flags may be specified:
.sp
.RS 10
o \fB-clear\fR - Causes images copied to the a Tk photo image to be cleared
before copying the image.   This is the default.
.sp
o \fB-noclear\fR - Causes images copied to the a Tk photo image to not
be cleared before copying the image.
The contents of the previously rendered image
will remain until a scanline is overwritten by the new image.  This is most
useful for overwriting a \fBLINE\fR rendered image with fully rendered image.
Only one of \fB-clear\fR or \fB-noclear\fR may be specified.
.RE
'\"@endhelp
'
.sp 2
.SH "PBMPLUS FILE ACCESS COMMANDS"
.PP
These commands are used for creating and accessing files defined by the
PBMPlus toolkit, either PPM files for fully rendered images or PBM files for
line images. PBMPlus files are a class of image store.
The scanning direction is from top to bottom.  PBMPlus files can be
created in raw (binary) or plain (ASCII) formats.
Due to the weaknesses of the PBMPlus library, read access to these files is
not supported.
'
'\"@help: tsipp/pbm/PBMOpen
'\"@brief: Open or create a PBMPlus file.
.TP
\fBSippPBMOpen \fR[\fB-plain\fR|\fB-raw\fR] \fIfilename access\fR
.br
Open the PBMPlus file \fIfilename\fR.  PBM files are created if rendered in
\fBLINE\fR mode, PPM files are created for all other rendering modes.
\fIAccess\fR is any of the standard access modes allowed by the Tcl \fBopen\fR
command.  Read access is not currently supported.
The command returns a handle to the PBM file.  
\fIFilename\fR follows the full semantics of the file name
argument to the Tcl \fBopen\fR command.  Thus the PBM file may be sent to a
pipeline, which could be a viewer or post-processing program.
.br
If \fB-plain\fR is specified, a plain (ASCII) format file is created.
If \fB-raw\fR is specified, a raw (binary) format file is created.
If neither are specified, raw format is used.  
'\"These flags may not be specified when reading a file.
'\"@endhelp
'
'\"@help: tsipp/pbm/PBMClose
'\"@brief: Close a PBMPlus file.
.TP
\fBSippPBMClose \fIpbmhandle\fR
.br
Close the PBM file specified by \fIpbmhandle\fR.
'\"@endhelp
'
.sp 2
.SH "RLE FILE ACCESS COMMANDS"
.PP
These commands are used to creating, accessing and manipulate RLE files as
defined by the Utah Raster Toolkit.  RLE files are a class of image store.
The scanning direction is from bottom to top.  An RLE file 
can store multiple images.  If TSIPP was not linked with the RLE libraries,
these commands will return an error.
'
'\"@help: tsipp/rle/RLEOpen
'\"@brief: Open or create a RLE file.
.TP
\fBSippRLEOpen \fIfilename [mode]\fR
.br
Open the RLE file \fIfilename\fR.
\fIMode\fR is any of the standard modes allowed by the Tcl \fBopen\fR command.
\fIAccess\fR is any of the standard access modes allowed by the Tcl \fBopen\fR
command. If \fIaccess\fR is omitted, then read access is used.
\fIFilename\fR follows the full semantics of the file name argument to the
Tcl \fBopen\fR command.  Thus the RLE file may be sent to a pipeline, which
could be a viewer or post-processing program.  Thus to render an image directly
into the RLE X11 file viewer on a 256 color display:
.sp
.nf
.ft CW
    set rle [SippRLEOpen "|rlequant | getx11" "w"]
.ft R
.fi
.sp
'\"@endhelp
'
'\"@help: tsipp/rle/RLEClose
'\"@brief: Close a RLE file.
.TP
\fBSippRLEClose \fIrlehandle\fR
.br
Close the RLE file specified by \fIrlehandle\fR.
'\"@endhelp
'
'\"@help: tsipp/rle/RLEInfo
'\"@brief: Returns information about an RLE file.
.TP
\fBSippRLEInfo \fIrlehandle attribute\fR
.br
Return various pieces of information about an RLE file specified by
\fIrlehandle\fR.  The following values may be specified for \fIattribute\fR:
.sp
.RS 10
o \fBEOF\fR - Returns 1 if a file open for reading is at the end of file,
returns 0 if more images remain in the file
.sp
o \fBXSIZE\fR - Returns the number of rows in the current image.
.sp
o \fBYSIZE\fR - Returns the number of scanlines (columns) in the current image.
.RE
'\"@endhelp
'
.sp 2
.SH "PIXEL MAPS ACCESS COMMANDS"
.PP
These commands are used to creating, accessing and manipulate memory resident
image called pixmaps.  Pixmaps are a class of image store
that provide for memory buffering of rendered or inputed (copied) images.
Pixmaps can be scanned either top-down or bottom-up.  When rendering a line
image, the images are actually stored internally as bit maps rather than pixel
maps.
'
'\"@help: tsipp/pixmap/PixMapCreate
'\"@brief: Create a pixmap object.
.TP
\fBSippPixMapCreate\fR
.br
Create a new pixmap object.  A pixmap handle is returned.
.br
If a pixmap object is reused to store subsequent images, either
through rendering or copying, old comments will not be cleared.
Either delete and recreate the pixmap or delete all comments with
\fBSippCommentDelete\fR.
'\"@endhelp
'
'\"@help: tsipp/pixmap/PixMapUnref
'\"@brief: Remove a reference to a pixmap and delete the handle.
.TP
\fBSippPixMapUnref \fIpixmapList\fR
.br
Delete the handles in the list \fIpixmaplist\fR which are assocated with
images.  This lowers the reference count on the pixmaps.  The data associated
with the pixmaps will not be deleted until there are no more references. If
\fBALL\fR is specified, then all pixmap handles are unreferenced.
.br
(NOTE: Currently there is no other way to keep a reference on a pixmap
except throuh its handle. This command always deletes pixmaps.)
'\"@endhelp
'
'\"@help: tsipp/pixmap/PixMapInfo
'\"@brief: Returns information about a pixmap.
.TP
\fBSippPixMapInfo \fIpixmapHandle attribute\fR
.br
Return various pieces of information about an pixmap specified by
\fIpixmapHandle\fR.
The following values may be specified for \fIattribute\fR:
.sp
.RS 10
o \fBXSIZE\fR - Returns the number of rows in the current image.
.sp
o \fBYSIZE\fR - Returns the number of scanlines (columns) in the current image.
.RE
'\"@endhelp
'
.sp 2
.SH "IMAGE COMMENTS"
'\"@help: tsipp/Intro/comments
'\"@brief: Comments that can be saved with images.
.PP
Image comments are similar to environment variables. They have a name and
optional value and are saved along with an image.  They can be used to save
human-readable information or data for use by other programs.  RLE files
store comments in a manner were they are retrievable.   Comments may also
be added to pixmaps.  If a pixmap is then copied to
an RLE file, the comments are also copied.  The following comments are added
automatically by TSIPP:
.IP
o \fBTSIPP_VERSION\fR - The TSIPP version number.
.IP
o \fBTSIPP_IMAGE\fR - Data describing the image in the form of a Tcl list.
The list consists of the following elements:
.sp
.RS 10
o The rendering mode: FLAT, GOURAUD, PHONG or LINE.  UNKNOWN24 is used
for 24 bit images were the rendering mode is unknown.
.sp
o The oversampling factor used to render the image.
.sp
o BOTH, ODD or EVEN, indicates which scanlines are included in the
image.
.sp
o The background color, in the standard TSIPP format.
.sp
o The line color, in the standard TSIPP format.  This useful only
if LINE mode is specified.
.RE
.IP
o \fBHISTORY\fR - A variable containing the command line that was used to
render the image.  This is a standard RLE comment, all image filters add
their command lines to the comment, so it serves as a history of how the image
was processed.
.PP
The following comments are standard to the Utah Raster Toolkit.  They are not
added by TSIPP, but it is suggested that you use them.
.IP
o \fBimage_gamma\fR - The image gamma.  I have found that a value of 0.4
produces \fBgetx11\fR displays that are similar to the default photo image
displays.
.IP
o \fBimage_title\fR - A title for the image.
'\"@endhelp
'
'
'\"@help: tsipp/comments/CommentPut
'\"@brief: Set a comment in an image.
.TP
\fBSippCommentPut \fIhandle name\fR [\fIvalue\fR]
.br
Set a comment in the image store specified by \fIhandle\fR.
This sets a comment of name \fIname\fR to the string \fIvalue\fR.
If \fIvalue\fR is omitted, then the name is saved without an empty value.
.br
Random access image store, such as pixmaps, don't automatically clear comments
if reused to store subsequent images, either through rendering or copying,
Either delete and recreate the image store or delete all comments with
\fBSippCommentDelete\fR.
.br
If the image store does not support comments, a error is returned.
'\"@endhelp
'
'\"@help: tsipp/comments/CommentGet
'\"@brief: Return the value of comment from an image.
.TP
\fBSippCommentGet handle [name] [retvar | {}]
.br
Return the value of comment identified by \fIname\fR from the image store
specified by \fIhandle\fR.
If \fIretvar\fR is specified the command returns `\fB1\fR' if the comment is
found and `\fB0\fR' if it is not found.
The value of the comment is returned in the variable \fIretvar\fR, unless
\fIretvar\fR is `{}', in which the only the presence of the variable is
returned, but not value.
If \fIretvar\fR is not specified, the command returns the value of the comment
or an empty string if the comment has no value or is not.
If no arguments are specified, then a keyed list of all comments is returned.
.br
If the image store does not support comments, a error is returned.
'\"@endhelp
'
'\"@help: tsipp/comments/CommentDelete
'\"@brief: Delete a comment from an image store.
.TP
\fBSippCommentDelete \fIhandle \fR[\fIname\fR]
.br
Delete the comment \fIname\fR from the image store specified by
\fIpixmapHandle\fR.
If \fIname\fR is omitted, all comments associated with the image store are
deleted.
.br
If the image store does not support comments, a error is returned.
'\"@endhelp
'
.sp 2
.SH "MISCELLANEOUS COMMANDS"
.PP
A few more useful commands.
'
'\"@help: tsipp/misc/Background
'\"@brief: Set the background color of the image.
.TP
\fBSippBackground \fR[\fIcolor\fR]
.br
Set the background color of the image to \fIcolor\fR.  If \fIcolor\fR is not
specified, the the current background color is returned.
'\"@endhelp
'
'\"@help: tsipp/misc/Color
'\"@brief: Transform color values
.TP
\fBSippColor tox|tosipp \fR[\fIcolor\fR]
.br
This command does valious operations on colors.  The following subcommands
are available:
.IP
o \fBtox\fR - Convert a TSIPP color to an an X11 color in the form
"rgb:<red>/<green>/<blue>."  Where each of the color values is a 4 digit
hexadecimal number.
.IP
o \fBtosipp\fR - Convert a X color to a TSIPP color.  See the
\fBXLookupColor\fR manual page for a description of the valid X color
specifications.
.sp
This command is only available in tksipp.
'\"@endhelp
'
'\"@help: tsipp/misc/LineColor
'\"@brief: Set the color for lines in RLE line mode rendering.
.TP
\fBSippLineColor \fR[\fIcolor\fR]
.br
Set the color of lines to use in line mode rendering to \fIcolor\fR.  This
is only useful when rendering to RLE files, it is ignored for PBM files.
If \fIcolor\fR is not specified, the the current line color is returned.
'\"@endhelp
'
'\"@help: tsipp/misc/ShowBackFaces
'\"@brief:  Enable or disable backface culling or hidden line elimination.
.TP
\fBSippShowBackFaces \fR[\fIflag\fR]
.br
This command can be used if backface culling should not be performed. If the
boolean \fIflag\fR is true backfacing polygons are "inverted" (their surface
normal is negated) and then rendered. This is useful if an object description
does not use a consequent ordering of the vertices.  This also controls hidden
line elimination when doing line rendering.  The rendering step can be slowed
down considerably since, on the average, twice as many polygons need to be
rendered. The default is that backface culling should be performed (false).
If \fIflag\fR is not specified, the current value is returned.
'\"@endhelp
'
'\"@help: tsipp/misc/AbortRender
'\"@brief:  Abort a rendering that is in progress.
.TP
\fBSippAbortRender\fR
.br
Set a flag that will cause the current rendering to terminate.  This is
only useful when called during an event update while rendering in Tk.
If it is called while a rendering is not in process, nothing is done.
This is designed for implementing an abort button in a Tk application, it
can also be used in a signal handler.
'\"@endhelp
'
'\"@help: tsipp/misc/Info
'\"@brief:  Return various pieces of info about TSIPP.
.TP
\fBSippInfo \fIattribute\fR
.br
This command returns various pieces of information about TSIPP.  The
following values may be specified for \fIattribute\fR:
.sp
.RS 10
o \fBVERSION\fR - Returns the full TSIPP version.  This includes the SIPP
library version number, a suffix indicating the TSIPP interface version and
the TSIPP interface patch level (if its been patched).
.sp
o \fBSIPPVERSION\fR - Returns the SIPP library version number that TSIPP is
linked with.
.sp
o \fBTSIPPVERSION\fR - Returns the TSIPP interface version (the suffix in
the full version).
.sp
o \fBTSIPPPATCHLEVEL\fR - Returns the TSIPP interface patchlevel.
.sp
o \fBRLE\fR - Returns 1 if TSIPP is linked with the RLE libraries and can
product RLE files.  Returns 0 if RLE file support is not available.
.sp
o \fBRENDERING\fR - Returns 1 if a rendering is currently being performed, 0
otherwise.  Useful only for code that might run as results of event updates
during a rendering process.  Modifying SIPP state during a rendering is
prohibited.
.sp
o \fBTK\fR - Returns 1 if this version of TSIPP is linked with Tk
rendering and display support.
.RE
'\"@endhelp
'
'\"@help: tsipp/misc/Reset
'\"@brief: Reset the SIPP environment.
.TP
\fBSippReset\fR
.br
This Tcl procedure completely resets the SIPP environment.  All objects,
surfaces, shaders, lights and cameras are released.  The \fBSTDCAMERA\fR,
background and line color, and the show backfaces flag are all reset.  All
allocate TSIPP handles are no longer valid.
'\"@endhelp
'
.sp 2
.SH "SIPP IN A TK APPLICATION"
'\"@help: tsipp/tk/Intro
'\"@brief: Introduction to using TSIPP in Tk
.PP
The program \fBtksipp\fR is a Tk wish shell containing TSIPP
and additional support for render to Tk photo images.
.PP
During the rendering of a scene, events will be handled via the Tk update
mechanism.  The update period may be specified as an option to any of the
rendering commands.  The unit of the update period is approximately the
amount of time taken to render one pixel in an average scene.  The updates
themselves occur at points in the rendering other than the direct rendering
of the picture, this time unit is just a guideline.  Specifying
a smaller period makes an application more responsive, specifying a larger
period speeds the rendering.
.PP
SIPP commands may not be used by events that occur while rendering, they will
return an error if this is attempted.  The \fBSippInfo\fR command may be used
to determine if rendering is in progress.  An application should normally
disable any widgets that might invoke a command that result in SIPP commands
being executed while a during rendering.  It is possible to abort a rendering
in progress with the \fBSippAbortRender\fR command.
.PP
See the Tk photo image manual page for details in how to create and manage
Tk photo images.  Photo images may have images displayed on it
with the \fBSippRender\fR or the \fBSippCopy\fR commands.  The images is
initialized to the background color at the start of the output.  Photo images
have no internal background color, by default, they are transparent.  I maybe
desirable to set the background color of the widget that the image is to be
displayed on to be background color that will be used during rendering.
The \fBSippColor\fR command can be useful for this.
'\"@endhelp
'
.sp 2
.SH "VECTOR AND MATRIX COMMANDS"
.PP
The following commands are available to manipulate vectors and matrices.
'\"@help: tsipp/geometric/MkVector
'\"@brief: Create a vector between two points.
.TP
\fBSippMkVector \fIpoint0 point1\fR
.br
Create a vector in the director going from the \fIpoint0\fR to \fIpoint1\fR.
'\"@endhelp
'
'\"@help: tsipp/geometric/VecNegate
'\"@brief: Negate a vector.
.TP
\fBSippVecNegate \fIvector\fR
.br
Negate all components of \fIvector\fR.
'\"@endhelp
'
'\"@help: tsipp/geometric/VecDot
'\"@brief: Return the dot product of the two Vectors.
.TP
\fBSippVecDot \fIvector0 vector1\fR
.br
Return the dot product of the two vectors \fIvector0\fR and \fIvector1\fR.
'\"@endhelp
'
'\"@help: tsipp/geometric/VecLen
'\"@brief: Return the length of a vector.
.TP
\fBSippVecLen \fIvector\fR
.br
Return the length of \fIvector\fR.
'\"@endhelp
'
'\"@help: tsipp/geometric/VecAdd
'\"@brief: Add two Vectors
.TP
\fBSippVecAdd \fIvector0 vector1\fR
.br
Add two vectors, \fIvector0\fR and \fIvector1\fR.
'\"@endhelp
'
'\"@help: tsipp/geometric/VecSub
'\"@brief: Subtract two Vectors
.TP
\fBSippVecSub \fIvector0 vector1\fR
.br
Subtract \fIvector1\fR from \fIvector0\fR.
'\"@endhelp
'
'\"@help: tsipp/geometric/VecMult
'\"@brief: Multiply a vector by a scalar.
.TP
\fBSippVecMult \fInumber vector\fR
.br
Multiply \fIvector\fR with the scalar \fInumber\fR.
'\"@endhelp
'
'\"@help: tsipp/geometric/VecCross
'\"@brief: Cross multiply two vectors.
.TP
\fBSippVecCross \fIvector0 vector1\fR
.br
Cross multiple \fIvector0\fR by \fIvector1\fR.
'\"@endhelp
'
'\"@help: tsipp/geometric/VecNorm
'\"@brief: Normalize a vector.
.TP
\fBSippVecNorm \fIvector\fR
.br
Normalize \fIvector\fR.
'\"@endhelp
'
'\"@help: tsipp/misc/MatIdent
'\"@brief: Return an identity matrix.
.TP
\fBSippMatIdent\fR
.br
Return an identity matrix.
'\"@endhelp
'
'\"@help: tsipp/misc/MatTranslate
'\"@brief: Translate a matrix along a vector.
.TP
\fBSippMatTranslate \fImatrix vector\fR
.br
Return \fImatrix\fR translated along \fIvector\fR.
'\"@endhelp
'
'\"@help: tsipp/misc/MatRotateX
'\"@brief: Rotate a matrix around the X axis.
.TP
\fBSippMatRotateX \fImatrix angle\fR
.br
Return the rotation of \fImatrix\fR \fIangle\fR radians or degrees about the
X axis.
If \fIangle\fR is prefixed with "R", then it is in radians.  If it
is prefixed with "D", then it is in degrees.  If no prefix is supplied, then
radians is assumed.
'\"@endhelp
'
'\"@help: tsipp/misc/MatRotateY
'\"@brief: Rotate a matrix around the Y axis.
.TP
\fBSippMatRotateY \fImatrix angle\fR
.br
Return the rotation of \fImatrix\fR \fIangle\fR radians or degrees about the
Y axis.
If \fIangle\fR is prefixed with "R", then it is in radians.  If it
is prefixed with "D", then it is in degrees.  If no prefix is supplied, then
radians is assumed.
'\"@endhelp
'
'\"@help: tsipp/misc/MatRotateZ
'\"@brief: Rotate a matrix around the Z axis.
.TP
\fBSippMatRotateZ \fImatrix angle\fR
.br
Return the rotation of \fImatrix\fR \fIangle\fR radians or degrees about the
Z axis.
If \fIangle\fR is prefixed with "R", then it is in radians.  If it
is prefixed with "D", then it is in degrees.  If no prefix is supplied, then
radians is assumed.
'\"@endhelp
'
'\"@help: tsipp/geometric/MatRotate
'\"@brief: Rotate a matrix about an arbitrary line.
.TP
\fBSippMatRotate \fImatrix point vector angle\fR
.br
Return the rotation rotation of \fImatrix\fR \fIangle\fR radians or degrees
about the line given by \fIpoint\fR and \fIvector\fR starting in that point.
If \fIangle\fR is prefixed with "R", then it is in radians.  If it
is prefixed with "D", then it is in degrees.  If no prefix is supplied, then
radians is assumed.
'\"@endhelp
'
'\"@help: tsipp/geometric/MatScale
'\"@brief: Scale a matrix.
.TP
\fBSippMatScale \fImatrix factor\fR|{\fIxfactor yfactor zfactor\fR}
.br
Return \fImatrix\fR scale by the scaling factors \fIxfactor\fR
\fIyfactor\fR and \fIzfactor\fR in the main directions respectively.  If only 
\fIfactor\fR is specified, it is used as the scaling factors in all directions.
'\"@endhelp
'
'\"@help: tsipp/geometric/MatMirrorPlane
'\"@brief: Return a mirroring of a matrix.
.TP
\fBSippMatMirrorPlane \fImatrix point normal\fR
.br
Return mirroring of \fImatrix\fR in the plane defined by \fIpoint\fR and
the vector \fInormal\fR.
'\"@endhelp
'
'\"@help: tsipp/geometric/MatMult
'\"@brief: Multiply two matrices.
.TP
\fBSippMatMult \fImatrix0 matrix1\fR
.br
Multiply \fImatrix0\fR by \fImatrix1\fR.
'\"@endhelp
'
'\"@help: tsipp/geometric/PointTransform
'\"@brief: Transform a point with a matrix.
.TP
\fBSippPointTransform \fIpoint matrix\fR
.br
Transform \fIpoint\fR with \fImatrix\fR.
'\"@endhelp
'
'\"@help: tsipp/geometric/AngleDegrees
'\"@brief: Convert an angle to degrees.
.TP
\fBSippAngleDegrees \fIangle\fR
.br
Convert \fIangle\fR to degrees,
If \fIangle\fR is prefixed with "R", then it is in radians.  If it
is prefixed with "D", then it is in degrees.  If no prefix is supplied, then
radians is assumed.
'\"@endhelp
'
'\"@help: tsipp/geometric/AngleRadians
'\"@brief: Convert an angle to radians.
.TP
\fBSippAngleRadians \fIangle\fR
.br
Convert \fIangle\fR to radians,
If \fIangle\fR is prefixed with "R", then it is in radians.  If it
is prefixed with "D", then it is in degrees.  If no prefix is supplied, then
degrees is assumed.
'\"@endhelp
'
.sp 2
.SH "NOTES"
o Although the Tcl interface tries to validate argument data, SIPP itself is
not very bullet proof.  Incorrect or out of range data can cause unfriendly
failures, including core dumps.  These problems are most likely to occur
if you are defining your own polygons.
.sp
o No command is implement using the mask shader.
'
.SH "AUTHOR OF THE TCL INTERFACE"
Mark Diekhans (markd@grizzly.com).  Please e-mail questions, comments
and suggestions to this address.
'
.SH "AUTHORS OF OTHER SOFTWARE IN THIS DISTRIBUTION"
The SIPP 3D rendering package was developed by:
.sp
   Jonas Yngvesson (jonas-y@isy.liu.se)
.sp
   Inge Wallin (ingwa@isy.liu.se)
.sp
of Linkoping Institute of Technology, Sweden
.sp
The Utah Raster Toolkit RLE library developed by
.sp
   Spencer W. Thomas and others
.sp
Thanks for the great software!!!
