/*
 *=============================================================================
 *                                tSippCamera.c
 *-----------------------------------------------------------------------------
 * Tcl commands to manage SIPP camera.
 *-----------------------------------------------------------------------------
 * Copyright 1992-1995 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tSippCamera.c,v 5.2 1995/01/06 09:35:52 markd Exp $
 *=============================================================================
 */

#include "tSippInt.h"

/*
 * This is not in sipp.h, its in an internal include file.
 */
Camera  *sipp_current_camera;

/*
 * Internal function prototypes.
 */
static Camera *
CameraHandleToPtr _ANSI_ARGS_((tSippGlob_t    *tSippGlobPtr,
                               char           *handle));

static void
CameraHandleCleanup _ANSI_ARGS_((tSippGlob_t   *tSippGlobPtr));

static bool
ProcessCameraParms _ANSI_ARGS_((tSippGlob_t    *tSippGlobPtr,
                                Camera         *cameraPtr,
                                int             argc,
                                char          **argv));

/*=============================================================================
 * CameraHandleToPtr --
 *   Utility procedure to convert a camera handle to a camera pointer. Handles
 * special STDCAMERA handle.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o handle (I) - A camera handle.
 * Returns:
 *   A pointer to the camera, or NULL if an error occured.
 *-----------------------------------------------------------------------------
 */
static Camera *
CameraHandleToPtr (tSippGlobPtr, handle)
    tSippGlob_t    *tSippGlobPtr;
    char           *handle;
{
    Camera **cameraEntryPtr;

    if (STREQU (handle, "STDCAMERA"))
        return sipp_camera;

    cameraEntryPtr = (Camera **)
        Tcl_HandleXlate (tSippGlobPtr->interp, 
                         tSippGlobPtr->cameraTblPtr, handle);
    if (cameraEntryPtr == NULL)
        return NULL;
    return *cameraEntryPtr;

}

/*=============================================================================
 * CameraHandleCleanup --
 *    Delete all camera handles that are defined.  The current camera is 
 * reset to STDCAMERA.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static void
CameraHandleCleanup (tSippGlobPtr)
    tSippGlob_t   *tSippGlobPtr;
{
    int      walkKey = -1;
    Camera **cameraEntryPtr;

    while (TRUE) {
        cameraEntryPtr = Tcl_HandleWalk (tSippGlobPtr->cameraTblPtr, &walkKey);
        if (cameraEntryPtr == NULL)
            break;

        camera_destruct (*cameraEntryPtr);

        Tcl_HandleFree (tSippGlobPtr->cameraTblPtr, cameraEntryPtr);
    }
    camera_use (sipp_camera);

}

/*=============================================================================
 * ProcessCameraParms --
 *    Process the arguments for the SippCameraCreate and SippCameraParms
 * commands.  These arguments are in the form:
 *    [position] [point] [upvector] [focal]
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o cameraPtr (I) - The camera apply the parameters to.
 *   o argc (I) - The argument count, only including the arguments to process.
 *   o argv (I) - The argument vector, starting with the first option to
 *     process.
 * Returns:
 *   TRUE is ok, FALSE if an error occured (message in interp).
 *-----------------------------------------------------------------------------
 */
static bool
ProcessCameraParms (tSippGlobPtr, cameraPtr, argc, argv)
    tSippGlob_t    *tSippGlobPtr;
    Camera         *cameraPtr;
    int             argc;
    char          **argv;
{
    Vector  position, point, upVector, viewVector;
    double  focal, dotProduct;
    bool    setPosition, setPoint, setUp, setFocal;

    setPosition = setPoint = setUp = setFocal = FALSE;
    if ((argc > 0) && (argv [0][0] != '\0'))
        setPosition = TRUE;
    if ((argc > 1) && (argv [1][0] != '\0'))
        setPoint = TRUE;
    if ((argc > 2) && (argv [2][0] != '\0'))
        setUp = TRUE;
    if ((argc > 3) && (argv [3][0] != '\0'))
        setFocal = TRUE;

    if (setPosition && !TSippConvertVertex (tSippGlobPtr, argv [0], &position))
        return FALSE;
    if (setPoint && !TSippConvertVertex (tSippGlobPtr, argv [1], &point))
        return FALSE;
    if (setUp && !TSippConvertVertex (tSippGlobPtr, argv [2], &upVector))
        return FALSE;
    if (setFocal && (Tcl_GetDouble (tSippGlobPtr->interp, argv [3],
                                    &focal) != TCL_OK))
        return FALSE;

    /*
     * Check the the position and point are not the same and that the view
     * and up vectors are not parallel.  First gather the current camera
     * settings for the values that were not specified.
     */
    if (!setPosition)
        position = cameraPtr->position;
    if (!setPoint)
        point = cameraPtr->lookat;
    if (!setUp)
        upVector = cameraPtr->up;

    if (FEQUAL (position.x, point.x) && FEQUAL (position.y, point.y) && 
        FEQUAL (position.z, point.z)) {
        Tcl_AppendResult (tSippGlobPtr->interp, "The position and point ",
                          "may not be the same", (char *) NULL);
        return FALSE;
    }
        
    viewVector.x = point.x - position.x;
    viewVector.y = point.y - position.y;
    viewVector.z = point.z - position.z;
    vecnorm (&viewVector);
    vecnorm (&upVector);

    dotProduct = fabs (VecDot (viewVector, upVector));
    if (FEQUAL (dotProduct, 1.0)) {
        Tcl_AppendResult (tSippGlobPtr->interp, "The view vector (position ",
                          "to point) and up vector may not be parallel",
                          (char *) NULL);
        return FALSE;
    }

    /*
     * Now set the requested values.
     */
    if (setPosition)
        camera_position (cameraPtr, position.x, position.y, position.z);
    if (setPoint)
        camera_look_at (cameraPtr, point.x, point.y, point.z);
    if (setUp)
        camera_up (cameraPtr, upVector.x, upVector.y, upVector.z);
    if (setFocal)
        camera_focal (cameraPtr, focal);
    return TRUE;

}

/*=============================================================================
 * SippCameraCreate --
 *   Implements the command:
 *     SippCameraCreate [position] [point] [upvector] [focal]
 *
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippCameraCreate (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t   *tSippGlobPtr = (tSippGlob_t *) clientData;
    Camera        *cameraPtr;
    Camera       **cameraEntryPtr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc > 5) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " [position] [point] [upvector] [focal]",
                          (char *) NULL);
        return TCL_ERROR;
    }
    cameraPtr = camera_create ();

    if (!ProcessCameraParms (tSippGlobPtr, cameraPtr, argc - 1, &argv [1])) {
        camera_destruct (cameraPtr);
        return TCL_ERROR;
    }
    cameraEntryPtr = (Camera **) Tcl_HandleAlloc (tSippGlobPtr->cameraTblPtr,
                                                 tSippGlobPtr->interp->result);
    *cameraEntryPtr = cameraPtr;
    return TCL_OK;

}

/*=============================================================================
 * SippCameraDestruct --
 *   Implements the command:
 *     SippCameraDestruct cameralist|ALL
 *
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippCameraDestruct (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t    *tSippGlobPtr = (tSippGlob_t *) clientData;
    handleList_t    cameraList;
    handleList_t    cameraEntryList;
    int             idx;
    Camera         *cameraPtr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " cameralist|ALL", (char *) NULL);
        return TCL_ERROR;
    }

    if (STREQU (argv [1], "ALL")) {
        CameraHandleCleanup (tSippGlobPtr);
        return TCL_OK;
    }

    if (!TSippHandleListConvert (tSippGlobPtr, tSippGlobPtr->cameraTblPtr,
                                 argv [1], &cameraList, &cameraEntryList,
                                 NULL))
        return TCL_ERROR;

    /*
     * Delete the cameras.  If the current camera is deleted, sipp_destruct
     * resets the current camera to sipp_camera (STDCAMERA).
     */
    for (idx = 0; idx < cameraList.len; idx++) {
        cameraPtr = (Camera *) cameraList.ptr [idx];

        camera_destruct (cameraPtr);
        Tcl_HandleFree (tSippGlobPtr->cameraTblPtr, cameraEntryList.ptr [idx]);
    }

    TSippHandleListFree (&cameraList);
    TSippHandleListFree (&cameraEntryList);
    return TCL_OK;

}

/*=============================================================================
 * SippCameraParams --
 *   Implements the command:
 *     SippCameraParams camera [position] [point] [upvector] [focal]
 *
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippCameraParams (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t   *tSippGlobPtr = (tSippGlob_t *) clientData;
    Camera        *camera;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 2) || (argc > 6)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " camera [position] [point] [upvector] [focal]",
                          (char *) NULL);
        return TCL_ERROR;
    }
    camera = CameraHandleToPtr (tSippGlobPtr, argv [1]);
    if (camera == NULL)
        return TCL_ERROR;

    if (!ProcessCameraParms (tSippGlobPtr, camera, argc - 2, &argv [2])) {
        return TCL_ERROR;
    }
    return TCL_OK;

}

/*=============================================================================
 * SippCameraUse --
 *   Implements the command:
 *     SippCameraUse camera
 *
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippCameraUse (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t   *tSippGlobPtr = (tSippGlob_t *) clientData;
    Camera        *camera;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], " camera",
                          (char *) NULL);
        return TCL_ERROR;
    }
    camera = CameraHandleToPtr (tSippGlobPtr, argv [1]);
    if (camera == NULL)
        return TCL_ERROR;

    camera_use (camera);
    return TCL_OK;

}

/*=============================================================================
 * TSippCameraInit --
 *   Initialized the camera commands and set up STDCAMERA as the current
 * camera.
 *
 * Parameters:
 *   o tclSippGlobP (I) - Pointer to the top level global data structure.
 *     (currently unused).
 *-----------------------------------------------------------------------------
 */
void
TSippCameraInit (tSippGlobPtr)
    tSippGlob_t    *tSippGlobPtr;
{

    static tSippTclCmdTbl_t cmdTable [] = {
        {"SippCameraCreate",   (Tcl_CmdProc *) SippCameraCreate},
        {"SippCameraDestruct", (Tcl_CmdProc *) SippCameraDestruct},
        {"SippCameraParams",   (Tcl_CmdProc *) SippCameraParams},
        {"SippCameraUse",      (Tcl_CmdProc *) SippCameraUse},
        {NULL,                 NULL}
    };

    tSippGlobPtr->cameraTblPtr = 
        Tcl_HandleTblInit ("camera", sizeof (Camera *), 24);

    TSippInitCmds (tSippGlobPtr, cmdTable);

}

/*=============================================================================
 * TSippCameraCleanUp --
 *   Cleanup the camera table and release all associated resources.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
void
TSippCameraCleanUp (tSippGlobPtr)
    tSippGlob_t  *tSippGlobPtr;
{
    CameraHandleCleanup (tSippGlobPtr);

    Tcl_HandleTblRelease (tSippGlobPtr->cameraTblPtr);
    tSippGlobPtr->cameraTblPtr = NULL;

}

