/*
 *=============================================================================
 *                                  tSippPBM.c
 *-----------------------------------------------------------------------------
 * Tcl commands to manipulate and render to PBMPlus PPM and PBM files.
 *-----------------------------------------------------------------------------
 * Copyright 1992-1995 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tSippPBM.c,v 5.7 1995/01/15 05:46:35 markd Exp $
 *=============================================================================
 */

#include "tSippInt.h"

/*
 * Type used to hold a file handle "fileNNN".
 */
typedef char fileHandle_t [10];


/*
 * PBM file table entry.
 */
typedef struct {
    fileHandle_t  handle;
    FILE         *filePtr;
    int           perms;
    bool          raw;
} pbmFile_t;


/*
 * Client-data for pixel setting call back used to hold a single row of data.
 * The data will be outputed when the last pixel is written.
 */
typedef struct {
    pbmFile_t *pbmFilePtr;
    FILE      *filePtr;
    int        xSize;
    bool       raw;
} renderData_t;

/*
 * Internal prototypes.
 */
static pbmFile_t *
PBMHandleToPtr _ANSI_ARGS_((tSippGlob_t *tSippGlobPtr,
                            char        *handle));

static bool
PBMWriteError _ANSI_ARGS_((tSippGlob_t *tSippGlobPtr,
                           pbmFile_t    *pbmFilePtr));

static bool
ClosePBMFile _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                          char         *command,
                          pbmFile_t    *pbmFilePtr));

static bool
OutputComments _ANSI_ARGS_((tSippGlob_t *tSippGlobPtr,
                            pbmFile_t  *pbmFilePtr,
                            char       **comments));

static void *
PBMOutputStart _ANSI_ARGS_((tSippGlob_t         *tSippGlobPtr,
                            tSippOutputParms_t  *outputParmsPtr,
                            char                *handle,
                            char               **comments));

static bool
PBMOutputLine _ANSI_ARGS_((tSippGlob_t *tSippGlobPtr,
                           void_pt      clientData,
                           int          y,
                           u_char      *rowPtr));

static bool
PBMOutputBitMap _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                             void_pt       clientData,
                             Sipp_bitmap  *bitMapPtr));

static bool
PBMOutputEnd _ANSI_ARGS_((tSippGlob_t        *tSippGlobPtr,
                          tSippOutputParms_t *outputParmsPtr,
                          void_pt             clientData));

/*=============================================================================
 * PBMHandleToPtr --
 *   Utility procedure to convert a PBM file handle to a Tcl OpenFile
 * structure.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o handle (I) - A PBM handle.
 * Returns:
 *   A pointer to the PBM file entry, or NULL if an error occured.
 *-----------------------------------------------------------------------------
 */
static pbmFile_t  *
PBMHandleToPtr (tSippGlobPtr, handle)
    tSippGlob_t  *tSippGlobPtr;
    char         *handle;
{
    pbmFile_t  *pbmFilePtr;

    pbmFilePtr = (pbmFile_t *)
        Tcl_HandleXlate (tSippGlobPtr->interp,
                         tSippGlobPtr->pbmTblPtr, handle);
    if (pbmFilePtr == NULL)
        return NULL;
    return pbmFilePtr;
}

/*=============================================================================
 * PBMWriteError --
 *   Handle a PBM write error.  This is a hack, since the PBM library doesn't
 * report I/O errors.  This should be called when ferror reports an error.
 * 
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.  The
 *     error message is returned in interp->result.
 *   o pbmFilePtr (I) - PBM file that the error occured on.
 * Globals:
 *   o errno (I) - Should contain the error number of the error that occured.
 * Returns:
 *   Always returns FALSE.
 *-----------------------------------------------------------------------------
 */
static bool
PBMWriteError (tSippGlobPtr, pbmFilePtr)
    tSippGlob_t *tSippGlobPtr;
    pbmFile_t    *pbmFilePtr;
{
    Tcl_AppendResult (tSippGlobPtr->interp, "write error on PBM file: ",
                      Tcl_PosixError (tSippGlobPtr->interp),
                      (char *) NULL);

    clearerr (pbmFilePtr->filePtr);
    return FALSE;
}

/*=============================================================================
 * ClosePBMFile --
 *   Close an PBMPlus file, deleting the handle entry.
 * 
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o command (I) - The command that called this function (argv [0]).
 *   o PBMFilePtr (I) - Pointer to the file entry to close.
 * Returns:
 *   TRUE if all is ok, FALSE if an error occured.
 *-----------------------------------------------------------------------------
 */
static bool
ClosePBMFile (tSippGlobPtr, command, pbmFilePtr)
    tSippGlob_t  *tSippGlobPtr;
    char         *command;
    pbmFile_t    *pbmFilePtr;
{
    int   result;

    result = TSippCallCmd (tSippGlobPtr,
                           "close", 
                           pbmFilePtr->handle,
                           (char *) NULL);

    Tcl_HandleFree (tSippGlobPtr->pbmTblPtr, pbmFilePtr);
    return (result == TCL_OK);
}

/*=============================================================================
 * SippPBMOpen --
 *   Implements the command:
 *     SippPBMOpen [-plain|-raw] filename access
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.  We cheat by calling
 * the Tcl "open" command executor to actually do the open, that way we get the
 * full semantics of the open command.
 *-----------------------------------------------------------------------------
 */
static int
SippPBMOpen (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t  *tSippGlobPtr = (tSippGlob_t *) clientData;
    fileHandle_t  fileHandle;
    pbmFile_t    *pbmFilePtr;
    int           argIdx;
    bool          raw = TRUE;
    bool          gotModeFlag = FALSE;
    FILE         *filePtr;
    int           perms;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    /*
     * Parse off argument flags (only one allowed right now).
     */
    for (argIdx = 1; (argIdx < argc) && (argv [argIdx][0] == '-'); argIdx++) {
        if (STREQU (argv [argIdx], "-plain")) {
            if (gotModeFlag)
                goto wrongArgs;
            raw = FALSE;
            gotModeFlag = TRUE;
            continue;
        }
        if (STREQU (argv [argIdx], "-raw")) {
            if (gotModeFlag)
                goto wrongArgs;
            raw = TRUE;
            gotModeFlag = TRUE;
            continue;
        }
        Tcl_AppendResult (interp, "invalid flag \"", argv [argIdx],
                          "\", expected one of \"-plain\" or \"-raw\"",
                          (char *) NULL);
        return TCL_ERROR;
    }

    /*
     * Validate the number of remaining arguments, which should be valid
     * for the open command.
     */
    if (argc - argIdx != 2)
         goto wrongArgs;

    /*
     * Use the open command to actually open the file.
     */
    if (TSippCallCmd (tSippGlobPtr,
                      "open", 
                      argv [argIdx],
                      argv [argIdx + 1],
                      (char *) NULL) == TCL_ERROR)
        return TCL_ERROR;
    strcpy (fileHandle, interp->result);
    Tcl_ResetResult (interp);

    if (Tcl_GetOpenFile (interp, fileHandle, FALSE, FALSE, &filePtr) != TCL_OK)
        return TCL_ERROR;  /* Should never happen */

    perms = Tcl_FilePermissions (filePtr);

    /*
     * We don't currently support reading of PBMPlus files because the
     * PBMPlus libraries exit on error rather than returning status!!!
     */
    if (perms & TCL_FILE_READABLE) {
        TSippCallCmd (tSippGlobPtr,
                      "close", 
                      fileHandle,
                      (char *) NULL);
        Tcl_ResetResult (interp);
        Tcl_AppendResult (interp, "read access to PBMPlus files is not ",
                          "supported", (char *) NULL);
        return TCL_ERROR;
    }

    /*
     * Setup the PBM file entry.
     */
    pbmFilePtr = (pbmFile_t *) Tcl_HandleAlloc (tSippGlobPtr->pbmTblPtr,
                                               interp->result);
    strcpy (pbmFilePtr->handle, fileHandle);
    pbmFilePtr->perms = perms;
    pbmFilePtr->filePtr = filePtr;
    pbmFilePtr->raw = raw;

    return TCL_OK;

  wrongArgs:
    Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                      " [-plain|-raw] filename access", (char *) NULL);
    return TCL_ERROR;
}

/*=============================================================================
 * SippPBMClose --
 *   Implements the command:
 *     SippPBMClose pbmhandle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippPBMClose (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t *tSippGlobPtr = (tSippGlob_t *) clientData;
    pbmFile_t   *pbmFilePtr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " pbmhandle", (char *) NULL);
        return TCL_ERROR;
    }

    pbmFilePtr = PBMHandleToPtr (tSippGlobPtr, argv [1]);
    if (pbmFilePtr == NULL)
        return TCL_ERROR;

    if (ClosePBMFile (tSippGlobPtr, argv [0], pbmFilePtr))
        return TCL_OK;
    else
        return TCL_ERROR;
}

/*=============================================================================
 * OutputComments --
 *   Write comments from a startard comment table to a PBMPlus file.  Since
 * comments are terminated by newlines in PBMPlus, these comments containing
 * newlines must be split.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o pbmFilePtr (I) - A pointer to the PBM file structure.
 *   o comments (I) - Comments to write.  If NULL, none are written.
 * Returns:
 *   TRUE if all is ok, FALSE and an error message in tSippGlobPtr->interp->
 * result if an error occurs.
 *-----------------------------------------------------------------------------
 */
static bool
OutputComments (tSippGlobPtr, pbmFilePtr, comments)
    tSippGlob_t *tSippGlobPtr;
    pbmFile_t   *pbmFilePtr;
    char       **comments;
{
    int   idx;
    char *strPtr;

    if (comments == NULL)
        return  TRUE;

    fprintf (pbmFilePtr->filePtr, "#\n");

    /*
     * Output comments, being careful to split comments containing '\n'.
     */
    for (idx = 0; comments [idx] != NULL; idx++) {
        strPtr = comments [idx];
        fputs ("#@ ", pbmFilePtr->filePtr);
        while (*strPtr != '\0') {
            if (*strPtr == '\n')
                fputs ("\n#  ", pbmFilePtr->filePtr);
            else
                fputc (*strPtr, pbmFilePtr->filePtr);
            strPtr++;
        }
        fputc ('\n', pbmFilePtr->filePtr);
    }
    fprintf (pbmFilePtr->filePtr, "#\n");
    if (ferror (pbmFilePtr->filePtr)) {
        return PBMWriteError (tSippGlobPtr,
                              pbmFilePtr);
    }

    return TRUE;
}

/*=============================================================================
 * PBMOutputStart --
 *   Start output to an PPM or PBM file.  This routine is pointed to by the
 * PBM image storage class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o handle (I) - Pointer to the PBM handle for the file to write.
 *   o comments (I) - If the source image has comments associated with it, they
 *     are passed here.  If the destination object can store them, it possible.
 *     NULL if no comments available.
 * Returns:
 *   A pointer to be passed back into the other PBM output routines or NULL
 *   an a message in tSippGlobPtr->interp->result if an error occurs.
 *-----------------------------------------------------------------------------
 */
static void *
PBMOutputStart (tSippGlobPtr, outputParmsPtr, handle, comments)
    tSippGlob_t         *tSippGlobPtr;
    tSippOutputParms_t  *outputParmsPtr;
    char                 *handle;
    char                **comments;
{
    pbmFile_t    *pbmFilePtr;
    renderData_t *renderDataPtr;

    pbmFilePtr = PBMHandleToPtr (tSippGlobPtr, handle);
    if (pbmFilePtr == NULL)
        return NULL;

    if ((pbmFilePtr->perms & TCL_FILE_WRITABLE) == 0) {
        TSippNotWritable (tSippGlobPtr->interp, handle);
        return NULL;
    }

    /*
     * Output the PBM or PPM header, using the plain or raw magic number.
     */
    if (outputParmsPtr->bitMapOutput) {
        fprintf (pbmFilePtr->filePtr, "%s\n",
                 pbmFilePtr->raw ? "P4" : "P1");

        fprintf (pbmFilePtr->filePtr,
                 "#Image rendered with Tcl-SIPP %s\n",  TSIPP_VERSION);
        if (!OutputComments (tSippGlobPtr, 
                             pbmFilePtr,
                             comments))
            return NULL;

        fprintf (pbmFilePtr->filePtr, "%d\n%d\n",
                 outputParmsPtr->imgData.xSize,
                 outputParmsPtr->imgData.ySize);
    } else {
        fprintf (pbmFilePtr->filePtr, "%s\n",
                 pbmFilePtr->raw ? "P6" : "P3");
        fprintf (pbmFilePtr->filePtr, "#Image rendered with Tcl-SIPP %s\n",
                 TSIPP_VERSION);
        if (!OutputComments (tSippGlobPtr, 
                             pbmFilePtr,
                             comments))
            return NULL;

        switch (outputParmsPtr->imgData.field) {
          case BOTH:
            fprintf (pbmFilePtr->filePtr, "%d\n%d\n255\n",
                     outputParmsPtr->imgData.xSize,
                     outputParmsPtr->imgData.ySize);
            break;

          case EVEN:
            fprintf (pbmFilePtr->filePtr,
                     "#Image field containing EVEN lines\n");
            fprintf (pbmFilePtr->filePtr, "%d\n%d\n255\n",
                     outputParmsPtr->imgData.xSize, 
                     (outputParmsPtr->imgData.ySize & 1) ?
                         (outputParmsPtr->imgData.ySize >> 1) + 1 :
                          outputParmsPtr->imgData.ySize >> 1);
            break;

          case ODD:
            fprintf (pbmFilePtr->filePtr,
                     "#Image field containing ODD lines\n");
            fprintf (pbmFilePtr->filePtr, "%d\n%d\n255\n",
                     outputParmsPtr->imgData.xSize, 
                     outputParmsPtr->imgData.ySize >> 1);
            break;
        }
    }
    if (ferror (pbmFilePtr->filePtr)) {
        PBMWriteError (tSippGlobPtr,
                       pbmFilePtr);
        return NULL;
    }

    /*
     * Setup the clientdata to pass back to the other rendering routines.
     */
    renderDataPtr = (renderData_t *) smalloc (sizeof (renderData_t));

    renderDataPtr->pbmFilePtr = pbmFilePtr;
    renderDataPtr->filePtr    = pbmFilePtr->filePtr;
    renderDataPtr->xSize      = outputParmsPtr->imgData.xSize;
    renderDataPtr->raw        = pbmFilePtr->raw;

    return renderDataPtr;
}

/*=============================================================================
 * PBMOutputLine --
 *   Output a rendered line to an PPM file.  This routine is pointed to by the
 * PBM output target class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o clientData (I) - Actually pointer to the PBM rendering clientdata.
 *   o y (I) - The scan line that was just rendered.
 *   o rowPtr (I) - The pixels for the scanline that was just rendered.
 * Returns:
 *   TRUE if all is ok, FALSE and an error message in tSippGlobPtr->interp->
 * result if an error occurs.
 *-----------------------------------------------------------------------------
 */
static bool
PBMOutputLine (tSippGlobPtr, clientData, y, rowPtr)
    tSippGlob_t *tSippGlobPtr;
    void_pt      clientData;
    int          y;
    u_char      *rowPtr;
{
    renderData_t *renderDataPtr = (renderData_t *) clientData;
    int           x, outCnt;

    /*
     * Output either in binary or ASCII.  ASCII is limit to 70 characters
     * per line.  Upto 12 characters are outputed per pixel.
     */
    if (renderDataPtr->raw) {
        fwrite (rowPtr, sizeof (u_char), renderDataPtr->xSize * 3,
                renderDataPtr->filePtr);
    } else {
        outCnt = 0;
        for (x = 0; x < renderDataPtr->xSize; x++) {
            fprintf (renderDataPtr->filePtr, "%d %d %d",
                     rowPtr [TSIPP_RED], rowPtr [TSIPP_GREEN],
                     rowPtr [TSIPP_BLUE]);
            rowPtr += 3;

            outCnt++;
            if (outCnt < 5) {
                fputc (' ', renderDataPtr->filePtr);
            } else {
                fputc ('\n', renderDataPtr->filePtr);
                outCnt = 0;
            }
        }
        if (outCnt != 0)
            fputc ('\n', renderDataPtr->filePtr);
    }
    if (ferror (renderDataPtr->filePtr)) {
        return PBMWriteError (tSippGlobPtr,
                              renderDataPtr->pbmFilePtr);
    }
    return TRUE;
}

/*=============================================================================
 * PBMOutputBitMap --
 *   Output a SIPP bit map to an PBM file.  This routine is pointed to by the
 * PBM output target class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o clientData (I) - Actually pointer to the PBM rendering clientdata.
 *   o bitMapPtr (I) - Pointer to the SIPP bit map structure.
 * Returns:
 *   TRUE if all is ok, FALSE and an error message in tSippGlobPtr->interp->
 * result if an error occurs.
 *-----------------------------------------------------------------------------
 */
static bool
PBMOutputBitMap (tSippGlobPtr, clientData, bitMapPtr)
    tSippGlob_t *tSippGlobPtr;
    void_pt      clientData;
    Sipp_bitmap *bitMapPtr;
{
    renderData_t *renderDataPtr = (renderData_t *) clientData;
    int           x, y, outCnt;
    u_char       *rowPtr;
    unsigned      bit;

    rowPtr = bitMapPtr->buffer;

    /*
     * Output either in binary or ASCII.  ASCII is limit to 70 characters
     * per line. Two characters are outputed per pixel.
     */
    if (renderDataPtr->raw) {
        for (y = 0; y < bitMapPtr->height; y++) {
            fwrite (rowPtr, 1, bitMapPtr->width_bytes, renderDataPtr->filePtr);
            rowPtr += bitMapPtr->width_bytes;
        }
    } else {
        outCnt = 0;
        for (y = 0; y < bitMapPtr->height; y++) {
            for (x = 0; x < bitMapPtr->width; x++) {
                bit = (rowPtr [x >> 3] >> (7-(x & 7))) & 1;

                fputc (bit ? '1' : '0', renderDataPtr->filePtr);

                outCnt++;
                if (outCnt < 34) {
                    fputc (' ', renderDataPtr->filePtr);
                } else {
                    fputc ('\n', renderDataPtr->filePtr);
                    outCnt = 0;
                }
            }
            rowPtr += bitMapPtr->width_bytes;
        }
        if (outCnt != 0)
            fputc ('\n', renderDataPtr->filePtr);
    }
    if (ferror (renderDataPtr->filePtr)) {
        return PBMWriteError (tSippGlobPtr,
                              renderDataPtr->pbmFilePtr);
    }
    return TRUE;
}

/*=============================================================================
 * PBMOutputEnd --
 *   Finish up output to an PBM file.  This routine is pointed to by the
 * PBM image storage class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o clientData (I) - Actually a pointer to the PBM rendering clientdata.
 * Returns:
 *   Always returns TRUE, since no errors are currently returned.
 *-----------------------------------------------------------------------------
 */
static bool
PBMOutputEnd (tSippGlobPtr, outputParmsPtr, clientData)
    tSippGlob_t        *tSippGlobPtr;
    tSippOutputParms_t *outputParmsPtr;
    void_pt             clientData;
{
    renderData_t *renderDataPtr = (renderData_t *) clientData;

    fflush (renderDataPtr->filePtr);
    if (ferror (renderDataPtr->filePtr)) {
        return PBMWriteError (tSippGlobPtr,
                              renderDataPtr->pbmFilePtr);
    }

    sfree (renderDataPtr);
    return TRUE;
}

/*=============================================================================
 * TSippPBMInit --
 *   Initialized the pbm and PBM commands.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - Pointer to the top level global data structure.
 *     (currently unused).
 *-----------------------------------------------------------------------------
 */
void
TSippPBMInit (tSippGlobPtr)
    tSippGlob_t  *tSippGlobPtr;
{
    static tSippTclCmdTbl_t cmdTable [] = {
        {"SippPBMOpen",   (Tcl_CmdProc *) SippPBMOpen},
        {"SippPBMClose",  (Tcl_CmdProc *) SippPBMClose},
        {NULL,           NULL}
    };

    static tSippStorageClass_t storageClass = {
        "pbmfile",                  /* handlePrefix  */
        7,                          /* prefixSize    */
        NULL,                       /* identFunc     */
        TSIPP_SCAN_TOP_DOWN,        /* scanDirection */
        TRUE,                       /* bitMapOptimal */
        PBMOutputStart,
        PBMOutputLine,
        PBMOutputBitMap,
        PBMOutputEnd,
        NULL,
        NULL,
        NULL,
        NULL,
        NULL
    };

    tSippGlobPtr->pbmTblPtr =
        Tcl_HandleTblInit ("pbmfile", sizeof (pbmFile_t), 4);

    TSippInitCmds (tSippGlobPtr, cmdTable);
    TSippAddStorageClass (tSippGlobPtr, &storageClass);
}

/*=============================================================================
 * TSippPBMCleanUp --
 *   Close all PBMPlus files and release all associated resources.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
void
TSippPBMCleanUp (tSippGlobPtr)
    tSippGlob_t  *tSippGlobPtr;
{
    int         walkKey = -1;
    pbmFile_t  *  pbmFilePtr;

    while (TRUE) {
        pbmFilePtr = Tcl_HandleWalk (tSippGlobPtr->pbmTblPtr, &walkKey);
        if (pbmFilePtr == NULL)
            break;
        if (!ClosePBMFile (tSippGlobPtr, "TSippPBMCleanUp", pbmFilePtr))
            Tcl_ResetResult (tSippGlobPtr->interp);
    }
    Tcl_HandleTblRelease (tSippGlobPtr->pbmTblPtr);
    tSippGlobPtr->pbmTblPtr = NULL;
}

