/*
 *=============================================================================
 *                                  tSippPrim.c
 *-----------------------------------------------------------------------------
 * Tcl commands to create basic SIPP primitive objects. 
 *-----------------------------------------------------------------------------
 * Copyright 1992-1995 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tSippPrim.c,v 5.3 1995/05/01 05:39:00 markd Exp $
 *=============================================================================
 */

#include "tSippInt.h"

/*
 * Prototype of function used to generate teapot parts.
 */
typedef Object *(teapotFunc_t) _ANSI_ARGS_((int            resolution,
                                            void          *surface,
                                            Shader        *shader,
                                            int            texture));

/*
 * Internal prototypes.
 */
static int
GenerateTeapot _ANSI_ARGS_((tSippGlob_t   *tSippGlobPtr,
                            Tcl_Interp    *interp,
                            int            argc,
                            char         **argv,
                            teapotFunc_t   teapotFunc));

/*=============================================================================
 * SippTorus --
 *   Implements the command:
 *     SippTorus bigradius smallradius radialres tuberes shaderhandle [texture]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
int
SippTorus (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t   *tSippGlobPtr = (tSippGlob_t *) clientData;
    Shader        *shaderPtr;
    void          *surfDescPtr;
    double         bigRadius, smallRadius;
    unsigned       radialRes, tubeRes;
    int            texture = WORLD;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 6) || (argc > 7)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " bigradius smallradius radialres tuberes ",
                          "shaderhandle [texture]", (char *) NULL);
        return TCL_ERROR;
    }                     

    if (!TSippConvertUnsignedDbl (tSippGlobPtr, argv [1], &bigRadius))
        return TCL_ERROR;
    if (!TSippConvertUnsignedDbl (tSippGlobPtr, argv [2], &smallRadius))
        return TCL_ERROR;
    if (smallRadius >= bigRadius) {
        Tcl_AppendResult (interp, "big radius must be greater than small ",
                          "radius", (char *) NULL);
        return TCL_ERROR;
    }
    if (!TSippConvertPosUnsigned (tSippGlobPtr, argv [3], &radialRes))
        return TCL_ERROR;
    if (!TSippConvertPosUnsigned (tSippGlobPtr, argv [4], &tubeRes))
        return TCL_ERROR;

    shaderPtr = TSippShaderHandleToPtr (tSippGlobPtr, argv [5],
                                        &surfDescPtr);
    if (shaderPtr == NULL)
        return TCL_ERROR;

    if (argc == 7) {
        if (!TSippParseTextureMapping (tSippGlobPtr, argv [6], &texture, NULL))
            return TCL_ERROR;
    }

    TSippBindObjectToHandle (tSippGlobPtr, 
                             sipp_torus (bigRadius, smallRadius, radialRes,
                                         tubeRes, surfDescPtr, shaderPtr,
                                         texture)); 
                      
    return TCL_OK;

}

/*=============================================================================
 * SippCone --
 *   Implements the command:
 *     SippCone bottomradius topradius length resolution shaderhandle [texture]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
int
SippCone (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t   *tSippGlobPtr = (tSippGlob_t *) clientData;
    Shader        *shaderPtr;
    void          *surfDescPtr;
    double         bottomRadius, topRadius, length;
    unsigned       resolution;
    int            texture = WORLD;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 6) || (argc > 7)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], " bottomgradius",
                          " topradius length resolution shaderhandle ",
                          "[texture]", (char *) NULL);
        return TCL_ERROR;
    }                     

    if (!TSippConvertUnsignedDbl (tSippGlobPtr, argv [1], &bottomRadius))
        return TCL_ERROR;
    if (!TSippConvertUnsignedDbl (tSippGlobPtr, argv [2], &topRadius))
        return TCL_ERROR;
    if (!TSippConvertUnsignedDbl (tSippGlobPtr, argv [3], &length))
        return TCL_ERROR;
    if (!TSippConvertPosUnsigned (tSippGlobPtr, argv [4], &resolution))
        return TCL_ERROR;

    shaderPtr = TSippShaderHandleToPtr (tSippGlobPtr, argv [5],
                                        &surfDescPtr);
    if (shaderPtr == NULL)
        return TCL_ERROR;

    if (argc == 7) {
        if (!TSippParseTextureMapping (tSippGlobPtr, argv [6], &texture, NULL))
            return TCL_ERROR;
    }

    TSippBindObjectToHandle (tSippGlobPtr, 
                             sipp_cone (bottomRadius, topRadius, length,
                                        resolution, surfDescPtr, shaderPtr,
                                        texture)); 
                      
    return TCL_OK;

}

/*=============================================================================
 * SippCylinder --
 *   Implements the command:
 *     SippCylinder radius length resolution shaderhandle [texture]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
int
SippCylinder (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t   *tSippGlobPtr = (tSippGlob_t *) clientData;
    Shader        *shaderPtr;
    void          *surfDescPtr;
    double         radius, length;
    int            resolution, texture = WORLD;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 5) || (argc > 6)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " radius length resolution shaderhandle [texture]",
                          (char *) NULL);
        return TCL_ERROR;
    }                     

    if (!TSippConvertUnsignedDbl (tSippGlobPtr, argv [1], &radius))
         return TCL_ERROR;
    if (!TSippConvertUnsignedDbl (tSippGlobPtr, argv [2], &length))
        return TCL_ERROR;
    if (!TSippConvertPosUnsigned (tSippGlobPtr, argv [3], &resolution))
        return TCL_ERROR;

    shaderPtr = TSippShaderHandleToPtr (tSippGlobPtr, argv [4],
                                        &surfDescPtr);
    if (shaderPtr == NULL)
        return TCL_ERROR;

    if (argc == 6) {
        if (!TSippParseTextureMapping (tSippGlobPtr, argv [5], &texture, NULL))
            return TCL_ERROR;
    }

    TSippBindObjectToHandle (tSippGlobPtr,
                             sipp_cylinder (radius, length, resolution,
                                            surfDescPtr, shaderPtr,
                                            texture)); 
    return TCL_OK;

}

/*=============================================================================
 * SippEllipsoid --
 *   Implements the command:
 *     SippEllipsoid {xradius yradius zradius} resolution shaderhandle
 *                   [texture]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
int
SippEllipsoid (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t   *tSippGlobPtr = (tSippGlob_t *) clientData;
    Shader        *shaderPtr;
    void          *surfDescPtr;
    Vector         radius;
    unsigned       resolution;
    int            texture = WORLD;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 4) || (argc > 5)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " {xradius yradius zradius} resolution ",
                          "shaderhandle [texture]", (char *) NULL);
        return TCL_ERROR;
    }                     
 
    if (!TSippConvertVertex (tSippGlobPtr, argv [1], &radius))
        return TCL_ERROR;
    if ((radius.x <= 0.0) || (radius.y <= 0.0) || (radius.z <= 0.0)) {
        Tcl_AppendResult (tSippGlobPtr->interp, " radius X Y and Z must be ",
                          " > 0.0", (char *) NULL);
        return TCL_ERROR;
    }
    if (!TSippConvertPosUnsigned (tSippGlobPtr, argv [2], &resolution))
        return TCL_ERROR;

    shaderPtr = TSippShaderHandleToPtr (tSippGlobPtr, argv [3],
                                        &surfDescPtr);
    if (shaderPtr == NULL)
        return TCL_ERROR;

    if (argc == 5) {
        if (!TSippParseTextureMapping (tSippGlobPtr, argv [4], &texture, NULL))
            return TCL_ERROR;
    }

    TSippBindObjectToHandle (tSippGlobPtr, 
                             sipp_ellipsoid (radius.x, radius.y, radius.z,
                                             resolution, surfDescPtr,
                                             shaderPtr, texture)); 
    return TCL_OK;

}

/*=============================================================================
 * SippSphere --
 *   Implements the command:
 *     SippSphere radius resolution shaderhandle [texture]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
int
SippSphere (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t   *tSippGlobPtr = (tSippGlob_t *) clientData;
    Shader        *shaderPtr;
    void          *surfDescPtr;
    double         radius;
    int            resolution, texture = WORLD;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 4) || (argc > 5)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " radius resolution shaderhandle [texture]",
                          (char *) NULL);
        return TCL_ERROR;
    }                     

    if (!TSippConvertUnsignedDbl (tSippGlobPtr, argv [1], &radius))
         return TCL_ERROR;
    if (!TSippConvertPosUnsigned (tSippGlobPtr, argv [2], &resolution))
        return TCL_ERROR;

    shaderPtr = TSippShaderHandleToPtr (tSippGlobPtr, argv [3],
                                        &surfDescPtr);
    if (shaderPtr == NULL)
        return TCL_ERROR;

    if (argc == 5) {
        if (!TSippParseTextureMapping (tSippGlobPtr, argv [4], &texture, NULL))
            return TCL_ERROR;
    }

    TSippBindObjectToHandle (tSippGlobPtr,
                             sipp_sphere (radius, resolution,
                                          surfDescPtr, shaderPtr, texture)); 
    return TCL_OK;

}

/*=============================================================================
 * SippPrism --
 *   Implements the command:
 *     SippPrism 2dpointlist length shaderhandle [texture]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
int
SippPrism (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t   *tSippGlobPtr = (tSippGlob_t *) clientData;
    Shader        *shaderPtr;
    void          *surfDescPtr;
    double         length;
    int            listArgc, idx;
    char         **listArgv;
    Vector        *pointList;
    int            texture = WORLD;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 4) || (argc > 5)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " 2dpointlist length shaderhandle [texture]",
                          (char *) NULL);
        return TCL_ERROR;
    }                     

    if (Tcl_SplitList (tSippGlobPtr->interp, argv [1], &listArgc,
                       &listArgv) != TCL_OK)
        return TCL_ERROR;

    if (listArgc < 3) {
        sfree (listArgv);
        Tcl_AppendResult (interp, "2d point list must contain at least 3 ",
                          "points", (char *) NULL);
        return TCL_ERROR;
    }

    pointList = (Vector *) smalloc (listArgc * sizeof (Vector));

    for (idx = 0; idx < listArgc; idx++) {
        if (!TSippConvert2DPoint (tSippGlobPtr, listArgv [idx], 
                                  &pointList [idx].x, &pointList [idx].y))
            goto errorExit;
        pointList [idx].z = 0.0;
    }

    if (!TSippConvertUnsignedDbl (tSippGlobPtr, argv [2], &length))
        goto errorExit;
    shaderPtr = TSippShaderHandleToPtr (tSippGlobPtr, argv [3],
                                        &surfDescPtr);
    if (shaderPtr == NULL)
        goto errorExit;

    if (argc == 5) {
        if (!TSippParseTextureMapping (tSippGlobPtr, argv [4], &texture, NULL))
            goto errorExit;
    }

    TSippBindObjectToHandle (tSippGlobPtr,
                             sipp_prism (listArgc, pointList, length,
                                         surfDescPtr, shaderPtr, texture)); 
    sfree (pointList);
    sfree (listArgv);
    return TCL_OK;
errorExit:
    sfree (pointList);
    sfree (listArgv);
    return TCL_ERROR;

}

/*=============================================================================
 * SippBlock --
 *   Implements the command:
 *      SippBlock {xsize ysize zsize} shaderhandle [texture]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
int
SippBlock (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t   *tSippGlobPtr = (tSippGlob_t *) clientData;
    Shader        *shaderPtr;
    void          *surfDescPtr;
    Vector         sizes;
    int            texture = WORLD;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 3) || (argc > 4)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " {xsize ysize zsize} shaderhandle [texture]",
                          (char *) NULL);
        return TCL_ERROR;
    }                     

    if (!TSippConvertVertex (tSippGlobPtr, argv [1], &sizes))
        return TCL_ERROR;
    if ((sizes.x <= 0.0) || (sizes.y <= 0.0) || (sizes.z <= 0.0)) {
        Tcl_AppendResult (tSippGlobPtr->interp, " sizes X Y and Z must be ",
                          " > 0.0", (char *) NULL);
        return TCL_ERROR;
    }

    shaderPtr = TSippShaderHandleToPtr (tSippGlobPtr, argv [2],
                                        &surfDescPtr);
    if (shaderPtr == NULL)
        return TCL_ERROR;

    if (argc == 4) {
        if (!TSippParseTextureMapping (tSippGlobPtr, argv [3], &texture, NULL))
            return TCL_ERROR;
    }

    TSippBindObjectToHandle (tSippGlobPtr, 
                             sipp_block (sizes.x, sizes.y, sizes.z, 
                                         surfDescPtr, shaderPtr, texture)); 
    return TCL_OK;

}

/*=============================================================================
 * SippCube --
 *   Implements the command:
 *      SippCube size shaderhandle [texture]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
int
SippCube (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t   *tSippGlobPtr = (tSippGlob_t *) clientData;
    Shader        *shaderPtr;
    void          *surfDescPtr;
    double         size;
    int            texture = WORLD;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 3) || (argc > 4)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " size shaderhandle [texture]", (char *) NULL);
        return TCL_ERROR;
    }                     
    if (!TSippConvertUnsignedDbl (tSippGlobPtr, argv [1], &size))
         return TCL_ERROR;

    shaderPtr = TSippShaderHandleToPtr (tSippGlobPtr, argv [2],
                                        &surfDescPtr);
    if (shaderPtr == NULL)
        return TCL_ERROR;

    if (argc == 4) {
        if (!TSippParseTextureMapping (tSippGlobPtr, argv [3], &texture, NULL))
            return TCL_ERROR;
    }
    TSippBindObjectToHandle (tSippGlobPtr, 
                             sipp_cube (size, surfDescPtr, shaderPtr,
                                        texture)); 
    return TCL_OK;

}

/*=============================================================================
 * GenerateTeapot --
 *   
 *  Command executor for all of the SippTeapot* commands.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o interp (I) - Errors are returned in result.
 *   o argc, argv (I) - Command arguments.
 *   o teapotFunc (I) - SIPP function to call to generate the teapot.
 * Returns:
 *   TCL_OK or TCL_ERROR;
 *-----------------------------------------------------------------------------
 */
static int
GenerateTeapot (tSippGlobPtr, interp, argc, argv, teapotFunc)
    tSippGlob_t   *tSippGlobPtr;
    Tcl_Interp    *interp;
    int            argc;
    char         **argv;
    teapotFunc_t   teapotFunc;
{
    Shader        *shaderPtr;
    void          *surfDescPtr;
    unsigned       resolution;
    int            texture = WORLD;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 3) || (argc > 4)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " resolution shaderhandle [texture]", (char *) NULL);
        return TCL_ERROR;
    }                     
    if (!TSippConvertPosUnsigned (tSippGlobPtr, argv [1], &resolution))
        return TCL_ERROR;

    shaderPtr = TSippShaderHandleToPtr (tSippGlobPtr, argv [2],
                                        &surfDescPtr);
    if (shaderPtr == NULL)
        return TCL_ERROR;

    if (argc == 4) {
        if (!TSippParseTextureMapping (tSippGlobPtr, argv [3], &texture, NULL))
            return TCL_ERROR;
    }
    TSippBindObjectToHandle (tSippGlobPtr, 
                             teapotFunc (resolution, surfDescPtr, shaderPtr,
                                         texture)); 
    return TCL_OK;

}

/*=============================================================================
 * SippTeapot --
 *   Implements the command:
 *      SippTeapot resolution shaderhandle [texture]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
int
SippTeapot (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    return GenerateTeapot ((tSippGlob_t *) clientData,
                           interp,
                           argc, argv,
                           sipp_teapot);
}

/*=============================================================================
 * SippTeapotBody --
 *   Implements the command:
 *      SippTeapotBody resolution shaderhandle [texture]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
int
SippTeapotBody (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    return GenerateTeapot ((tSippGlob_t *) clientData,
                           interp,
                           argc, argv,
                           sipp_teapot_body);
}

/*=============================================================================
 * SippTeapotLid --
 *   Implements the command:
 *      SippTeapotLid resolution shaderhandle [texture]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
int
SippTeapotLid (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    return GenerateTeapot ((tSippGlob_t *) clientData,
                           interp,
                           argc, argv,
                           sipp_teapot_lid);
}

/*=============================================================================
 * SippTeapotHandle --
 *   Implements the command:
 *      SippTeapotHandle resolution shaderhandle [texture]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
int
SippTeapotHandle (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    return GenerateTeapot ((tSippGlob_t *) clientData,
                           interp,
                           argc, argv,
                           sipp_teapot_handle);
}

/*=============================================================================
 * SippTeapotSpout --
 *   Implements the command:
 *      SippTeapotSpout resolution shaderhandle [texture]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
int
SippTeapotSpout (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    return GenerateTeapot ((tSippGlob_t *) clientData,
                           interp,
                           argc, argv,
                           sipp_teapot_spout);
}

/*=============================================================================
 * TSippPrimInit --
 *   Initialized the primitve object creation commands.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - Pointer to the top level global data structure.
 *-----------------------------------------------------------------------------
 */
void
TSippPrimInit (tSippGlobPtr)
    tSippGlob_t  *tSippGlobPtr;
{
    static tSippTclCmdTbl_t cmdTable [] = {
        {"SippTorus",        (Tcl_CmdProc *) SippTorus},
        {"SippCone",         (Tcl_CmdProc *) SippCone},
        {"SippCylinder",     (Tcl_CmdProc *) SippCylinder},
        {"SippEllipsoid",    (Tcl_CmdProc *) SippEllipsoid},
        {"SippSphere",       (Tcl_CmdProc *) SippSphere},
        {"SippPrism",        (Tcl_CmdProc *) SippPrism},
        {"SippBlock",        (Tcl_CmdProc *) SippBlock},
        {"SippCube",         (Tcl_CmdProc *) SippCube},
        {"SippTeapot",       (Tcl_CmdProc *) SippTeapot},
        {"SippTeapotBody",   (Tcl_CmdProc *) SippTeapotBody},
        {"SippTeapotLid",    (Tcl_CmdProc *) SippTeapotLid},
        {"SippTeapotHandle", (Tcl_CmdProc *) SippTeapotHandle},
        {"SippTeapotSpout",  (Tcl_CmdProc *) SippTeapotSpout},
        {NULL,                NULL}
    };

    TSippInitCmds (tSippGlobPtr, cmdTable);
}


