/*
 *=============================================================================
 *                              tSippRender.c
 *-----------------------------------------------------------------------------
 * Tcl rendering command to render to one or more image stores.  Also 
 * contains a command to copy from one image store to another.
 *-----------------------------------------------------------------------------
 * Copyright 1992-1995 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tSippRender.c,v 5.10 1995/08/15 08:38:59 markd Exp $
 *=============================================================================
 */

#include "tSippInt.h"

/*
 * Output image store information for each of the objects being written.  Some
 * information is cached from the class entry for speed.  Handle is stored as
 * part of this memory allocation unit.
 */
typedef struct outputInfo_t {
    void                 *clientData;     /* Client data for output instance */
    outputLine_t         *outputLine;     /* Call for each scan line.        */
    tSippStorageClass_t  *classPtr;       /* Class entry.                    */
    char                 *handle;         /* Handle associated with object.  */
    bool                  initialized;    /* Startup code has been run.      */
    struct outputInfo_t  *nextPtr;        /* Next output object.             */
} outputInfo_t;

/*
 * Clientdata for pixel setting call back used to hold a scanline and
 * the output store callback information.
 */
typedef struct {
    tSippGlob_t   *tSippGlobPtr;  /* Globals, used for error handling.     */
    int            xMax;          /* Maximum pixel number in a row.        */
    u_char        *rowPtr;        /* Buffer containing a scanline.         */
    u_char        *nextPixelPtr;  /* Next pixel in the scan line to write. */
    outputInfo_t  *outputListPtr; /* Output object to write scan lines to. */
} renderData_t;


/*
 * Internal prototypes.
 */
static void
PixelRender _ANSI_ARGS_((renderData_t  *renderDataPtr,
                         int            x,
                         int            y,
                         int            red,
                         int            green,
                         int            blue));

static void
RenderImage _ANSI_ARGS_((tSippGlob_t         *tSippGlobPtr,
                         tSippOutputParms_t  *outputParmsPtr,
                         outputInfo_t        *outputListPtr));

static void
DrawImage _ANSI_ARGS_((tSippGlob_t         *tSippGlobPtr,
                       tSippOutputParms_t  *outputParmsPtr,
                       outputInfo_t        *outputListPtr));

static void
AddStandardComment _ANSI_ARGS_((tSippOutputParms_t     *outputParmsPtr,
                                char                 ***commentsPtr));


static bool
ReturnScanDirectionError _ANSI_ARGS_((tSippGlob_t         *tSippGlobPtr,
                                      tSippOutputParms_t  *outputParmsPtr,
                                      tSippStorageClass_t *storageClassPtr,
                                      outputInfo_t        *outputListPtr));

static bool
SetupOutputEntry _ANSI_ARGS_((tSippGlob_t          *tSippGlobPtr,
                              tSippOutputParms_t   *outputParmsPtr,
                              char                 *handle,
                              outputInfo_t        **outputListPtrPtr));

static outputInfo_t*
SetupOutputObjects _ANSI_ARGS_((tSippGlob_t         *tSippGlobPtr,
                                tSippOutputParms_t  *outputParmsPtr,
                                char                *handleList));

static bool
FinishUpOutputObjects _ANSI_ARGS_((tSippGlob_t         *tSippGlobPtr,
                                   tSippOutputParms_t  *outputParmsPtr,
                                   outputInfo_t        *outputListPtr));

static bool
RequiresArgError _ANSI_ARGS_((tSippGlob_t        *tSippGlobPtr,
                              char                *flag));

static bool
ParseRenderFlags _ANSI_ARGS_((tSippGlob_t          *tSippGlobPtr,
                              char                **argv,
                              tSippOutputParms_t   *outputParmsPtr,
                              int                  *nextArgPtr));

static bool
ParseRenderMode _ANSI_ARGS_((tSippGlob_t         *tSippGlobPtr,
                             char                *modeStr,
                             tSippOutputParms_t  *outputParmsPtr));

static bool
ParseRenderParms _ANSI_ARGS_((tSippGlob_t          *tSippGlobPtr,
                              int                   argc,
                              char                **argv,
                              tSippOutputParms_t   *outputParmsPtr,
                              char                **handleListPtr));

/*=============================================================================
 * TSippHandleRenderingError --
 *   Save an error that occurs during rendering in the delayedError field
 * of the globals and abort the rendering.
 *
 * Parameters:
 *   o tSippGlobPtr (I/O) - A pointer to the Tcl SIPP global structure.  The
 *     error message should be in the interp->result, which will be cleared.
 *-----------------------------------------------------------------------------
 */
void
TSippHandleRenderingError (tSippGlobPtr)
    tSippGlob_t  *tSippGlobPtr;
{
    if (tSippGlobPtr->delayedError == NULL)
        tSippGlobPtr->delayedError = strdup (tSippGlobPtr->interp->result);

    Tcl_ResetResult (tSippGlobPtr->interp);
    sipp_render_terminate ();
}

/*=============================================================================
 * PixelRender --
 *   Pixel rendering call-back procedure.  SIPP doesn't have the concept of 
 * background and always invokes the call back for every pixel.
 *
 * Parameters:
 *   o renderDataPtr (I) - A pointer to the clientdata, It contains a buffer
 *     with the block of data currently being rendered.
 *   o x, y (I) - The pixel coordinate.
 *   o red, green, blue - (I) - The color values for the pixel.  These are
 *     in the range 0..255 (really should be u_char, but causes the compiler
 *     to complain about argument promotion).
 *-----------------------------------------------------------------------------
 */
static void
PixelRender (renderDataPtr, x, y, red, green, blue)
    renderData_t  *renderDataPtr;
    int            x;
    int            y;
    int            red;
    int            green;
    int            blue;
{
    tSippGlob_t   *tSippGlobPtr;
    outputInfo_t  *outputPtr;

    renderDataPtr->nextPixelPtr [TSIPP_RED]   = red;
    renderDataPtr->nextPixelPtr [TSIPP_GREEN] = green;
    renderDataPtr->nextPixelPtr [TSIPP_BLUE]  = blue;
    renderDataPtr->nextPixelPtr += 3;

    if (x < renderDataPtr->xMax)
        return;  /* Row not complete */

    /*
     * Filled a scan line, output it to all outputs.
     */
    tSippGlobPtr = renderDataPtr->tSippGlobPtr;
    outputPtr = renderDataPtr->outputListPtr;
    while (outputPtr != NULL) {
        if (!(*outputPtr->outputLine) (tSippGlobPtr,
                                       outputPtr->clientData,
                                       y,
                                       renderDataPtr->rowPtr)) {
            TSippHandleRenderingError (tSippGlobPtr);
            break;
        }
        outputPtr = outputPtr->nextPtr;
    }
    renderDataPtr->nextPixelPtr = renderDataPtr->rowPtr;
}

/*=============================================================================
 * RenderImage --
 *   Render an image to a set of output objects.  Errors will be
 * saved in the tSippGlob delayedError field for later processing.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o outputListPtr (I) - A pointer to a list of information on the outputs
 *     that the image is to be rendered to.
 *-----------------------------------------------------------------------------
 */
static void
RenderImage (tSippGlobPtr, outputParmsPtr, outputListPtr)
    tSippGlob_t         *tSippGlobPtr;
    tSippOutputParms_t  *outputParmsPtr;
    outputInfo_t        *outputListPtr;
{
    renderData_t renderData;

    /*
     * Set up the clientdata SIPP will pass to our rendering function.
     */
    renderData.tSippGlobPtr  = tSippGlobPtr;
    renderData.xMax          = outputParmsPtr->imgData.xSize - 1;
    renderData.rowPtr        = (u_char *) 
        alloca (outputParmsPtr->imgData.xSize * 3);
    renderData.nextPixelPtr  = renderData.rowPtr;
    renderData.outputListPtr = outputListPtr;

    if (outputParmsPtr->imgData.field == BOTH)
        render_image_func (outputParmsPtr->imgData.xSize,
                           outputParmsPtr->imgData.ySize,
                           PixelRender,
                           &renderData, 
                           outputParmsPtr->imgData.mode,
                           outputParmsPtr->imgData.overSampling);
    else
        render_field_func (outputParmsPtr->imgData.xSize,
                           outputParmsPtr->imgData.ySize,
                           PixelRender,
                           &renderData, 
                           outputParmsPtr->imgData.mode,
                           outputParmsPtr->imgData.overSampling,
                           outputParmsPtr->imgData.field);
}

/*=============================================================================
 * DrawImage --
 *   Render a line image to a list of output image store.  Errors will be
 * saved in the tSippGlob delayedError field for later processing.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o outputListPtr (I) - A pointer to a list of information on the outputs
 *     that the image is to be rendered to.
 *-----------------------------------------------------------------------------
 */
static void
DrawImage (tSippGlobPtr, outputParmsPtr, outputListPtr)
    tSippGlob_t         *tSippGlobPtr;
    tSippOutputParms_t  *outputParmsPtr;
    outputInfo_t        *outputListPtr;
{
    Sipp_bitmap   *bitMapPtr;
    outputInfo_t  *outputPtr;
    bool           status;

    bitMapPtr = sipp_bitmap_create (outputParmsPtr->imgData.xSize,
                                    outputParmsPtr->imgData.ySize);

    render_image_func (outputParmsPtr->imgData.xSize,
                       outputParmsPtr->imgData.ySize,
                       sipp_bitmap_line,
                       bitMapPtr,
                       LINE, 0);

    outputPtr = outputListPtr;
    while (outputPtr != NULL) {
        status = (*outputPtr->classPtr->outputBitMap) (tSippGlobPtr,
                                                       outputPtr->clientData,
                                                       bitMapPtr);
        if (!status) {
            TSippHandleRenderingError (tSippGlobPtr);
            break;
        }
        outputPtr = outputPtr->nextPtr;
    }

    sipp_bitmap_destruct (bitMapPtr);
}

/*=============================================================================
 * AddStandardComment --
 *    Add the standard Tcl-SIPP comments to an image.
 *
 * Parameters:
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o commentsPtr (I/O) - Pointer to comment array.
 *-----------------------------------------------------------------------------
 */
static void
AddStandardComment (outputParmsPtr, commentsPtr)
    tSippOutputParms_t     *outputParmsPtr;
    char                 ***commentsPtr;
{
    char  *infoArgv [5], *infoList;
    char   overSamplingStr [16];
    char   backgroundColorStr [3 * TCL_DOUBLE_SPACE];
    char   lineColorStr [3 * TCL_DOUBLE_SPACE];

    TSippCommentPut (commentsPtr,
                     "TSIPP_VERSION",
                     TSIPP_VERSION);

    if (outputParmsPtr->argv != NULL)
        TSippAddHist (commentsPtr, outputParmsPtr->argv);

    switch (outputParmsPtr->imgData.mode) {
      case FLAT:
        infoArgv [0] = "FLAT";
        break;
      case GOURAUD:
        infoArgv [0] = "GOURAUD";
        break;
      case PHONG:
        infoArgv [0] = "PHONG";
        break;
      case LINE:
        infoArgv [0] = "LINE";
        break;
      case MODE_UNKNOWN24:
        infoArgv [0] = "UNKNOWN24";
        break;
    }

    sprintf (overSamplingStr, "%d", outputParmsPtr->imgData.overSampling);
    infoArgv [1] = overSamplingStr;

    switch (outputParmsPtr->imgData.field) {
      case BOTH:
        infoArgv [2] = "BOTH";
        break;
      case ODD:
        infoArgv [2] = "ODD";
        break;
      case EVEN:
        infoArgv [2] = "EVEN";
        break;
    }

    /*
     * Save the background color.
     */
    TSippFormatIntColor (outputParmsPtr->tSippGlobPtr,
                         outputParmsPtr->imgData.backgroundColor,
                         backgroundColorStr);
    infoArgv [3] = backgroundColorStr;

    /*
     * Save the line color.
     */
    TSippFormatIntColor (outputParmsPtr->tSippGlobPtr,
                         outputParmsPtr->imgData.lineColor,
                         lineColorStr);
    infoArgv [4] = lineColorStr;

    infoList = Tcl_Merge (5, infoArgv);

    TSippCommentPut (commentsPtr,
                     "TSIPP_IMAGE",
                     infoList);
    sfree (infoList);
}

/*=============================================================================
 * ReturnScanDirectionError --
 *   Determine which other image storage class or command line specification,
 * that the current image storage class conflicts with and return a meaningful
 * error message.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - Error returned in interp->result
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o storageClassPtr (I) - Storage class that conflicts with the current
 *     scan direction.
 *   o outputListPtr (I) - The head of the output list.
 * Returns:
 *   Always returnes FALSE.
 *-----------------------------------------------------------------------------
 */
static bool
ReturnScanDirectionError (tSippGlobPtr, outputParmsPtr, storageClassPtr,
                          outputListPtr)
    tSippGlob_t          *tSippGlobPtr;
    tSippOutputParms_t   *outputParmsPtr;
    tSippStorageClass_t  *storageClassPtr;
    outputInfo_t         *outputListPtr;
{
    char *flag;

    while (outputListPtr != NULL) {
        if (outputListPtr->classPtr->scanDirection != TSIPP_SCAN_RANDOM)
            break;
        outputListPtr = outputListPtr->nextPtr;
    }

    if (outputListPtr != NULL) { 
        Tcl_AppendResult (tSippGlobPtr->interp, storageClassPtr->handlePrefix,
                          " scanning direction conflicts with ",
                          outputListPtr->classPtr->handlePrefix,
                          (char *) NULL);
    } else {
        if (outputParmsPtr->scanDirection == TSIPP_SCAN_TOP_DOWN) {
            flag = "-scandown";
        } else {
            flag = "-scanup";
        }

        Tcl_AppendResult (tSippGlobPtr->interp, storageClassPtr->handlePrefix,
                          " scanning direction conflicts with ", flag,
                          (char *) NULL);
    }

    return FALSE;
}

/*=============================================================================
 * SetupOutputEntry --
 *   Given an image store handle, locate the image store class in the
 * class table and set up a output list entry.  The initialization code is not
 * run at this time, since the scan direction might not be known.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o handle (I) - The output handle.
 *   o outputListPtrPtr (I/O) - A pointer to the head of the output list.
 * Returns:
 *   TRUE if all is OK, FALSE if an error occured.
 *-----------------------------------------------------------------------------
 */
static bool
SetupOutputEntry (tSippGlobPtr, outputParmsPtr, handle, outputListPtrPtr)
    tSippGlob_t          *tSippGlobPtr;
    tSippOutputParms_t   *outputParmsPtr;
    char                 *handle;
    outputInfo_t        **outputListPtrPtr;
{
    tSippStorageClass_t *storageClassPtr;
    outputInfo_t        *outputInfoPtr;

    storageClassPtr = TSippFindStorageClass (tSippGlobPtr,
                                             handle);
    if (storageClassPtr == NULL)
        return FALSE;

    /*
     * Make sure the image store's rendering direction does not conflict
     * with the direction of any other object.  If there is a conflict,
     * find the conflicting class and output a helpful error message.
     */
    if ((outputParmsPtr->scanDirection != TSIPP_SCAN_RANDOM) &&
        (outputParmsPtr->scanDirection != storageClassPtr->scanDirection)) {

        return ReturnScanDirectionError (tSippGlobPtr,     outputParmsPtr,
                                         storageClassPtr, *outputListPtrPtr);
    }

    /*
     * Stash the scanning direction, if its not SIPP_BOTH, this sets it.
     */
    outputParmsPtr->scanDirection = storageClassPtr->scanDirection;
    
    /*
     * Set up output info, save the handle as part of it.  Add it into the
     * list.
     */
    outputInfoPtr = (outputInfo_t*) smalloc (sizeof (outputInfo_t) +
                                             strlen (handle) + 1);
    
    outputInfoPtr->clientData  = NULL;
    outputInfoPtr->outputLine  = storageClassPtr->outputLine;
    outputInfoPtr->classPtr    = storageClassPtr;
    outputInfoPtr->initialized = FALSE;
    outputInfoPtr->handle      = PTR_ADD (outputInfoPtr,
                                          sizeof (outputInfo_t));
    strcpy (outputInfoPtr->handle, handle);

    outputInfoPtr->nextPtr = *outputListPtrPtr;
    *outputListPtrPtr      = outputInfoPtr;

    return TRUE;
}

/*=============================================================================
 * SetupOutputObjects --
 *   Run through the output list and setup the output info entries for all 
 * output objects that were specified on the command line.  The startup
 * code is run for each object.  The standard comments will be passed to each
 * object.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o handleList (I) - String containing the list of output object handles.
 * Returns:
 *   A pointer to a list of output objects, or NULL if an error occured.
 *-----------------------------------------------------------------------------
 */
static outputInfo_t*
SetupOutputObjects (tSippGlobPtr, outputParmsPtr, handleList)
    tSippGlob_t         *tSippGlobPtr;
    tSippOutputParms_t  *outputParmsPtr;
    char                *handleList;
{
    int             handleArgc, idx;
    char          **handleArgv, *errorMsg;
    void           *outputClientData;
    outputInfo_t   *outputListPtr = NULL, *outputInfoPtr;
    char          **comments = NULL;

    if (Tcl_SplitList (tSippGlobPtr->interp, handleList,
                       &handleArgc, &handleArgv) != TCL_OK) {
        return NULL;
    }

    /*
     * Run through and set up each of the outputs in the list.  This also
     * determines the scanning direction.
     */
    for (idx = 0; idx < handleArgc; idx++) {
        if (!SetupOutputEntry (tSippGlobPtr, outputParmsPtr, handleArgv [idx],
                               &outputListPtr))
            goto errorCleanUp;
    }

    AddStandardComment (outputParmsPtr, &comments);

    /*
     * Call the output output classes set up routines.
     */
    outputInfoPtr = outputListPtr;
    while (outputInfoPtr != NULL) {
        outputClientData =
            (*outputInfoPtr->classPtr->outputStart) (tSippGlobPtr,
                                                     outputParmsPtr,
                                                     outputInfoPtr->handle,
                                                     comments);
        if (outputClientData == NULL)
            goto errorCleanUp;
        outputInfoPtr->clientData  = outputClientData;
        outputInfoPtr->initialized = TRUE;

        outputInfoPtr = outputInfoPtr->nextPtr;
    }

    TSippCommentFree (&comments);
    sfree (handleArgv);
    return outputListPtr;

    /*
     * Got an error, run through and do the finish up operation.  At the least,
     * this will clean up each of the entries.  Save the error message as
     * FinishUpOutputObjects might overwrite it.
     */
  errorCleanUp:
    TSippCommentFree (&comments);
    sfree (handleArgv);

    errorMsg = strdup (tSippGlobPtr->interp->result);
    Tcl_ResetResult (tSippGlobPtr->interp);

    FinishUpOutputObjects (tSippGlobPtr, outputParmsPtr, outputListPtr);

    Tcl_ResetResult (tSippGlobPtr->interp);
    Tcl_SetResult (tSippGlobPtr->interp, errorMsg, TCL_DYNAMIC);
    return NULL;
}

/*=============================================================================
 * FinishUpOutputObjects --
 *   Run through the output object list and run the renderEnd functions for
 * each object, if its been initialized, and release the output info entry.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o outputListPtr (I) - The list of output info entries.
 * Returns:
 *   TRUE if all is ok, FALSE and an error message in
 *  tSippGlobPtr->interp->result if an error occurs.
 *-----------------------------------------------------------------------------
 */
static bool
FinishUpOutputObjects (tSippGlobPtr, outputParmsPtr, outputListPtr)
    tSippGlob_t         *tSippGlobPtr;
    tSippOutputParms_t  *outputParmsPtr;
    outputInfo_t        *outputListPtr;
{
    outputInfo_t  *freeEntryPtr;
    bool           status;
    char          *errorMsg = NULL;

    /*
     * Loop through doing output ends.  The first error message is saved and
     * returned.
     */
    while (outputListPtr != NULL) {
        if (outputListPtr->initialized) {
            status =(*outputListPtr->classPtr->
                     outputEnd) (tSippGlobPtr,
                                 outputParmsPtr,
                                 outputListPtr->clientData);
            if (!status) {
                if (errorMsg == NULL)
                    errorMsg = strdup (tSippGlobPtr->interp->result);
                Tcl_ResetResult (tSippGlobPtr->interp);
            }
        }
        freeEntryPtr = outputListPtr;
        outputListPtr = outputListPtr->nextPtr;
        sfree (freeEntryPtr);
    }

    if (errorMsg != NULL) {
        Tcl_SetResult (tSippGlobPtr->interp, errorMsg, TCL_DYNAMIC);
        return FALSE;
    }
    return TRUE;
}

/*=============================================================================
 * RequiresArgError --
 *   Return a error indicating that a flag requires an argument.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *     Errors are returned in interp->result.
 *   o flag (I) - Flag that requires an argument.
 * Returns:
 *   FALSE, so function can be called in a return.
 *-----------------------------------------------------------------------------
 */
static bool
RequiresArgError (tSippGlobPtr, flag)
    tSippGlob_t        *tSippGlobPtr;
    char                *flag;
{
    Tcl_AppendResult (tSippGlobPtr->interp, "\"", flag,
                      "\" flag requires an argument", (char *) NULL);
    return FALSE;
}

/*=============================================================================
 * ParseRenderFlags --
 *   Parse the flags to the rendering commands.  Flags must be the first
 *  arguments.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *     Errors are returned in interp->result.
 *   o argv (I) - Command argument vector to parse. 
 *   o outputParmsPtr (O) - The parameters describing the image to output. 
 *     Fields are initialized from the flags.
 *   o nextArgPtr (O) - The index of the next argument after the flags is
 *     returned here.
 * Returns:
 *   TRUE if all is ok, FALSE if an error occured.
 *-----------------------------------------------------------------------------
 */
static bool
ParseRenderFlags (tSippGlobPtr, argv, outputParmsPtr, nextArgPtr)
    tSippGlob_t          *tSippGlobPtr;
    char                **argv;
    tSippOutputParms_t   *outputParmsPtr;
    int                  *nextArgPtr;
{
    char *argPtr;
    int   nextArg = 1;
    bool  gotField, gotClear, gotDirection;

    gotField = gotClear = gotDirection = FALSE;

    for (; (argv [nextArg] != NULL) && argv [nextArg][0] == '-'; nextArg++) {
        argPtr = argv [nextArg];

        if (STREQU (argPtr, "-both")) {
            if (gotField)
                goto dupField;
            gotField = TRUE;
            outputParmsPtr->imgData.field = BOTH;
            continue;
        }
        if (STREQU (argPtr, "-odd")) {
            if (gotField)
                goto dupField;
            gotField = TRUE;
            outputParmsPtr->imgData.field = ODD;
            continue;
        }
        if (STREQU (argPtr, "-even")) {
            if (gotField)
                goto dupField;
            gotField = TRUE;
            outputParmsPtr->imgData.field = EVEN;
            continue;
        }
        if (STREQU (argPtr, "-clear")) {
            if (gotClear)
                goto dupClear;
            gotClear = TRUE;
            outputParmsPtr->clear = TRUE;
            continue;
        }
        if (STREQU (argPtr, "-noclear")) {
            if (gotClear)
                goto dupClear;
            gotClear = TRUE;
            outputParmsPtr->clear = FALSE;
            continue;
        }
        if (STREQU (argPtr, "-either")) {
            if (gotDirection)
                goto dupDirection;
            gotDirection = TRUE;
            outputParmsPtr->scanDirection = TSIPP_SCAN_RANDOM;
            continue;
        }
        if (STREQU (argPtr, "-topdown")) {
            if (gotDirection)
                goto dupDirection;
            gotDirection = TRUE;
            outputParmsPtr->scanDirection = TSIPP_SCAN_TOP_DOWN;
            continue;
        }
        if (STREQU (argPtr, "-bottomup")) {
            if (gotDirection)
                goto dupDirection;
            gotDirection = TRUE;
            outputParmsPtr->scanDirection = TSIPP_SCAN_BOTTOM_UP;
            continue;
        }
        if (STREQU (argPtr, "-update")) {
            unsigned update;

            if (argv [nextArg + 1] == NULL) {
                return RequiresArgError (tSippGlobPtr, argv [nextArg]);
            }
            if (Tcl_GetUnsigned (tSippGlobPtr->interp, argv [nextArg + 1],
                                 &update) != TCL_OK)
                return FALSE;
            outputParmsPtr->update = update;
            nextArg++;
            continue;
        }
        /*
         * Made it here, valid flag was not found.
         */
        Tcl_AppendResult (tSippGlobPtr->interp,
                          "invalid flag \"", argv [nextArg],
                          "\", expected one of \"-both\", \"-odd\", ",
                          "\"-even\", \"-update\", \"-clear\", \"-noclear\", ",
                          "\"-either\", \"-topdown\", or \"-bottomup\"",
                          (char *) NULL);
        return FALSE;
    }

    *nextArgPtr = nextArg;
    return TRUE;

    /*
     * Error return code for duplicated flags.
     */

  dupField:
    Tcl_AppendResult (tSippGlobPtr->interp,
                      "only one of \"-both\", \"-odd\", or ",
                      "\"-even\" may be specified", (char *) NULL);
    return FALSE;

  dupClear:
    Tcl_AppendResult (tSippGlobPtr->interp,
                      "only one of \"-clear\", or \"-noclear\"",
                      " may be specified", (char *) NULL);
    return FALSE;

  dupDirection:
    Tcl_AppendResult (tSippGlobPtr->interp,
                      "only one of \"-either\", \"-topdown\", or ",
                      "\"-bottomup\" may be specified", (char *) NULL);
    return FALSE;
}

/*=============================================================================
 * ParseRenderMode --
 *   Convert a string representing the SIPP render mode.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *     Errors are returned in interp->result.
 *   o modeStr (I) - The mode string to convert.
 *   o outputParmsPtr (O) - The output parameters to fill in from the command
 *     line.
 * Returns:
 *   TRUE if all is ok, FALSE if an error occured.
 *-----------------------------------------------------------------------------
 */
static bool
ParseRenderMode (tSippGlobPtr, modeStr, outputParmsPtr)
    tSippGlob_t         *tSippGlobPtr;
    char                *modeStr;
    tSippOutputParms_t  *outputParmsPtr;
{
    if (STREQU (modeStr, "PHONG"))
        outputParmsPtr->imgData.mode = PHONG;
    else
    if (STREQU (modeStr, "GOURAUD"))
        outputParmsPtr->imgData.mode = GOURAUD;
    else
    if (STREQU (modeStr, "FLAT"))
        outputParmsPtr->imgData.mode = FLAT;
    else
    if (STREQU (modeStr, "LINE")) {
        outputParmsPtr->imgData.mode = LINE;
    } else {
        Tcl_AppendResult (tSippGlobPtr->interp, "invalid rendering mode, ",
                          "expect one of `PHONG', `GOURAUD', `FLAT', ",
                          "or `LINE', got: ", modeStr,
                          (char *) NULL);
        return FALSE;
    }
    return TRUE;
}

/*=============================================================================
 * ParseRenderParms --
 *   Procedure to parse the render command parameters commands.  All
 * parameters are handled except for the handle list.
 * The command is in the form:
 *
 *  SippRender [-flags] outputlist xsize ysize [mode] [oversample]
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *     Errors are returned in interp->result.
 *   o argc, argv (I) - Arguments to the command.
 *   o outputParmsPtr (O) - The parameters describing the image to output.
 *     Info filled in from the command line args.
 *   o handleListPtr (O) - A pointer to the string from the argv containing the
 *     output object handle list.
 * Returns:
 *   TRUE if all is ok, FALSE if an error occured.
 *-----------------------------------------------------------------------------
 */
static bool
ParseRenderParms (tSippGlobPtr, argc, argv, outputParmsPtr, handleListPtr)
    tSippGlob_t          *tSippGlobPtr;
    int                   argc;
    char                **argv;
    tSippOutputParms_t   *outputParmsPtr;
    char                **handleListPtr;
{
    int nextArg;

    /*
     * Default all fields that might need it up front.
     */
    outputParmsPtr->imgData.mode         = PHONG;
    outputParmsPtr->imgData.overSampling = 1;
    outputParmsPtr->imgData.field        = BOTH;

    outputParmsPtr->imgData.backgroundColor [TSIPP_RED] =
        tSippGlobPtr->backgroundColor.red * 255;
    outputParmsPtr->imgData.backgroundColor [TSIPP_GREEN] =
        tSippGlobPtr->backgroundColor.grn * 255;
    outputParmsPtr->imgData.backgroundColor [TSIPP_BLUE] =
        tSippGlobPtr->backgroundColor.blu * 255;

    outputParmsPtr->imgData.lineColor [TSIPP_RED] =
        tSippGlobPtr->lineColor.red * 255;
    outputParmsPtr->imgData.lineColor [TSIPP_GREEN] =
        tSippGlobPtr->lineColor.grn * 255;
    outputParmsPtr->imgData.lineColor [TSIPP_BLUE] =
        tSippGlobPtr->lineColor.blu * 255;

    outputParmsPtr->tSippGlobPtr = tSippGlobPtr;
    outputParmsPtr->argv = argv;
    outputParmsPtr->update = -1;
    outputParmsPtr->clear = TRUE;
    outputParmsPtr->bitMapOutput = FALSE;
    outputParmsPtr->scanDirection = TSIPP_SCAN_RANDOM;

    if (!ParseRenderFlags (tSippGlobPtr, argv, outputParmsPtr, &nextArg))
        return FALSE;

    if (((argc - nextArg) < 3) || ((argc - nextArg) > 5)) {
        Tcl_AppendResult (tSippGlobPtr->interp, "wrong # args: ", argv [0],
                          " [-flags] outputlist xsize ysize [mode] ",
                          "[oversample]", (char *) NULL);
        return FALSE;
    }

    *handleListPtr = argv [nextArg];
    nextArg++;

    if (!TSippConvertPosUnsigned (tSippGlobPtr, argv [nextArg],
                                  &outputParmsPtr->imgData.xSize))
        return FALSE;
    nextArg++;

    if (!TSippConvertPosUnsigned (tSippGlobPtr, argv [nextArg],
                                  &outputParmsPtr->imgData.ySize))
        return FALSE;
    nextArg++;

    if ((nextArg < argc) && (argv [nextArg][0] != '\0')) {
        if (!ParseRenderMode (tSippGlobPtr, argv [nextArg], outputParmsPtr))
            return FALSE;
    }
    nextArg++;

    if ((nextArg < argc) && (argv [nextArg][0] != '\0')) {
        if (!TSippConvertPosUnsigned (tSippGlobPtr, argv [nextArg],
                                      &outputParmsPtr->imgData.overSampling))
            return FALSE;
    }

    if (outputParmsPtr->imgData.mode == LINE) {
        if (outputParmsPtr->imgData.field != BOTH) {
            Tcl_AppendResult (tSippGlobPtr->interp, "can't specify \"-odd\" ",
                              "or \"-even\" with a mode of \"LINE\"",
                              (char *) NULL);
            return FALSE;
        }
        outputParmsPtr->imgData.overSampling = 1;
        outputParmsPtr->bitMapOutput = TRUE;
    }
    if (outputParmsPtr->update < 0)
        outputParmsPtr->update = outputParmsPtr->imgData.xSize;

    return TRUE;
}

/*=============================================================================
 * SippRender --
 *   Implements the command:
 *     SippRender [-flags] outputList xsize ysize [mode] [oversample]
 *
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *
 * Great care must be take with the updates and the abort flag.  If we did
 * an update before rendering begins and an abort takes place in that update,
 * then the abort might be lost, as the rendering clears it at the beginning.
 *-----------------------------------------------------------------------------
 */
static int
SippRender (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t        *tSippGlobPtr = (tSippGlob_t *) clientData;
    tSippOutputParms_t  outputParms;
    outputInfo_t       *outputListPtr;
    char               *handleList;
    bool                ok;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);
                          
    if (!ParseRenderParms (tSippGlobPtr, argc, argv, &outputParms,
                           &handleList))
        return TCL_ERROR;

    /*
     * Setup the list of output objects.
     */
    outputListPtr = SetupOutputObjects (tSippGlobPtr, &outputParms,
                                        handleList);
    if (outputListPtr == NULL)
        return TCL_ERROR;

    /*
     * Flag rendering in progress to prevent modifying SIPP during an Tk event
     * update.
     */
    tSippGlobPtr->rendering = TRUE;

    /*
     * Set the scanning direction if not decided and tell SIPP about it.
     */
    if (outputParms.scanDirection == TSIPP_SCAN_RANDOM) {
        outputParms.scanDirection = TSIPP_SCAN_TOP_DOWN;
    }

    if (outputParms.scanDirection == TSIPP_SCAN_BOTTOM_UP)
        sipp_render_direction (BOTTOM_TO_TOP);
    else
        sipp_render_direction (TOP_TO_BOTTOM);

    if (tSippGlobPtr->updateProc != NULL && outputParms.update > 0)
        sipp_set_update_callback (tSippGlobPtr->updateProc,
                                  tSippGlobPtr,
                                  outputParms.update);

    /*
     * Do the rendering.
     */
    if (outputParms.imgData.mode == LINE)
        DrawImage (tSippGlobPtr, &outputParms, outputListPtr);
    else
        RenderImage (tSippGlobPtr, &outputParms, outputListPtr);

    ok = FinishUpOutputObjects (tSippGlobPtr, &outputParms, outputListPtr);

    sipp_set_update_callback (NULL, NULL, 0);

    /*
     * Finish up with an update.  Aborts in here would not matter, since
     * we are done.  Don't execute if we have an error message in
     * tSippGlobPtr->interp, as it might wipe it out.
     */
    if (ok && (tSippGlobPtr->updateProc != NULL) && (outputParms.update > 0))
        (*tSippGlobPtr->updateProc) (tSippGlobPtr);

    /*
     * An error due to a signal in a non-Tk environment will be stashed in the
     * globals.
     */
    if (tSippGlobPtr->delayedError != NULL) {
        Tcl_ResetResult (interp);
        Tcl_SetResult (interp, tSippGlobPtr->delayedError, TCL_DYNAMIC);
        tSippGlobPtr->delayedError = NULL;
        ok = FALSE;
    }

    tSippGlobPtr->rendering = FALSE;

    return ok ? TCL_OK : TCL_ERROR;
}

/*=============================================================================
 * SippCopy --
 *   Implements the command:
 *     SippCopy [-flags] inputHandle outputHandle
 *
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippCopy (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t          *tSippGlobPtr = (tSippGlob_t *) clientData;
    tSippStorageClass_t  *inputClassPtr, *outputClassPtr;
    int                   nextArg = 1;
    bool                  ok, clear = TRUE;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc < 2)
        goto wrongArgs;

    if (argv [1][0] == '-') {
        if (STREQU (argv [1], "-clear")) {
            clear = TRUE;
        } else if (STREQU (argv [1], "-noclear")) {
            clear = FALSE;
        } else {
            Tcl_AppendResult (tSippGlobPtr->interp,
                              "invalid flag \"", argv [1],
                              "\", expected one of  \"-clear\", \"-noclear\"",
                              (char *) NULL);
            return TCL_ERROR;
        }
        nextArg = 2;
    }

    if ((argc - nextArg) != 2)
        goto wrongArgs;

    /*
     * Find the output classes for each handle.
     */

    inputClassPtr = TSippFindStorageClass (tSippGlobPtr,
                                           argv [nextArg]);
    if (inputClassPtr == NULL)
        return TCL_ERROR;
    
    outputClassPtr = TSippFindStorageClass (tSippGlobPtr,
                                            argv [nextArg + 1]);
    if (outputClassPtr == NULL)
        return TCL_ERROR;

    /*
     * Make sure that the scanning direction of the two classes don't
     * conflict and the you can copy from the input image.
     */
    if ((inputClassPtr->scanDirection != TSIPP_SCAN_RANDOM) &&
        (outputClassPtr->scanDirection != TSIPP_SCAN_RANDOM) &&
        (inputClassPtr->scanDirection != outputClassPtr->scanDirection)) {
        Tcl_AppendResult (interp, argv [nextArg], 
                          " scanning direction conflicts with ",
                          argv [nextArg + 1], (char *) NULL);
        return TCL_ERROR;
    }

    if (inputClassPtr->copyImage == NULL) {
        Tcl_AppendResult (interp, argv [nextArg],
                          " can't be used as a copy source", (char *) NULL);
        return TCL_ERROR;
    }

    ok = (*inputClassPtr->copyImage) (tSippGlobPtr,
                                      argv [nextArg],
                                      argv [nextArg + 1],
                                      outputClassPtr,
                                      clear);

    return ok ? TCL_OK : TCL_ERROR;

  wrongArgs:
    Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                      " [-flags] inputHandle outputHandle",
                      (char *) NULL);
    return TCL_ERROR;
}

/*=============================================================================
 * TSippRenderInit --
 *   Initialized rendering commands.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
void
TSippRenderInit (tSippGlobPtr)
    tSippGlob_t  *tSippGlobPtr;
{
    static tSippTclCmdTbl_t cmdTable [] = {
        {"SippRender", (Tcl_CmdProc *) SippRender},
        {"SippCopy",   (Tcl_CmdProc *) SippCopy},
        {NULL,        NULL}
    };

    TSippInitCmds (tSippGlobPtr, cmdTable);
}

