/*
 *=============================================================================
 *                                tSippTk.c
 *-----------------------------------------------------------------------------
 * Tcl-SIPP Tk utilities.
 *-----------------------------------------------------------------------------
 * Copyright 1992-1995 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tSippTk.c,v 5.4 1995/01/06 09:35:52 markd Exp $
 *=============================================================================
 */

#include "../src/tSippInt.h"

#include "tk.h"
#include <X11/X.h>
#include <X11/Xlib.h>

/*
 * Maximum value of an X11 color value.
 */
#define X_COLOR_MAX 65535


/*
 * Internal prototypes.
 */
static void
TkUpdate _ANSI_ARGS_((void_pt  clientData));

static int 
SippColorToXColor _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                               int           argc,
                               char        **argv));

static int 
XColorToSippColor _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                               int           argc,
                               char        **argv));

/*=============================================================================
 * TkUpdate -- 
 *   Allow Tk to handle pending events, also checks for signals.
 *
 * Parameters:
 *   o clientData (I) - Actually a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static void
TkUpdate (clientData)
     void_pt  clientData;
{
    tSippGlob_t *tSippGlobPtr = (tSippGlob_t *) clientData;

    if (tcl_AsyncReady) {
        if (Tcl_AsyncInvoke (tSippGlobPtr->interp, TCL_OK) == TCL_ERROR)
            Tk_BackgroundError (tSippGlobPtr->interp);
        Tcl_ResetResult (tSippGlobPtr->interp);
    }

    TSippCallCmd (tSippGlobPtr,
                  "update",
                  (char *) NULL);
}

/*=============================================================================
 * SippColorToXColor --
 *   Implements the "tox" subcommand of SippColor, converting a SIPP color to
 * a X color.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o argc - Count of arguments to SippColor.
 *   o argv - Arguments to SippColor.
 * Returns:
 *   TCL_OK or TCL_ERROR.
 *-----------------------------------------------------------------------------
 */
static int 
SippColorToXColor (tSippGlobPtr, argc, argv)
    tSippGlob_t  *tSippGlobPtr;
    int           argc;
    char        **argv;
{
    Color  color; 
    char   red [TCL_DOUBLE_SPACE];
    char   green [TCL_DOUBLE_SPACE];
    char   blue [TCL_DOUBLE_SPACE];
    
    if (argc != 3) {
        Tcl_AppendResult (tSippGlobPtr->interp, "wrong # args: ", argv [0],
                          " ", argv [1], " color", (char *) NULL);
        return TCL_ERROR;
    }
    if (!TSippConvertColor (tSippGlobPtr,
                            argv [2],
                            &color))
        return TCL_ERROR;

    sprintf (red, "%04x", (int) (color.red * X_COLOR_MAX));
    sprintf (green, "%04x", (int) (color.grn * X_COLOR_MAX));
    sprintf (blue, "%04x", (int) (color.blu * X_COLOR_MAX));
    
    Tcl_AppendResult (tSippGlobPtr->interp, "rgb:", red, "/",
                      green, "/", blue, (char *) NULL);
    return TCL_OK;
}

/*=============================================================================
 * XColorToSippColor --
 *   Implements the "tosipp" subcommand of SippColor, converting an X color
 * to a SIPP color.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o argc - Count of arguments to SippColor.
 *   o argv - Arguments to SippColor.
 * Returns:
 *   TCL_OK or TCL_ERROR.
 *-----------------------------------------------------------------------------
 */
static int 
XColorToSippColor (tSippGlobPtr, argc, argv)
    tSippGlob_t  *tSippGlobPtr;
    int           argc;
    char        **argv;
{
    XColor exactColor, mapColor;
    Color  color;
    char   colorList [3 * TCL_DOUBLE_SPACE];

    if (argc != 3) {
        Tcl_AppendResult (tSippGlobPtr->interp, "wrong # args: ", argv [0],
                          " ", argv [1], " xcolor", (char *) NULL);
        return TCL_ERROR;
    }

    if (XLookupColor (Tk_Display (tSippGlobPtr->tkMainWindow),
                      Tk_Colormap (tSippGlobPtr->tkMainWindow),
                      argv [2],
                      &exactColor,
                      &mapColor) == 0) {
        Tcl_AppendResult (tSippGlobPtr->interp, "invalid X11 color \"",
                          argv [2], "\"", (char *) NULL);
        return TCL_ERROR;
    }
    color.red = (double) exactColor.red / (double) X_COLOR_MAX;
    color.grn = (double) exactColor.green / (double) X_COLOR_MAX;
    color.blu = (double) exactColor.blue / (double) X_COLOR_MAX;

    TSippFormatColor (tSippGlobPtr,
                      color,
                      colorList);

    Tcl_AppendResult (tSippGlobPtr->interp, colorList, (char *) NULL);
    return TCL_OK;
}

/*=============================================================================
 * TkSipp_Init -- 
 *   Initialize the Tcl/Sipp environment including Tk commands.  This is called
 * in place of TSippInitialize in a Tk application.
 *
 * Parameters:
 *   o interp (I) - Pointer to the Tcl interpreter.
 * Returns:
 *   TCL_OK or TCL_ERROR.
 *-----------------------------------------------------------------------------
 */
int
TkSipp_Init (interp)
    Tcl_Interp *interp;
{
    static char  *aTkCommand = "button";
    Tcl_CmdInfo   cmdInfo;
    tSippGlob_t  *tSippGlobPtr;

    /*
     * Locate the TK main window by finding a Tk command's client data.
     */
    if (!Tcl_GetCommandInfo (interp, aTkCommand, &cmdInfo)) {
        Tcl_AppendResult (interp, "unable to find Tk command \"",
                          aTkCommand, "\" need to locate main Tk window for ",
                          "Tcl-SIPP", (char *) NULL);
        return TCL_ERROR;
    }

    /*
     * Initialize the basic TSIPP commands.
     */
    tSippGlobPtr = TSippInitialize (interp);
    if (tSippGlobPtr == NULL)
        return TCL_ERROR;

    /*
     * Save Tk information in the TSIPP globals and set up the photo image
     * access.
     */
    tSippGlobPtr->tkMainWindow = cmdInfo.clientData;
    tSippGlobPtr->updateProc = TkUpdate;
    tSippGlobPtr->sippColorToXColor = SippColorToXColor;
    tSippGlobPtr->xColorToSippColor = XColorToSippColor;
    TSippPhotoInit (tSippGlobPtr);

    return TCL_OK;
}
