#!/usr/local/bin/wish

# Old Hacker's Clock - by Enrico Colombini (erix@mclink.it) Oct 1997

# I put this program into the public domain. Please add your name
# if you change the code.

# This is my first Tcl/Tk script, I wrote it on a Linux box after reading
# John Ousterout's excellent book ("Tcl and the Tk toolkit", Addison-Wesley).
# The clock will sometimes skip a beat, since the total cycle time is
# slightly longer than a second, but the time indicated will always be 
# right (+- 1 sec). I could have fixed this by continuously reading the 
# time and updating only on changes, but that would have been a waste of 
# system resources.
# The window is not resizeable because of a few puzzling (at least for me)
# side effects with my configuration: Linux 2.0.3, XFree 3.3, fvwm, Tk 8.0.
# Windows 9x situation seems to be somewhat worse, even if the standard 
# clock configuration works.
#
# Have a nice time :-)
#
#   Enrico


# --- globals -----------------------------------------------------------

# version

set vers "v 1.0, Oct 97"

# command line help, valid options

set help " #\n\
	# Old Hacker's Clock $vers - by Enrico Colombini - public domain\n #\n\
	valid options:\n\
	-align <frameAlign> <bitAlign>  #h|v, default: h v\n\
	\ example: -align v h\n\
	-size <bitHsize> <bitVsize>     #usual Tk size units, default: 4 4\n\
	\ example: -size 14 2c\n\
	-ampm                           #enable am/pm 12h mode, default: 24h\n\
	\ example: -ampm\n"

# colors	

set bgColor #e0e0c0 ;#background color = bit "off" color
set hColor  #0000ff ;#bit "on" colors for the 3 frames
set mColor  #ff0000
set sColor  #00e000
set apColor #c0c000 ;#color for am/pm flag

# global vars (options and their defaults)

set frameAlign h
set bitAlign   v
set hSize      8
set vSize      8
set ampm       no


# --- initialization/reconfiguration ------------------------------------

# build a 6-bit frame with all bits off, alignment could be changed later

proc makeFrame {fr} {
    global bgColor
    frame $fr -relief sunken -borderwidth 1 -bg $bgColor
    foreach bit {b0 b1 b2 b3 b4 b5} {
	frame $fr.$bit -relief raised -borderwidth 1 -bg $bgColor
	pack $fr.$bit -side bottom -padx 1m -pady 0.8m -expand yes -fill both
    }
}

# create all frames and bits

proc initVisual {} {
#    wm title . "Oh-clock" ;# trouble under X, at least with fvwm & Tk 8.0
    wm resizable . no no ;#same as above
    global bgColor
    makeFrame .h
    makeFrame .m
    makeFrame .s
    pack .h .m .s -side left -expand yes -fill both
}

# set frames & bits alignment, remember them in globals

proc setAlignment {f b} {
    global frameAlign bitAlign
    if {$f == "h"} {
	pack configure .h .m .s -side left
    } elseif {$f == "v"} {
	pack configure .h .m .s -side top
    }
    foreach fr {.h .m .s} {
	foreach bit {b0 b1 b2 b3 b4 b5} {
	    if {$b == "h"} {
		pack configure $fr.$bit -side right
	    } elseif {$b == "v"} {
		pack configure $fr.$bit -side bottom
	    }
	}
    }
    set frameAlign $f
    set bitAlign $b
}

# set bit size, remember it in globals (just to keep things in sync)

proc setSize {hs vs} {
    global hSize vSize
    foreach fr {.h .m .s} {
	foreach bit {b0 b1 b2 b3 b4 b5} {
	    $fr.$bit configure -width $hs -height $vs
	}
    }
    set hSize $hs
    set vSize $vs
}

# set am/pm mode, remember it in global

proc setAmpm {ap} {
    global ampm bgColor apColor
    if {$ap} {
	.h.b4 configure -bg $bgColor -relief flat
	.h.b5 configure -bg $bgColor -relief raised
    } else {
	.h.b5 configure -bg $bgColor -relief flat
	.h.b4 configure -bg $bgColor -relief raised
    }
    set ampm $ap
}


# --- time updating -----------------------------------------------------

# set bits of frame $fr to value $val, use $color for "on" bits
# doesn't do harm in .h because .h.b5 is always zero on Earth

proc setBits {fr val color} {
    global bgColor
    foreach bit {b0 b1 b2 b3 b4 b5} {
	if {$val & 0x01} {
	    $fr.$bit configure -bg $color
	} else {
	    $fr.$bit configure -bg $bgColor
	}
	set val [expr $val >> 1]
    }
}

# update displayed h:m:s time, ask to be recalled 1 sec later

proc updateTime {} {
    global hColor mColor sColor apColor ampm
    set hms [clock format [clock seconds] -format %H:%M:%S]
    scan $hms "%d:%d:%d" h m s ;#force decimal interpreration of "0n"
    set pm [expr $h >= 12]
    if {$ampm} {
	if {$pm} {set h [expr $h - 12]}
	if {$h == 0} {set h 12}
    }
    setBits .h $h $hColor
    if {$ampm && $pm} {
	.h.b5 configure -bg $apColor
    }
    setBits .m $m $mColor
    setBits .s $s $sColor
    after 1000 {updateTime}
}


# --- menus & accessory procs -------------------------------------------

# 'local' globals

set alignTemp 0 ;#just for documentation

# build menus

proc initMenus {} {
    global vers
    menu .mn
    .mn add checkbutton -label "AM/PM mode" -variable ampm\
	    -command {setAmpm $ampm; updateTime}
    .mn add separator
    .mn add radiobutton -label "Horizontal strip" -variable alignTemp\
	    -value hh -command {setAlignment h h}
    .mn add radiobutton -label "Vertical strip" -variable alignTemp\
	    -value vv -command {setAlignment v v}
    .mn add radiobutton -label "By lines" -variable alignTemp\
	    -value vh -command {setAlignment v h}
    .mn add radiobutton -label "By columns" -variable alignTemp\
	    -value hv -command {setAlignment h v}
    .mn add separator
    .mn add cascade -label "Standard sizes" -menu .mn.mss
    .mn add cascade -label "Other sizes" -menu .mn.mos
    .mn add separator
    .mn add cascade -label "Info" -menu .mn.info
    .mn add command -label "Kill time" -command {exit}
    menu .mn.mss
    .mn.mss add command -label "Nano (1x1)" -command {setSize 1 1}
    .mn.mss add command -label "Micro (2x2)" -command {setSize 2 2}
    .mn.mss add command -label "Milli (4x4)" -command {setSize 4 4}
    .mn.mss add command -label "Unit (8x8)" -command {setSize 8 8}
    .mn.mss add command -label "Kilo (16x16)" -command {setSize 16 16}
    .mn.mss add command -label "Mega (32x32)" -command {setSize 32 32}
    .mn.mss add command -label "Giga (64x64)" -command {setSize 64 64}
    .mn.mss add command -label "Tera (128x128)" -command {setSize 128 128}
    menu .mn.mos
    .mn.mos add command -label "H-LED (12x8)" -command {setSize 12 8}
    .mn.mos add command -label "V-LED (8x12)" -command {setSize 8 12}
    .mn.mos add command -label "H-strip (32x4)" -command {setSize 32 4}
    .mn.mos add command -label "V-strip (4x32)" -command {setSize 4 32}
    .mn.mos add command -label "H-block (64x16)" -command {setSize 64 16}
    .mn.mos add command -label "V-block (16x64)" -command {setSize 16 64}
    .mn.mos add separator
    .mn.mos add command -label "Hint: you can choose any size you like\
	    by using -size x y on the command line" -command {}
    menu .mn.info -tearoff no
    .mn.info add command -label "$vers - by Enrico Colombini\
	    (erix@mclink.it) - public domain"
}

# show pop-up menu in response to event

proc popMenu {x y} {
    global alignTemp frameAlign bitAlign
    set alignTemp $frameAlign$bitAlign
    .mn post $x $y
}


# --- main script & options ---------------------------------------------

# get options from command line, show help lines if invalid options
# (some invalid formats are ignored and could trigger an error later)
# ...almost as readable as obfuscated C...

proc getOptions {} {
    global argc argv frameAlign bitAlign hSize vSize ampm help
    set args [string tolower [join $argv]]
    set validOpt [expr [\
	    regexp {\-align ([hv]) ([hv])} $args x frameAlign bitAlign]\
	    + [regexp {\-size ([0-9\.cimp]+) ([0-9\.cimp]+)}\
	      $args x hSize vSize]\
	    + [set ampm [regexp {\-ampm} $args]]]
    if {$argc > 0 && ! $validOpt} {
	puts $help
	exit
    }
}

# startup script

getOptions
initVisual
setAlignment $frameAlign $bitAlign
setSize $hSize $vSize
setAmpm $ampm
initMenus
bind . <1> {popMenu %X %Y}
updateTime
