# man2tcl.tcl --
#
#   This file contains a Tcl source that turns a man page of the
#   form used for Tcl and Tk into a Tcl script that invokes
#   a Tcl command for each construct in the man page. The
#   script can then be eval'ed to translate the manual entry
#   into some other format such as MIF or HTML.
#
# Usage:
#
#   man2tcl fileName



# QuoteText --
#
#   Return the "string" argument, adding quote characters around
#   any special Tcl characters so that they'll just be treated
#   as ordinary text.
#
# Results:
#   Returns generated text.
#
# Side effects:
#   None.

proc QuoteText text {
    if {$text == ""} {
	return
    }
    regsub -all {[ ;$\[\{"\\]} $text {\\&} text
    set text "text $text"
    if [regsub -all \t $text "\ntab\ntext " text] { ;# interior tabs
	regsub -all "text \n" $text "" text	    ;# delete blank text lines
	regsub  "\ntext $" $text "" text	    ;# delete trailing blank
    }
    return $text\n
}



# DoText --
#
#   This procedure is called to handle a line of troff text.
#   It parses the text, generating Tcl commands for text and
#   for formatting stuff such as font changes.
#
# Results:
#   Returns the generated text.
#
# Side effects:
#   None.

proc DoText line {
    # the regexp splits 'line' into three parts:
    #	a -- text preceeding the first '\'
    #	b -- the char immediately following the '\'
    #	line -- the remaining text
    #
    while {[regexp {([^\]*)\\(.)((.)*)} $line {} a b line]} {
	append data [QuoteText $a]
	switch -- $b {
	    f { 
		append data "font [string index $line 0]\n"
		set line [string range $line 1 end]
	    }
	    e { 
		append data "text \\\\\n"
	    }
	    - { 
		append data "dash\n"
	    }
	    . { 
		append data "dash\n"
	    }
	    ( { 
		append data "char \{\\\([string range $line 0 1]\}\n"
		set line [string range $line 2 end]
	    }
	    default {
		append data "char \{\\$b\}\n"
	    }
	}
    }
    append data [QuoteText $line]
    append data newline\n
}





# DoMacro --
#
#   This procedure is called to handle a macro invocation.
#   It parses the arguments to the macro and generates a
#   Tcl command to handle the invocation.
#
# Results:
#   Returns the generated text.
#
# Side effects:
#   None.

proc DoMacro line {
    regsub -all {[;$\[\{\\]} $line {\\&} line
    regsub -all {\\ } $line { } line
    if {$line=="."} return ""
    return "macro [string range $line 1 end]\n"
}



# man2tcl --
#
#   This procedure is the main program, which does all of the work.
#
# Results:
#   Returns a Tcl script.
#
# Side effects:
#   None. 

proc man2tcl file {
    global line_number inDE
    set inDE 0
    set line_number 0	
    set fpin [open $file r]
    while {[gets $fpin line] != -1} {
	incr line_number
	switch -regexp -- $line {
	    {^\.\.} {
		set inDE 0
	    }
	    {^\.de} {
		set inDE 1
	    }
	    {^\.} {
		if !$inDE {
		    append data [DoMacro $line]
		}
	    }
	    {^'} {
	    }
	    {^( )*$} {
		append data "macro sp\n"
	    }
	    default {
		if !$inDE {
		    append data [DoText $line]
		}
	    }
	}
    }
    close $fpin
    return $data
}

return


set file "../man/tcl8.0a1/doc/history.n"
puts [man2tcl $file]
# eval [man2tcl $file]


return

set file "/Net & Scripting/neotcl7.5/itcl2.1/itcl/doc/body.n"

set fp [open $file.tcl w]
puts $fp [man2tcl $file]
close $fp

