if [catch {

# man2html.tcl --
#
# This file contains procedures that work in conjunction with 
# man2tcl.tcl, man2html1.tcl and man2html2.tcl files to generate
# HTML files from Tcl manual entries. The 'main' procedure,
# defined in this file, is the entry point for the convertion
# process.
#
# Copyright (c) 1996 by Sun Microsystems, Inc.



# Global variables used by these scripts:
#
# footer -	Info to insert at end of each page.
#
# package -	package directory name. Man pages are assumed to be in
#		$package/doc directory.
#
# NAME_file -	array indexed by command NAME and containing its file name.
#		Used for inserting hyperlinks and creating the contents file. 
#		Created during pass 1.
#
# NAME_chg -	array indexed by command NAME and containing an asterick 
#		(*) if page has change bars. Created during pass 1.
#
# KEY_file -	array indexed by KEYWORD and containing file names used
#		for hyperlinks. Created during pass 1; read only by doXfer.
#
# THinfo -	Array containing information about the current man page
#		found in the TH macro. For example:
#		    THinfo(fname)   - Tcl_UpVar
#		    THinfo(page)    - 3
#		    THinfo(vers)    - 7.4
#		    THinfo(lib)     - Tcl
#    		    THinfo(pname)   - {Tcl Library Procedures}




# sarray -
#
# Similar to parray. Save arrays to a file so they can be sourced. 
# Debug aid.
#
# Arguments:
#   file -	Name of the output file
#   args -	Name of the arrays to save
#
proc sarray {file args} {
    set file [open $file w]
    foreach a $args {
	upvar 1 $a array
	if ![array exists array] {
	    puts "sarray: \"$a\" isn't an array"
	    break
	}
	foreach name [lsort [array names array]] {
	    regsub -all " " $name "\\ " name1
	    puts $file "set ${a}($name1) \{$array($name)\}"
	}
    }
    close $file
}



# doListing --
#
# Writes a multi-column list to a file (much like 'ls'). Searches an array 
# for entries that match the input pattern.
#
# Arguments:
#   file -	Output file pointer.
#   array -	the array to search
#   pattern -	glob style match pattern

proc doListing {file pattern} {
    global NAME_file NAME_chg

    set pattern [string tolower $pattern]
    set max_len 0
    foreach name [lsort [array names NAME_file]] {
	set ref $NAME_file($name)
	if [string match $pattern $ref] {
	    lappend type $name
       	    if {[string length $name] > $max_len} {
       		set max_len [string length $name]
	    }
 	}
    }
    if [catch {llength $type} ] {
	puts stderr "	   doListing: no names matched pattern ($pattern)"
	return
    }
    incr max_len 2
    set ncols [expr 90/$max_len]
    set nrows [expr int( ceil( [llength $type] / $ncols. ) ) ]

    set index 0
    foreach f $type {
	lappend row([expr $index % $nrows]) $f
	incr index
    }

    puts -nonewline $file "<PRE>"
    for {set i 0} {$i<$nrows} {incr i} {
	set num_spaces 0
	foreach name $row($i) {
	    if [catch {set chg $NAME_chg($name)}] {set chg " "}
	    set spaces [format "%-*s" $num_spaces ""]
	    set ref [lindex [split $NAME_file($name) /] end]
	    puts -nonewline $file "$spaces$chg<A HREF=\"$ref.html\">$name</A>"
	    set num_spaces [expr $max_len - [string length $name] - 1]
	}
	puts $file {}
    }
    puts $file "</PRE>"
}



# doContents --
#
# Generates a HTML contents file using an array as its input database.
#
# Arguments:
#   array -	array indexed by command name and whose elements are the 
#		file reference.
#   packageName - string used in the title and sub-heads of the HTML page. 
#		Normally name of the package without version numbers.

proc doContents {array file packageName} {
    global footer

    upvar $array the_array

    set file [open $file w]

    puts $file "<HTML><HEAD><TITLE>$packageName Manual</TITLE></HEAD><BODY>"

    puts $file "<H3>$packageName Applications</H3>"
    doListing $file "$packageName*.1"

    puts $file "<HR><H3>$packageName Built-In Commands</H3>"
    doListing $file "$packageName*.n"

    puts $file "<HR><H3>$packageName Library Procedures</H3>"
    doListing $file "$packageName*.3"

    puts $file "<P>* Revised pages."
    puts $file $footer
    puts $file "</BODY></HTML>"
    close $file
}



# doXref --
#
#
# Arguments:
#   fileName -	    Name of the output file.

proc doXref dir {
    global KEY_file footer

    file mkdir $dir
	
    # create the main cross-reference page
    #
    set fp [open $dir/xref.html w]
    puts $fp "<head>"
    puts $fp "<title>Cross-reference</title>"
    puts $fp "</head>"
    puts $fp "<FRAMESET ROWS=\"70,*,35\">"
    puts $fp "    <FRAME SRC=\"a2z.html\" SCROLLING=\"no\" NORESIZE>"
    puts $fp "    <FRAME SRC=\"xref_A.html\" NAME=\"xref\">"
    puts $fp "    <FRAME SRC=\"footer.html\" SCROLLING=\"no\" NORESIZE>"
    puts $fp "</FRAMESET>"
    puts $fp "<NOFRAMES>"
    puts $fp "<H1>Index</H1>"
    puts $fp "The Index has been designed to work best with a frames-compatible browser."
    puts $fp "Unfortunately your browser  cannot handle frames."
    puts $fp "</body>"
    puts $fp "</NOFRAMES>"
    puts $fp "</HTML>"
    close $fp

    catch {unset alpha}
    for {set i 1} {$i<=26} {incr i} {
	lappend alpha [format "%c" [expr $i+0x40]]
    }

    # create content for the top frame: the A to Z selector
    #
    set fp [open $dir/a2z.html w]
    puts $fp "<HTML>"
    puts $fp "<BASE TARGET=\"xref\">"
    puts $fp "<CENTER>"
    puts $fp "<H2>Keyword cross-reference</H2>"
    puts $fp "<H3>"
    foreach char $alpha {
	puts $fp "<A HREF=\"xref_$char.html\">$char </A>"
    }
    puts $fp "</H3>"
    puts $fp "</CENTER>"
    puts $fp "</HTML>"
    close $fp


    # create content for the bottom frame: one for each letter of the alphabet
    #
    foreach char $alpha {
	set fp [open $dir/xref_$char.html w]
	puts $fp "<HTML>"
	puts $fp "<BASE TARGET=\"_top\">"
	puts $fp "<H2> $char </H2>"
	set names [lsort [array names KEY_file \[[string tolower $char]$char\]*]]
	if {[llength names] > 0} {
#	    puts $fp "<UL>"
	    foreach li $names {
		set line {}
		puts $fp "<LI> $li -- "
		foreach file [lsort $KEY_file($li)] {
		    regexp -nocase {/[a-z_0-9]+} $file name
		    set name [string range $name 1 end]
		    lappend line "<A HREF=\"../$file.html\">$name</A>"
		}
		puts $fp [join $line ", \n"]
	    }
#	    puts $fp "</UL>"
	}
	puts $fp "</HTML>"
	close $fp
    }
}



# contents --
#
# Builds the main contents HTML page.
#
# Arguments:
#   file -	file name of the contents HTML page
#   packages -	list of package names
#   libs -	list of package library names

proc contents {file packages libs} {

    set fp [open $file w]
    puts $fp "<HTML><HEAD><TITLE>Tcl/Tk Contents</TITLE></HEAD><BODY>"
    puts $fp "<H2><CENTER>Tcl/Tk Documentation</CENTER></H2>"
    puts $fp "<PRE>\n</PRE>"

    set i 0
    foreach package $packages {
	puts $fp "<H3><A HREF=\"$package/contents.html\">$package</A></H3>"
	puts $fp "Table of contents for [lindex $libs $i] commands and library calls."
	puts $fp "<PRE>\n</PRE>"
	incr i
    }
    puts $fp "<H3><A HREF=\"xref/xref.html\">Cross-reference</A></H3>"
    puts $fp "An index built from the 'Keywords' section of the man pages."
    puts $fp "<PRE>"
    puts $fp " "
    puts $fp " "
    puts $fp "</PRE>"
    puts $fp "Send comments, suggestions and bug reports to me at "
    puts $fp "<A HREF=\"mailto:tom-l-davis@ti.com\">tom-l-davis@ti.com</a>."
    puts $fp "<P>"
    puts $fp "<HR>"
    puts $fp "<FONT SIZE=\"SMALL\">"
    puts $fp "Copyright &#169; 1989-1994 The Regents of the University of California."
    puts $fp "<BR>"
    puts $fp "Copyright &#169; 1994-1996 Sun Microsystems, Inc."
    puts $fp "</FONT>"
    puts $fp "</BODY></HTML>"

    close $fp

}



# foot --
#
# Builds the footer info for the cross-reference page.
#
# Arguments:
#   file -	file name of the cross-reference HTML page
#   packages -	list of package names

proc foot {file packages} {

    foreach package $packages {
	lappend h "<A HREF=\"../$package/contents.html\">$package</A>"
    }

    set fp [open $file w]
    puts $fp "<BASE TARGET=\"_top\">"
    puts $fp "<TABLE CELLSPACING=0 CELLPADDING=0 border=0>"
    puts $fp "<TR height=30>"
    puts $fp "<TD NOWRAP>"
    puts $fp "<B>\[ </B>"
    puts $fp "[join $h " <B> | </B> "]"
    puts $fp "<B> \]</B>"
    puts $fp "</TD>"
    puts $fp "<TD NOWRAP width=10>"
    puts $fp "</TD>"
    puts $fp "<TD NOWRAP>"
    puts $fp "<FONT SIZE=\"SMALL\">"
    puts $fp "Copyright &#169; 1989-1994 The Regents of the University of California."
    puts $fp "<BR>"
    puts $fp "Copyright &#169; 1994-1996 Sun Microsystems, Inc."
    puts $fp "</FONT>"
    puts $fp "</TD>"
    puts $fp "</TR>"
    puts $fp "</TABLE>"
    close $fp
}



# footer --
#
# Builds footer info for all HTML pages except for the cross_reference.
#
# Arguments:
#   packages	- list of package names

proc footer {packages} {

    foreach package $packages {
	lappend h "<A HREF=\"../$package/contents.html\">$package</A>"
    }
    lappend h "<A HREF=\"../xref/xref.html\">X-ref</A>"

    return "<HR>
<center><B>\[</B> [join $h " <B>|</B> "] <B>\]</B></center>
<P>
<FONT SIZE=\"SMALL\">
Copyright &#169; 1989-1994 The Regents of the University of California.
<BR>
Copyright &#169; 1994-1996 Sun Microsystems, Inc.
</FONT>"
}



# main --
#
# Main entry point for building all the HTML pages.
#
# Arguments:
#   tcl_dir -	The top level directory containing all the Tcl distributions.
#		For example, the unconverted man pages for Tcl 7.5 are expected 
#		to be located in directory $tcl_dir/tcl7.5/man.
#   html_dir -	html versions of the man pages are put in this directory
#		The script creates this directory. After the conversion, it
#		will contain
#   packages -	Packages to convert. For example tcl7.5 and tk4.1. 


proc main {tcl_dir html_dir args} {
    global footer NAME_file KEY_file NAME_chg package THinfo

    set packages $args

    # make sure there are doc directories for each package
    #
    foreach i $packages {
	if ![file isdirectory $tcl_dir/$i/doc] {
	    puts stderr "Error: $tcl_dir/$i/doc for package $i is missing or not a directory"
	    return
	}
    }

    # we want to start with a clean sheet
    #
    if [file exists $html_dir] {
	puts stderr "Error: file or directory for HTML output already exists: $html_dir"
	return
    } else {
	file mkdir $html_dir
    }

    catch {unset NAME_file}
    catch {unset KEY_file}
    source man2tcl.tcl	

    # make the hyperlink arrays for all packages
    #
    source man2html1.tcl
    foreach package $packages {
	catch {file mkdir $html_dir/$package}
	
	# each package adds to the hyperlink database arrays: NAME_file 
	# and KEY_file
	#
	puts "\nScanning man pages in $tcl_dir/$package/doc..."
	doPass1 $tcl_dir/$package/doc
    }

    # clean up the NAME_file and KEY_file database arrays
    #
    foreach name [lsort [array names NAME_file]] {
	set file_name $NAME_file($name)
	if {[llength $file_name] > 1} {
	    set file_name [lsort $file_name]
	    puts stderr "Warning: '$name' multiply defined in: $file_name; using last"
	    set NAME_file($name) [lindex $file_name end]
	}
    }
    catch {unset KEY_file()}

    # debug aids
    #
    #	sarray $html_dir/xref.tcl NAME_file KEY_file NAME_chg
    # 	source $html_dir/xref.tcl

    # define macros for pass 2
    #
    source man2html2.tcl
    set footer [footer $packages]

    foreach package $packages {
	
	# now translate the man pages to HTML pages
	#
	puts "\nBuilding html pages from man pages in $tcl_dir/$package/doc..."
	doPass2 $tcl_dir/$package/doc $html_dir

	# build the contents file from the changed array
	#
	puts "\nGenerating contents.html for $package"; update
	doContents changed $html_dir/$package/contents.html $THinfo(lib)
	lappend libs $THinfo(lib)
    }

    # build the cross-reference file
    #
    puts "\nGenerating xref.html"; update
    doXref $html_dir/xref
    foot $html_dir/xref/footer.html $packages

    # build the main contents file
    #
    contents $html_dir/Contents.html $packages $libs
}


} result] {
    global errorInfo
    puts stderr $result
    puts stderr "in"
    puts stderr $errorInfo
}

# main ../man7.6  ../Tcl-Tk_docs tcl7.6 tk4.2
