# man2html2.tcl --
#
# This file defines procedures that are used during the second pass of the
# man page to html conversion process. It is sourced by man2html.tcl.
#
# Copyright (c) 1996 by Sun Microsystems, Inc.



# Global variables used by these scripts:
#
# manFile -	name of the current man page file, e.g. 'array.n'.
#
# file -	Where to output the generated HTML. (file pointer)
#
# package -	package directory name. Man pages are assumed to be in
#		$package/doc directory.
#
# THinfo -	Array containing information about the current man page
#		found in the TH macro. For example:
#		    THinfo(fname)   - Tcl_UpVar
#		    THinfo(page)    - 3
#		    THinfo(vers)    - 7.4
#		    THinfo(lib)     - Tcl
#    		    THinfo(pname)   - {Tcl Library Procedures}
#
# NAME_file -	array indexed by NAME and containing file names used
#		for hyperlinks. Read only during pass 2.
#
# state -	state variable defining behavior of 'text' proc during
#		pass 2.
#
# inDT -	set by 'TPmacro', cleared by 'newline'. Used to terminate
#		a dictionary term <DT> and start the dictionary definition
#		<DD> while in a dictionary list <DL>.
#
# inVS -	In change bar section. See the VS and VE macros.
#
# noFill - 	When set, newlines cause a linebreak <BR>.
#
# inPRE -	When set, text is pre-formated <PRE>. 
#
# paraStk -	stack oriented list containing currently active
#		HTML tags (UL, OL, DL). Local to 'para' proc.
#
# curFont -	Name of special font that is currently in
#		use.  Null means the default paragraph font
#		is being used.
#
# charCnt -	Number of characters on current line. Used by tab proc
#
# tabString -	String defining the current tab settings. Set by setTabs
#		and used by the tab proc.
#
# footer -	Info to insert at end of each page.




# initGlobals --
#
# This procedure is invoked to set the initial values of all of the
# global variables local to pass 2, before processing a man page.
#
# Arguments:
#   None.

proc initGlobals {} {
    global state curFont inPRE inVS inDT noFill

    para init
    set inDT 0
    set inPRE 0
    set inVS 0
    set noFill 0
    set state 0
    set curFont ""
    setTabs 0.5i
}



# font --
#
# This procedure is invoked to handle font changes in the text
# being output.
#
# Arguments:
#   type -	Type of font: R, I, B, or S.

proc font type {
    global state
    switch $type {
	B {
	    beginFont B
	    if {$state == "INSERT"} {
      	      	set state REF
	    }
	}
	I {
	    beginFont I
	    if {$state == "REF"} {
		set state INSERT
	    }
	}
	P -
	R {
	    endFont
	    if {$state == "REF"} {
		set state INSERT
	    }
	}
	S {
	}
	default {
	    puts stderr "Unknown font: $type"
	}
    }
}



# beginFont --
#
# Arranges for future text to use a special font, rather than
# the default paragraph font.
#
# Arguments:
#   font -	Name of new font to use.

proc beginFont font {
    global curFont file

    if {$curFont == $font} {
	return
    }
    endFont
    puts -nonewline $file <$font>
    set curFont $font
}



# endFont --
#
# Reverts to the default font for the paragraph type.
#
# Arguments:
#   None.

proc endFont {} {
    global curFont file 

    if {$curFont != ""} {
	puts -nonewline $file </$curFont>
	set curFont ""
    }
}



# text --
#
# This procedure adds text to the current paragraph.  If this is
# the first text in the paragraph then header information for the
# paragraph is output before the text.
#
# Arguments:
#   string -	Text to output in the paragraph.

proc text string {
    global file state inDT inPRE charCnt

    incr charCnt [string length $string]
    regsub -all {&} $string {\&amp;}  string
    regsub -all {<} $string {\&lt;}  string
    regsub -all {>} $string {\&gt;}  string
    regsub -all {"} $string {\&quot;}  string
    switch $state {
	REF { 
	    if {!$inDT && !$inPRE} {
		set string [insertRef $string]
	    }
	}
	SEE {
 	    foreach i [split $string ,] {
		lappend str [insertRef [string trim $i]]
 	    }
 	    set string [join $str ", "]
	}
    }
    puts -nonewline $file "$string"
}



# insertRef --
#
#
# Arguments:
#   string -	    Text to output in the paragraph.

proc insertRef string {
    global NAME_file manFile package

    # does the string start with a word that matches a name in NAME_file?
    #
    regexp -nocase {[a-z_0-9]+} $string name
    if ![catch {set ref $NAME_file($name)} ] {
	if {$ref != "$package/$manFile"} {
	    set ref [split $ref /]
	    if {[lindex $ref 0] == $package} {
		set ref [lindex $ref 1]
	    } else {
		set ref "../[join $ref /]"
	    }
	    set string "<A HREF=\"$ref.html\">$string</A>"	
	}
    }
    return $string
}



# macro --
#
# This procedure is invoked to process macro invocations that start
# with "." (instead of ').
#
# Arguments:
#    name -	The name of the macro (without the ".").
#    args -	Any additional arguments to the macro.

proc macro {name args} {
    switch $name {
	AP {
	    if {[llength $args] != 3} {
		puts stderr "Bad .AP macro: .$name [join $args " "]"
	    }
	    setTabs {1.25i 2.5i 3.75i}
	    TPmacro {}
	    font B
	    text "[lindex $args 0]  "
	    font I
	    text "[lindex $args 1]"
	    font R
	    text " ([lindex $args 2])"
	    newline
	}
	AS {}				    ;# next page and previous page
	br {
	    newline
#	    lineBreak	
	}
	BS {}
	BE {}
	CE -
	DE {
	    global file inPRE noFill
	    puts $file </PRE></BLOCKQUOTE>
	    set inPRE 0
	    set noFill 0
	}
	CS -
	DS {				    ;# code section
	    global file inPRE
	    puts -nonewline $file <BLOCKQUOTE><PRE>
	    set inPRE 1
	}
	fi {				    ;# fill
	    global noFill
	    set noFill 0
	}
	IP {
	    IPmacro $args
	}
	LP -
	PP {
	    global file
	    para decr; 
	    para incr; 		
	    puts $file "<P>"

	}
	ne {}
	nf {				    ;#no fill
	    global noFill
	    set noFill 1
	}
	OP {
	    global inDT file inPRE 
	    if {[llength $args] != 3} {
		puts stderr "Bad .OP macro: .$name [join $args " "]"
	    }
      	    para item DL DT
	    set inPRE 1
	    puts -nonewline $file <PRE>		
     	    setTabs 4c
	    text "Command-Line Name:"
	    tab
	    font B
	    set x [lindex $args 0]
	    regsub -all {\\-} $x - x
	    text $x
	    newline
	    font R
	    text "Database Name:"
	    tab
	    font B
	    text [lindex $args 1]
	    newline
	    font R
	    text "Database Class:"
	    tab
	    font B
	    text [lindex $args 2]
	    font R
	    puts -nonewline $file </PRE>		
	    set inDT 1			;# next newline writes inDT 
	    set inPRE 0
	    newline
	}
	RE {
	    para decr	
	}
	RS {
	    para incr
	}
	SE {
	    global state inPRE file

	    font R
	    puts -nonewline $file </PRE>
	    set inPRE 0
	    para reset
	    puts $file "<P>"
	    text "See the "
	    font B
	    set temp $state
	    set state REF
	    text options
	    set state $temp
	    font R
	    text " manual entry for detailed descriptions of the above options."
	    puts $file ""
	}
	SH {
	    SHmacro $args
	}
	SO {
	    global inPRE file

	    SHmacro "STANDARD OPTIONS"
	    setTabs {4c 8c 12c}
	    puts -nonewline $file <PRE>
	    set inPRE 1
	    font B
	}
	so {
	    if {$args != "man.macros"} {
		puts stderr "Unknown macro: .$name [join $args " "]"
	    }
	}
	sp {				    ;# needs work
	    if {$args == ""} {
		set count 1
	    } else {
		set count [lindex $args 0]
	    }
	    while {$count > 0} {
#		lineBreak
		newline
		incr count -1
	    }
	}
	ta {
	    setTabs $args
	}
	TH {
	    THmacro $args
	}
	TP {
	    TPmacro $args
	}
	UL {				    ;# underline
	    global file
	    puts -nonewline $file "<U><B>"
	    text [lindex $args 0]
	    puts -nonewline $file "</B></U>"
	    if {[llength $args] == 2} {
		set x [lindex $args 1]
		if {$x == "\t"} {
		    font B
		    tab
		    font R
		} else {
		    puts "UL macro: found 2nd arg not a tab"
		    text [lindex $args 1]
		}
	    }
	}
	VE {
	    global file inVS
	    while {$inVS>0} {
		puts -nonewline $file "</FONT>"
		incr inVS -1
	    }

	}
	VS {
	    global file inVS
#	    if {[llength $args] > 0} {
#		puts -nonewline $file "<BR>"
#	    }
	    puts -nonewline $file "<FONT COLOR=\"007700\">"
	    incr inVS
	}
	default {
	    puts stderr "Unknown macro: .$name [join $args " "]"
	}
    }
}


# formattedText --
#
# Insert a text string that may also have \fB-style font changes
# and a few other backslash sequences in it.
#
# Arguments:
#   text -	Text to insert.

proc formattedText text {
    while {$text != ""} {
	set index [string first \\ $text]
	if {$index < 0} {
	    text $text
	    return
	}
	text [string range $text 0 [expr $index-1]]
	set c [string index $text [expr $index+1]]
	switch -- $c {
	    f {
		font [string index $text [expr $index+2]]
		set text [string range $text [expr $index+3] end]
	    }
	    e {
		text \\
		set text [string range $text [expr $index+2] end]
	    }
	    - {
		dash
		set text [string range $text [expr $index+2] end]
	    }
	    | {
		set text [string range $text [expr $index+2] end]
	    }
	    default {
		puts stderr "Unknown sequence: \\$c"
		set text [string range $text [expr $index+2] end]
	    }
	}
    }
}



# dash --
#
# This procedure is invoked to handle dash characters ("\-" in
# troff).  It outputs a special dash character.
#
# Arguments:
#   None.

proc dash {} {
    global state charCnt
    if {$state == "NAME"} {
	set state 0
    } elseif {$state == "REF"} {
	set state INSERT
    }
    incr charCnt
    text "-"
}



# tab --
#
# This procedure is invoked to handle tabs in the troff input.
#
# Arguments:
#   None.

proc tab {} {
    global file inPRE charCnt tabString
    if {$inPRE == 1} {
	set pos [expr $charCnt % [string length $tabString] ]
	set spaces [string first "1" [string range $tabString $pos end] ]
	text [format "%*s" [incr spaces] " "]
    } else {
	puts stderr "tab: found tab outside of <PRE> block"
    }
}



# setTabs --
#
# This procedure handles the ".ta" macro, which sets tab stops.
#
# Arguments:
#   tabList -	List of tab stops, each consisting of a number
#		followed by "i" (inch) or "c" (cm).

proc setTabs {tabList} {
    global file tabString

    set last 0
    set tabString {}
    set charsPerInch 14.
    set numTabs [llength $tabList]
    foreach arg $tabList {
	if {[scan $arg "%f%s" distance units] != 2} {
	    puts stderr "bad distance \"$arg\""
	    return 0
	}
	switch -- $units {
	    c {
		set distance [expr $distance * $charsPerInch / 2.54 ]
	    }
	    i {
		set distance [expr $distance * $charsPerInch]
	    }
	    default {
		puts stderr "bad units in distance \"$arg\""
		continue
	    }
	}
	lappend tabString [format "%*s1" [expr round($distance-$last-1)] " "]
	set last $distance
    }
    set tabString [join $tabString {}]
}



# lineBreak --
#
# Generates a line break in the HTML output.
#
# Arguments:
#   None.

proc lineBreak {} {
    global file
    puts $file "<BR>"
}



# newline --
#
# This procedure is invoked to handle newlines in the troff input.
# It outputs either a space character or a newline character, depending
# on fill mode.
#
# Arguments:
#    None.

proc newline {} {
    global file inDT charCnt noFill inPRE

    if {$inDT} {		    ;# terminate dictionary term
	puts $file "\n<DD>"	    ;# start dictionary definition
	set inDT 0
    } elseif {$noFill && !$inPRE} {
	lineBreak
    } else {
	puts $file {}
    }
    set charCnt 0
}



# char --
#
# This procedure is called to handle a special character.
#
# Arguments:
#   name -	Special character named in troff \x or \(xx construct.

proc char name {
    global file charCnt

    incr charCnt
#   puts "char: $name"
    switch -exact $name {
	\\0 {					    ;#  \0
	    puts -nonewline $file " "
	}
	\\\\ {					    ;#  \\
	    puts -nonewline $file "\\"
	}
	\\(+- { 				    ;#  +/-
	    puts -nonewline $file "&#177;"
	}
	\\% {}					    ;#  \%
	\\| {}					    ;#  \|
	default {
	    puts stderr "Unknown character: $name"
	}
    }
}



# macro2 --
#
# This procedure handles macros that are invoked with a leading "'"
# character instead of space.  Right now it just generates an
# error diagnostic.
#
# Arguments:
#   name -	The name of the macro (without the ".").
#   args -	Any additional arguments to the macro.

proc macro2 {name args} {
    puts stderr "Unknown macro: '$name [join $args " "]"
}



# SHmacro --
#
# Subsection head; handles the .SH macro.
#
# Arguments:
#   name -	Section name.

proc SHmacro argList {
    global file state charCnt noFill inPRE

    set args [join $argList " "]
    if {[llength $argList] < 1} {
	puts stderr "Bad .SH macro: .$name $args"
    }

    para reset
    set state 0
    set charCnt 0
    set noFill 0
    if $inPRE {
	puts $file </PRE>
	set inPRE 0
    }

    puts -nonewline $file "<H3>"
    text $args
    puts $file "</H3>"

    # control what the text proc does with text

    switch $args {
	NAME {set state NAME}
	SYNOPSIS {
	    set state 0
	    puts -nonewline $file <PRE>
	    set inPRE 1
	}
	"SEE ALSO" {set state SEE}
	default {set state INSERT}
    }
    set charCnt 0
}



# IPmacro --
#
# This procedure is invoked to handle ".IP" macros, which may take any
# of the following forms:
#
# IPmacro	    Translate to a bulleted list (UL).
# IPmacro x	    Translate to a numbered list (OL). x is ignored.
# IPmacro text n    Translate to a dictionary list (DL). n is ignored.
#
# Arguments:
#   argList -	    List of arguments to the I macro.

proc IPmacro argList {
    global file

    setTabs 0.5i
    set length [llength $argList]
    if {$length == 0} {
	para item UL LI
	return
    }
    if {$length == 1} {
	para item OL LI
	return
    }
    if {$length > 1} {
	para item DL DT
	formattedText [lindex $argList 0]
	puts $file "\n<DD>"
	return
    }
    puts stderr "Bad .IP macro: .IP [join $argList " "]"
}



# TPmacro --
#
# This procedure is invoked to handle ".TP" macros, which may take any
# of the following forms:
#
# TPmacro x	Translate to a dictionary list (DL). n is ignored.
#
# TPmacro 	Translate to a dictionary list (DL).
#
# Arguments:
#   argList	 List of arguments to the .TP macro (ignored).

proc TPmacro {argList} {
    global inDT
    para item DL DT
    set inDT 1		    ;# next newline writes inDT 
    setTabs 0.5i
}



# THmacro --
#
# This procedure handles the .TH macro.  For man pages, it generates 
# the non-scrolling header section for a given man page. For the html
# pages, it sets a five element array corresponding to its five arguments.
# The THmacro also writes the first few lines of html for the page.
# The .TH macro has the following form:
#
# THmacro fname page vers lib pname
#
# Arguments:
#   argList -	    List of 5 arguments to the .TH macro.

proc THmacro {argList} {
    global file THinfo

    if {[llength $argList] != 5} {
	set args [join $argList " "]
	puts stderr "Bad .TH macro: .$name $args"
    }
    set THinfo(fname) [lindex $argList 0]	;# Tcl_UpVar
    set THinfo(page)  [lindex $argList 1]	;# 3
    set THinfo(vers)  [lindex $argList 2]	;# 7.4
    set THinfo(lib)   [lindex $argList 3]	;# Tcl
    set THinfo(pname) [lindex $argList 4]	;# {Tcl Library Procedures}

    puts -nonewline $file "<HTML><HEAD><TITLE>"
    text "$THinfo(lib) - $THinfo(fname) ($THinfo(page))"
    puts $file "</TITLE></HEAD><BODY>\n"

    puts -nonewline $file "<H2><CENTER>"
    text $THinfo(pname)
    puts $file "</CENTER></H2>\n"
}



# para --
#
# This procedure takes care of inserting the tags associated with the
# IP, TP, RS, RE, LP and PP macros. Only 'para item' takes arguments.
#
# Arguments:
#   op -	    operation: item, incr, decr, reset, init
#   listStart -	    begin list tag: OL, UL, DL.
#   listItem -	    item tag:	    LI, LI, DT.

proc para {op {listStart "NEW"} {listItem {} } } {
    global file paraStk charCnt
    switch $op {
	item {
	    set top [lindex $paraStk end]
	    if {$top == "NEW" } {
		set paraStk [lreplace $paraStk end end $listStart]
		puts $file "<$listStart>"
	    } elseif {$top != $listStart} {
		puts stderr "para item: bad paragraph stack"
	    }
	    puts $file "<p>\n<$listItem>"
	}
	decr {
	    if {[llength $paraStk] == 0} {
		puts stderr "para error: para stack is empty"
		set paraStk NEW
	    }
	    set tag [lindex $paraStk end]
	    if {$tag != "NEW"} {
		puts $file "</$tag>"
	    }
	    set paraStk [lreplace $paraStk end end]
	}
	incr {			    ;# paired with & follows a decr
 	   lappend paraStk NEW
	}
	reset {
	    while {[llength $paraStk] > 0} {
		para decr
	    }
	    set paraStk NEW
	}
	init {
	    set paraStk NEW
	}
    }
    set charCnt 0
}



# do --
#
# This is the toplevel procedure that generates an HTML page.
# It sources $fileName.tcl (which was generated during pass1) 
# to generate the HTML file.
#
# Arguments:
#   fileName -	    Name of the file to translate.
#   html_dir -	    HTML pages are stored in this directory.

proc do {fileName html_dir} {
    global file manFile package footer THinfo

    set manFile "[file tail $fileName]"
    set file [open "$html_dir/$package/$manFile.html" w]
    puts "  Pass 2 -- $fileName"
    update
    initGlobals
    if [catch {
	eval [man2tcl [glob $fileName]]
#	source $fileName.tcl
#	file delete $fileName.tcl
    } msg] {
	global errorInfo
	puts stderr $msg
	puts "in"
	puts stderr $errorInfo
	close $file
	return
    }
    para reset
    macro VE
    puts $file "<p>"
    puts $file "<center>Last change: $THinfo(vers)</center>"
    puts $file $footer
    puts $file "</BODY></HTML>"
    close $file
}



# doPass2 --
#
# This is the toplevel procedure for pass 2 processing of a package. 
# Given a directory as argument, generate all the man pages in that 
# directory.
#
# Arguments:
#   dir -	    Name of the directory containing man pages.
#   html_dir -	    Name of the directory to save HTML pages

proc doPass2 {dir html_dir} {
    foreach f [lsort [glob [file join $dir/*.\[13n\]]]] {
	do $f $html_dir
	update
    }
}


