# man2tcl.tcl --
#
#   This file contains a Tcl source that turns a man page of the
#   form used for Tcl and Tk into a Tcl script that invokes
#   a Tcl command for each construct in the man page. The
#   script can then be eval'ed to translate the manual entry
#   into some other format such as MIF or HTML.
#
# Usage:
#
#   man2tcl fileName



# QuoteText --
#
#   Return the "string" argument, adding quote characters around
#   any special Tcl characters so that they'll just be treated
#   as ordinary text.
#
# Results:
#   Returns generated text.
#
# Side effects:
#   None.

proc QuoteText text {
    if {$text == ""} {
	return
    }
    regsub -all {[ ;$\[\{"\\]} $text {\\&} text
    set text "text $text"
    if [regsub -all \t $text "\ntab\ntext " text] { ;# interior tabs
	regsub -all "text \n" $text "" text	    ;# delete blank text lines
	regsub  "\ntext $" $text "" text	    ;# delete trailing blank
    }
    return $text\n
}



# DoText --
#
#   This procedure is called to handle a line of troff text.
#   It parses the text, generating Tcl commands for text and
#   for formatting stuff such as font changes.
#
# Results:
#   Returns the generated text.
#
# Side effects:
#   None.

proc DoText line {
    set line [split $line \\]
    if { [lindex $line 0] != "" } {
	append data [QuoteText [lindex $line 0]]
    }

    foreach phrase [lrange $line 1 end] {
	switch -- [string index $phrase 0] {
	    f { 
		append data "font [string index $phrase 1]\n"
		append data [QuoteText [string range $phrase 2 end]]
	    }
	    e { 
		append data "text \\\\\n"
		append data [QuoteText [string range $phrase 1 end]]
	    }
	    - { 
		append data "dash\n"
		append data [QuoteText [string range $phrase 1 end]]
	    }
	    . { 
		append data "dash\n"
	    }
	    & { 
		append data [QuoteText [string range $phrase 1 end]]
	    }
	    ( { 
		append data "char \{\\\([string range $phrase 1 2]\}\n"
		append data [QuoteText [string range $phrase 3 end]]
	    }
	    {} {}
	    default {
		append data "char \{\\[string index $phrase 0]\}\n"
		append data [QuoteText [string range $phrase 1 end]]
	    }
	}
    }
    append data newline\n
}



# DoMacro --
#
#   This procedure is called to handle a macro invocation.
#   It parses the arguments to the macro and generates a
#   Tcl command to handle the invocation.
#
# Results:
#   Returns the generated text.
#
# Side effects:
#   None.

proc DoMacro line {
    regsub -all {[;$\[\{\\]} $line {\\&} line
    if {$line=="."} return ""
    return "macro [string range $line 1 end]\n"
}



# man2tcl --
#
#   This procedure is the main program, which does all of the work
#   of the program.
#
# Results:
#   None.
#
# Side effects:
#   A Tcl script is output to 'file'. 

proc man2tcl file {
    global fp line_number
    set fpin [open $file r]
    set line_number 0	
    while {[gets $fpin line] != -1} {
	incr line_number
	if {[string index $line 0] == "."} {
	    append data [DoMacro $line]
	} elseif ![regexp {^'\\"} $line] { 
	    append data [DoText $line]
	}
    }
    close $fpin
    return $data
}

return


set file "man/tcl7.6b1/doc/Interp.3"

set fp [open $file.tcl w]
puts $fp [man2tcl $file]
close $fp

