#include "global.h"


class CPU CPU;

//***********************************************************************************************
//
// CPU : management of global CPU structure
//
//***********************************************************************************************

//
// Init to create registers
//
void CPU::Create(void)
{
int i;

	D[0] = new Register;
	D[1] = new Register;
	D[2] = new Register;
	D[3] = new Register;
	D[4] = new Register;
	D[5] = new Register;
	D[6] = new Register;
	D[7] = new Register;

	A[0] = new RegisterAddr;
	A[1] = new RegisterAddr;
	A[2] = new RegisterAddr;
	A[3] = new RegisterAddr;
	A[4] = new RegisterAddr;
	A[5] = new RegisterAddr;
	A[6] = new RegisterAddr;

	USP = new RegisterSP;
	SSP = new RegisterSP;

	A[7] = USP;

	SR = new Register;

	// to set a value an initialization
	for (i=0; i<8; i++) {
		D[i]->Value = (U32) 0;
		D[i]->Changed = 0;
		A[i]->Value = (U32) 0;
		A[i]->Changed = 0;
	}

	PC = (U32) 0;
	VBR = (U32) 0;

	USP->Value = (U32) 0;
	USP->Changed = 0;
	SSP->Value = (U32) 0;
	SSP->Changed = (U16) 0;

	SR->Value = (U32) 0;
	SR->Changed = 0;
}

//
// Reset the CPU the default value
//
void CPU::Init(U32 pc, U32 usp, U32 ssp)
{
int i;

	for (i=0; i<8; i++) {
		D[i]->Value = (U32) 0;
		D[i]->Changed = 0;
		A[i]->Value = (U32) 0;
		A[i]->Changed = 0;
	}

	PC = (U32) pc;

	USP->Value = (U32) usp;
	USP->Changed = (U16) TRUE;
	SSP->Value = (U32) ssp;
	SSP->Changed = (U16) TRUE;

	SR->Value = (U32) 0x2700;
	SR->Changed = 0;
	A[7] = SSP;
}

//
// Set the flags of the status register
//
void CPU::SetCCR(int C, int V, int Z, int N, int X)
{
	if (C != -1) {
		if (C == 0)
			SR->Value &= 0xFFFE;
		else
			SR->Value |= 0x0001; 
		SR->Changed |= 0x0001; 
	}
	if (V != -1) {
		if (V == 0)
			SR->Value &= 0xFFFD;
		else
			SR->Value |= 0x0002; 
		SR->Changed |= 0x0002; 
	}
	if (Z != -1) {
		if (Z == 0)
			SR->Value &= 0xFFFB;
		else
			SR->Value |= 0x0004; 
		SR->Changed |= 0x0004; 
	}
	if (N != -1) {
		if (N == 0)
			SR->Value &= 0xFFF7;
		else
			SR->Value |= 0x0008; 
		SR->Changed |= 0x0008; 
	}
	if (X != -1) {
		if (X == 0)
			SR->Value &= 0xFFEF;
		else
			SR->Value |= 0x0010; 
		SR->Changed |= 0x0010; 
	}
}


//***********************************************************************************************
//
// Registers
//
//***********************************************************************************************

//
// For (An)+ modes
//
U32 Register::Plus(int size)
{
U32 save = Value;

	switch (size) {
		case BYTE:	Value += 1; break;
		case WORD:	Value += 2; break;
		case LONG:	Value += 4; break;

		default :	break;
	}
	Changed = (U16) TRUE;
	return save;
}

//
// For -(An) modes
//
U32 Register::Minus(int size)
{
	switch (size) {
		case BYTE:	Value -= 1; break;
		case WORD:	Value -= 2; break;
		case LONG:	Value -= 4; break;

		default :	break;
	}
	Changed = (U16) TRUE;
	return Value;
}

//
// Set the contents of register to new value
//
void Register::Set(U32 value, int size)
{
	switch (size) {
		case BYTE:	Value = (Value & 0xFFFFFF00) | (value & 0x000000FF); break;
		case WORD:	Value = (Value & 0xFFFF0000) | (value & 0x0000FFFF); break;
		case LONG:	Value = value; break;
	}

	Changed = (U16) TRUE;
}


//-----------------------------------------------------------------------------------------------

//
// Set the contents of register to new value
//
void RegisterAddr::Set(U32 value, int size)
{
	switch (size) {
		case BYTE:	Error(ERR_INVALID_SIZE, 0, "movea || addq || subq");
		case WORD:
		case LONG:	Value = value; break;
	}

	Changed = (U16) TRUE;
}


//-----------------------------------------------------------------------------------------------

//
// For (SP)+ modes : byte are similar to word
//
U32 RegisterSP::Plus(int size)
{
U32 save = Value;

	switch (size) {
		case BYTE:	Value += 2; save += 1; break;
		case WORD:	Value += 2; break;
		case LONG:	Value += 4; break;

		default :	break;
	}
	Changed = (U16) TRUE;
	return save;
}

//
// For -(SP) modes : byte are similar to word
//
U32 RegisterSP::Minus(int size)
{
U32 save = Value;

	switch (size) {
		case BYTE:	Value -= 2; save -= 1; Poke(Value, BYTE, 0); break;
		case WORD:	Value -= 2; save -= 2; break;
		case LONG:	Value -= 4; save -= 4; break;

		default :	break;
	}
	Changed = (U16) TRUE;
	return save;
}
