#include "global.h"

//
// Global variables
//

char Text_Disassamble[100];
char Text_Opcode[100];
char Text_EA[100];
extern U16 Opcode;
U32 PC, PCTmp;

//
// Formats for the addressing modes
//

char *FMT_DataImmByte = "#0x%02x";
char *FMT_DataImmWord = "#0x%04x";
char *FMT_DataImmLong = "#0x%08x";
char *FMT_ImmByte = "0x%02x";
char *FMT_ImmWord = "0x%04x";
char *FMT_ImmLong = "0x%08x";
char *FMT_PC8 = "%d(PC,%c%d.%c)";		//"0x%x(PC,%c%d.%c)";
char *FMT_PC16 = "%d(PC)";				//"0x%x(PC)";
char *FMT_AbsLong = "0x%08x";
char *FMT_AbsWord = "0x%04x";
char *FMT_An16 = "%d(a%d)";				//"0x%x(a%d)";
char *FMT_An8 = "%d(a%d,%c%d.%c)";		//"0x%x(a%d,%c%d.%c)";
char *FMT_iLogicCCR = "0x%02x";
char *FMT_iLogicSR = "0x%04x";
char *FMT_Addr = "0x%08x";
char *FMT_Moveq = "%d";


//***********************************************************************************************
//
// Compute Effective Address (mode + register = <ea>)
//
//***********************************************************************************************

void Disassamble_EA(int mode, int reg, int size=LONG)
{
U32 addr;
U16 index;
char *label;

	switch (mode) {
		// direct Dn
		case 0:	sprintf(Text_EA,"d%d",reg);
				break;

		// direct An
		case 1: sprintf(Text_EA,"a%d",reg);
				break;

		// indirect An
		case 2: sprintf(Text_EA,"(a%d)",reg);
				break;

		// indirect An post-incremented
		case 3: sprintf(Text_EA,"(a%d)+",reg);
				break;

		// indirect An pre-incremented
		case 4: sprintf(Text_EA,"-(a%d)",reg);
				break;

		// indirect An with 16-bits offset
		case 5: sprintf(Text_EA, FMT_An16, Peek(PCTmp , WORD) ,reg);
				PCTmp += 2;
				break;

		// indirect An with 8-bits offset and index
		case 6: index = Peek(PCTmp , WORD);
				PCTmp += 2;
				sprintf(Text_EA, FMT_An8, BITS(index,0,8), reg,
						BITS(index,15,1)?'a':'d', BITS(index,12,3), BITS(index,11,1)?'l':'w');
				break;

		// all other
		case 7:	switch (reg) {
					// absolute word
					case 0:	addr = Peek(PCTmp , WORD);
							PCTmp += 2;
							if ((label = Symbols.GetName(addr)) == NULL)
								sprintf(Text_EA, FMT_AbsWord,addr);
							else
								sprintf(Text_EA,"%s",label);
							break;

					// absolute long
					case 1:	addr = Peek(PCTmp , LONG);
							PCTmp += 4;
							if ((label = Symbols.GetName(addr)) == NULL)
								sprintf(Text_EA, FMT_AbsLong,addr);
							else
								sprintf(Text_EA,"%s",label);
							break;

					// indirect PC with 16-bits offset
					case 2: sprintf(Text_EA, FMT_PC16, Peek(PCTmp , WORD));
							PCTmp += 2;
							break;

					// indirect PC with 8-bits offset
					case 3: index = Peek(PCTmp , WORD);
							PCTmp += 2;
							sprintf(Text_EA, FMT_PC8, BITS(index,0,8), reg,
									BITS(index,15,1)?'a':'d', BITS(index,12,3),
									BITS(index,11,1)?'l':'w');
							break;

					// immediate
					case 4: switch (size) {
								case BYTE:	sprintf(Text_EA, FMT_DataImmByte, 
													Peek(PCTmp , WORD) & 0x00FF);
											PCTmp += 2;
											break;
								case WORD:	sprintf(Text_EA, FMT_DataImmWord, 
													Peek(PCTmp , WORD) & 0xFFFF);
											PCTmp += 2;
											break;
								case LONG:	sprintf(Text_EA, FMT_DataImmLong, Peek(PCTmp, LONG));
											PCTmp += 4;
											break;
							}
							break;

					default:	sprintf(Text_EA,"???");
				}
				break;
	}
} 


//***********************************************************************************************
//
// Group 0 : immediate logic + bit functions + MOVEP
//
//***********************************************************************************************

void dis_immediate(char *inst)
{
char nb[50];
char size = '?';

	switch (BITS(Opcode,6,2)) {
		case BYTE:	size = 'b';
					sprintf(nb, FMT_ImmByte, Peek(PCTmp, WORD) & 0x00FF);
					PCTmp += 2;
					break;
		case WORD:	size = 'w';
					sprintf(nb, FMT_ImmWord, Peek(PCTmp, WORD) &0xFFFF);
					PCTmp += 2;
					break;
		case LONG:	size = 'l';
					sprintf(nb, FMT_ImmLong, Peek(PCTmp, LONG));
					PCTmp += 4;
					break;
	}
	Disassamble_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), BITS(Opcode,6,2));
	sprintf(Text_Opcode, inst, size, nb, Text_EA);
}							   

void dis_ilogic_ccr(char *inst)
{
char nb[50];

	sprintf(nb, FMT_iLogicCCR, Peek(PCTmp, WORD) & 0x00FF);
	PCTmp += 2;	
	sprintf(Text_Opcode, inst, nb);
}

void dis_ilogic_sr(char *inst)
{
char nb[50];

	sprintf(nb, FMT_iLogicSR, Peek(PCTmp, WORD) & 0xFFFF);
	PCTmp += 2;	
	sprintf(Text_Opcode, inst, nb);
}

void dis_bitfn(char *inst)
{
char tmp[10],size;

	if (BITS(Opcode,8,1)) {
		sprintf(tmp, "d%d", BITS(Opcode,9,3));		// dynamic
	} else {
		sprintf(tmp, "#%d", Peek(PCTmp, WORD));		// static
		PCTmp += 2;
	}
	
	Disassamble_EA(BITS(Opcode,3,3), BITS(Opcode,0,3));
	if (BITS(Opcode,3,3) == 0 || BITS(Opcode,3,3) == 1) {
		size = 'l';		// LONG on register
	} else {
		size = 'b';		// BYTE on memory
	}

	sprintf(Text_Opcode, inst, size, tmp, Text_EA);		
}

void dis_movep(char *inst)
{
char data[10], addr[50], size;

	sprintf(data, "d%d", BITS(Opcode,9,3));
	sprintf(addr,"0x%x(a%d)", Peek(PCTmp , WORD), BITS(Opcode,0,3));
	PCTmp += 2;

	size = (BITS(Opcode,6,1) ? 'l' : 'w');
	if (BITS(Opcode,7,1))
		sprintf(Text_Opcode, inst, size, data, addr);
	else
		sprintf(Text_Opcode, inst, size, addr, data);
}

//***********************************************************************************************
//
// Group 1 2 3 : MOVE
//
//***********************************************************************************************

void dis_move(char *inst)
{
char text_EA_tmp[100];
int size=BYTE;

	// size
	switch (BITS(Opcode,12,2)) {
		case 1: size = BYTE; break;
		case 2: size = LONG; break;
		case 3: size = WORD; break;
	}

	// source
	Disassamble_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), size);
	strcpy(text_EA_tmp, Text_EA);

	// dest
	Disassamble_EA(BITS(Opcode,6,3), BITS(Opcode,9,3));
	
	sprintf(Text_Opcode, inst, BITS(Opcode,6,3)==1?"a":"",
							   size==BYTE?'b':size==WORD?'w':'l',
							   BITS(Opcode,6,3)==1?"":" ",
							   text_EA_tmp, Text_EA);  
}


//***********************************************************************************************
//
// Group 4 : MISC
//
//***********************************************************************************************

void dis_std_unary_size(char *inst)
{
char size = '?';

	Disassamble_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), BITS(Opcode,6,2));
	switch (BITS(Opcode,6,2)) {
		case BYTE:	size = 'b'; break;
		case WORD:	size = 'w'; break;
		case LONG:	size = 'l'; break;
	}
	sprintf(Text_Opcode, inst, size, Text_EA);
}

void dis_move_sr(char *inst)
{
	Disassamble_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), WORD);
	if (BITS(Opcode,8,4))
		sprintf(Text_Opcode, inst, Text_EA, "SR");
	else
		sprintf(Text_Opcode, inst, "SR", Text_EA);
}

void dis_std_unary(char *inst)
{
	Disassamble_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), WORD);
	sprintf(Text_Opcode, inst, Text_EA);
}

void dis_std_ea_dn(char *inst)
{
	Disassamble_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), WORD);
	sprintf(Text_Opcode, inst, Text_EA, BITS(Opcode,9,3)); 
}

void dis_lea(char *inst)
{
	Disassamble_EA(BITS(Opcode,3,3), BITS(Opcode,0,3));
	sprintf(Text_Opcode,inst,Text_EA,BITS(Opcode,9,3));
}

void dis_swap(char *inst)
{
	sprintf(Text_Opcode,inst, BITS(Opcode,0,3));
}

void dis_ext(char *inst)
{
	sprintf(Text_Opcode,inst, BITS(Opcode,6,1)?'l':'w', BITS(Opcode,0,3));		
}

void dis_movem(char *inst)
{
char list[50], *where;
char size = '?';
U16 reg, left, right;
int i, cont, flag;


#define SET_LIST(c)	if (cont != -1) {										\
						if (cont == i-1)									\
							sprintf(where, "%c%d/", c, cont);				\
						else												\
							sprintf(where, "%c%d-%c%d/", c,cont, c,i-1);	\
						where = list + strlen(list);						\
						cont = -1;											\
				 	}



	reg = Peek(PCTmp , WORD);
	PCTmp += 2;	

	Disassamble_EA(BITS(Opcode,3,3), BITS(Opcode,0,3));
	size = (BITS(Opcode,6,1) ? 'l' : 'w');

	if (BITS(Opcode,3,3) == 4) {
		// mode -(An) = need to transpose
		for (left=0x8000, right=0x1; left > right; left>>=1, right<<=1)
		{
			flag = (reg & left);
			reg = ((reg & right) ? reg | left : reg & ~left);
			reg = ((flag) ? reg | right : reg & ~right);
		}
	}
		
	// decode the list of registers
	where = list;
	cont = -1;
	for (i=0; i<8; i++) {
		if (reg & 0x0001) {
			if (cont == -1)
				cont = i;
		} else {
			SET_LIST('d')
		}
		reg >>= 1;
	}
	SET_LIST('d')
	for (i=0; i<8; i++) {
		if (reg & 0x0001) {
			if (cont == -1)
				cont = i;
		} else {
			SET_LIST('a')
		}
		reg >>= 1;
	}
	SET_LIST('a')
	list[strlen(list)-1] = '\0';

	if (BITS(Opcode,10,1))
		sprintf(Text_Opcode, inst, size, Text_EA, list);
	else
		sprintf(Text_Opcode, inst, size, list, Text_EA);
}

void dis_direct(char *inst)
{
	sprintf(Text_Opcode, inst);
}

void dis_trap(char *inst)
{
	sprintf(Text_Opcode, inst, BITS(Opcode,0,4));
}

void dis_link(char *inst)
{
char nb[50];

	if ((Opcode & 0xFF00) == 0x4E00) {
		sprintf(nb, "%d", Peek(PCTmp , WORD));
		PCTmp += 2;
	} else {
		sprintf(nb, "%d", Peek(PCTmp , LONG));
		PCTmp += 4;
	}
	sprintf(Text_Opcode, inst, BITS(Opcode,0,3), nb);
}

void dis_unlink(char *inst)
{
	sprintf(Text_Opcode, inst, BITS(Opcode,0,3));
}

void dis_move_usp(char *inst)
{
char tmp[10];

	sprintf(tmp, "a%d", BITS(Opcode,0,3));
	if (BITS(Opcode,3,1))
		sprintf(Text_Opcode, inst, "USP", tmp);
	else
		sprintf(Text_Opcode, inst, tmp, "USP");
}

void dis_movec(char *inst)
{
char tmp[10], reg[10];
U16 control;

	control = (U16) Peek(PCTmp , WORD);
	PCTmp += 2;

	switch (BITS(control,0,12)) {
		case 0x000:	sprintf(tmp, "SFC"); break;
		case 0x001:	sprintf(tmp, "DFC"); break;
		case 0x002:	sprintf(tmp, "CACR"); break;
		case 0x800:	sprintf(tmp, "USP"); break;	
		case 0x801:	sprintf(tmp, "VBR"); break;
		case 0x802:	sprintf(tmp, "CAAR"); break;
		case 0x803:	sprintf(tmp, "MSP"); break;
		case 0x804:	sprintf(tmp, "ISP"); break;

		default:	sprintf(tmp, "???"); break;
	}
	sprintf(reg, "%c%d", BITS(control,15,1)?'a':'d', BITS(control,12,3));

	if (BITS(Opcode,0,1))
		sprintf(Text_Opcode, inst, reg,tmp);
	else
		sprintf(Text_Opcode, inst, tmp,reg);
}

void dis_stop(char *inst)
{
char nb[50];

	sprintf(nb, "%d", Peek(PCTmp , WORD));
	PCTmp += 2;
	sprintf(Text_Opcode, inst, nb);	
}


//***********************************************************************************************
//
// Group 5 : DB.. + S.. + ADDQ + SUBQ
//
//***********************************************************************************************

void dis_arith_q(char *inst)
{
char size = '?', nb[50];
int data;

	Disassamble_EA(BITS(Opcode,3,3), BITS(Opcode,0,3));
	switch (BITS(Opcode,6,2)) {
		case BYTE:	size = 'b'; break;
		case WORD:	size = 'w'; break;
		case LONG:	size = 'l'; break;
	}
	data = BITS(Opcode,9,3);
	if (data==0)  data = 8;
	sprintf(nb, "%d", data);
	sprintf(Text_Opcode, inst, size, nb, Text_EA);
}

void dis_scc(char *inst)
{
	Disassamble_EA(BITS(Opcode,3,3), BITS(Opcode,0,3));

	switch (BITS(Opcode,8,4)) {
		case  0:	sprintf(Text_Opcode, inst, "t", Text_EA); break;
		case  1:	sprintf(Text_Opcode, inst, "f", Text_EA); break;
		case  2:	sprintf(Text_Opcode, inst, "hi", Text_EA); break;
		case  3:	sprintf(Text_Opcode, inst, "ls", Text_EA); break;
		case  4:	sprintf(Text_Opcode, inst, "cc", Text_EA); break;
		case  5:	sprintf(Text_Opcode, inst, "cs", Text_EA); break;
		case  6:	sprintf(Text_Opcode, inst, "ne", Text_EA); break;
		case  7:	sprintf(Text_Opcode, inst, "eq", Text_EA); break;
		case  8:	sprintf(Text_Opcode, inst, "vc", Text_EA); break;
		case  9:	sprintf(Text_Opcode, inst, "vs", Text_EA); break;
		case 10:	sprintf(Text_Opcode, inst, "pl", Text_EA); break;
		case 11:	sprintf(Text_Opcode, inst, "mi", Text_EA); break;
		case 12:	sprintf(Text_Opcode, inst, "ge", Text_EA); break;
		case 13:	sprintf(Text_Opcode, inst, "lt", Text_EA); break;
		case 14:	sprintf(Text_Opcode, inst, "gt", Text_EA); break;
		case 15:	sprintf(Text_Opcode, inst, "le", Text_EA); break;
	}
}

void dis_dbcc(char *inst)
{
char reg[10], *label;
S32 addr;

	sprintf(reg, "d%d", BITS(Opcode,0,3));
	addr = (S16) Peek(PCTmp , WORD);
	PCTmp += 2;
	addr += PC+2;	// relative address
	if ((label = Symbols.GetName(addr)) == NULL)
		sprintf(Text_EA, FMT_Addr,addr);
	else
		sprintf(Text_EA,"%s",label);

	switch (BITS(Opcode,8,4)) {
		case  0:	sprintf(Text_Opcode, inst, "t", reg, Text_EA); break;
		case  1:	sprintf(Text_Opcode, inst, "ra", reg, Text_EA); break;
		case  2:	sprintf(Text_Opcode, inst, "hi", reg, Text_EA); break;
		case  3:	sprintf(Text_Opcode, inst, "ls", reg, Text_EA); break;
		case  4:	sprintf(Text_Opcode, inst, "cc", reg, Text_EA); break;
		case  5:	sprintf(Text_Opcode, inst, "cs", reg, Text_EA); break;
		case  6:	sprintf(Text_Opcode, inst, "ne", reg, Text_EA); break;
		case  7:	sprintf(Text_Opcode, inst, "eq", reg, Text_EA); break;
		case  8:	sprintf(Text_Opcode, inst, "vc", reg, Text_EA); break;
		case  9:	sprintf(Text_Opcode, inst, "vs", reg, Text_EA); break;
		case 10:	sprintf(Text_Opcode, inst, "pl", reg, Text_EA); break;
		case 11:	sprintf(Text_Opcode, inst, "mi", reg, Text_EA); break;
		case 12:	sprintf(Text_Opcode, inst, "ge", reg, Text_EA); break;
		case 13:	sprintf(Text_Opcode, inst, "lt", reg, Text_EA); break;
		case 14:	sprintf(Text_Opcode, inst, "gt", reg, Text_EA); break;
		case 15:	sprintf(Text_Opcode, inst, "le", reg, Text_EA); break;
	}
}


//***********************************************************************************************
//
// Group 6 : B..
//
//***********************************************************************************************

void dis_bcc(char *inst)
{
char *label;
S32 addr;

	addr = (S8) BITS(Opcode,0,8);
	if (addr == 0) {
		addr = (S16) Peek(PCTmp , WORD);
		PCTmp += 2;
	} else if ((addr == -1) && (CurrentModel > 68000)) {
		addr = (S32) Peek(PCTmp , LONG);
		PCTmp += 4;
	}

	addr += PC+2;	// relative address
	if ((label = Symbols.GetName(addr)) == NULL)
		sprintf(Text_EA, FMT_Addr,addr);
	else
		sprintf(Text_EA,"%s",label);

	switch (BITS(Opcode,8,4)) {
		case  0:	sprintf(Text_Opcode, inst, "ra", Text_EA); break;
		case  1:	sprintf(Text_Opcode, inst, "sr", Text_EA); break;
		case  2:	sprintf(Text_Opcode, inst, "hi", Text_EA); break;
		case  3:	sprintf(Text_Opcode, inst, "ls", Text_EA); break;
		case  4:	sprintf(Text_Opcode, inst, "cc", Text_EA); break;
		case  5:	sprintf(Text_Opcode, inst, "cs", Text_EA); break;
		case  6:	sprintf(Text_Opcode, inst, "ne", Text_EA); break;
		case  7:	sprintf(Text_Opcode, inst, "eq", Text_EA); break;
		case  8:	sprintf(Text_Opcode, inst, "vc", Text_EA); break;
		case  9:	sprintf(Text_Opcode, inst, "vs", Text_EA); break;
		case 10:	sprintf(Text_Opcode, inst, "pl", Text_EA); break;
		case 11:	sprintf(Text_Opcode, inst, "mi", Text_EA); break;
		case 12:	sprintf(Text_Opcode, inst, "ge", Text_EA); break;
		case 13:	sprintf(Text_Opcode, inst, "lt", Text_EA); break;
		case 14:	sprintf(Text_Opcode, inst, "gt", Text_EA); break;
		case 15:	sprintf(Text_Opcode, inst, "le", Text_EA); break;
	}
}


//***********************************************************************************************
//
// Group 7 : MOVEQ
//
//***********************************************************************************************

void dis_moveq(char *inst)
{
char nb[50];

	sprintf(nb, FMT_Moveq, (S8) BITS(Opcode,0,8));
	sprintf(Text_Opcode, inst, nb, BITS(Opcode,9,3));
}


//***********************************************************************************************
//
// Group 8 : SBCD + DIV + OR
//
//***********************************************************************************************

void dis_sbcd_abcd(char *inst)
{
char src[10], dest[10];

	if (BITS(Opcode,3,1)) {
		sprintf(src,"-(a%d)", BITS(Opcode,0,3));
		sprintf(dest,"-(a%d)", BITS(Opcode,9,3));
	} else {
		sprintf(src,"d%d", BITS(Opcode,0,3));
		sprintf(dest,"d%d", BITS(Opcode,9,3));
	}
	sprintf(Text_Opcode, inst, src, dest);
}

void dis_std_ea_dn_swap(char *inst)
{
char reg[10], size='?';

	Disassamble_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), BITS(Opcode,6,2));
	sprintf(reg, "d%d", BITS(Opcode,9,3));

	switch (BITS(Opcode,6,2)) {
		case BYTE:	size = 'b'; break;
		case WORD:	size = 'w'; break;
		case LONG:	size = 'l'; break;
	}

	if (BITS(Opcode,8,1))
		sprintf(Text_Opcode, inst, size, reg, Text_EA);
	else
		sprintf(Text_Opcode, inst, size, Text_EA, reg);
}


//***********************************************************************************************
//
// Group 9/D : SUB / ADD
//
//***********************************************************************************************

void dis_suba_adda(char *inst)
{
char size;

	Disassamble_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), BITS(Opcode,8,1) ? LONG : WORD);
	size = (BITS(Opcode,8,1) ? 'l' : 'w');
	sprintf(Text_Opcode, inst, size, Text_EA, BITS(Opcode,9,3));	
}

void dis_subx_addx(char *inst)
{
char src[10], dest[10], size='?';

	switch (BITS(Opcode,6,2)) {
		case BYTE:	size = 'b'; break;
		case WORD:	size = 'w'; break;
		case LONG:	size = 'l'; break;
	}

	if (BITS(Opcode,3,1)) {
		sprintf(src,"-(a%d)", BITS(Opcode,0,3));
		sprintf(dest,"-(a%d)", BITS(Opcode,9,3));
	} else {
		sprintf(src,"d%d", BITS(Opcode,0,3));
		sprintf(dest,"d%d", BITS(Opcode,9,3));
	}
	sprintf(Text_Opcode, inst, size, src, dest);
}


//***********************************************************************************************
//
// Group A/F : Line A/F
//
//***********************************************************************************************

void dis_lineAF(char *inst)
{
	sprintf(Text_Opcode, inst, Opcode);
}


//***********************************************************************************************
//
// Group B : CMP + EOR
//
//***********************************************************************************************

void dis_cmpm(char *inst)
{
char size = '?';

	switch (BITS(Opcode,6,2)) {
		case BYTE:	size = 'b'; break;
		case WORD:	size = 'w'; break;
		case LONG:	size = 'l'; break;
	}
	sprintf(Text_Opcode, inst, size, BITS(Opcode,0,3), BITS(Opcode,9,3));
}

void dis_cmpa(char *inst)
{
char reg[10], size='?';

	Disassamble_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), BITS(Opcode,8,1)?LONG:WORD);
	sprintf(reg, "a%d", BITS(Opcode,9,3));

	if (BITS(Opcode,8,1))
		size = 'l';
	else
		size = 'w';

	sprintf(Text_Opcode, inst, size, Text_EA, reg);
}


//***********************************************************************************************
//
// Group C : EXG + ABCD + MUL + AND
//
//***********************************************************************************************

void dis_exg(char *inst)
{
char src[10],dest[10];

	switch (BITS(Opcode,3,5)) {
		case 0x08:	sprintf(src, "d%d", BITS(Opcode,0,3));
					sprintf(dest,"d%d", BITS(Opcode,9,3));
					break;
		case 0x09:	sprintf(src, "a%d", BITS(Opcode,0,3));
					sprintf(dest,"a%d", BITS(Opcode,9,3));
					break;
		case 0x11:	sprintf(src, "a%d", BITS(Opcode,0,3));
					sprintf(dest,"d%d", BITS(Opcode,9,3));
					break;
		default :	sprintf(src, "??");
					sprintf(dest,"??");
	}
	sprintf(Text_Opcode, inst, dest, src);
}


//***********************************************************************************************
//
// Group E : shifts
//
//***********************************************************************************************

void dis_shift(char *inst)
{
char size='?', src[10];
int nb;

	switch (BITS(Opcode,6,2)) {
		case BYTE:	size = 'b'; break;
		case WORD:	size = 'w'; break;
		case LONG:	size = 'l'; break;
	}

	if (BITS(Opcode,5,1))
		sprintf(src, "d%d", BITS(Opcode,9,3));
	else {
		nb = BITS(Opcode,9,3);
		if (nb == 0) nb = 8;
		sprintf(src, "#%d", nb);
	}

	sprintf(Text_Opcode, inst, size, src, BITS(Opcode,0,3));
}


//***********************************************************************************************
//
// Disassemble : entry point, put result in string Text_Opcode
//
//***********************************************************************************************

U32 Disassamble(U32 start)
{
int i,j;
char *label;


	PC = start;

	if (GetPeek(PC) & 0xFF00) {
		strcpy(Text_Disassamble, "0xffffffff\n");
		return PC+2;
	}

	// Get the opcode
	Opcode = Peek(PC , WORD);
	PCTmp = PC+2;

	// Search class of opcode
	i = Opcode_Group[(Opcode & 0xF000) >> 12];
	j = Opcode_Group[ ((Opcode & 0xF000) >> 12) + 1];
	while (i<j) {
		if ((Opcode & Opcode_List[i].Mask) == Opcode_List[i].Value)
			break;
		else
			i++;
	}

	// Disassamble intruction
	if ((i == j) || (Opcode_List[i].Model > CurrentModel))
		strcpy(Text_Opcode, "????????");
	else
		(*Opcode_List[i].Dis)(Opcode_List[i].Name);

    label=Symbols.GetName(PC);
 	sprintf(Text_Disassamble, "0x%08x  %-20s %s\n",PC, label==NULL?"":label, Text_Opcode);

	return PCTmp;
}
