#include "global.h"


//
// Defines the addressing modes
//

#define EA_Dn		0x0001
#define EA_An		0x0002
#define EA_An_ind	0x0004
#define EA_An_inc	0x0008
#define EA_An_dec	0x0010
#define EA_An_d16	0x0020
#define EA_An_d8_Xi	0x0040
#define EA_Abs_W	0x0080
#define EA_Abs_L	0x0100
#define EA_PC_d16	0x0200
#define EA_PC_d8_Xi	0x0400
#define EA_Imm		0x0800
#define EA_unknow	0x1000

//
// Macro for getting value of a status flag
//

#define srC	(CPU.SR->Value & 0x0001)
#define srV	(CPU.SR->Value & 0x0002)
#define srZ	(CPU.SR->Value & 0x0004)
#define srN	(CPU.SR->Value & 0x0008)
#define srX	(CPU.SR->Value & 0x0010)


U16 Opcode;
U32	Addr_EA;

int ExceptionDetected;
int HookOpcode;


//***********************************************************************************************
//
// Check proprieties / abnormal execution (trap, ...)
//
//***********************************************************************************************

void Exception(int number)
{
U16 vector;


	CPU.A[7] = CPU.SSP;			// switch to supervisor stack

	vector = (number*4 & 0x0000FFFF);

	if (CurrentModel > 68000) {
		switch (number) {
			case EXCEP_CHK:
			case EXCEP_TRAPV:
			case EXCEP_TRACE:
			case EXCEP_DIV:
				Poke(CPU.A[7]->Minus(LONG), CPU.PC, LONG);		// instruction address
				vector |= 0x2000;
		}
	}

	Poke(CPU.A[7]->Minus(WORD), vector, WORD);				// save format + vector
	Poke(CPU.A[7]->Minus(LONG), Disassamble(CPU.PC), LONG);	// save PC of next instruction
	Poke(CPU.A[7]->Minus(WORD), CPU.SR->Value, WORD);
	
	CPU.SR->Value |= 0x2000;	// only flag affected = S
	CPU.SR->Changed |= 0x2000;

	if (CurrentModel > 68000)
		CPU.PC = CPU.PCTmp = Peek(number*4 + CPU.VBR, LONG);
	else
		CPU.PC = CPU.PCTmp = Peek(number*4 , LONG);

	if (number>=EXCEP_TRAP && number<=EXCEP_TRAP+16)
		sprintf(what,".info.text insert end \"Exception TRAP.\n\"");
	else if (number>=EXCEP_AUTO && number<=EXCEP_AUTO+7)
		sprintf(what,".info.text insert end \"Exception AUTO.\n\"");
	else if (number>=EXCEP_USER)
		sprintf(what,".info.text insert end \"Exception USER.\n\"");
	else sprintf(what,".info.text insert end \"Exception %s.\n\"",
			number==EXCEP_BUS?"BUS ERROR":
			number==EXCEP_ADDR?"ADDRESS ERROR":
			number==EXCEP_ILLEGAL?"ILLEGAL":
			number==EXCEP_DIV?"DIVISION BY ZERO":
			number==EXCEP_CHK?"CHK":
			number==EXCEP_TRAPV?"TRAPV":
			number==EXCEP_PROTECT?"PRIVILEGE VIOLATION":
			number==EXCEP_LINE_A?"LINE A":
			number==EXCEP_LINE_F?"LINE F":
			number==EXCEP_TRAP?"TRAP":"????");
	Tcl_Eval(output, what);

	ExceptionDetected = TRUE;
}

int CheckSupervisor(void)
{
	if ((CPU.SR->Value & 0x2000) == 0) {
		Exception(EXCEP_PROTECT);
		return FALSE;
	}
	return TRUE;
}


//***********************************************************************************************
//
// Compute Effective Address (mode + register = <ea>)
//
//***********************************************************************************************

U16 Compute_EA(int mode, int reg, int size)
{
U16 index;


	switch (mode) {
		// direct Dn
		case 0:	Addr_EA = reg;
				return EA_Dn;

		// direct An
		case 1:	Addr_EA = reg;
				return EA_An;

		// indirect An
		case 2:	Addr_EA = CPU.A[reg]->Value;
				return EA_An_ind;

		// indirect An post-incremented
		case 3:	Addr_EA = CPU.A[reg]->Plus(size);
				return EA_An_inc;

		// indirect An pre-decremented
		case 4:	Addr_EA = CPU.A[reg]->Minus(size);
				return EA_An_dec;

		// indirect An with 16-bits offset
		case 5: Addr_EA = (S16) Peek(CPU.PCTmp , WORD);
				CPU.PCTmp += 2;
				Addr_EA += CPU.A[reg]->Value;
				return EA_An_d16;

		// indirect An with 8-bits offset and index
		case 6: index = Peek(CPU.PCTmp , WORD);
				CPU.PCTmp += 2;
				if (BITS(index,15,1)) {
					if (BITS(index,11,1))
						Addr_EA = CPU.A[BITS(index,12,3)]->Value;
					else
						Addr_EA = (S16) (CPU.A[BITS(index,12,3)]->Value & 0x0000FFFF);
				} else {
					if (BITS(index,11,1))
						Addr_EA = CPU.D[BITS(index,12,3)]->Value;
					else
						Addr_EA = (S16) (CPU.D[BITS(index,12,3)]->Value & 0x0000FFFF);
				}
				Addr_EA += (S8) BITS(index,0,8);
				Addr_EA += CPU.A[reg]->Value;
				return EA_An_d8_Xi;

		// all other
		case 7:	switch (reg) {
					// absolute word
					case 0:	Addr_EA = (S16) Peek(CPU.PCTmp , WORD);
							CPU.PCTmp += 2;
							return EA_Abs_W;

					// absolute long
					case 1:	Addr_EA = Peek(CPU.PCTmp , LONG);
							CPU.PCTmp += 4;
							return EA_Abs_L;

					// indirect PC with 16-bits offset
					case 2: Addr_EA = (S16) Peek(CPU.PCTmp , WORD);
							CPU.PCTmp += 2;
							Addr_EA += CPU.PC;
							return EA_PC_d16;

					// indirect PC with 16-bits offset
					case 3: index = Peek(CPU.PCTmp , WORD);
							CPU.PCTmp += 2;
							if (BITS(index,15,1)) {
								if (BITS(index,11,1))
									Addr_EA = CPU.A[BITS(index,12,3)]->Value;
								else
									Addr_EA = (S16)(CPU.A[BITS(index,12,3)]->Value & 0x0000FFFF);
							} else {
								if (BITS(index,11,1))
									Addr_EA = CPU.D[BITS(index,12,3)]->Value;
								else
									Addr_EA = (S16)(CPU.D[BITS(index,12,3)]->Value & 0x0000FFFF);
							}
							Addr_EA += (S8) BITS(index,0,8);
							Addr_EA += CPU.PC;
							return EA_PC_d8_Xi;

					// immediate
					case 4: Addr_EA = CPU.PCTmp;
							switch (size) {
								case BYTE:	CPU.PCTmp += 2; Addr_EA += 1; break;
								case WORD:	CPU.PCTmp += 2; break;
								case LONG:	CPU.PCTmp += 4; break;
							}
							return EA_Imm;
				}
				break;
	}

	Exception(EXCEP_ILLEGAL);		// not a known format
	return EA_unknow;
} 


//***********************************************************************************************
//
// Utility functions
//
//***********************************************************************************************

//
// Return the Most Significant Bit (7,15 or 31 according to size)
//
int MSB(U32 value, int size)
{
	switch (size) {
		case BYTE:	return ((value>>7) & 0x00000001);
		case WORD:	return ((value>>15) & 0x00000001);
		case LONG:	return ((value>>31) & 0x00000001);
	}
	return 0;
}

//
// Get a immediate data when needed in an opcode
//
U32 GetImmediate(int size)
{
U32 data = 0;

	switch (size) {
		case BYTE:	data = (S8) (Peek(CPU.PCTmp, WORD) & 0x000000FF);
					CPU.PCTmp += 2;
					break;
		case WORD:	data = (S16) (Peek(CPU.PCTmp, WORD) &0x0000FFFF);
					CPU.PCTmp += 2;
					break;
		case LONG:	data = Peek(CPU.PCTmp, LONG);
					CPU.PCTmp += 4;
					break;
	}
	return data;
}

//
// Return the value associated with a given addressing mode
//
U32 GetValue(U16 ea, int size, U32 addr = Addr_EA)
{
U32 value;

	if (ea == EA_Dn)
		value = CPU.D[addr]->Value;	// Dn
	else if (ea == EA_An)
		return CPU.A[addr]->Value;	// An : no extension of sign
	else
		value = Peek(addr , size);	// memory


 	switch (size) {
		case BYTE: value = (S8) (value & 0x000000FF); break;
		case WORD: value = (S16) (value & 0x0000FFFF); break;
	}

	return value;
}

//
// Set a value according to a given addressing mode
//
void SetValue(U16 ea, int size, U32 value, U32 addr = Addr_EA)
{
	if (ea == EA_Dn)
		CPU.D[addr]->Set(value,size);	// Dn
	else if (ea == EA_An)
		CPU.A[addr]->Set(value,size);	// An
	else
		Poke(addr, value , size);		// memory
}


//***********************************************************************************************
//
// Group 0 : immediate logic + bit functions + MOVEP
//
//***********************************************************************************************

//-----------------------------------------------------------------------------------------------
// Imm. Logic simple
//-----------------------------------------------------------------------------------------------

void fn_ilogic(char what, char *name)
{
U16 ea;
U32 data, value;
int size;

	// get immediate data
	size = BITS(Opcode,6,2);
	data = GetImmediate(size);

	// ea of destination
	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), size);
	if ((ea & 0x01FD) == 0) {
		Error(ERR_INVALID_EA, 0x01FD, name);
		return;
	}

	// execute operation
	value = GetValue(ea, size);
	switch (what) {
		case '|' :	value = value | data; break;
		case '&' :	value = value & data; break;
		case '^' :	value = value ^ data; break;
	}
	SetValue(ea, size, value);

	// set CCR
	CPU.SetCCR(0, 0, value==0 ?1:0, MSB(value,size), -1);
}

void fn_ori(void)  { fn_ilogic('|',"ori"); }
void fn_andi(void) { fn_ilogic('&',"andi"); }
void fn_eori(void) { fn_ilogic('^',"eori"); }

//-----------------------------------------------------------------------------------------------
// Imm. Logic to CCR or SR
//-----------------------------------------------------------------------------------------------

void fn_ilogicSR(U32 value, char what)
{
	switch (what) {
		case '|' :	CPU.SR->Value = CPU.SR->Value | value; break;
		case '&' :	CPU.SR->Value = CPU.SR->Value & value; break;
		case '^' :	CPU.SR->Value = CPU.SR->Value ^ value; break;
	}
	CPU.SR->Changed = (U16) TRUE; 
}

void fn_ori_ccr(void)  { fn_ilogicSR(GetImmediate(BYTE), '|'); }
void fn_andi_ccr(void) { fn_ilogicSR(GetImmediate(BYTE), '&'); }
void fn_eori_ccr(void) { fn_ilogicSR(GetImmediate(BYTE), '^'); }

void fn_ori_sr(void)  { if (!CheckSupervisor()) return; fn_ilogicSR(GetImmediate(WORD), '|'); }
void fn_andi_sr(void) { if (!CheckSupervisor()) return; fn_ilogicSR(GetImmediate(WORD), '&');}
void fn_eori_sr(void) { if (!CheckSupervisor()) return; fn_ilogicSR(GetImmediate(WORD), '^');}

//-----------------------------------------------------------------------------------------------
// Bit functions
//-----------------------------------------------------------------------------------------------

void fn_bitfn(char what, U16 ea_valid, char *name)
{
int bit=0, size;
U16 ea;
U32 value;
int flagZ;

	if (BITS(Opcode,8,1))
		bit = CPU.D[BITS(Opcode,9,3)]->Value;		// dynamic
	else
		bit = GetImmediate(BYTE);	// static

	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), BYTE);
	if ((ea & ea_valid) == 0) {
		Error(ERR_INVALID_EA, ea_valid, name);
		return;
	}
	
	if (ea == EA_Dn || ea == EA_An)
		size = LONG;
	else
		size = BYTE;

	value = GetValue(ea, size);
	flagZ = (value & (1<<(bit%32))) ? 0 : 1;	// set Z if bit was 0

	switch (what) {
		//case 'T' :	nothing to do
		case 'S' :	value = value | (1<<(bit%32)); break;
		case 'C' :	value = value & ~(1<<(bit%32)); break;
		case 'X' :	value = value ^ (1<<(bit%32)); break;
	}
	SetValue(ea, size, value);

	// set CCR
	CPU.SetCCR(-1, -1, flagZ, -1, -1);	
}

void fn_btst(void) { fn_bitfn('T', 0x0FFD, "btst"); }
void fn_bchg(void) { fn_bitfn('X', 0x01FD, "bchg"); }
void fn_bclr(void) { fn_bitfn('C', 0x01FD, "bclr"); }
void fn_bset(void) { fn_bitfn('S', 0x01FD, "bset"); }

//-----------------------------------------------------------------------------------------------
// Others
//-----------------------------------------------------------------------------------------------

void fn_movep(void)
{
U32 reg;
U32 addr, a;
int offset, bitval;

	reg = CPU.D[BITS(Opcode,9,3)]->Value;
	addr = (S16) Peek(CPU.PCTmp , WORD);
	CPU.PCTmp += 2;
	addr += CPU.A[BITS(Opcode,0,3)]->Value;

	offset = BITS(Opcode,6,1) ? 6 : 2;
	bitval = addr & 0x00000001;
	if (BITS(Opcode,7,1)) {
		// register to memory
		for (a = addr + offset; a >= addr; reg>>=8, a -= 2)
			Poke(a + bitval, reg & 0xFF, BYTE);
	} else {
		// memory to register
		reg = 0;
		for (a = addr; a <= addr + offset; a += 2);
			reg = (reg << 8) | (Peek(a+bitval, BYTE) & 0x000000FF);
		
		CPU.D[BITS(Opcode,9,3)]->Set(reg,BITS(Opcode,6,1)?LONG:WORD);	// set value
	}

	// MOVEP doesn't affect CCR 
}

void fn_addi(void)
{
U16 ea;
U32 data, value, result;
int size, s,d,r;

	// get immediate data
	size = BITS(Opcode,6,2);
	data = GetImmediate(size);

	// ea of destination
	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), size);
	if ((ea & 0x01FD) == 0) {
		Error(ERR_INVALID_EA, 0x01FD, "addi");
		return;
	}

	// execute operation
	value = GetValue(ea, size);
	result = value + data;
	SetValue(ea, size, result);

	// set CCR
	s = MSB(data, size);
	d = MSB(value, size);
	r = MSB(result, size);
	CPU.SetCCR((s & d) | (s & ~r) | (d & ~r) ?1:0,	// C
			   (s & d & ~r) | (~s & ~d & r) ?1:0,	// V
			   result==0 ?1:0,						// Z
			   r,									// N
			   (s & d) | (s & ~r) | (d & ~r) ?1:0);	// X
}
void fn_subi_cmpi(char what, char *name)
{
U16 ea;
U32 data, value, result;
int size, s,d,r;

	// get immediate data
	size = BITS(Opcode,6,2);
	data = GetImmediate(size);

	// ea of destination
	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), size);
	if ((ea & 0x01FD) == 0) {
		Error(ERR_INVALID_EA, 0x01FD, name);
		return;
	}

	// execute operation
	value = GetValue(ea, size);
	result = value - data;
	if (what == 'S')
		SetValue(ea, size, result);

	// set CCR
	s = MSB(data, size);
	d = MSB(value, size);
	r = MSB(result, size);
	CPU.SetCCR((s & ~d) | (s & r) | (~d & r) ?1:0,	// C
			   (~s & d & ~r) | (s & ~d & r) ?1:0,	// V
			   result==0 ?1:0,						// Z
			   r,									// N
			   (s & ~d) | (s & r) | (~d & r) ?1:0);	// X
}

void fn_subi(void) { fn_subi_cmpi('S', "subi"); }
void fn_cmpi(void) { fn_subi_cmpi('C', "cmpi"); }


//***********************************************************************************************
//
// Group 1 2 3 : MOVE
//
//***********************************************************************************************

void fn_move(void)
{
U16 ea_src, ea_dest;
U32 addr_src, value;
int size;

	// size
	switch (BITS(Opcode,12,2)) {
		case 1: size = BYTE; break;
		case 2: size = LONG; break;
		case 3: size = WORD; break;

		default: size = 2;
	}

	// source
	ea_src = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), size);
	addr_src = Addr_EA;
	value = GetValue(ea_src, size, addr_src);

	// dest
	ea_dest = Compute_EA(BITS(Opcode,6,3), BITS(Opcode,9,3), size);
	if ((ea_dest & 0x01FF) == 0) {
		Error(ERR_INVALID_EA, 0x01FF, "move");
		return;
	}

	// move (src) -> (dest)
	SetValue(ea_dest, size, value);

	// set CCR (warning : MOVEA doesn't affect CCR)
	if (ea_dest != EA_An)
		CPU.SetCCR(0, 0, value==0 ?1:0, MSB(value,size), -1);
}


//***********************************************************************************************
//
// Group 4 : MISC
//
//***********************************************************************************************

void fn_move_sr(void)
{
U16 ea;

	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), WORD);

	if (BITS(Opcode,9,1)) {
		// move to SR
		if (!CheckSupervisor())
			return;

		if ((ea & 0x0FFD) == 0) {
			Error(ERR_INVALID_EA, 0x0FFD, "move to sr");
			return;
		}

		CPU.SR->Value = (U16) GetValue(ea, WORD);
		CPU.SR->Changed = (U16) TRUE;
		if (CPU.SR->Value & 0x0200)
			CPU.A[7] = CPU.SSP;
		else
			CPU.A[7] = CPU.USP;
	} else {
		// move from SR
		if ((ea & 0x01FD) == 0) {
			Error(ERR_INVALID_EA, 0x01FD, "move from sr");
			return;
		}

		SetValue(ea, WORD, CPU.SR->Value);
		// doesn't affect CCR
	}
}

void fn_move_ccr(void)
{
U16 ea;
U32 value;

	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), WORD);
	if ((ea & 0x0FFD) == 0) {
		Error(ERR_INVALID_EA, 0x0FFD, "move to ccr");
		return;
	}
		
	value = GetValue(ea, WORD);

	CPU.SR->Value = (CPU.SR->Value & 0xFF00) | (value & 0x001F);
	CPU.SR->Changed = 0x001F;	
}

void fn_chk(void)
{
U16 ea;
S16 data, reg;

	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), WORD);
	if ((ea & 0x0FFD) == 0) {
		Error(ERR_INVALID_EA, 0x01FF, "chk");
		return;
	}

	reg = (S16) (CPU.D[BITS(Opcode,9,3)]->Value & 0x0000FFFF);
	data = GetValue(ea, WORD);

	// set CCR
	CPU.SetCCR(-1, -1, -1, reg<0 ? 1 : reg>data ? 0:-1, -1);

	if (reg<0 || reg>data)
		Exception(EXCEP_CHK);
}

void fn_lea(void)
{
U16 ea;

	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), LONG);
	if ((ea & 0x07E4) == 0) {
		Error(ERR_INVALID_EA, 0x07E4, "lea");
		return;
	}

	CPU.A[BITS(Opcode,9,3)]->Set(Addr_EA, LONG);
	// LEA doesn't affect CCR
}

void fn_pea(void)
{
U16 ea;

	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), LONG);
	if ((ea & 0x07E4) == 0) {
		Error(ERR_INVALID_EA, 0x07E4, "pea");
		return;
	}

	Poke(CPU.A[7]->Minus(LONG), Addr_EA, LONG);
	// PEA doesn't affect CCR
}

void fn_clr(void)
{
int size;
U16 ea;

	size = BITS(Opcode,6,2);
	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), size);
	if ((ea & 0x01FD) == 0) {
		Error(ERR_INVALID_EA, 0x01FD, "clr");
		return;
	}
	
	SetValue(ea, size, 0);
	// set CCR
	CPU.SetCCR(0, 0, 1, 0, -1);
}

void fn_negation(char what, char *name)
{
int size, d, r;
U16 ea;
U32 value;

	size = BITS(Opcode,6,2);
	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), size);
	if ((ea & 0x01FD) == 0) {
		Error(ERR_INVALID_EA, 0x01FD, name);
		return;
	}

	value = GetValue(ea, size);
	d = MSB(value,size);
	value = 0 - value - (what=='X' ? (srX?1:0) : 0);
	r = MSB(value,size);
	SetValue(ea, size, value);

	// set CCR
	if (what=='X')
		CPU.SetCCR(d | r, d & r, value==0 ?1:-1, r, d | r);
	else
		CPU.SetCCR(value==0? 0:1, d & r, value==0 ?1:0, r, value==0? 0:1);
}

void fn_neg(void)  { fn_negation('N',"neg"); }
void fn_negx(void) { fn_negation('X',"negx"); }

void fn_not(void) { void fn_logic(char ,char *); fn_logic('~',"not"); }

void fn_swap(void)
{
int reg;
U32 value;

	reg = BITS(Opcode,0,3);
	value = CPU.D[reg]->Value;
	// swap the words
	value = ((value & 0xFFFF0000) >> 16)  |  ((value & 0x0000FFFF) << 16);
	CPU.D[reg]->Set(value, LONG);
	
	// set CCR
	CPU.SetCCR(0, 0, value==0 ?1:0, MSB(value, LONG), value&0x1);
													  // make undefined (pseudo random)
}

void fn_ext(void)
{
U16 ea;
U32 value;
int size;

	ea = EA_Dn;
	Addr_EA = BITS(Opcode,0,3);

	if (BITS(Opcode,6,1)) {
		// .W -> .L
		value = GetValue(ea, WORD);
		SetValue(ea, LONG, value);
		size = LONG;
	} else {
		// .B -> .W
		value = GetValue(ea, BYTE);
		SetValue(ea, WORD, value);
		size = WORD;
	}
	// set CCR
	CPU.SetCCR(0, 0, value==0 ?1:0, MSB(value, size), -1);
}

void fn_movem(void)
{
int i, size;
U16 reg, ea;
U32 start, addr, oldA[8];


	for (i=0; i<8; i++)
		oldA[i] = CPU.A[i]->Value;

	reg = GetImmediate(WORD);

	size = BITS(Opcode,6,1) ? LONG : WORD;
	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), size);
	if (BITS(Opcode,10,1)) {
		if ((ea & 0x07EC) == 0) {
			Error(ERR_INVALID_EA, 0x7EC, "movem (mem->reg)");
			return;
		}
	} else {
		if ((ea & 0x01F4) == 0) {
			Error(ERR_INVALID_EA, 0x01F4, "movem (reg->mem)");
			return;
		}
	}

	start = Addr_EA;

	if (ea == EA_An_dec) {
		// for mode -(An) : inverse order
		for (i=7; i>=0; i--) {
			if (reg & 0x0001) {
				SetValue(ea, size, oldA[i]);
				Addr_EA = CPU.A[BITS(Opcode,0,3)]->Minus(size);
			}
			reg >>= 1;
		}
		for (i=7; i>=0; i--) {
			if (reg & 0x0001) {
				SetValue(ea, size, CPU.D[i]->Value);
				Addr_EA = CPU.A[BITS(Opcode,0,3)]->Minus(size);
			}
			reg >>= 1;
		}
		CPU.A[BITS(Opcode,0,3)]->Plus(size);  // we do one time too much

		addr = start + (size==LONG ? 4:2);
		start = CPU.A[BITS(Opcode,0,3)]->Value;
	} else {
		// others modes
		addr = Addr_EA;
		for (i=0; i<8; i++) {
			if (reg & 0x0001) {
				if (BITS(Opcode,10,1)) {
					// mem -> reg
					CPU.D[i]->Set(GetValue(EA_An_ind, size, addr), LONG);
				} else {
					// reg -> mem
					SetValue(EA_An_ind, size, CPU.D[i]->Value, addr);
				}
				addr += size==LONG ? 4 : 2;
			}
			reg >>= 1;
		}
		for (i=0; i<8; i++) {
			if (reg & 0x0001) {
				if (BITS(Opcode,10,1)) {
					// mem -> reg
					CPU.A[i]->Set(GetValue(EA_An_ind, size, addr), LONG);
				} else {
					// reg -> mem
					SetValue(EA_An_ind, size, oldA[i], addr);
				}
				addr += size==LONG ? 4 : 2;
			}
			reg >>= 1;
		}
		if (ea == EA_An_inc)
			CPU.A[BITS(Opcode,0,3)]->Set(addr,LONG);
	}


	// look if poke affects memory dumping windows

	if (BITS(Opcode,10,1) == 0) {
		for (i=0; i<NB_DUMP; i++) {
			if (start>=MD_start[i]  &&  start<=(MD_start[i]+MD_len[i])) {
				MD_where[i] = start;
				if (addr<=(MD_start[i]+MD_len[i]))
					MD_size[i] = start-addr;	// negative = block
				else
					MD_size[i] = start-(MD_start[i]+MD_len[i]);	// negative = block
			} else if (addr>=MD_start[i]  &&  addr<=(MD_start[i]+MD_len[i])) {
				MD_where[i] = MD_start[i];
				MD_size[i] = MD_start[i]-addr;	// negative = block
			}
		}
	}
	

	// MOVEM doesn't affect CCR
}

void fn_tst(void)
{
int size;
U16 ea;
U32 value;

	size = BITS(Opcode,6,2);
	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), size);
	if ((ea & 0x01FD) == 0) {
		Error(ERR_INVALID_EA, 0x01FD, "tst");
		return;
	}
	
	value = GetValue(ea, size);

	// set CCR
	CPU.SetCCR(0, 0, value==0 ?1:0, MSB(value,size), -1);
}

void fn_tas(void)
{
U16 ea;
U32 value;

	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), BYTE);
	if ((ea & 0x01FD) == 0) {
		Error(ERR_INVALID_EA, 0x01FD, "tas");
		return;
	}
	
	value = GetValue(ea, BYTE);
	
	// set CCR
	CPU.SetCCR(0, 0, value==0 ?1:0, MSB(value, BYTE), -1);

	value |= 0x80;	// set bit 7
	SetValue(ea, BYTE, value);
}

void fn_illegal(void)
{
	Exception(EXCEP_ILLEGAL);
}

void fn_trap(void)
{
int number;

	number = BITS(Opcode,0,4);
	Exception(EXCEP_TRAP + number);
}

void fn_link(void)
{
int reg;
U32 offset;

	reg = BITS(Opcode,0,3);

	if ((Opcode & 0xFF00) == 0x4E00) {
		offset = (S16) Peek(CPU.PCTmp , WORD);
		CPU.PCTmp += 2;
	} else {
		offset = (S32) Peek(CPU.PCTmp , LONG);
		CPU.PCTmp += 4;		
	}

	Poke(CPU.A[7]->Minus(LONG), CPU.A[reg]->Value, LONG);
	CPU.A[reg]->Set(CPU.A[7]->Value, LONG);
	offset = CPU.A[7]->Value + offset;
	CPU.A[7]->Set(offset, LONG);

	// LINK doesn't affect CCR
}

void fn_unlink(void)
{
int reg;

	reg = BITS(Opcode,0,3);
	CPU.A[7]->Set(CPU.A[reg]->Value, LONG);
	CPU.A[reg]->Set(Peek(CPU.A[7]->Plus(LONG), LONG), LONG);

	// UNLINK doesn't affect CCR		
}

void fn_move_usp(void)
{
	if (!CheckSupervisor())
		return;

	if (BITS(Opcode,3,1))
		CPU.A[BITS(Opcode,0,3)]->Set(CPU.USP->Value, LONG);
	else
		CPU.USP->Set(CPU.A[BITS(Opcode,0,3)]->Value, LONG);

	// MOVE USP doesn't affect CCR
}

void fn_movec(void)
{
U16 control;


	if (!CheckSupervisor())
		return;

	// where to go
	control = (U16) Peek(CPU.PCTmp , WORD);
	CPU.PCTmp += 2;

	if (BITS(Opcode,0,1)) {
		switch (BITS(control,0,12)) {
			case 0x000:
			case 0x001:
			case 0x002:
			case 0x800:
			case 0x802:
			case 0x803:
			case 0x804:	break;
			case 0x801:	CPU.VBR = (BITS(control,15,1) ? CPU.A[BITS(control,12,3)]->Value
													  : CPU.D[BITS(control,12,3)]->Value);
						sprintf(what, "0x%08X", CPU.VBR);
						Tcl_SetVar(output, "VBR_text", what, TCL_GLOBAL_ONLY);
						break;

			default:	Exception(EXCEP_ILLEGAL);
		}
	} else {
		switch (BITS(control,0,12)) {
			case 0x000:
			case 0x001:
			case 0x002:
			case 0x800:
			case 0x802:
			case 0x803:
			case 0x804:	break;
			case 0x801: if (BITS(control,15,1))
							CPU.A[BITS(control,12,3)]->Set(CPU.VBR , LONG);
						else
							CPU.D[BITS(control,12,3)]->Set(CPU.VBR , LONG);
						break;

			default:	Exception(EXCEP_ILLEGAL);
		}
	}
}

void fn_reset(void)
{
	if (!CheckSupervisor())
		return;

	Tcl_Eval(output, ".info.text insert end \"Reset done ... \n\"");
}


void fn_nop(void)
{
	// does nothing ....
	// NOP doesn't affect CCR
}

void fn_stop(void)
{
	if (!CheckSupervisor())
		return;
	
	CPU.SR->Value = Peek(CPU.PCTmp, WORD) &0x0000FFFF;
	CPU.SR->Changed = (U16) TRUE;
	CPU.PCTmp += 2;
	
	Tcl_Eval(output, ".info.text insert end \"Stop not supported in simulator ... \n\"");
}

void fn_rte(void)
{
	if (!CheckSupervisor())
		return;

	CPU.SR->Value = Peek(CPU.A[7]->Plus(WORD), WORD);
	CPU.SR->Changed = (U16) TRUE;
	CPU.PCTmp = Peek(CPU.A[7]->Plus(LONG), LONG);

	// if format was Six Word Stack Frame then pop Instruction Address
	if (Peek(CPU.A[7]->Plus(WORD), WORD) & 0xF000)
		Peek(CPU.A[7]->Plus(LONG), LONG);

	if (CPU.SR->Value & 0x0200)
		CPU.A[7] = CPU.SSP;
	else
		CPU.A[7] = CPU.USP;
}

void fn_rts(void)
{
	CPU.PCTmp = Peek(CPU.A[7]->Plus(LONG), LONG);
	// RTS doesn't affect CCR
}

void fn_trapv(void)
{
	if (srV)
		Exception(EXCEP_TRAPV);
}

void fn_rtr(void)
{
	CPU.SR->Value = (CPU.SR->Value & 0xFF00) | (Peek(CPU.A[7]->Plus(WORD), WORD) & 0x00FF);
	CPU.SR->Changed = 0x00FF;
	CPU.PCTmp = Peek(CPU.A[7]->Plus(LONG), LONG);
}

void fn_jsr(void)
{
U16 ea;

	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), LONG);
	if ((ea & 0x07E4) == 0) {
		Error(ERR_INVALID_EA, 0x07E4, "jsr");
		return;
	}

	Poke(CPU.A[7]->Minus(LONG), CPU.PCTmp, LONG);	// save old PC in stack
	CPU.PCTmp = Addr_EA;
	// JMP doesn't affect CCR
}

void fn_jmp(void)
{
U16 ea;

	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), LONG);
	if ((ea & 0x07E4) == 0) {
		Error(ERR_INVALID_EA, 0x07E4, "jmp");
		return;
	}

	CPU.PCTmp = Addr_EA;
	// JMP doesn't affect CCR
}


//***********************************************************************************************
//
// Group 5 : DB.. + S.. + ADDQ + SUBQ
//
//***********************************************************************************************

int Condition(int what)
{
	switch (what) {
		case  0:	return TRUE;										// f / bra
		case  1:	return FALSE;										// f / dbra / bsr
		case  2:	return (!srC && !srZ);								// hi
		case  3:	return (srC || srZ);								// ls
		case  4:	return (!srC);										// cc
		case  5:	return (srC);										// cs
		case  6:	return (!srZ);										// ne
		case  7:	return (srZ);										// eq
		case  8:	return (!srV);										// vc
		case  9:	return (srV);										// vs
		case 10:	return (!srN);										// pl
		case 11:	return (srN);										// mi
		case 12:	return ((srN && srV) || (!srN && !srV));			// ge
		case 13:	return ((srN && !srV) || (!srN && srV));			// lt
		case 14:	return ( ((srN && srV) || (!srN && !srV)) && !srZ);	// gt
		case 15:	return (srZ || (srN && !srV) || (!srN && srV));		// le
	}
	return 0;
}

//-----------------------------------------------------------------------------------------------
// Contional set / decr.branchment
//-----------------------------------------------------------------------------------------------

void fn_set(void)
{
U16 ea;
U32 result;

	// get ea
	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), BYTE);
	if ((ea & 0x01FD) == 0) {
		Error(ERR_INVALID_EA, 0x01FD, "set");
		return;
	}

	// test condition
	if (Condition(BITS(Opcode,8,4)))
		result = 0xFFFFFFFF;
	else
		result = 0;

	// set result
	SetValue(ea, BYTE, result);

	// Scc doesn't affect CCR
}

void fn_dbcc(void)
{
U32 addr;
U16 reg;

	// where to go
	addr = (S16) Peek(CPU.PCTmp , WORD);
	CPU.PCTmp += 2;

	// test condition
	if (!Condition(BITS(Opcode,8,4))) {
		reg = (CPU.D[BITS(Opcode,0,3)]->Value & 0x0000FFFF) -1;
		CPU.D[BITS(Opcode,0,3)]->Set(reg, WORD);

		if (reg != (U16) -1)
			CPU.PCTmp = CPU.PC + addr + 2;
	}

	// DBcc doesn't affect CCR
}

//-----------------------------------------------------------------------------------------------
// Quick arithmetic
//-----------------------------------------------------------------------------------------------

void fn_addq(void)
{
U16 ea;
U32 data, value, result;
int size, d,r;

	// get data
	size = BITS(Opcode,6,2);
	data = BITS(Opcode,9,3);
	if (data == 0)	data=8;

	// ea of destination
	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), size);
	if ((ea & 0x01FF) == 0) {
		Error(ERR_INVALID_EA, 0x01FF, "addq");
		return;
	}

	// execute operation
	value = GetValue(ea, size);
	result = value + data;
	SetValue(ea, size, result);

	// set CCR
	d = MSB(value, size);
	r = MSB(result, size);
	CPU.SetCCR(d & ~r ?1:0,		// C
			   ~d & r ?1:0,		// V
			   result==0 ?1:0,	// Z
			   r,				// N
			   d & ~r ?1:0);	// X
}
void fn_subq(void)
{
U16 ea;
U32 data, value, result;
int size, d,r;

	// get data
	size = BITS(Opcode,6,2);
	data = BITS(Opcode,9,3);
	if (data == 0)	data=8;

	// ea of destination
	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), size);
	if ((ea & 0x01FF) == 0) {
		Error(ERR_INVALID_EA, 0x01FF, "subq");
		return;
	}

	// execute operation
	value = GetValue(ea, size);
	result = value - data;
	SetValue(ea, size, result);

	// set CCR
	d = MSB(value, size);
	r = MSB(result, size);
	CPU.SetCCR(~d & r ?1:0,		// C
			   d & ~r ?1:0,		// V
			   result==0 ?1:0,	// Z
			   r,				// N
			   ~d & r ?1:0);	// X
}


//***********************************************************************************************
//
// Group 6 : B..
//
//***********************************************************************************************

void fn_bcc(void)
{
U32 addr;

	// where to go
	addr = (S8) BITS(Opcode,0,8);
	if (addr == 0) {
		addr = (S16) Peek(CPU.PCTmp , WORD);
		CPU.PCTmp += 2;
	} else if (((signed int) addr == -1) && (CurrentModel > 68000)) {
		addr = (S32) Peek(CPU.PCTmp , LONG);
		CPU.PCTmp += 4;
	}
	
	// test condition
	if (BITS(Opcode,8,4) == 1) {	// bsr
		Poke(CPU.A[7]->Minus(LONG), CPU.PCTmp, LONG);
		CPU.PCTmp = CPU.PC + addr + 2;
	} else {						// bcc	
		if (Condition(BITS(Opcode,8,4)))
			CPU.PCTmp = CPU.PC + addr + 2;
	}

	// Bcc doesn't affect CCR
}


//***********************************************************************************************
//
// Group 7 : MOVEQ
//
//***********************************************************************************************

void fn_moveq(void)
{
S8 value;

	value = BITS(Opcode,0,8);
	CPU.D[BITS(Opcode,9,3)]->Set(value, LONG);
	// set CCR
	CPU.SetCCR(0, 0, value==0 ?1:0, value<0 ?1:0, -1);
}


//***********************************************************************************************
//
// Group 8 : SBCD + DIV + OR
//
//***********************************************************************************************


void fn_logic(char what, char *name)
{
U16 ea_dest, ea_src;
U32	addr_dest, addr_src;
char tmp[30];
int size;
U32 src,dest;


	size = BITS(Opcode,6,2);

	// ea of source and destination
	if (what == '^') {
		// mode for EOR
		ea_src = EA_Dn;
		addr_src = BITS(Opcode,9,3);
		ea_dest = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), size);
		addr_dest = Addr_EA;
		if ((ea_dest & 0x01FD) == 0) {
			Error(ERR_INVALID_EA, 0x01FD, name);
			return;
		}
	} else if (what == '~') {
		// mode for NOT
		ea_src = EA_Dn;					// not used
		addr_src = BITS(Opcode,9,3);	// not used
		ea_dest = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), size);
		addr_dest = Addr_EA;
		if ((ea_dest & 0x01FD) == 0) {
			Error(ERR_INVALID_EA, 0x01FD, name);
			return;
		}
	} else {
		// mode for AND and OR
		if (BITS(Opcode,8,1)) {
			// opcode Dn,<ea>
			ea_src = EA_Dn;
			addr_src = BITS(Opcode,9,3);
			ea_dest = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), size);
			addr_dest = Addr_EA;
			sprintf(tmp,"%s (dest)",name);
			if ((ea_dest & 0x01FC) == 0) {
				Error(ERR_INVALID_EA, 0x01FC, tmp);
				return;
			}
		} else {
			// opcode <ea>,Dn
			ea_src = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), size);
			addr_src = Addr_EA;
			ea_dest = EA_Dn;
			addr_dest = BITS(Opcode,9,3);
			sprintf(tmp,"%s (src)",name);
			if ((ea_src & 0x0FFD) == 0) {
				Error(ERR_INVALID_EA, 0x0FFD, tmp);
				return;
			}
		}
	}

	src = GetValue(ea_src, size, addr_src);
	dest = GetValue(ea_dest, size, addr_dest);

	switch (what) {
		case '&':	dest = dest & src; break;
		case '|':	dest = dest | src; break;
		case '^':	dest = dest ^ src; break;
		case '~':	dest = ~dest; break;
	}

	SetValue(ea_dest, size, dest, addr_dest);

	// set CCR
	CPU.SetCCR(0, 0, dest==0 ?1:0, MSB(dest,size), what=='~' ? -1 : dest&0x1);
}

void fn_or(void) { fn_logic('|',"or"); }

void fn_divu(void)
{
int flag = 0;		// for tracking overflow
U16 ea;
U32 src,dest,result;

	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), WORD);
	if ((ea & 0x0FFD) == 0) {
		Error(ERR_INVALID_EA, 0x0FFD, "divu");
		return;
	}
	
	src = GetValue(ea, WORD) & 0x0000FFFF;
	if (src == 0) {
		Exception(EXCEP_DIV);
		return;
	}

	dest = GetValue(EA_Dn, LONG, BITS(Opcode,9,3));

	result = dest/src;
	if (result > 0xFFFF)
		flag = 1;
	else
		SetValue(EA_Dn, LONG,
				  (((dest % src) & 0xFFFF)<<16) | (result & 0xFFFF), BITS(Opcode,9,3));

	// set CCR
	CPU.SetCCR(0, flag, result==0 ?1:0, MSB(result,WORD), -1);
}

void fn_divs(void) 
{
int flag = 0;		// for tracking overflow
U16 ea;
S32 src,dest,result;

	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), WORD);
	if ((ea & 0x0FFD) == 0) {
		Error(ERR_INVALID_EA, 0x0FFD, "divs");
		return;
	}
	
	src = GetValue(ea, WORD);
	if (src == 0) {
		Exception(EXCEP_DIV);
		return;
	}

	dest = GetValue(EA_Dn, LONG, BITS(Opcode,9,3));

	result = dest/src;
	if (result > 0xFFFF)
		flag = 1;
	else
		SetValue(EA_Dn, LONG,
				  (((dest % src) & 0xFFFF)<<16) | (result & 0xFFFF), BITS(Opcode,9,3));

	// set CCR
	CPU.SetCCR(0, flag, result==0 ?1:0, MSB(result,WORD), -1);
}

//-----------------------------------------------------------------------------------------------
// Binary Coded Decimal Arithmetic
//-----------------------------------------------------------------------------------------------

void fn_sbcd(void)
{
U16 ea_src, ea_dest;
U32 addr_src, addr_dest, src, dest;
S32 result;

	if (BITS(Opcode,3,1)) {
		// mem + mem
		ea_src = EA_An_dec;
		addr_src = CPU.A[BITS(Opcode,0,3)]->Minus(BYTE);
		ea_dest = EA_An_dec;
		addr_dest = CPU.A[BITS(Opcode,9,3)]->Minus(BYTE);
	} else {
		// reg + reg
		ea_src = EA_Dn;
		addr_src = BITS(Opcode,0,3);
		ea_dest = EA_Dn;
		addr_dest = BITS(Opcode,9,3);
	}

	src = GetValue(ea_src, BYTE, addr_src) & 0x00FF;
	src = (src >> 4) * 10 + (src & 0xF);
	dest = GetValue(ea_dest, BYTE, addr_dest) & 0x00FF;
	dest = (dest >> 4) * 10 + (dest & 0xF);

	result = dest - src - (srX?1:0);

	// set CCR
	CPU.SetCCR(result<0 ?1:0, result&1, result==0 ?-1:0, result&1, result<0 ?1:0);

	if (result<0)	result += 100;
	result = ((result / 10) << 4) | (result % 10);
	SetValue(ea_dest, BYTE, result, addr_dest);
}

void fn_abcd(void)
{
U16 ea_src, ea_dest;
U32 addr_src, addr_dest, src, dest, result;

	if (BITS(Opcode,3,1)) {
		// mem + mem
		ea_src = EA_An_dec;
		addr_src = CPU.A[BITS(Opcode,0,3)]->Minus(BYTE);
		ea_dest = EA_An_dec;
		addr_dest = CPU.A[BITS(Opcode,9,3)]->Minus(BYTE);
	} else {
		// reg + reg
		ea_src = EA_Dn;
		addr_src = BITS(Opcode,0,3);
		ea_dest = EA_Dn;
		addr_dest = BITS(Opcode,9,3);
	}

	src = GetValue(ea_src, BYTE, addr_src) & 0x00FF;
	src = (src >> 4) * 10 + (src & 0xF);
	dest = GetValue(ea_dest, BYTE, addr_dest) & 0x00FF;
	dest = (dest >> 4) * 10 + (dest & 0xF);

	result = src + dest + (srX?1:0);

	// set CCR
	CPU.SetCCR(result>99 ?1:0, result&1, result==0 ?-1:0, result&1, result>99 ?1:0);

	result %= 100;
	result = ((result / 10) << 4) | (result % 10);
	SetValue(ea_dest, BYTE, result, addr_dest);
}

void fn_nbcd(void)
{
U16 ea;
U32 src;
S32 result;

	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), BYTE);
	if ((ea & 0x01FD) == 0) {
		Error(ERR_INVALID_EA, 0x01FD, "nbcd");
		return;
	}

	src = GetValue(ea, BYTE);
	src = (src >> 4) * 10 + (src & 0xF);
	result = 0 - src - (srX?1:0);

	CPU.SetCCR(result<0 ?1:0, result&1, result==0 ?-1:0, result&1, result<0 ?1:0);
	if (result<0)	result += 100;
	result = ((result / 10) << 4) | (result % 10);
	
	SetValue(ea, BYTE, result);
}


//***********************************************************************************************
//
// Group 9/D : SUB / ADD  +  CMP
//
//***********************************************************************************************

//-----------------------------------------------------------------------------------------------
// For address register
//-----------------------------------------------------------------------------------------------

void fn_subcomp_a(char what, char *name)
{
int size, s,d,r;
U16 ea;
U32 src,dest,result;

	size = BITS(Opcode,8,1) ? LONG : WORD;	

	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), size);
	// all mode authorized ...

	src = GetValue(ea, size);
	dest = GetValue(EA_An, LONG, BITS(Opcode,9,3));
	result = dest - src;
	if (what == 'S')
		SetValue(EA_An, LONG, result, BITS(Opcode,9,3));

	// set CCR only for CMPA. SUBA doesn't affect CCR
	if (what == 'C') {
		s = MSB(src, size);
		d = MSB(dest, size);
		r = MSB(result, size);

		CPU.SetCCR((s & ~d) | (s & r) | (~d & r) ?1:0,	// C
				   (~s & d & ~r) | (s & ~d & r) ?1:0,	// V
				   result==0 ?1:0,						// Z
				   r,									// N
				   -1);									// X
	}
}
void fn_suba(void) { fn_subcomp_a('S',"suba"); }
void fn_cmpa(void) { fn_subcomp_a('C',"cmpa"); }

void fn_adda(void)
{
int size;
U16 ea;
U32 src,dest,result;

	size = BITS(Opcode,8,1) ? LONG : WORD;	
	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), size);
	// all mode authorized ...

	src = GetValue(ea, size);
	dest = GetValue(EA_An, LONG, BITS(Opcode,9,3));
	result = dest + src;
	SetValue(EA_An, LONG, result, BITS(Opcode,9,3));

	// ADDA doesn't affect CCR
}

//-----------------------------------------------------------------------------------------------
// With X flag
//-----------------------------------------------------------------------------------------------

void fn_subx(void)
{
U16 ea_src,ea_dest;
U32 addr_src,addr_dest, src,dest,result;
int size, s,d,r;

	size = BITS(Opcode,6,2);

	if (BITS(Opcode,3,1)) {
		// memory
		ea_src = EA_An_ind;
		addr_src = CPU.A[BITS(Opcode,0,3)]->Minus(size);
		ea_dest = EA_An_ind;
		addr_dest = CPU.A[BITS(Opcode,9,3)]->Minus(size);
	} else {
		// register
		ea_src = EA_Dn;
		addr_src = BITS(Opcode,0,3);
		ea_dest = EA_Dn;
		addr_dest = BITS(Opcode,9,3);
	}

	src = GetValue(ea_src, size, addr_src);
	dest = GetValue(ea_dest, size, addr_dest);
	result = dest - src - (srX ? 1 : 0);

	s = MSB(src, size);
	d = MSB(dest, size);
	r = MSB(result, size);

	SetValue(ea_dest, size, result, addr_dest);

	// set CCR
	CPU.SetCCR((s & ~d) | (s & r) | (~d & r) ?1:0,	// C
			   (~s & d & ~r) | (s & ~d & r) ?1:0,	// V
			   result==0 ?1:0,						// Z
			   r,									// N
			   (s & ~d) | (s & r) | (~d & r) ?1:0);	// X
}

void fn_addx(void)
{
U16 ea_src,ea_dest;
U32 addr_src,addr_dest, src,dest,result;
int size, s,d,r;

	size = BITS(Opcode,6,2);

	if (BITS(Opcode,3,1)) {
		// memory
		ea_src = EA_An_ind;
		addr_src = CPU.A[BITS(Opcode,0,3)]->Minus(size);
		ea_dest = EA_An_ind;
		addr_dest = CPU.A[BITS(Opcode,9,3)]->Minus(size);
	} else {
		// register
		ea_src = EA_Dn;
		addr_src = BITS(Opcode,0,3);
		ea_dest = EA_Dn;
		addr_dest = BITS(Opcode,9,3);
	}

	src = GetValue(ea_src, size, addr_src);
	dest = GetValue(ea_dest, size, addr_dest);
	result = dest + src + (srX ? 1 : 0);

	s = MSB(src, size);
	d = MSB(dest, size);
	r = MSB(result, size);

	SetValue(ea_dest, size, result, addr_dest);

	// set CCR
	CPU.SetCCR((s & d) | (s & ~r) | (d & ~r) ?1:0,	// C
			   (s & d & ~r) | (~s & ~d & r) ?1:0,	// V
			   result==0 ?1:0,						// Z
			   r,									// N
			   (s & d) | (s & ~r) | (d & ~r) ?1:0);	// X
}

//-----------------------------------------------------------------------------------------------
// Simple
//-----------------------------------------------------------------------------------------------

void fn_subcmp(char what, char *name)
{
char tmp[30];
int size, s,d,r;
U16 ea_src, ea_dest;
U32 addr_src, addr_dest;
U32 src,dest,result;


	size = BITS(Opcode,6,2);

	if (BITS(Opcode,8,1)) {
		// opcode Dn,<ea>
		ea_src = EA_Dn;
		addr_src = BITS(Opcode,9,3);
		ea_dest = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), size);
		addr_dest = Addr_EA;
		sprintf(tmp,"%s (dest)",name);
		if ((ea_dest & 0x01FC) == 0) {
			Error(ERR_INVALID_EA, 0x01FC, tmp);
			return;
		}
	} else {
		// opcode <ea>,Dn
		ea_src = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), size);
		addr_src = Addr_EA;
		ea_dest = EA_Dn;
		addr_dest = BITS(Opcode,9,3);
		// all mode authorized
	}

	src = GetValue(ea_src, size, addr_src);
	dest = GetValue(ea_dest, size, addr_dest);
	result = dest - src;

	s = MSB(src, size);
	d = MSB(dest, size);
	r = MSB(result, size);

	if (what=='S')
		SetValue(ea_dest, size, result, addr_dest);

	// set CCR
	CPU.SetCCR((s & ~d) | (s & r) | (~d & r) ?1:0,	// C
			   (~s & d & ~r) | (s & ~d & r) ?1:0,	// V
			   result==0 ?1:0,						// Z
			   r,									// N
			   (s & ~d) | (s & r) | (~d & r) ?1:0);	// X
}
void fn_sub(void) { fn_subcmp('S',"sub"); }
void fn_cmp(void) { fn_subcmp('C',"cmp"); }

void fn_add(void)
{
int size, s,d,r;
U16 ea_src, ea_dest;
U32 addr_src, addr_dest;
U32 src,dest,result;


	size = BITS(Opcode,6,2);

	if (BITS(Opcode,8,1)) {
		// opcode Dn,<ea>
		ea_src = EA_Dn;
		addr_src = BITS(Opcode,9,3);
		ea_dest = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), size);
		addr_dest = Addr_EA;
		if ((ea_dest & 0x01FC) == 0) {
			Error(ERR_INVALID_EA, 0x01FC, "add (dest)");
			return;
		}
	} else {
		// opcode <ea>,Dn
		ea_src = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), size);
		addr_src = Addr_EA;
		ea_dest = EA_Dn;
		addr_dest = BITS(Opcode,9,3);
		// all mode authorized
	}

	src = GetValue(ea_src, size, addr_src);
	dest = GetValue(ea_dest, size, addr_dest);
	result = dest + src;

	s = MSB(src, size);
	d = MSB(dest, size);
	r = MSB(result, size);

	SetValue(ea_dest, size, result, addr_dest);

	// set CCR
	CPU.SetCCR((s & d) | (s & ~r) | (d & ~r) ?1:0,	// C
			   (s & d & ~r) | (~s & ~d & r) ?1:0,	// V
			   result==0 ?1:0,						// Z
			   r,									// N
			   (s & d) | (s & ~r) | (d & ~r) ?1:0);	// X
}

//-----------------------------------------------------------------------------------------------
// Comparaison in memory
//-----------------------------------------------------------------------------------------------

void fn_cmpm(void)
{
int as,ad, size, s,d,r;
U32 src,dest,result;

	as = BITS(Opcode,0,3);
	ad = BITS(Opcode,9,3);
	size = BITS(Opcode,6,2);

	src = GetValue(EA_An_ind, size, CPU.A[as]->Value);
	CPU.A[as]->Plus(size);
	dest = GetValue(EA_An_ind, size, CPU.A[ad]->Value);
	CPU.A[ad]->Plus(size);

	result = dest - src;

	s = MSB(src, size);
	d = MSB(dest, size);
	r = MSB(result, size);

	// set CCR
	CPU.SetCCR((s & ~d) | (s & r) | (~d & r) ?1:0,	// C
			   (~s & d & ~r) | (s & ~d & r) ?1:0,	// V
			   result==0 ?1:0,						// Z
			   r,									// N
			   -1);									// X
}


//***********************************************************************************************
//
// Group A/F : Line A/F
//
//***********************************************************************************************

void fn_lineA(void)
{
	Exception(EXCEP_LINE_A);
}

void fn_lineF(void)
{
	Exception(EXCEP_LINE_F);
}


//***********************************************************************************************
//
// Group B : CMP (put with sub) + EOR
//
//***********************************************************************************************

void fn_eor(void) { fn_logic('^',"eor"); }

//***********************************************************************************************
//
// Group C : EXG + ABCD + MUL + AND
//
//***********************************************************************************************

void fn_exg(void)
{
int src, dest;
U32 tmp;

	src = BITS(Opcode,0,3);
	dest = BITS(Opcode,9,3);

	switch (BITS(Opcode,3,5)) {
		case 0x08:	tmp = CPU.D[src]->Value;
					CPU.D[src]->Value = CPU.D[dest]->Value;
					CPU.D[dest]->Value = tmp;
					CPU.D[dest]->Changed = CPU.D[src]->Changed = (U16) TRUE;
					break;

		case 0x09:	tmp = CPU.A[src]->Value;
					CPU.A[src]->Value = CPU.A[dest]->Value;
					CPU.A[dest]->Value = tmp;
					CPU.A[dest]->Changed = CPU.A[src]->Changed = (U16) TRUE;
					break;

		case 0x11:	tmp = CPU.D[src]->Value;
					CPU.D[src]->Value = CPU.A[dest]->Value;
					CPU.A[dest]->Value = tmp;
					CPU.A[dest]->Changed = CPU.D[src]->Changed = (U16) TRUE;
					break;
	}

	// EXG doesn't affect CCR
}

void fn_and(void) { fn_logic('&',"and"); }

void fn_multiplication(char what, char *name)
{
U16 ea;
U32 src,dest,result;

	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), WORD);
	if ((ea & 0x0FFD) == 0) {
		Error(ERR_INVALID_EA, 0x0FFD, name);
		return;
	}
	
	src = GetValue(ea, WORD);
	dest = GetValue(EA_Dn, WORD, BITS(Opcode,9,3));

	if (what == 'S')
		result = (S32) ((S16)(src) * (S16)(dest));
	else
		result = (U32) ((U16)(src) * (U16)(dest));

	SetValue(EA_Dn, LONG, result, BITS(Opcode,9,3));

	// set CCR
	CPU.SetCCR(0, 0, result==0 ?1:0, MSB(result,LONG), -1);
}

void fn_mulu(void) { fn_multiplication('U',"mulu"); }
void fn_muls(void) { fn_multiplication('S',"muls"); }


//***********************************************************************************************
//
// Group E : Shifts
//
//***********************************************************************************************

//-----------------------------------------------------------------------------------------------
// Arithmetic Shift
//-----------------------------------------------------------------------------------------------

void fn_asr(void)
{
int count, size, i, flag;
U16 ea;
U32 value, mask;

	// test if counter is register or immediate
	if (BITS(Opcode,5,1))
		count = CPU.D[BITS(Opcode,9,3)]->Value % 64;
	else {
		count = BITS(Opcode,9,3);
		if (count == 0)	count=8;
	}

	size = BITS(Opcode,6,2);
	// simulate EA for getting value
	ea = EA_Dn;
	Addr_EA = BITS(Opcode,0,3);
	value = GetValue(ea, size);

	// make shifts
	if (count == 0) {
		CPU.SetCCR(0, -1, value==0 ?1:0, MSB(value,size), -1);
		return;
	} else {
		mask = 1 << (size==BYTE ? 7 : size==WORD ? 15 : 31);
		for (i=0; i<count-1; i++)		// -1 for getting C and X before last shift
			value = (value & mask) | (value >> 1);
		flag = value & 0x00000001;
			value = (value & mask) | (value >> 1);	// last shift
	}

	// set result
	SetValue(EA_Dn, size, value, BITS(Opcode,0,3));

	// set CCR
	CPU.SetCCR(flag, -1, value==0 ?1:0, MSB(value,size), flag);
}

void fn_asl(void) 
{
int count, size, i, flag, MSBchanged = FALSE, MSBinit;
U32 value;

	// test if counter is register or immediate
	if (BITS(Opcode,5,1))
		count = CPU.D[BITS(Opcode,9,3)]->Value % 64;
	else {
		count = BITS(Opcode,9,3);
		if (count == 0)	count=8;
	}

	size = BITS(Opcode,6,2);
	// simulate EA for getting value
	value = GetValue(EA_Dn, size, BITS(Opcode,0,3));

	// make shifts
	if (count == 0) {
		CPU.SetCCR(0, -1, value==0 ?1:0, MSB(value,size), -1);
		return;
	} else {
		MSBinit = MSB(value, size);
		for (i=0; i<count-1; i++) {		// -1 for getting C and X before last shift
			value <<= 1;
			if (MSB(value, size) != MSBinit)
				MSBchanged = TRUE;
		}
		flag = MSB(value, size);
			value <<= 1;	// last shift
			if (MSB(value, size) != MSBinit)
				MSBchanged = TRUE;
	}

	// set result
	SetValue(EA_Dn, size, value, BITS(Opcode,0,3));

	// set CCR
	CPU.SetCCR(flag, MSBchanged ?1:-1, value==0 ?1:0, MSB(value,size), flag);
}

void fn_asr_mem(void)
{
U16 ea;
U32 value;
int flag;

	// compute ea
	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), WORD);
	if ((ea & 0x01FC) == 0) {
		Error(ERR_INVALID_EA, 0x01FC, "asr (mem)");
		return;
	}
	
	// make shift
	value = GetValue(ea, WORD);
	flag = value & 0x00000001;
	value = (value & 0x00008000) | (value >> 1);
	SetValue(ea, WORD, value);
	// set CCR
	CPU.SetCCR(flag, -1, value==0 ?1:0, MSB(value,WORD), flag);
}

void fn_asl_mem(void)
{
U16 ea;
U32 value;
int flag;

	// compute ea
	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), WORD);
	if ((ea & 0x01FC) == 0) {
		Error(ERR_INVALID_EA, 0x01FC, "asl (mem)");
		return;
	}
	
	// make shift
	value = GetValue(ea, WORD);
	flag = MSB(value, WORD);
	value <<= 1;
	SetValue(ea, WORD, value);
	// set CCR
	CPU.SetCCR(flag, MSB(value,WORD)!=flag ? 1:0, value==0 ?1:0, MSB(value,WORD), flag);
}

//-----------------------------------------------------------------------------------------------
// Logical Shift
//-----------------------------------------------------------------------------------------------

void fn_lsr(void)
{
int count, size, flag;
U32 value;

	// test if counter is register or immediate
	if (BITS(Opcode,5,1))
		count = CPU.D[BITS(Opcode,9,3)]->Value % 64;
	else {
		count = BITS(Opcode,9,3);
		if (count == 0)	count=8;
	}

	size = BITS(Opcode,6,2);
	// simulate EA for getting value
	value = GetValue(EA_Dn, size, BITS(Opcode,0,3));

	// make shifts
	if (count == 0) {
		CPU.SetCCR(0, 0, value==0 ?1:0, MSB(value,size), -1);
		return;
	} else {
		switch (size) {
			case BYTE: value = value & 0x000000FF; break;
			case WORD: value = value & 0x0000FFFF; break;
		}	// be sure MSBs to be zero
		value >>= (count - 1);
		flag = value & 0x00000001;	// get flags
		value >>= 1;	// last shift
	}

	// set result
	SetValue(EA_Dn, size, value, BITS(Opcode,0,3));

	// set CCR
	CPU.SetCCR(flag, 0, value==0 ?1:0, MSB(value,size), flag);
}

void fn_lsl(void)
{
int count, size, flag;
U32 value;

	// test if counter is register or immediate
	if (BITS(Opcode,5,1))
		count = CPU.D[BITS(Opcode,9,3)]->Value % 64;
	else {
		count = BITS(Opcode,9,3);
		if (count == 0)	count=8;
	}

	size = BITS(Opcode,6,2);
	// simulate EA for getting value
	value = GetValue(EA_Dn, size, BITS(Opcode,0,3));

	// make shifts
	if (count == 0) {
		CPU.SetCCR(0, 0, value==0 ?1:0, MSB(value,size), -1);
		return;
	} else {
		value <<= (count - 1);
		flag = MSB(value, size);	// get flags
		value <<= 1;	// last shift
	}

	// set result
	SetValue(EA_Dn, size, value, BITS(Opcode,0,3));

	// set CCR
	CPU.SetCCR(flag, 0, value==0 ?1:0, MSB(value,size), flag);
}

void fn_lsr_mem(void)
{
U16 ea;
U32 value;
int flag;

	// compute ea
	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), WORD);
	if ((ea & 0x01FC) == 0) {
		Error(ERR_INVALID_EA, 0x01FC, "lsr (mem)");
		return;
	}
	
	// make shift
	value = GetValue(ea, WORD);
	flag = value & 0x00000001;
	value >>= 1;
	value &= 0x00007FFF;	// be sure MSB is zero
	SetValue(ea, WORD, value);
	// set CCR
	CPU.SetCCR(flag, 0, value==0 ?1:0, MSB(value,WORD), flag);
}

void fn_lsl_mem(void)
{
U16 ea;
U32 value;
int flag;

	// compute ea
	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), WORD);
	if ((ea & 0x01FC) == 0) {
		Error(ERR_INVALID_EA, 0x01FC, "lsl (mem)");
		return;
	}
	
	// make shift
	value = GetValue(ea, WORD);
	flag = MSB(value, WORD);
	value <<= 1;
	SetValue(ea, WORD, value);
	// set CCR
	CPU.SetCCR(flag, 0, value==0 ?1:0, MSB(value,WORD), flag);
}

//-----------------------------------------------------------------------------------------------
// Rotation
//-----------------------------------------------------------------------------------------------

void fn_roxr(void)
{
int count, size, i, flag;
U32 value, mask, ext;

	// test if counter is register or immediate
	if (BITS(Opcode,5,1))
		count = CPU.D[BITS(Opcode,9,3)]->Value % 64;
	else {
		count = BITS(Opcode,9,3);
		if (count == 0)	count=8;
	}

	size = BITS(Opcode,6,2);
	// simulate EA for getting value
	value = GetValue(EA_Dn, size, BITS(Opcode,0,3));

	// make shifts
	if (count == 0) {
		CPU.SetCCR(0, 0, value==0 ?1:0, MSB(value,size), -1);
		return;
	} else {
		mask = 1 << (size==BYTE ? 7 : size==WORD ? 15 : 31);
		ext = srX ? -1&mask : 0;
		for (i=0; i<count; i++) {		// -1 for getting C and X before last shift
			flag = value & 0x00000001;
			if (ext) {
				value = ext | (value >> 1);
			} else {
				value = ~ext & (value >>1);
			}
			ext = flag ? -1&mask : 0;
		}
	}
	flag = ext ? 1 : 0;

	// set result
	SetValue(EA_Dn, size, value, BITS(Opcode,0,3));

	// set CCR
	CPU.SetCCR(flag, 0, value==0 ?1:0, MSB(value,size), flag);
}

void fn_roxl(void)
{
int count, size, i, flag;
U32 value, ext;

	// test if counter is register or immediate
	if (BITS(Opcode,5,1))
		count = CPU.D[BITS(Opcode,9,3)]->Value % 64;
	else {
		count = BITS(Opcode,9,3);
		if (count == 0)	count=8;
	}

	size = BITS(Opcode,6,2);
	// simulate EA for getting value
	value = GetValue(EA_Dn, size, BITS(Opcode,0,3));

	// make shifts
	if (count == 0) {
		CPU.SetCCR(0, 0, value==0 ?1:0, MSB(value,size), -1);
		return;
	} else {
		ext = srX ? 1 : 0;
		for (i=0; i<count; i++) {		// -1 for getting C and X before last shift
			flag = MSB(value, size);
			value = ext | (value << 1);
			ext = flag ? 1 : 0;
		}
	}
	flag = ext ? 1 : 0;

	// set result
	SetValue(EA_Dn, size, value, BITS(Opcode,0,3));

	// set CCR
	CPU.SetCCR(flag, 0, value==0 ?1:0, MSB(value,size), flag);
}

void fn_ror(void)
{
int count, size, i, flag=0;
U32 value, mask;

	// test if counter is register or immediate
	if (BITS(Opcode,5,1))
		count = CPU.D[BITS(Opcode,9,3)]->Value % 64;
	else {
		count = BITS(Opcode,9,3);
		if (count == 0)	count=8;
	}

	size = BITS(Opcode,6,2);
	// simulate EA for getting value
	value = GetValue(EA_Dn, size, BITS(Opcode,0,3));

	// make shifts
	if (count == 0) {
		CPU.SetCCR(0, 0, value==0 ?1:0, MSB(value,size), -1);
		return;
	} else {
		mask = 1 << (size==BYTE ? 7 : size==WORD ? 15 : 31);
		for (i=0; i<count; i++) {
			flag = (value & 0x00000001) ? -1&mask : 0;
			if (flag) {
				value = flag | (value >> 1);
			} else {
				value = ~flag & (value >> 1);
			}
		}
	}
	flag = flag ? 1 : 0;

	// set result
	SetValue(EA_Dn, size, value, BITS(Opcode,0,3));

	// set CCR
	CPU.SetCCR(flag, 0, value==0 ?1:0, MSB(value,size), -1);
}

void fn_rol(void)
{
int count, size, i, flag=0;
U32 value;

	// test if counter is register or immediate
	if (BITS(Opcode,5,1))
		count = CPU.D[BITS(Opcode,9,3)]->Value % 64;
	else {
		count = BITS(Opcode,9,3);
		if (count == 0)	count=8;
	}

	size = BITS(Opcode,6,2);
	// simulate EA for getting value
	value = GetValue(EA_Dn, size, BITS(Opcode,0,3));

	// make shifts
	if (count == 0) {
		CPU.SetCCR(0, 0, value==0 ?1:0, MSB(value,size), -1);
		return;
	} else {
		for (i=0; i<count; i++) {
			flag = MSB(value, size);
			value = flag | (value << 1);
		}
	}

	// set result
	SetValue(EA_Dn, size, value, BITS(Opcode,0,3));

	// set CCR
	CPU.SetCCR(flag, 0, value==0 ?1:0, MSB(value,size), -1);
}

void fn_roxr_mem(void)
{
U16 ea;
U32 value;
int flag;

	// compute ea
	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), WORD);
	if ((ea & 0x01FC) == 0) {
		Error(ERR_INVALID_EA, 0x01FC, "roxr (mem)");
		return;
	}
	
	// make shift
	value = GetValue(ea, WORD);
	flag = value & 0x00000001;
	value >>= 1;
	if (srX) {
		value |= 0x00008000;
	} else {
		value &= 0x00007FFF;
	}
	SetValue(ea, WORD, value);
	// set CCR
	CPU.SetCCR(flag, 0, value==0 ?1:0, MSB(value,WORD), flag);
}

void fn_roxl_mem(void)
{
U16 ea;
U32 value;
int flag;

	// compute ea
	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), WORD);
	if ((ea & 0x01FC) == 0) {
		Error(ERR_INVALID_EA, 0x01FC, "roxl (mem)");
		return;
	}
	
	// make shift
	value = GetValue(ea, WORD);
	flag = MSB(value, WORD);
	value <<= 1;
	if (srX) {
		value |= 0x00000001;
	}
	SetValue(ea, WORD, value);
	// set CCR
	CPU.SetCCR(flag, 0, value==0 ?1:0, MSB(value,WORD), flag);
}

void fn_ror_mem(void)
{
U16 ea;
U32 value;
int flag;

	// compute ea
	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), WORD);
	if ((ea & 0x01FC) == 0) {
		Error(ERR_INVALID_EA, 0x01FC, "ror (mem)");
		return;
	}
	
	// make shift
	value = GetValue(ea, WORD);
	flag = value & 0x00000001;
	if (flag) {
		value = 0x00008000 | (value >> 1);
	} else {
		value = 0x00007FFF & (value >>1);
	}
	SetValue(ea, WORD, value);
	// set CCR
	CPU.SetCCR(flag, 0, value==0 ?1:0, MSB(value,WORD), -1);
}

void fn_rol_mem(void)
{
U16 ea;
U32 value;
int flag;

	// compute ea
	ea = Compute_EA(BITS(Opcode,3,3), BITS(Opcode,0,3), WORD);
	if ((ea & 0x01FC) == 0) {
		Error(ERR_INVALID_EA, 0x01FC, "rol (mem)");
		return;
	}
	
	// make shift
	value = GetValue(ea, WORD);
	flag = MSB(value, WORD);
	value = (value << 1) | flag;
	SetValue(ea, WORD, value);
	// set CCR
	CPU.SetCCR(flag, 0, value==0 ?1:0, MSB(value,WORD), -1);
}


//***********************************************************************************************
//
// Execute : entry point, execute next instruction
//
//***********************************************************************************************

void Execute(void)
{
int i, j;


	if (CPU.PC & 0x00000001) {
		Exception(EXCEP_ADDR);	// address error if opcode not at even address
		return;
	}

	// Get the opcode and increment PC
	Opcode = Peek(CPU.PC , WORD);
	CPU.PCTmp = CPU.PC+2;

	// Search class of the opcode
	i = Opcode_Group[(Opcode & 0xF000) >> 12];
	j = Opcode_Group[ ((Opcode & 0xF000) >> 12) + 1];
	while (i<j) {
		if ((Opcode & Opcode_List[i].Mask) == Opcode_List[i].Value)
			break;
		else
			i++;
	}

	if ((i == j) || (Opcode_List[i].Model > CurrentModel))
		Exception(EXCEP_ILLEGAL);	// not an intruction : exception
	else {
		if (HookOpcode) {
			extern U32 PC,PCTmp;
			extern U16 Opcode;
			extern char Text_Opcode[];
			char reg[10];

			PC = CPU.PC;
			Opcode = Peek(PC , WORD);
			PCTmp = PC+2;
			(*Opcode_List[i].Dis)(Opcode_List[i].Name);

			// hook executed before opcode
			for (j=0; j<8; j++) {
				sprintf(reg,"d%d",j);
				sprintf(what,"%d",CPU.D[j]->Value);
				Tcl_SetVar(output, reg,what, TCL_GLOBAL_ONLY);
			}
			for (j=0; j<7; j++) {
				sprintf(reg,"a%d",j);
				sprintf(what,"%d",CPU.A[j]->Value);
				Tcl_SetVar(output, reg,what, TCL_GLOBAL_ONLY);
			}
			sprintf(what, "%d", CPU.USP->Value);
			Tcl_SetVar(output, "usp",what, TCL_GLOBAL_ONLY);
			sprintf(what, "%d", CPU.SSP->Value);
			Tcl_SetVar(output, "ssp",what, TCL_GLOBAL_ONLY);
			sprintf(what, "%d", CPU.PC);
			Tcl_SetVar(output, "pc",what, TCL_GLOBAL_ONLY);
			sprintf(what, "%d", CPU.SR->Value & 0xFFFF);
			Tcl_SetVar(output, "sr",what, TCL_GLOBAL_ONLY);

			sprintf(what, "beforeOpcode %d \"%s\"", CPU.PC, Text_Opcode);
			Tcl_Eval(output, what);

			// execute instruction
			(*Opcode_List[i].Exec)();

			// hook executed after opcode
			for (j=0; j<8; j++) {
				sprintf(reg,"d%d",j);
				sprintf(what,"%d",CPU.D[j]->Value);
				Tcl_SetVar(output, reg,what, TCL_GLOBAL_ONLY);
			}
			for (j=0; j<7; j++) {
				sprintf(reg,"a%d",j);
				sprintf(what,"%d",CPU.A[j]->Value);
				Tcl_SetVar(output, reg,what, TCL_GLOBAL_ONLY);
			}
			sprintf(what, "%d", CPU.USP->Value);
			Tcl_SetVar(output, "usp",what, TCL_GLOBAL_ONLY);
			sprintf(what, "%d", CPU.SSP->Value);
			Tcl_SetVar(output, "ssp",what, TCL_GLOBAL_ONLY);
			sprintf(what, "%d", CPU.PC);
			Tcl_SetVar(output, "pc",what, TCL_GLOBAL_ONLY);
			sprintf(what, "%d", CPU.SR->Value & 0xFFFF);
			Tcl_SetVar(output, "sr",what, TCL_GLOBAL_ONLY);

			sprintf(what, "afterOpcode %d \"%s\"", CPU.PC, Text_Opcode);
			Tcl_Eval(output, what);
		} else
			// execute instruction
			(*Opcode_List[i].Exec)();
	}
	// Update PC
	CPU.PC = CPU.PCTmp;
}
