#include "global.h"

#define RAW   		0
#define ASCII 		1
#define SWIFT 		2
#define SRECORD		3


//-----------------------------------------------------------------------------------------------
//
// NEW COMMAND : memout start end file ?format?
//
// Action : writes a memory area in a file
//
//-----------------------------------------------------------------------------------------------

int MemoutCmd(ClientData clientData, Tcl_Interp *interp, int argc, char **argv)
{
static char *usage = {"Usage : memout start end file ?format?"};
int start, end, pc, count, chksum=0;
U16 peek;
unsigned char c;
FILE *out;
int format = SRECORD;


	if (argc < 4) {
		sprintf(interp->result, usage);
		return TCL_OK;
	}

	//
	// parsing start
	//
	if (!ParseAddr(interp,argv[1], &start, usage)) {
		sprintf(interp->result, ErrorMsg);
		return TCL_OK;
	}

	//
	// parsing end
	//
	if (!ParseAddr(interp,argv[2], &end, usage)) {
		sprintf(interp->result, ErrorMsg);
		return TCL_OK;
	}

	//
	// get format if specified (default = ASCII)
	//
	if (argc == 5) {
		if (strcmp(argv[4], "raw") == 0)			format = RAW;
		else if (strcmp(argv[4], "ascii") == 0)		format = ASCII;
		else if (strcmp(argv[4], "swift") == 0)		format = SWIFT;
		else if (strcmp(argv[4], "srecord") == 0)	format = SRECORD;
		else {
			sprintf(interp->result, "%s -> unkown format", usage);			
			return TCL_OK;
		}
	}

	// try to open file
	if ((out = fopen(argv[3], "w")) == NULL) {
		sprintf(interp->result, "Error : can't open file '%s'.", argv[3]);
		return TCL_OK;
	}

	// save data
	if (format == ASCII)
		fprintf(out, "/* start=0x%08X - end=0x%08X */\n", start,end);
	
	if (format == SRECORD) {
		fprintf(out, "S016000047656E6572617465642062792053696D36386B3D\n");
		fprintf(out, "S315%08X",start);
		chksum = 0x11;
		chksum += (start & 0xFF000000)>>24;
		chksum += (start & 0x00FF0000)>>16;
		chksum += (start & 0x0000FF00)>>8;
		chksum +=  start & 0x000000FF;
	}

	for (count = 1, pc=start; pc<=end; pc++, count++) {
		peek = GetPeek((U32) pc);
		c = peek & 0x00FF;
		switch (format) {
			case RAW:	if (peek < 0x100)
							fwrite(&c, 1, 1, out); 
						break;
			case ASCII:	if (peek < 0x100)
							fprintf(out, "%02X ", c);
						else
							fprintf(out, "** ");
						if ((count % 16) == 0)
							fprintf(out, "\n");
						break;
			case SWIFT:	if (peek < 0x100)
							fprintf(out, "%08X/%02X;\n", pc,c); 
						break;
			case SRECORD:	fprintf(out, "%02X", c);
							chksum += c;
							if (((count % 16) == 0) && (pc < end)) {
								fprintf(out, "%02X\nS%d%02X%08X", 255 - (chksum & 0x00FF), 
										pc+16<end ? 3:7,
										chksum = pc+16<end ? 21:end-pc+5,
										pc+1);
								chksum += ((pc+1) & 0xFF000000)>>24;
								chksum += ((pc+1) & 0x00FF0000)>>16;
								chksum += ((pc+1) & 0x0000FF00)>>8;
								chksum +=  (pc+1) & 0x000000FF;
							}
							break; 
		}
	}
	if (format == SRECORD)
		fprintf(out, "%02X\n", 255 - (chksum & 0x00FF));

	fclose(out);

	sprintf(interp->result, "Memory saved in '%s' (%s format).",
			argv[3],format==RAW?"RAW":format==ASCII?"ASCII":format==SWIFT?"SWIFT":"S-RECORD");
	return TCL_OK;
}


//-----------------------------------------------------------------------------------------------
//
// NEW COMMAND : memin file ?start end format?
//
// Action : writes a memory area in a file
//
//-----------------------------------------------------------------------------------------------

int MeminCmd(ClientData clientData, Tcl_Interp *interp, int argc, char **argv)
{
static char *usage = {"Usage : memin file ?start end format?"};
int start, end, pc, i, count, result, type;
unsigned char c;
FILE *in;
int format = SRECORD;


	if (argc != 2 && argc != 5) {
		sprintf(interp->result, usage);
		return TCL_OK;
	}

	end = 0; // wait user args

	// try to open file
	if ((in = fopen(argv[1], "r")) == NULL) {
		sprintf(interp->result, "Error : can't open file '%s'.", argv[3]);
		return TCL_OK;
	}
	
	if (fscanf(in, "/* start=%i - end=%i */\n", &start,&end) == 2) {
		format = ASCII;
	} else if (fscanf(in,  "%x/%x;\n", &start,&c) == 2) {
		format = SWIFT;
		end = 0x7FFFFFFF;	// no end specified = go to the end of file
	} else if (fscanf(in,  "S0%02X", &start) == 1) {
		format = SRECORD;
		end = 0x7FFFFFFF;	// no end specified = go to the end of file
		fscanf(in, "%199s\n", what);
		fscanf(in, "S%1d", &type);
		fscanf(in, "%02X", &count);
		switch (type) {
			case 1:	
			case 9:	fscanf(in, "%04X", &start); break;
			
			case 2:
			case 8:	fscanf(in, "%06X", &start); break;

			case 3:
			case 7:	fscanf(in, "%08X", &start); break;
		}
		count -= 5;		// we have read the address
	} else if (argc != 5) {
		sprintf(interp->result, "%s -> format not recognized", usage);			
		return TCL_OK;
	}

	//
	// optional args
	//
	if (argc == 5) {
		// parsing start
		if (!ParseAddr(interp,argv[2], &start, usage)) {
			sprintf(interp->result, ErrorMsg);
			return TCL_OK;
		}
		// parsing end
		if (!ParseAddr(interp,argv[3], &end, usage)) {
			sprintf(interp->result, ErrorMsg);
			return TCL_OK;
		}
		// parsing format
		if (strcmp(argv[4], "raw") == 0)			format = RAW;
		else if (strcmp(argv[4], "ascii") == 0)		format = ASCII;
		else if (strcmp(argv[4], "swift") == 0)		format = SWIFT;
		else if (strcmp(argv[4], "srecord") == 0)	format = SRECORD;
		else {
			sprintf(interp->result, "%s -> unkown format", usage);			
			return TCL_OK;
		}
	}

	// load data
	for (pc=start; pc<=end && (!feof(in)); pc++,count--) {
		switch (format) {
			case RAW:	fread(&c, 1, 1, in); 
						SetPoke(pc, (U32) c);
						break;
			case ASCII:	fscanf(in, "%x", &result);
						SetPoke(pc, (U32) result);
						break;
			case SWIFT:	fscanf(in, "%x/%x;\n", &pc,&result);
						SetPoke(pc, (U32) result);
						break;
			case SRECORD:	if (count == 0) {
								if (type > 5) {
									end = pc-- -10;	// was a terminating block : force stop
								} else {
									fscanf(in, "%02X\n", &result);	// read chksum
									fscanf(in, "S%1d", &type);
									fscanf(in, "%02X", &count);
									switch (type) {
 										case 1:	
										case 9:	fscanf(in, "%04X", &pc); break;
				
										case 2:
										case 8:	fscanf(in, "%06X", &pc); break;
	
										case 3:
										case 7:	fscanf(in, "%08X", &pc); break;
									}
									pc--;	// because of pc++ in for loop
									count -= 4;	// because we read the address
								}				
							} else {
								fscanf(in, "%02X", &result);
								SetPoke(pc, (U32) result);
							}
							break;
		}
	}
	fclose(in);
	end = pc;	// set real end

	// look if poke affects memory dumping windows

	for (i=0; i<NB_DUMP; i++) {
		if (start>=MD_start[i]  &&  start<=(MD_start[i]+MD_len[i])) {
			MD_where[i] = start;
			if (end<=(MD_start[i]+MD_len[i]))
				MD_size[i] = start-end;	// negative = block
			else
				MD_size[i] = start-(MD_start[i]+MD_len[i]);	// negative = block
		} else if (end>=MD_start[i]  &&  end<=(MD_start[i]+MD_len[i])) {
			MD_where[i] = MD_start[i];
			MD_size[i] = MD_start[i]-end;	// negative = block
		}
	}

	// update dumping windows + register dumping
	DumpAn(CPU.A[0]->Value, 0, interp);
	DumpAn(CPU.A[1]->Value, 1, interp);
	DumpAn(CPU.A[2]->Value, 2, interp);
	DumpAn(CPU.A[3]->Value, 3, interp);
	DumpAn(CPU.A[4]->Value, 4, interp);
	DumpAn(CPU.A[5]->Value, 5, interp);
	DumpAn(CPU.A[6]->Value, 6, interp);
	DumpAn(CPU.USP->Value,  7, interp);
	DumpAn(CPU.SSP->Value,  8, interp);
	UpdateWindowDump(interp);


	sprintf(interp->result, "Memory load from '%s' (%s format) at 0x%08X.",
			argv[1],format==RAW?"RAW":format==ASCII?"ASCII":format==SWIFT?"SWIFT":"S-RECORD",
			start);
	return TCL_OK;
}
