/* 
 * tkMain.c --
 *
 *	This file contains a generic main program for Tk-based applications.
 *	It can be used as-is for many applications, just by supplying a
 *	different appInitProc procedure for each specific application.
 *	Or, it can be used as a template for creating new main programs
 *	for Tk applications.
 *
 * Copyright (c) 1990-1994 The Regents of the University of California.
 * Copyright (c) 1994-1996 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * SCCS: @(#) tkMain.c 1.150 96/09/05 18:42:00
 */

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <tcl.h>
#include <tk.h>
#ifdef NO_STDLIB_H
#   include "../compat/stdlib.h"
#else
#   include <stdlib.h>
#endif

#if defined(_SYSTYPE_SVR4)  /* IRIX 5.3 */
#define _SVR4_SOURCE    /* for fileno() and SIGCLD */
#define sigfntype     SIG_PF
#else                 /* IRIX 4 */
#ifdef __cplusplus
#ifdef SUNOS41
typedef void (*sigfntype)(...);
#else /* !SUNOS41 */
typedef void (*sigfntype)(int);
#endif /* SUNOS41 */
#else /* !__cplusplus */
typedef void (*sigfntype)();
#endif /* __cplusplus */
#endif /* _SYSTYPE_SVR4 */

#ifndef USESTDIN
#include <readline/readline.h>
#endif /* USESTDIN */

#ifdef SUNOS41
EXTERN int on_exit _ANSI_ARGS_((void (*procp)(void), caddr_t arg));
#define atexit(a)	on_exit(a,0);
#endif /* SUNOS41 */

/*
 * Declarations for various library procedures and variables (don't want
 * to include tkInt.h or tkPort.h here, because people might copy this
 * file out of the Tk source directory to make their own modified versions).
 * Note: don't declare "exit" here even though a declaration is really
 * needed, because it will conflict with a declaration elsewhere on
 * some systems.
 */

EXTERN int		isatty _ANSI_ARGS_((int fd));
EXTERN int		read _ANSI_ARGS_((int fd, char *buf, size_t size));
EXTERN int	        write _ANSI_ARGS_((int fd, char *buf, size_t size));
EXTERN char *		strrchr _ANSI_ARGS_((CONST char *string, int c));

/*
 * Global variables used by the main program:
 */

static Tcl_Interp *interp;	/* Interpreter for this application. */
static Tcl_DString command;	/* Used to assemble lines of terminal input
				 * into Tcl commands. */
static Tcl_DString line;	/* Used to read the next line from the
                                 * terminal input. */
static int tty;			/* Non-zero means standard input is a
				 * terminal-like device.  Zero means it's
				 * a file. */

/*
 * Forward declarations for procedures defined later in this file.
 */

static void		Prompt _ANSI_ARGS_((Tcl_Interp *interp, int partial));
static void		StdinProc _ANSI_ARGS_((ClientData clientData,
			    int mask));

#ifdef DEBUG 
#define DBG(a)        a
#else /* !DEBUG */
#define DBG(a)        /* nothing */
#endif /* DEBUG */

#ifndef USESTDIN
#include <readline/readline.h>
#endif /* USESTDIN */


#ifndef USESTDIN
    /* in forker.c */
EXTERN int forker _ANSI_ARGS_((int *pprintwfd, int *pchildpid));

int readlinefd = -1;
int promptfd   = -1;
int rlchildpid = 0;

static void   RLPrompt _ANSI_ARGS_((Tcl_Interp *interp, int partial));
static void   ReadlineProc _ANSI_ARGS_((ClientData clientData,
                          int mask));
void tclexit(Tcl_Interp *interp)
{
    /*
     * Don't exit directly, but rather invoke the Tcl "exit" command.
     * This gives the application the opportunity to redefine "exit"
     * to do additional cleanup.
     */
    DBG((fprintf(stderr, "tkMain.c:tclexit()\n")));
    Tcl_Eval(interp, "exit");
    exit(1);
}

/* cast for function pointer to pass to signal(2) */
#include <sys/types.h>
#include <signal.h>   /* only if !USESTDIN, for SIGCLD */
#include <sys/wait.h>   /* for WNOHANG, for waitpid() */

void killchild(void)
{     /* function put on atexit queue to kill child (if no-one else has)
         at exit */
    DBG((fprintf(stderr, "tkMain.c:killchild() (atexit)\n")));
    /* If we've got a child reading stdin, make sure to kill it too */    
    if(rlchildpid)  {
      kill(rlchildpid, SIGKILL);
    }
}    

static Tcl_Interp *exitsiginterp = NULL;  /* use this interpreter for signal */

void tclexitsig(int sig)
{   /* handler passed to SIGNAL */
    int status;
    int pid;
    
    DBG((fprintf(stderr, "tkMain.c:tclexitsig(sig=%d)\n",sig)));
    if(rlchildpid)  {
      /* use waitpid to see if this SIGCHLD event is connected 
         with our readline proc.  If not (i.e. if it is a different child, 
         perhaps spawned by a different tcl function), we don't want to 
         deal with it here, as it's irrelevent, and there will be 
         a different thread waiting for it somewhere */
      pid = waitpid(rlchildpid, &status, WNOHANG);
      /* pid is returned as zero if the specified process has not 
         changed i.e. was not the cause of the SIGCHLD */
      DBG((fprintf(stderr, "tkMain.c:tclexitsig(pid=%d)\n",pid)));
      if(pid==rlchildpid)  {  /* was the readline child dying */
          /* child has already died so no need to have exit kill it */
          rlchildpid = 0;
          /* since the readline child has died, exit the interpreter */
          tclexit(exitsiginterp);
      }
    }
}

static int
ConsoleBlockMode(
    ClientData instanceData,		/* File state. */
    int mode)				/* The mode to set. Can be one of
                                         * TCL_MODE_BLOCKING or
                                         * TCL_MODE_NONBLOCKING. */
{
    DBG((fprintf(stderr, "ConsoleBlockMode(%s)\n", (mode==TCL_MODE_BLOCKING)?"blocking":"nonblocking")));
    return 0;
}

static int
ConsoleClose(
    ClientData instanceData,	/* Unused. */
    Tcl_Interp *interp)		/* Unused. */
{
    DBG((fprintf(stderr, "ConsoleClose\n")));
    return 0;
}

static int
ConsoleInput(
    ClientData instanceData,		/* Unused. */
    char *buf,				/* Where to store data read. */
    int bufSize,			/* How much space is available
                                         * in the buffer? */
    int *errorCode)			/* Where to store error code. */
{
    DBG((fprintf(stderr, "ConsoleInput\n")));
    return 0;			/* Always return EOF. */
}

static int
ConsoleOutput(
    ClientData instanceData,		/* Unused. */
    char *buf,				/* The data buffer. */
    int toWrite,			/* How many bytes to write? */
    int *errorCode)			/* Where to store error code. */
{
    *errorCode = 0;
    Tcl_SetErrno(0);

    DBG((fprintf(stderr, "ConsoleOutput: '%s'\n", buf)));
    return toWrite;
}

static void
ConsoleWatch(
    ClientData instanceData,		/* Device ID for the channel. */
    int mask)				/* flags for events of interest */
{
    Tcl_File *inoutpair = (Tcl_File *)instanceData;

    DBG((fprintf(stderr, "ConsoleWatch: %s %s %s\n", (mask&TCL_READABLE)?"read":"", (mask&TCL_WRITABLE)?"write":"", (mask&TCL_EXCEPTION)?"except":"")));
    /* this appears to be what Tcl7.5 does (in tclIO.c:ChannelHandlerSetupProc)
       that Tcl7.6 passes off to this routine instead */
    if(mask & TCL_READABLE) {
	Tcl_WatchFile(inoutpair[0], TCL_READABLE);
    }
    if(mask & TCL_WRITABLE) {
	Tcl_WatchFile(inoutpair[1], TCL_WRITABLE);
    }
}

#include <assert.h>
#include <fcntl.h>
#include <stropts.h>
#include <poll.h>

static int
ConsoleReady(
    ClientData instanceData,		/* Device ID for the channel. */
    int mask)				/* flags for events of interest */
{
    Tcl_File *inoutpair = (Tcl_File *)instanceData;
    struct pollfd pfd;
    int rc = 0;
    int type = TCL_UNIX_FD;

    if(mask&TCL_READABLE) {
	pfd.fd = (int)Tcl_GetFileInfo(inoutpair[0], &type);
	assert(type==TCL_UNIX_FD);
	pfd.events = POLLIN;
	poll(&pfd, 1, 0);
	if(pfd.revents & POLLIN) {
	    rc |= TCL_READABLE;
	}
    }
    if(mask&TCL_WRITABLE) {
	pfd.fd = (int)Tcl_GetFileInfo(inoutpair[1], &type);
	assert(type==TCL_UNIX_FD);
	pfd.events = POLLOUT;
	poll(&pfd, 1, 0);
	if(pfd.revents & POLLOUT) {
	    rc |= TCL_WRITABLE;
	}
    }
    DBG((fprintf(stderr, "ConsoleReady: %s/%s/%s = %d\n", (mask&TCL_READABLE)?"read":"", (mask&TCL_WRITABLE)?"write":"", (mask&TCL_EXCEPTION)?"except":"", rc)));
    return rc;
}

static Tcl_File
ConsoleFile(
    ClientData instanceData,		/* Device ID for the channel. */
    int direction)			/* TCL_READABLE or _WRITABLE */
{
    Tcl_File *inoutpair = (Tcl_File *)instanceData;
    DBG((fprintf(stderr, "ConsoleFile: %s\n", (direction==TCL_READABLE)?"read":"write")));
    if(direction == TCL_READABLE) {
	return inoutpair[0];
    } else {
	return inoutpair[1];
    }
}

static Tcl_ChannelType forkerChan = {
    "forker",			/* Type name. */
    ConsoleBlockMode,		/* set blocking mode */
    ConsoleClose,		/* Close proc. */
    ConsoleInput,		/* Input proc. */
    ConsoleOutput,		/* Output proc. */
    NULL,			/* Seek proc. */
    NULL,			/* Set option proc. */
    NULL,			/* Get option proc. */
    ConsoleWatch,		/* Watch for events on console. */
    ConsoleReady,		/* Are events present? */
    ConsoleFile,		/* Get a Tcl_File from the device. */
};

#endif /* USESTDIN */


/*
 *----------------------------------------------------------------------
 *
 * Tk_Main --
 *
 *	Main program for Wish and most other Tk-based applications.
 *
 * Results:
 *	None. This procedure never returns (it exits the process when
 *	it's done.
 *
 * Side effects:
 *	This procedure initializes the Tk world and then starts
 *	interpreting commands;  almost anything could happen, depending
 *	on the script being interpreted.
 *
 *----------------------------------------------------------------------
 */

void
Tk_Main(
    int argc,				/* Number of arguments. */
    char **argv,			/* Array of argument strings. */
    Tcl_AppInitProc *appInitProc)	/* Application-specific initialization
					 * procedure to call after most
					 * initialization but before starting
					 * to execute commands. */
{
    char *args, *fileName;
    char buf[20];
    int code;
    size_t length;
    Tcl_Channel inChannel, outChannel, errChannel;
#ifndef USESTDIN
    Tcl_File inoutpair[2];
#endif /* USESTDIN */

    Tcl_FindExecutable(argv[0]);
    interp = Tcl_CreateInterp();
#ifdef TCL_MEM_DEBUG
    Tcl_InitMemory(interp);
#endif

    /*
     * Parse command-line arguments.  A leading "-file" argument is
     * ignored (a historical relic from the distant past).  If the
     * next argument doesn't start with a "-" then strip it off and
     * use it as the name of a script file to process.
     */

    fileName = NULL;
    if (argc > 1) {
	length = strlen(argv[1]);
	if ((length >= 2) && (strncmp(argv[1], "-file", length) == 0)) {
	    argc--;
	    argv++;
	}
    }
    if ((argc > 1) && (argv[1][0] != '-')) {
	fileName = argv[1];
	argc--;
	argv++;
    }

    /*
     * Make command-line arguments available in the Tcl variables "argc"
     * and "argv".
     */

    args = Tcl_Merge(argc-1, argv+1);
    Tcl_SetVar(interp, "argv", args, TCL_GLOBAL_ONLY);
    ckfree(args);
    sprintf(buf, "%d", argc-1);
    Tcl_SetVar(interp, "argc", buf, TCL_GLOBAL_ONLY);
    Tcl_SetVar(interp, "argv0", (fileName != NULL) ? fileName : argv[0],
	    TCL_GLOBAL_ONLY);

    /*
     * Set the "tcl_interactive" variable.
     */

    /*
     * For now, under Windows, we assume we are not running as a console mode
     * app, so we need to use the GUI console.  In order to enable this, we
     * always claim to be running on a tty.  This probably isn't the right
     * way to do it.
     */

#ifdef __WIN32__
    tty = 1;
#else
    tty = isatty(0);
#endif
    Tcl_SetVar(interp, "tcl_interactive",
	    ((fileName == NULL) && tty) ? "1" : "0", TCL_GLOBAL_ONLY);

    /*
     * Invoke application-specific initialization.
     */

    if ((*appInitProc)(interp) != TCL_OK) {
	errChannel = Tcl_GetStdChannel(TCL_STDERR);
	if (errChannel) {
            Tcl_Write(errChannel,
		    "application-specific initialization failed: ", -1);
            Tcl_Write(errChannel, interp->result, -1);
            Tcl_Write(errChannel, "\n", 1);
        }
    }

    /*
     * Invoke the script specified on the command line, if any.
     */

    if (fileName != NULL) {
	code = Tcl_EvalFile(interp, fileName);
	if (code != TCL_OK) {
	    goto error;
	}
	tty = 0;
    } else {

	/*
	 * Evaluate the .rc file, if one has been specified.
	 */

	Tcl_SourceRCFile(interp);

	/*
	 * Establish a channel handler for stdin.
	 */

#ifdef USESTDIN
	inChannel = Tcl_GetStdChannel(TCL_STDIN);
	if (inChannel) {
	    Tcl_CreateChannelHandler(inChannel, TCL_READABLE, StdinProc,
		    (ClientData) inChannel);
	}
	if (tty) {
	    Prompt(interp, 0);
	}
#else /* !USESTDIN */
       readlinefd = forker(&promptfd, &rlchildpid);
	inoutpair[0] = Tcl_GetFile((ClientData)readlinefd, TCL_UNIX_FD);
	inoutpair[1] = Tcl_GetFile((ClientData)promptfd, TCL_UNIX_FD);
	DBG((fprintf(stderr, "fds: in=%d, out=%d\n", readlinefd, promptfd)));
       inChannel  = Tcl_CreateChannel(&forkerChan, "forker_channel", 
			  (ClientData)inoutpair, TCL_READABLE | TCL_WRITABLE);
	if (inChannel) {
	    Tcl_CreateChannelHandler(inChannel, TCL_READABLE, ReadlineProc,
		    (ClientData) inChannel);
	}
	if (tty) {
	    RLPrompt(interp, 0);
	}
       /* also, be sure to exit if the child does */
       {
           /* sigset(SIGCLD, (sigfntype)tclexitsig); */
           /* Use POSIX signal handling fns, need waitpid to distinguish 
              death of our readline handler from any other assorted 
              processes that tcl might fork off */
           struct sigaction sa;
           sigemptyset(&sa.sa_mask);
           sa.sa_handler = (sigfntype)tclexitsig;
           sa.sa_flags = 0;
           sigaction(SIGCHLD, &sa, NULL);
       }
	exitsiginterp = interp;
	/* also, add a command to make sure the child is killed 
	   if we exit on our own */
	atexit(killchild);
#endif /* USESTDIN */
    }

    outChannel = Tcl_GetStdChannel(TCL_STDOUT);
    if (outChannel) {
	Tcl_Flush(outChannel);
    }
    Tcl_DStringInit(&command);
    Tcl_DStringInit(&line);
    Tcl_ResetResult(interp);

    /*
     * Loop infinitely, waiting for commands to execute.  When there
     * are no windows left, Tk_MainLoop returns and we exit.
     */

    Tk_MainLoop();
    Tcl_DeleteInterp(interp);
    Tcl_Exit(0);

error:
    /*
     * The following statement guarantees that the errorInfo
     * variable is set properly.
     */

    Tcl_AddErrorInfo(interp, "");
    errChannel = Tcl_GetStdChannel(TCL_STDERR);
    if (errChannel) {
        Tcl_Write(errChannel, Tcl_GetVar(interp, "errorInfo", TCL_GLOBAL_ONLY),
		-1);
        Tcl_Write(errChannel, "\n", 1);
    }
    Tcl_DeleteInterp(interp);
    Tcl_Exit(1);
}

/*
 *----------------------------------------------------------------------
 *
 * StdinProc --
 *
 *	This procedure is invoked by the event dispatcher whenever
 *	standard input becomes readable.  It grabs the next line of
 *	input characters, adds them to a command being assembled, and
 *	executes the command if it's complete.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Could be almost arbitrary, depending on the command that's
 *	typed.
 *
 *----------------------------------------------------------------------
 */

    /* ARGSUSED */
static void
StdinProc(
    ClientData clientData,		/* The channel we are listening to. */
    int mask)				/* Not used. */
{
    static int gotPartial = 0;
    char *cmd;
    int code, count;
    Tcl_Channel chan = (Tcl_Channel) clientData;

    count = Tcl_Gets(chan, &line);

    if (count < 0) {
	if (!gotPartial) {
	    if (tty) {
		Tcl_Exit(0);
	    } else {
		Tcl_DeleteChannelHandler(chan, StdinProc, (ClientData) chan);
	    }
	    return;
	} else {
	    count = 0;
	}
    }

    (void) Tcl_DStringAppend(&command, Tcl_DStringValue(&line), -1);
    cmd = Tcl_DStringAppend(&command, "\n", -1);
    Tcl_DStringFree(&line);
    
    if (!Tcl_CommandComplete(cmd)) {
        gotPartial = 1;
        goto prompt;
    }
    gotPartial = 0;

    /*
     * Disable the stdin channel handler while evaluating the command;
     * otherwise if the command re-enters the event loop we might
     * process commands from stdin before the current command is
     * finished.  Among other things, this will trash the text of the
     * command being evaluated.
     */

    Tcl_CreateChannelHandler(chan, 0, StdinProc, (ClientData) chan);
    code = Tcl_RecordAndEval(interp, cmd, TCL_EVAL_GLOBAL);
    Tcl_CreateChannelHandler(chan, TCL_READABLE, StdinProc,
	    (ClientData) chan);
    Tcl_DStringFree(&command);
    if (*interp->result != 0) {
	if ((code != TCL_OK) || (tty)) {
	    /*
	     * The statement below used to call "printf", but that resulted
	     * in core dumps under Solaris 2.3 if the result was very long.
             *
             * NOTE: This probably will not work under Windows either.
	     */

	    puts(interp->result);
	}
    }

    /*
     * Output a prompt.
     */

    prompt:
    if (tty) {
	Prompt(interp, gotPartial);
    }
    Tcl_ResetResult(interp);
}

/*
 *----------------------------------------------------------------------
 *
 * Prompt --
 *
 *	Issue a prompt on standard output, or invoke a script
 *	to issue the prompt.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	A prompt gets output, and a Tcl script may be evaluated
 *	in interp.
 *
 *----------------------------------------------------------------------
 */

static void
Prompt(
    Tcl_Interp *interp,			/* Interpreter to use for prompting. */
    int partial)			/* Non-zero means there already
					 * exists a partial command, so use
					 * the secondary prompt. */
{
    char *promptCmd;
    int code;
    Tcl_Channel outChannel, errChannel;

    DBG((fprintf(stderr, "Prompt\n"))); DBG((fflush(stderr)));

    errChannel = Tcl_GetChannel(interp, "stderr", NULL);

    promptCmd = Tcl_GetVar(interp,
	partial ? "tcl_prompt2" : "tcl_prompt1", TCL_GLOBAL_ONLY);
    if (promptCmd == NULL) {
defaultPrompt:
	if (!partial) {

            /*
             * We must check that outChannel is a real channel - it
             * is possible that someone has transferred stdout out of
             * this interpreter with "interp transfer".
             */

	    outChannel = Tcl_GetChannel(interp, "stdout", NULL);
            if (outChannel != (Tcl_Channel) NULL) {
                Tcl_Write(outChannel, "% ", 2);
            }
	}
    } else {
	code = Tcl_Eval(interp, promptCmd);
	if (code != TCL_OK) {
	    Tcl_AddErrorInfo(interp,
		    "\n    (script that generates prompt)");
            /*
             * We must check that errChannel is a real channel - it
             * is possible that someone has transferred stderr out of
             * this interpreter with "interp transfer".
             */
            
	    errChannel = Tcl_GetChannel(interp, "stderr", NULL);
            if (errChannel != (Tcl_Channel) NULL) {
                Tcl_Write(errChannel, interp->result, -1);
                Tcl_Write(errChannel, "\n", 1);
            }
	    goto defaultPrompt;
	}
    }
    outChannel = Tcl_GetChannel(interp, "stdout", NULL);
    if (outChannel != (Tcl_Channel) NULL) {
        Tcl_Flush(outChannel);
    }
}

#ifndef USESTDIN

static void
ReadlineProc(ClientData  clientData, int mask)
{
#define BUFFER_SIZE 4000
    char input[BUFFER_SIZE+1];
    static int gotPartial = 0;
    char *cmd;
    int code, count;
    Tcl_Channel chan = (Tcl_Channel) clientData;

    DBG((fprintf(stderr, "ReadlineProc\n"))); DBG((fflush(stderr)));

    count = read(readlinefd, input, BUFFER_SIZE-1);
    DBG((fprintf(stderr, "RLproc: got %d '%s'\n", count, input)));
    if (count <= 0) {
	if (!gotPartial) {
	    if (tty) {
		Tcl_Eval(interp, "exit");
		DBG((fprintf(stderr, "tkMain.c:ReadlineProc()\n")));
		exit(1);
	    } else {
		Tcl_DeleteChannelHandler(chan, ReadlineProc, (ClientData)chan);
	    }
	    return;
	} else {
	    count = 0;
	}
    }
    (void) Tcl_DStringAppend(&command, input, count);
    cmd = Tcl_DStringAppend(&command, "\n", -1);
    DBG((fprintf(stderr, "cmd: %s", Tcl_DStringValue(&command))));
    if (!Tcl_CommandComplete(cmd)) {
	gotPartial = 1;
	goto prompt;
    }
    gotPartial = 0;

    /*
     * Disable the file handler while evaluating the command;
     * otherwise if the command re-enters the event loop we might
     * process commands from stdin before the current command is
     * finished.  Among other things, this will trash the text of the
     * command being evaluated.
     */

    Tcl_CreateChannelHandler(chan, 0, ReadlineProc,
			     (ClientData) chan);
    code = Tcl_RecordAndEval(interp, cmd, 0);
    Tcl_CreateChannelHandler(chan, TCL_READABLE, ReadlineProc,
			     (ClientData) chan);

    Tcl_DStringFree(&command);
    if (*interp->result != 0) {
	if ((code != TCL_OK) || (tty)) {
	    puts(interp->result);
	}
    }

    /*
     * Output a prompt.
     */

    prompt:
    if (tty) {
	RLPrompt(interp, gotPartial);
    }
    Tcl_ResetResult(interp);
}

static void
RLPrompt(Tcl_Interp *interp, int partial) /* Non-zero means there already
					 * exists a partial command, so use
					 * the secondary prompt. */
{
    char *promptCmd;
    int code;
    char *prompt = "% ";
    char *partprompt = "+ ";

    DBG((fprintf(stderr, "RLPrompt\n"))); DBG((fflush(stderr)));

    /* The readline child process *waits* for a prompt to appear on 
       the prompt pipe *before* starting to listen for the next 
       input.  Therefore, it is critical that some non-zero-length 
       prompt be written to the pipe, else the child will block.  
       This routine ensures this is so. */

    /* The tcl variables tcl_prompt1 and tcl_prompt2 used to be 
       required to 'puts' the prompts to which they correspond. 
       But since we need the prompt to be written to our 
       special pipe, they are redefined to *return* the desired 
       prompt string, which is then put.  Sorry about this! */

    promptCmd = Tcl_GetVar(interp,
	partial ? "tcl_prompt2" : "tcl_prompt1", TCL_GLOBAL_ONLY);
    if (promptCmd == NULL) {
	defaultPrompt:
	if (!partial) {
	    write(promptfd, prompt, strlen(prompt));
	} else {
	    write(promptfd, partprompt, strlen(partprompt));
	}
    } else {
	code = Tcl_Eval(interp, promptCmd);
	if (code != TCL_OK) {
	    Tcl_AddErrorInfo(interp,
		    "\n    (script that generates prompt)");
	    fprintf(stderr, "%s\n", interp->result);
	    goto defaultPrompt;
	}
	/* this is not the standard, but for now, prompt functions 
	   *return* the string to use as a prompt, but don't 
	   print it themselves */
	if(strlen(interp->result)>0)  {
	    write(promptfd, interp->result, strlen(interp->result));
	} else {
	    write(promptfd, " ", 1);
	}
    }
}

#endif /* !USESTDIN */
