/* 
 * tclSSL.c
 *
 * Unix sockets that support SSL.  Much of this file is taken from 
 *      tclUnixChan.c.  My intent was to use as much of the core tcp
 *      code as posible.  The since many of the functions defined in
 *      tclUnixChan.c are static and I want to create a loadable
 *      module, I need to include them here but they
 *      are unchanged.
 *
 * Copyright (c) 1995-1997  Tony Bringardner
 *
 */
#define DBUG /* fprintf(stderr,"%s (%d)\n",__FILE__,__LINE__);fflush(stderr);  */

/*  tclUnixChan.c start  */
#include	"tclInt.h"	/* Internal definitions for Tcl. */
#include	"tclPort.h"	/* Portability features for Tcl. */
#include	"ssl.h"		/* SSL include  */
#define TCLSSLCERT "tclSSL.pem"

/*
 * sys/ioctl.h has already been included by tclPort.h.  Including termios.h
 * or termio.h causes a bunch of warning messages because some duplicate
 * (but not contradictory) #defines exist in termios.h and/or termio.h
 */
#undef NL0
#undef NL1
#undef CR0
#undef CR1
#undef CR2
#undef CR3
#undef TAB0
#undef TAB1
#undef TAB2
#undef XTABS
#undef BS0
#undef BS1
#undef FF0
#undef FF1
#undef ECHO
#undef NOFLSH
#undef TOSTOP
#undef FLUSHO
#undef PENDIN

#ifdef USE_TERMIOS
#   include <termios.h>
#else	/* !USE_TERMIOS */
#ifdef USE_TERMIO
#   include <termio.h>
#else	/* !USE_TERMIO */
#ifdef USE_SGTTY
#   include <sgtty.h>
#endif	/* USE_SGTTY */
#endif	/* !USE_TERMIO */
#endif	/* !USE_TERMIOS */

/*  The following are static functions in generic/tclIOCmd.c */

/*
 * Return at most this number of bytes in one call to Tcl_Read:
 */

#define	TCL_READ_CHUNK_SIZE	4096

/*
 * Callback structure for accept callback in a TCP server.
 */

typedef struct AcceptCallback {
    char *script;			/* Script to invoke. */
    Tcl_Interp *interp;			/* Interpreter in which to run it. */
} AcceptCallback;

/*
 * Static functions for this file:
 */

static void	AcceptCallbackProc _ANSI_ARGS_((ClientData callbackData,
	            Tcl_Channel chan, char *address, int port));
static void	RegisterTcpServerInterpCleanup _ANSI_ARGS_((Tcl_Interp *interp,
	            AcceptCallback *acceptCallbackPtr));
static void	TcpAcceptCallbacksDeleteProc _ANSI_ARGS_((
		    ClientData clientData, Tcl_Interp *interp));
static void	TcpServerCloseProc _ANSI_ARGS_((ClientData callbackData));
static void	UnregisterTcpServerInterpCleanupProc _ANSI_ARGS_((
		    Tcl_Interp *interp, AcceptCallback *acceptCallbackPtr));



/*
 *----------------------------------------------------------------------
 *
 * TcpAcceptCallbacksDeleteProc --
 *
 *	Assocdata cleanup routine called when an interpreter is being
 *	deleted to set the interp field of all the accept callback records
 *	registered with	the interpreter to NULL. This will prevent the
 *	interpreter from being used in the future to eval accept scripts.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Deallocates memory and sets the interp field of all the accept
 *	callback records to NULL to prevent this interpreter from being
 *	used subsequently to eval accept scripts.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
static void
TcpAcceptCallbacksDeleteProc(clientData, interp)
    ClientData clientData;	/* Data which was passed when the assocdata
                                 * was registered. */
    Tcl_Interp *interp;		/* Interpreter being deleted - not used. */
{
    Tcl_HashTable *hTblPtr;
    Tcl_HashEntry *hPtr;
    Tcl_HashSearch hSearch;
    AcceptCallback *acceptCallbackPtr;

DBUG
    hTblPtr = (Tcl_HashTable *) clientData;
DBUG
    for (hPtr = Tcl_FirstHashEntry(hTblPtr, &hSearch);
             hPtr != (Tcl_HashEntry *) NULL;
             hPtr = Tcl_NextHashEntry(&hSearch)) {
DBUG
        acceptCallbackPtr = (AcceptCallback *) Tcl_GetHashValue(hPtr);
DBUG
        acceptCallbackPtr->interp = (Tcl_Interp *) NULL;
DBUG
    }
DBUG
    Tcl_DeleteHashTable(hTblPtr);
DBUG
    ckfree((char *) hTblPtr);
DBUG
}

/*
 *----------------------------------------------------------------------
 *
 * RegisterTcpServerInterpCleanup --
 *
 *	Registers an accept callback record to have its interp
 *	field set to NULL when the interpreter is deleted.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	When, in the future, the interpreter is deleted, the interp
 *	field of the accept callback data structure will be set to
 *	NULL. This will prevent attempts to eval the accept script
 *	in a deleted interpreter.
 *
 *----------------------------------------------------------------------
 */

static void
RegisterTcpServerInterpCleanup(interp, acceptCallbackPtr)
    Tcl_Interp *interp;		/* Interpreter for which we want to be
                                 * informed of deletion. */
    AcceptCallback *acceptCallbackPtr;
    				/* The accept callback record whose
                                 * interp field we want set to NULL when
                                 * the interpreter is deleted. */
{
    Tcl_HashTable *hTblPtr;	/* Hash table for accept callback
                                 * records to smash when the interpreter
                                 * will be deleted. */
    Tcl_HashEntry *hPtr;	/* Entry for this record. */
    int new;			/* Is the entry new? */

DBUG
    hTblPtr = (Tcl_HashTable *) Tcl_GetAssocData(interp,
            "tclTCPAcceptCallbacks",
            NULL);
DBUG
    if (hTblPtr == (Tcl_HashTable *) NULL) {
DBUG
        hTblPtr = (Tcl_HashTable *) ckalloc((unsigned) sizeof(Tcl_HashTable));
        Tcl_InitHashTable(hTblPtr, TCL_ONE_WORD_KEYS);
        (void) Tcl_SetAssocData(interp, "tclTCPAcceptCallbacks",
                TcpAcceptCallbacksDeleteProc, (ClientData) hTblPtr);
DBUG
    }
DBUG
    hPtr = Tcl_CreateHashEntry(hTblPtr, (char *) acceptCallbackPtr, &new);
DBUG
    if (!new) {
        panic("RegisterTcpServerCleanup: damaged accept record table");
    }
DBUG
    Tcl_SetHashValue(hPtr, (ClientData) acceptCallbackPtr);
DBUG
}

/*
 *----------------------------------------------------------------------
 *
 * UnregisterTcpServerInterpCleanupProc --
 *
 *	Unregister a previously registered accept callback record. The
 *	interp field of this record will no longer be set to NULL in
 *	the future when the interpreter is deleted.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Prevents the interp field of the accept callback record from
 *	being set to NULL in the future when the interpreter is deleted.
 *
 *----------------------------------------------------------------------
 */

static void
UnregisterTcpServerInterpCleanupProc(interp, acceptCallbackPtr)
    Tcl_Interp *interp;		/* Interpreter in which the accept callback
                                 * record was registered. */
    AcceptCallback *acceptCallbackPtr;
    				/* The record for which to delete the
                                 * registration. */
{
    Tcl_HashTable *hTblPtr;
    Tcl_HashEntry *hPtr;

DBUG
    hTblPtr = (Tcl_HashTable *) Tcl_GetAssocData(interp,
            "tclTCPAcceptCallbacks", NULL);
DBUG
    if (hTblPtr == (Tcl_HashTable *) NULL) {
        return;
    }
DBUG
    hPtr = Tcl_FindHashEntry(hTblPtr, (char *) acceptCallbackPtr);
DBUG
    if (hPtr == (Tcl_HashEntry *) NULL) {
        return;
    }
DBUG
    Tcl_DeleteHashEntry(hPtr);
DBUG
}

/*
 *----------------------------------------------------------------------
 *
 * AcceptCallbackProc --
 *
 *	This callback is invoked by the TCP channel driver when it
 *	accepts a new connection from a client on a server socket.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Whatever the script does.
 *
 *----------------------------------------------------------------------
 */

static void
AcceptCallbackProc(callbackData, chan, address, port)
    ClientData callbackData;		/* The data stored when the callback
                                         * was created in the call to
                                         * Tcl_OpenSSLServer. */
    Tcl_Channel chan;			/* Channel for the newly accepted
                                         * connection. */
    char *address;			/* Address of client that was
                                         * accepted. */
    int port;				/* Port of client that was accepted. */
{
    AcceptCallback *acceptCallbackPtr;
    Tcl_Interp *interp;
    char *script;
    char portBuf[10];
    int result;

    acceptCallbackPtr = (AcceptCallback *) callbackData;

    /*
     * Check if the callback is still valid; the interpreter may have gone
     * away, this is signalled by setting the interp field of the callback
     * data to NULL.
     */
DBUG
    
    if (acceptCallbackPtr->interp != (Tcl_Interp *) NULL) {
DBUG

        script = acceptCallbackPtr->script;
DBUG
        interp = acceptCallbackPtr->interp;
DBUG
        
        Tcl_Preserve((ClientData) script);
DBUG
        Tcl_Preserve((ClientData) interp);
DBUG

	TclFormatInt(portBuf, port);
DBUG
        Tcl_RegisterChannel(interp, chan);
DBUG
        result = Tcl_VarEval(interp, script, " ", Tcl_GetChannelName(chan),
                " ", address, " ", portBuf, (char *) NULL);
DBUG
        if (result != TCL_OK) {
DBUG
            Tcl_BackgroundError(interp);
DBUG
	    Tcl_UnregisterChannel(interp, chan);
DBUG
        }
DBUG
        Tcl_Release((ClientData) interp);
DBUG
        Tcl_Release((ClientData) script);
DBUG
    } else {

DBUG
        /*
         * The interpreter has been deleted, so there is no useful
         * way to utilize the client socket - just close it.
         */

        Tcl_Close((Tcl_Interp *) NULL, chan);
DBUG
    }
DBUG
}

/*
 *----------------------------------------------------------------------
 *
 * TcpServerCloseProc --
 *
 *	This callback is called when the TCP server channel for which it
 *	was registered is being closed. It informs the interpreter in
 *	which the accept script is evaluated (if that interpreter still
 *	exists) that this channel no longer needs to be informed if the
 *	interpreter is deleted.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	In the future, if the interpreter is deleted this channel will
 *	no longer be informed.
 *
 *----------------------------------------------------------------------
 */

static void
TcpServerCloseProc(callbackData)
    ClientData callbackData;	/* The data passed in the call to
                                 * Tcl_CreateCloseHandler. */
{
    AcceptCallback *acceptCallbackPtr;
    				/* The actual data. */

DBUG
    acceptCallbackPtr = (AcceptCallback *) callbackData;
DBUG
    if (acceptCallbackPtr->interp != (Tcl_Interp *) NULL) {
DBUG
        UnregisterTcpServerInterpCleanupProc(acceptCallbackPtr->interp,
                acceptCallbackPtr);
DBUG
    }
DBUG
    Tcl_EventuallyFree((ClientData) acceptCallbackPtr->script, TCL_DYNAMIC);
DBUG
    ckfree((char *) acceptCallbackPtr);
DBUG
}
/*  End of tclIOCmd.c  */

/*
 * This structure describes per-instance state of a tcp based channel.
 */

typedef struct TcpState {
    int flags;				/* ORed combination of the
                                         * bitfields defined below. */
    Tcl_File sock;			/* The socket itself. */
    Tcl_TcpAcceptProc *acceptProc;	/* Proc to call on accept. */
    ClientData acceptProcData;		/* The data for the accept proc. */
    SSL *ssl;				/* Struct for SSL processing */
    SSL_CTX  *ctx;                      /* Contect for SSL    */
    char *key;                   	/* Private key so we don't need a passwd */
    char *cert; 			/* Certificate  */
    int sslerr;				/* I'm sure there another way to do this, but... */
    int server;
} TcpState;

/*
 * These bits may be ORed together into the "flags" field of a TcpState
 * structure.
 */

#define TCP_ASYNC_SOCKET	(1<<0)	/* Asynchronous socket. */
#define TCP_ASYNC_CONNECT	(1<<1)	/* Async connect in progress. */

/*
 * The following defines the maximum length of the listen queue. This is
 * the number of outstanding yet-to-be-serviced requests for a connection
 * on a server socket, more than this number of outstanding requests and
 * the connection request will fail.
 */

#ifndef	SOMAXCONN
#define SOMAXCONN	100
#endif

#if	(SOMAXCONN < 100)
#undef	SOMAXCONN
#define	SOMAXCONN	100
#endif

/*
 * The following defines how much buffer space the kernel should maintain
 * for a socket.
 */

#define SOCKET_BUFSIZE	4096

/*
 * Static routines for this file:
 *        these are pulled directly from tclUnixChan.c
 */

static TcpState *	CreateSocket _ANSI_ARGS_((Tcl_Interp *interp,
			    int port, char *host, int server,
			    char *myaddr, int myport, int async, int needssl,char *key,char *cert));
static int		CreateSocketAddress _ANSI_ARGS_(
			    (struct sockaddr_in *sockaddrPtr,
			    char *host, int port));
static void		TcpAccept _ANSI_ARGS_((ClientData data, int mask));
static int		TcpBlockModeProc _ANSI_ARGS_((ClientData data,
        		    int mode));
static int		TcpCloseProc _ANSI_ARGS_((ClientData instanceData,
			    Tcl_Interp *interp));
static Tcl_File		TcpGetProc _ANSI_ARGS_((ClientData instanceData,
		            int direction));
static int		TcpGetOptionProc _ANSI_ARGS_((ClientData instanceData,
                            char *optionName, Tcl_DString *dsPtr));
static int		TcpInputProc _ANSI_ARGS_((ClientData instanceData,
		            char *buf, int toRead,  int *errorCode));
static int		TcpOutputProc _ANSI_ARGS_((ClientData instanceData,
		            char *buf, int toWrite, int *errorCode));
static int		TcpReadyProc _ANSI_ARGS_((ClientData instanceData,
		            int mask));
static void		TcpWatchProc _ANSI_ARGS_((ClientData instanceData,
		            int mask));
static int		WaitForConnect _ANSI_ARGS_((TcpState *statePtr,
		            int *errorCodePtr));

/*
 * This structure describes the channel type structure for SSL socket
 * based IO: (Same as core TCP based )
 */

static Tcl_ChannelType tcpChannelType = {
    "tcp",				/* Type name. */
    TcpBlockModeProc,			/* Set blocking/nonblocking mode.*/
    TcpCloseProc,			/* Close proc. */
    TcpInputProc,			/* Input proc. */
    TcpOutputProc,			/* Output proc. */
    NULL,				/* Seek proc. */
    NULL,				/* Set option proc. */
    TcpGetOptionProc,			/* Get option proc. */
    TcpWatchProc,			/* Initialize notifier. */
    TcpReadyProc,			/* Are there events? */
    TcpGetProc,				/* Get Tcl_Files out of channel. */
};


/*
 *----------------------------------------------------------------------
 *
 * TcpBlockModeProc --
 *
 *	This procedure is invoked by the generic IO level to set blocking
 *	and nonblocking mode on a SSL socket based channel.
 *
 * Results:
 *	0 if successful, errno when failed.
 *
 * Side effects:
 *	Sets the device into blocking or nonblocking mode.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
static int
TcpBlockModeProc(instanceData, mode)
    ClientData instanceData;		/* Socket state. */
    int mode;				/* The mode to set. Can be one of
                                         * TCL_MODE_BLOCKING or
                                         * TCL_MODE_NONBLOCKING. */
{
    TcpState *statePtr;
    int sock;
    int setting;
    
    statePtr = (TcpState *) instanceData;
    sock = (int) Tcl_GetFileInfo(statePtr->sock, NULL);
#ifndef	USE_FIONBIO
    setting = fcntl(sock, F_GETFL);
    if (mode == TCL_MODE_BLOCKING) {
        statePtr->flags &= (~(TCP_ASYNC_SOCKET));
        setting &= (~(O_NONBLOCK));
    } else {
        statePtr->flags |= TCP_ASYNC_SOCKET;
        setting |= O_NONBLOCK;
    }
    if (fcntl(sock, F_SETFL, setting) < 0) {
        return errno;
    }
#endif

#ifdef	USE_FIONBIO
    if (mode == TCL_MODE_BLOCKING) {
        statePtr->flags &= (~(TCP_ASYNC_SOCKET));
        setting = 0;
        if (ioctl(sock, (int) FIONBIO, &setting) == -1) {
            return errno;
        }
    } else {
        statePtr->flags |= TCP_ASYNC_SOCKET;
        setting = 1;
        if (ioctl(sock, (int) FIONBIO, &setting) == -1) {
            return errno;
        }
    }
#endif

    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * WaitForConnect --
 *
 *	Waits for a connection on an asynchronously opened socket to
 *	be completed.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The socket is connected after this function returns.
 *
 *----------------------------------------------------------------------
 */

static int
WaitForConnect(statePtr, errorCodePtr)
    TcpState *statePtr;		/* State of the socket. */
    int *errorCodePtr;		/* Where to store errors? */
{
    int sock;			/* The socket itself. */
    int timeOut;		/* How long to wait. */
    int state;			/* Of calling TclWaitForFile. */
    int flags;			/* fcntl flags for the socket. */

    /*
     * If an asynchronous connect is in progress, attempt to wait for it
     * to complete before reading.
     */
    
DBUG
    if (statePtr->flags & TCP_ASYNC_CONNECT) {
        if (statePtr->flags & TCP_ASYNC_SOCKET) {
            timeOut = 0;
        } else {
            timeOut = -1;
        }
        errno = 0;
        state = TclWaitForFile(statePtr->sock, TCL_WRITABLE | TCL_EXCEPTION,
                timeOut);
        if (!(statePtr->flags & TCP_ASYNC_SOCKET)) {
            sock = (int) Tcl_GetFileInfo(statePtr->sock, NULL);
#ifndef	USE_FIONBIO
            flags = fcntl(sock, F_GETFL);
            flags &= (~(O_NONBLOCK));
            (void) fcntl(sock, F_SETFL, flags);
#endif

#ifdef	USE_FIONBIO
            flags = 0;
            (void) ioctl(sock, FIONBIO, &flags);
#endif
        }
        if (state & TCL_EXCEPTION) {
            return -1;
        }
        if (state & TCL_WRITABLE) {
            statePtr->flags &= (~(TCP_ASYNC_CONNECT));
        } else if (timeOut == 0) {
            *errorCodePtr = errno = EWOULDBLOCK;
            return -1;
        }
    }
    if ( statePtr->ssl ) {
DBUG
	if ( statePtr->server == 2 ) {
DBUG
		if((statePtr->sslerr = SSL_accept(statePtr->ssl)) <= 0 ) {
        		fprintf(stderr, "SSL AcceptError (%d) (%s)\n",
				statePtr->sslerr, ERR_error_string(ERR_get_error()));
			statePtr->sslerr = -1;
		} else {
			statePtr->sslerr = 0;
		}
		statePtr->server = 3;
	}
DBUG
        return statePtr->sslerr;
    }
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * TcpInputProc --
 *
 *	This procedure is invoked by the generic IO level to read input
 *	from a SSL socket based channel.
 *
 * Results:
 *	The number of bytes read is returned or -1 on error. An output
 *	argument contains the POSIX error code on error, or zero if no
 *	error occurred.
 *
 * Side effects:
 *	Reads input from the input device of the channel.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
static int
TcpInputProc(instanceData, buf, bufSize, errorCodePtr)
    ClientData instanceData;		/* Socket state. */
    char *buf;				/* Where to store data read. */
    int bufSize;			/* How much space is available
                                         * in the buffer? */
    int *errorCodePtr;			/* Where to store error code. */
{
    TcpState *statePtr;			/* The state of the socket. */
    int bytesRead;			/* How many bytes were read? */
    int sock;				/* The OS handle. */
    int state;				/* Of waiting for connection. */

    *errorCodePtr = 0;
    statePtr = (TcpState *) instanceData;
    sock = (int) Tcl_GetFileInfo(statePtr->sock, NULL);
DBUG

    state = WaitForConnect(statePtr, errorCodePtr);
DBUG
    if (state != 0) {
        return -1;
    }
	if ( statePtr->ssl ) {
DBUG
    		bytesRead = SSL_read(statePtr->ssl, buf, bufSize);
DBUG
	} else {
    		bytesRead = recv(sock, buf, bufSize,0);
	}
    if (bytesRead > -1) {
        return bytesRead;
    }
    if (errno == ECONNRESET) {

        /*
         * Turn ECONNRESET into a soft EOF condition.
         */
        
DBUG
        return 0;
    }
    *errorCodePtr = errno;
DBUG
    return -1;
}

/*
 *----------------------------------------------------------------------
 *
 * TcpOutputProc --
 *
 *	This procedure is invoked by the generic IO level to write output
 *	to a SSL socket based channel.
 *
 * Results:
 *	The number of bytes written is returned. An output argument is
 *	set to a POSIX error code if an error occurred, or zero.
 *
 * Side effects:
 *	Writes output on the output device of the channel.
 *
 *----------------------------------------------------------------------
 */

static int
TcpOutputProc(instanceData, buf, toWrite, errorCodePtr)
    ClientData instanceData;		/* Socket state. */
    char *buf;				/* The data buffer. */
    int toWrite;			/* How many bytes to write? */
    int *errorCodePtr;			/* Where to store error code. */
{
    TcpState *statePtr;
    int written;
    int sock;				/* The OS handle. */
    int state;				/* Of waiting for connection. */

    *errorCodePtr = 0;
    statePtr = (TcpState *) instanceData;
    sock = (int) Tcl_GetFileInfo(statePtr->sock, NULL);
    state = WaitForConnect(statePtr, errorCodePtr);

    if (state != 0) {
        return -1;
    }
	if ( statePtr->ssl ) {
    		written = SSL_write(statePtr->ssl, buf, toWrite);
	} else {
    		written = send(sock, buf, toWrite,0);
	}
    if (written > -1) {
        return written;
    }
    *errorCodePtr = errno;
    return -1;
}

/*
 *----------------------------------------------------------------------
 *
 * TcpCloseProc --
 *
 *	This procedure is invoked by the generic IO level to perform
 *	channel-type-specific cleanup when a SSL socket based channel
 *	is closed.
 *
 * Results:
 *	0 if successful, the value of errno if failed.
 *
 * Side effects:
 *	Closes the socket of the channel.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
static int
TcpCloseProc(instanceData, interp)
    ClientData instanceData;	/* The socket to close. */
    Tcl_Interp *interp;		/* For error reporting - unused. */
{
    TcpState *statePtr;
    Tcl_File sockFile;
    int sock;
    int errorCode = 0;

DBUG
    statePtr = (TcpState *) instanceData;
DBUG
    sockFile = statePtr->sock;
DBUG
    sock = (int) Tcl_GetFileInfo(sockFile, NULL);
    
    /*
     * Delete a file handler that may be active for this socket if this
     * is a server socket - the file handler was created automatically
     * by Tcl as part of the mechanism to accept new client connections.
     * Channel handlers are already deleted in the generic IO channel
     * closing code that called this function, so we do not have to
     * delete them here.
     */
    
DBUG
     	if ( statePtr->ssl ) { SSL_free(statePtr->ssl); }
DBUG
	if ( statePtr->key ) { ckfree(statePtr->key); }
DBUG
	if ( statePtr->cert ) { ckfree(statePtr->cert); }
DBUG
    	Tcl_DeleteFileHandler(sockFile);
DBUG


    ckfree((char *) statePtr);
DBUG
    
    /*
     * We assume that inFile==outFile==sockFile and so
     * we only clean up sockFile.
     */

    Tcl_FreeFile(sockFile);

DBUG
    if (close(sock) < 0) {
	errorCode = errno;
    }
DBUG

    return errorCode;
}

/*
 *----------------------------------------------------------------------
 *
 * TcpGetOptionProc --
 *
 *	Computes an option value for a SSL socket based channel, or a
 *	list of all options and their values.
 *
 *	Note: This code is based on code contributed by John Haxby.
 *
 * Results:
 *	A standard Tcl result. The value of the specified option or a
 *	list of all options and	their values is returned in the
 *	supplied DString.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
TcpGetOptionProc(instanceData, optionName, dsPtr)
    ClientData instanceData;		/* Socket state. */
    char *optionName;			/* Name of the option to
                                         * retrieve the value for, or
                                         * NULL to get all options and
                                         * their values. */
    Tcl_DString *dsPtr;			/* Where to store the computed
                                         * value; initialized by caller. */
{
    TcpState *statePtr;
    struct sockaddr_in sockname;
    struct sockaddr_in peername;
    struct hostent *hostEntPtr;
    int sock;
    size_t size = sizeof(struct sockaddr_in);
    size_t len = 0;
    char buf[128];

DBUG
    statePtr = (TcpState *) instanceData;
    sock = (int) Tcl_GetFileInfo(statePtr->sock, NULL);
    if (optionName != (char *) NULL) {
        len = strlen(optionName);
    }

    if ((len == 0) ||
            ((len > 1) && (optionName[1] == 'p') &&
                    (strncmp(optionName, "-peername", len) == 0))) {
        if (getpeername(sock, (struct sockaddr *) &peername, &size) >= 0) {
            if (len == 0) {
                Tcl_DStringAppendElement(dsPtr, "-peername");
                Tcl_DStringStartSublist(dsPtr);
            }
            Tcl_DStringAppendElement(dsPtr, inet_ntoa(peername.sin_addr));
            hostEntPtr = gethostbyaddr((char *) &(peername.sin_addr),
                    sizeof(peername.sin_addr), AF_INET);
            if (hostEntPtr != (struct hostent *) NULL) {
                Tcl_DStringAppendElement(dsPtr, hostEntPtr->h_name);
            } else {
                Tcl_DStringAppendElement(dsPtr, inet_ntoa(peername.sin_addr));
            }
            sprintf(buf, "%d", ntohs(peername.sin_port));
            Tcl_DStringAppendElement(dsPtr, buf);
            if (len == 0) {
                Tcl_DStringEndSublist(dsPtr);
            } else {
                return TCL_OK;
            }
        }
    }

    if ((len == 0) ||
            ((len > 1) && (optionName[1] == 's') &&
                    (strncmp(optionName, "-sockname", len) == 0))) {
        if (getsockname(sock, (struct sockaddr *) &sockname, &size) >= 0) {
            if (len == 0) {
                Tcl_DStringAppendElement(dsPtr, "-sockname");
                Tcl_DStringStartSublist(dsPtr);
            }
            Tcl_DStringAppendElement(dsPtr, inet_ntoa(sockname.sin_addr));
            hostEntPtr = gethostbyaddr((char *) &(sockname.sin_addr),
                    sizeof(peername.sin_addr), AF_INET);
            if (hostEntPtr != (struct hostent *) NULL) {
                Tcl_DStringAppendElement(dsPtr, hostEntPtr->h_name);
            } else {
                Tcl_DStringAppendElement(dsPtr, inet_ntoa(sockname.sin_addr));
            }
            sprintf(buf, "%d", ntohs(sockname.sin_port));
            Tcl_DStringAppendElement(dsPtr, buf);
            if (len == 0) {
                Tcl_DStringEndSublist(dsPtr);
            } else {
                return TCL_OK;
            }
        }
    }

    if (len > 0) {
        Tcl_SetErrno(EINVAL);
        return TCL_ERROR;
    }

DBUG
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TcpWatchProc --
 *
 *	Initialize the notifier to watch Tcl_Files from this channel.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Sets up the notifier so that a future event on the channel will
 *	be seen by Tcl.
 *
 *----------------------------------------------------------------------
 */

static void
TcpWatchProc(instanceData, mask)
    ClientData instanceData;		/* The socket state. */
    int mask;				/* Events of interest; an OR-ed
                                         * combination of TCL_READABLE,
                                         * TCL_WRITABEL and TCL_EXCEPTION. */
{
    TcpState *statePtr = (TcpState *) instanceData;
DBUG

    Tcl_WatchFile(statePtr->sock, mask);
DBUG
}

/*
 *----------------------------------------------------------------------
 *
 * TcpReadyProc --
 *
 *	Called by the notifier to check whether events of interest are
 *	present on the channel.
 *
 * Results:
 *	Returns OR-ed combination of TCL_READABLE, TCL_WRITABLE and
 *	TCL_EXCEPTION to indicate which events of interest are present.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
TcpReadyProc(instanceData, mask)
    ClientData instanceData;		/* The socket state. */
    int mask;				/* Events of interest; an OR-ed
                                         * combination of TCL_READABLE,
                                         * TCL_WRITABLE and TCL_EXCEPTION. */
{
	int i;
    TcpState *statePtr = (TcpState *) instanceData;
DBUG

    i = Tcl_FileReady(statePtr->sock, mask);
DBUG
	return i;
}

/*
 *----------------------------------------------------------------------
 *
 * TcpGetProc --
 *
 *	Called from Tcl_GetChannelFile to retrieve Tcl_Files from inside
 *	a SSL socket based channel.
 *
 * Results:
 *	The appropriate Tcl_File or NULL if not present. 
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
static Tcl_File
TcpGetProc(instanceData, direction)
    ClientData instanceData;		/* The socket state. */
    int direction;			/* Which Tcl_File to retrieve? */
{
    TcpState *statePtr = (TcpState *) instanceData;
DBUG

    return statePtr->sock;
}

/*
 *----------------------------------------------------------------------
 *
 * CreateSocket --
 *
 *	This function opens a new socket in client or server mode
 *	and initializes the TcpState structure.
 *
 * Results:
 *	Returns a new TcpState, or NULL with an error in interp->result,
 *	if interp is not NULL.
 *
 * Side effects:
 *	Opens a socket.
 *
 *----------------------------------------------------------------------
 */

static TcpState *
CreateSocket(interp, port, host, server, myaddr, myport, async,needssl, key, cert)
    Tcl_Interp *interp;		/* For error reporting; can be NULL. */
    int port;			/* Port number to open. */
    char *host;			/* Name of host on which to open port.
				 * NULL implies INADDR_ANY */
    int server;			/* 1 if socket should be a server socket,
				 * else 0 for a client socket. */
    char *myaddr;		/* Optional client-side address */
    int myport;			/* Optional client-side port */
    int async;			/* If nonzero and creating a client socket,
                                 * attempt to do an async connect. Otherwise
                                 * do a synchronous connect or bind. */
    int  needssl;		/* configure for a secure socket  */
    char *key;                  /* Private key for SSL server     */
    char *cert;                 /* Certificate for SSL server */
{
    int status, sock, asyncConnect, curState, origState;
    struct sockaddr_in sockaddr;	/* socket address */
    struct sockaddr_in mysockaddr;	/* Socket address for client */
    TcpState *statePtr;
    SSL *ssl = (SSL *)0;
    SSL_CTX *ctx = (SSL_CTX *)0;

    sock = -1;
    origState = 0;
    if (! CreateSocketAddress(&sockaddr, host, port)) {
	goto addressError;
    }
    if ((myaddr != NULL || myport != 0) &&
	    ! CreateSocketAddress(&mysockaddr, myaddr, myport)) {
	goto addressError;
    }

    sock = socket(AF_INET, SOCK_STREAM, 0);
    if (sock < 0) {
	goto addressError;
    }

    if ( needssl ) {
	ctx = SSL_CTX_new();
	ssl = SSL_new(ctx);
	SSL_set_fd(ssl,sock);
    }

    /*
     * Set the close-on-exec flag so that the socket will not get
     * inherited by child processes.
     */

    fcntl(sock, F_SETFD, FD_CLOEXEC);
    
    /*
     * Set kernel space buffering
     */

    TclSockMinimumBuffers(sock, SOCKET_BUFSIZE);

    asyncConnect = 0;
    status = 0;
    if (server) {

	/*
	 * Set up to reuse server addresses automatically and bind to the
	 * specified port.
	 * Nothing to do for SSL.  Server SSL stuff we be done at connect time
	 */
    
	status = 1;
	(void) setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (char *) &status,
		sizeof(status));
	status = bind(sock, (struct sockaddr *) &sockaddr,
                sizeof(struct sockaddr));
	if (status != -1) {
	    status = listen(sock, SOMAXCONN);
	} 
    } else {
	if (myaddr != NULL || myport != 0) { 
	    curState = 1;
	    (void) setsockopt(sock, SOL_SOCKET, SO_REUSEADDR,
                    (char *) &curState, sizeof(curState));
	    status = bind(sock, (struct sockaddr *) &mysockaddr,
		    sizeof(struct sockaddr));
	    if (status < 0) {
		goto bindError;
	    }
	}

	/*
	 * Attempt to connect. The connect may fail at present with an
	 * EINPROGRESS but at a later time it will complete. The caller
	 * will set up a file handler on the socket if she is interested in
	 * being informed when the connect completes.
	 */

        if (async) {
#ifndef	USE_FIONBIO
            origState = fcntl(sock, F_GETFL);
            curState = origState | O_NONBLOCK;
            status = fcntl(sock, F_SETFL, curState);
#endif

#ifdef	USE_FIONBIO
            curState = 1;
            status = ioctl(sock, FIONBIO, &curState);
#endif            
        } else {
            status = 0;
        }
        if (status > -1) {
            status = connect(sock, (struct sockaddr *) &sockaddr,
                    sizeof(sockaddr));
            if (status < 0) {
                if (errno == EINPROGRESS) {
                    asyncConnect = 1;
                    status = 0;
                } 
            }
        }
        if (needssl && (status > -1)) {
    		if ( (status = SSL_connect(ssl)) != 1) {
			status = -1;
        		if (interp != NULL) {
        			Tcl_AppendResult(interp, "SSL connect error ",NULL);
			}
    		}
	}
    }

bindError:
    if (status < 0) {
	if ( ssl ) { 
		SSL_free(ssl); 
	} 
        if (interp != NULL) {
            Tcl_AppendResult(interp, "couldn't open socket: ",
                    Tcl_PosixError(interp), (char *) NULL);
        }
        if (sock != -1) {
            close(sock);
        }
        return NULL;
    }

    /*
     * Allocate a new TcpState for this socket.
     */

    statePtr = (TcpState *) ckalloc((unsigned) sizeof(TcpState));
    statePtr->flags = 0;
    if (asyncConnect) {
        statePtr->flags = TCP_ASYNC_CONNECT;
    }
    statePtr->sock = Tcl_GetFile((ClientData) sock, TCL_UNIX_FD);
    statePtr->ssl = ssl;
    statePtr->ctx = ctx;
    statePtr->sslerr = 0;
    statePtr->server = 0;
    if ( key ) { 
    	statePtr->key = ckalloc((unsigned) (strlen(key)+1));
	strcpy(statePtr->key,key);
    } else {
    	statePtr->key = NULL;
    }
    if ( cert ) { 
    	statePtr->cert = ckalloc((unsigned) (strlen(cert)+1));
	strcpy(statePtr->cert,cert);
    } else {
    	statePtr->cert= NULL;
    }
    
    return statePtr;

addressError:
    if (sock != -1) {
        close(sock);
    }
    if (interp != NULL) {
	Tcl_AppendResult(interp, "couldn't open socket: ",
		Tcl_PosixError(interp), (char *) NULL);
    }
	if ( ssl ) { 
		SSL_free(ssl); 
	} 
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * CreateSocketAddress --
 *
 *	This function initializes a sockaddr structure for a host and port.
 *
 * Results:
 *	1 if the host was valid, 0 if the host could not be converted to
 *	an IP address.
 *
 * Side effects:
 *	Fills in the *sockaddrPtr structure.
 *
 *----------------------------------------------------------------------
 */

static int
CreateSocketAddress(sockaddrPtr, host, port)
    struct sockaddr_in *sockaddrPtr;	/* Socket address */
    char *host;				/* Host.  NULL implies INADDR_ANY */
    int port;				/* Port number */
{
    struct hostent *hostent;		/* Host database entry */
    struct in_addr addr;		/* For 64/32 bit madness */

    (void) memset((VOID *) sockaddrPtr, '\0', sizeof(struct sockaddr_in));
    sockaddrPtr->sin_family = AF_INET;
    sockaddrPtr->sin_port = htons((unsigned short) (port & 0xFFFF));
    if (host == NULL) {
	addr.s_addr = INADDR_ANY;
    } else {
        addr.s_addr = inet_addr(host);
        if (addr.s_addr == -1) {
            hostent = gethostbyname(host);
            if (hostent != NULL) {
                memcpy((VOID *) &addr,
                        (VOID *) hostent->h_addr_list[0],
                        (size_t) hostent->h_length);
            } else {
#ifdef	EHOSTUNREACH
                errno = EHOSTUNREACH;
#else
#ifdef ENXIO
                errno = ENXIO;
#endif
#endif
                return 0;	/* error */
            }
        }
    }
        
    /*
     * NOTE: On 64 bit machines the assignment below is rumored to not
     * do the right thing. Please report errors related to this if you
     * observe incorrect behavior on 64 bit machines such as DEC Alphas.
     * Should we modify this code to do an explicit memcpy?
     */

    sockaddrPtr->sin_addr.s_addr = addr.s_addr;
    return 1;	/* Success. */
}


static void doSSLAccept (sockState,newSockState,newsock)
    TcpState *sockState;		/* Client data of server socket. */
    TcpState *newSockState;		/* Client data of server socket. */
	int newsock ;
{
		char ssl_file[1024];
		char *cert = sockState->cert;
		char *key  = sockState->key ;
		newSockState->sslerr = 0;

		if ( !key )
			key = cert;

DBUG
		newSockState->ctx = SSL_CTX_new();
DBUG
		newSockState->ssl = SSL_new(newSockState->ctx);
DBUG
    		SSL_set_fd(newSockState->ssl, newsock);
DBUG

		sprintf(ssl_file,"%s/%s",X509_get_default_cert_dir(), cert );
DBUG
		if (SSL_use_certificate_file(newSockState->ssl,ssl_file, X509_FILETYPE_PEM)<=0) {
DBUG
			if (SSL_use_certificate_file(newSockState->ssl,cert, X509_FILETYPE_PEM)<=0) {
DBUG
				newSockState->sslerr = -1;
               			fprintf(stderr,
					"SSL Invalid certificate:(%s) (%s)\n",
					cert,ERR_error_string(ERR_get_error()));
				
			}
DBUG
		}
DBUG

    		/* specify private key */
		if ( newSockState->sslerr == 0 && 
			(key && SSL_use_RSAPrivateKey_file(newSockState->ssl,key, X509_FILETYPE_PEM)<=0)) {
DBUG
			newSockState->sslerr = -2;
               			fprintf(stderr,
					"SSL Invalid key: (%s) (%s)\n",key,ERR_error_string(ERR_get_error()));
		}
DBUG
}

/*
 *----------------------------------------------------------------------
 *
 * TcpAccept --
 *	Accept a SSL socket connection.  This is called by the event loop.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Creates a new connection socket. Calls the registered callback
 *	for the connection acceptance mechanism.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
static void
TcpAccept(data, mask)
    ClientData data;			/* Callback token. */
    int mask;				/* Not used. */
{
    TcpState *sockState;		/* Client data of server socket. */
    int newsock;			/* The new client socket */
    Tcl_File newFile;			/* Its file. */
    TcpState *newSockState;		/* State for new socket. */
    struct sockaddr_in addr;		/* The remote address */
    size_t len;				/* For accept interface */
    Tcl_Channel chan;			/* Channel instance created. */
    char channelName[20];

    sockState = (TcpState *) data;

    len = sizeof(struct sockaddr_in);
    newsock = accept((int) Tcl_GetFileInfo(sockState->sock, NULL),
	    (struct sockaddr *)&addr, &len);
    if (newsock < 0) {
        return;
    }

	DBUG
    /*
     * Set close-on-exec flag to prevent the newly accepted socket from
     * being inherited by child processes.
     */

    (void) fcntl(newsock, F_SETFD, FD_CLOEXEC);
   
    
     
    newFile = Tcl_GetFile((ClientData) newsock, TCL_UNIX_FD);
    if (newFile) {
        newSockState = (TcpState *) ckalloc((unsigned) sizeof(TcpState));

        newSockState->flags = 0;
        newSockState->sock = newFile;
        newSockState->acceptProc = (Tcl_TcpAcceptProc *) NULL;
        newSockState->acceptProcData = (ClientData) NULL;
        newSockState->cert = NULL;
        newSockState->key  = NULL;
        newSockState->server = 2;

	DBUG
        
        sprintf(channelName, "sock%d", (int) newsock);
        chan = Tcl_CreateChannel(&tcpChannelType, channelName,
                (ClientData) newSockState, (TCL_READABLE | TCL_WRITABLE));
        if (chan == (Tcl_Channel) NULL) {
            ckfree((char *) newSockState);
            close(newsock);
            Tcl_FreeFile(newFile);
	} else {
            if (Tcl_SetChannelOption((Tcl_Interp *) NULL, chan, "-translation",
                    "auto crlf") == TCL_ERROR) {
                Tcl_Close((Tcl_Interp *) NULL, chan);
            } 

	    if ( sockState->ssl ) 
		doSSLAccept(sockState,newSockState,newsock);

            if (sockState->acceptProc != (Tcl_TcpAcceptProc *) NULL) {
                (sockState->acceptProc) (sockState->acceptProcData, chan,
                        inet_ntoa(addr.sin_addr), ntohs(addr.sin_port));
            }
	}
    }
}

/*  tclUnixChan.c  ends here  */

/*----------------------------------------------------------------------
 *  The following functions are madified versions of the Tcp functions
 *  in the Tcl core that do very little to add SSL.
 *---------------------------------------------------------------------/

/*
 *----------------------------------------------------------------------
 *
 * Tcl_OpenSSLClient --
 *
 *	Opens a SSL client socket and creates a channel around it.
 *
 * Results:
 *	The channel or NULL if failed.  An error message is returned
 *	in the interpreter on failure.
 *
 * Side effects:
 *	Opens a client socket and creates a new channel.
 *
 *----------------------------------------------------------------------
 */

Tcl_Channel
Tcl_OpenSSLClient(interp, port, host, myaddr, myport, async,needssl)
    Tcl_Interp *interp;			/* For error reporting; can be NULL. */
    int port;				/* Port number to open. */
    char *host;				/* Host on which to open port. */
    char *myaddr;			/* Client-side address */
    int myport;				/* Client-side port */
    int async;				/* If nonzero, attempt to do an
                                         * asynchronous connect. Otherwise
                                         * we do a blocking connect. */
    int needssl;                        /* Do a secure client */
{
    Tcl_Channel chan;
    TcpState *statePtr;
    char channelName[20];

    /*
     * Create a new client socket and wrap it in a channel.
     */

    statePtr = CreateSocket(interp, port, host, 0, myaddr, myport, async, needssl,NULL,NULL);
    if (statePtr == NULL) {
	return NULL;
    }

    statePtr->acceptProc = NULL;
    statePtr->acceptProcData = (ClientData) NULL;
    
    sprintf(channelName, "sock%d",
	    (int) Tcl_GetFileInfo(statePtr->sock, NULL));

    chan = Tcl_CreateChannel(&tcpChannelType, channelName,
            (ClientData) statePtr, (TCL_READABLE | TCL_WRITABLE));

    if (Tcl_SetChannelOption(interp, chan, "-translation", "auto crlf") == TCL_ERROR) {
        Tcl_Close((Tcl_Interp *) NULL, chan);
        return NULL;
    }

    return chan;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_OpenSSLServer --
 *
 *	Opens a SSL server socket and creates a channel around it.
 *
 * Results:
 *	The channel or NULL if failed. If an error occurred, an
 *	error message is left in interp->result if interp is
 *	not NULL.
 *
 * Side effects:
 *	Opens a server socket and creates a new channel.
 *
 *----------------------------------------------------------------------
 */

Tcl_Channel
Tcl_OpenSSLServer(interp, port, myHost, acceptProc, acceptProcData,needssl,key,cert)
    Tcl_Interp *interp;			/* For error reporting - may be
                                         * NULL. */
    int port;				/* Port number to open. */
    char *myHost;			/* Name of local host. */
    Tcl_TcpAcceptProc *acceptProc;	/* Callback for accepting connections
                                         * from new clients. */
    ClientData acceptProcData;		/* Data for the callback. */
    int needssl;                        /* Create a secure server  */
    char *key;
    char *cert;
{
    Tcl_Channel chan;
    TcpState *statePtr;
    char channelName[20];

    /*
     * Create a new client socket and wrap it in a channel.
     */

    statePtr = CreateSocket(interp, port, myHost, 1, NULL, 0, 0,needssl,key,cert);
    if (statePtr == NULL) {
	return NULL;
    }

    statePtr->server = 1;
    statePtr->acceptProc = acceptProc;
    statePtr->acceptProcData = acceptProcData;

    /*
     * Set up the callback mechanism for accepting connections
     * from new clients.
     */

    Tcl_CreateFileHandler(statePtr->sock, TCL_READABLE, TcpAccept,
            (ClientData) statePtr);
    sprintf(channelName, "sock%d",
	    (int) Tcl_GetFileInfo(statePtr->sock, NULL));
    chan = Tcl_CreateChannel(&tcpChannelType, channelName,
            (ClientData) statePtr, 0);
    return chan;
}

/*
 *----------------------------------------------------------------------
 *
 * replacement for Tcl_SocketCmd --
 *
 *	This procedure is invoked to process the "socket" Tcl command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	Creates a socket based channel.
 *
 *----------------------------------------------------------------------
 */

int
Tcl_SSLSocketCmd(notUsed, interp, argc, argv)
    ClientData notUsed;			/* Not used. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    int a, server, port;
    char *arg, *copyScript, *host, *script;
    char *myaddr = NULL;
    int myport = 0;
    int async = 0;
    int ssl = 0;
    char *key = NULL;
    char *cert = NULL;

    Tcl_Channel chan;
    AcceptCallback *acceptCallbackPtr;
    
    server = 0;
    script = NULL;

    if (TclHasSockets(interp) != TCL_OK) {
	return TCL_ERROR;
    }

    for (a = 1; a < argc; a++) {
        arg = argv[a];
	if (arg[0] == '-') {
	    if (strcmp(arg, "-server") == 0) {
                if (async == 1) {
                    Tcl_AppendResult(interp,
                            "cannot set -async option for server sockets",
                            (char *) NULL);
                    return TCL_ERROR;
                }
		server = 1;
		a++;
		if (a >= argc) {
		    Tcl_AppendResult(interp,
			    "no argument given for -server option",
                            (char *) NULL);
		    return TCL_ERROR;
		}
                script = argv[a];
            } else if (strcmp(arg, "-ssl") == 0) {
		ssl = 1;
            } else if (strcmp(arg, "-cert") == 0) {
		a++;
                if (a >= argc) {
		    Tcl_AppendResult(interp,
			    "no argument given for -cert option",
                            (char *) NULL);
		    return TCL_ERROR;
		}
                cert = argv[a];
            } else if (strcmp(arg, "-key") == 0) {
		a++;
                if (a >= argc) {
		    Tcl_AppendResult(interp,
			    "no argument given for -key option",
                            (char *) NULL);
		    return TCL_ERROR;
		}
                key = argv[a];
            } else if (strcmp(arg, "-myaddr") == 0) {
		a++;
                if (a >= argc) {
		    Tcl_AppendResult(interp,
			    "no argument given for -myaddr option",
                            (char *) NULL);
		    return TCL_ERROR;
		}
                myaddr = argv[a];
            } else if (strcmp(arg, "-myport") == 0) {
		a++;
                if (a >= argc) {
		    Tcl_AppendResult(interp,
			    "no argument given for -myport option",
                            (char *) NULL);
		    return TCL_ERROR;
		}
		if (TclSockGetPort(interp, argv[a], "tcp", &myport)
                    != TCL_OK) {
		    return TCL_ERROR;
		}
            } else if (strcmp(arg, "-async") == 0) {
                if (server == 1) {
                    Tcl_AppendResult(interp,
                            "cannot set -async option for server sockets",
                            (char *) NULL);
                    return TCL_ERROR;
                }
                async = 1;
	    } else {
		Tcl_AppendResult(interp, "bad option \"", arg,
                        "\", must be -async, -myaddr, -myport, or -server",
                        (char *) NULL);
		return TCL_ERROR;
	    }
	} else {
	    break;
	}
    }
    if (server) {
        host = myaddr;		/* NULL implies INADDR_ANY */
	if (myport != 0) {
	    Tcl_AppendResult(interp, "Option -myport is not valid for servers",
		    NULL);
	    return TCL_ERROR;
	}
    } else if (a < argc) {
	host = argv[a];
	a++;
    } else {
wrongNumArgs:
	Tcl_AppendResult(interp, "wrong # args: should be either:\n",
		argv[0],
                " ?-ssl? ?-myaddr addr? ?-myport myport? ?-async? host port\n",
		argv[0],
                " -server command ?-ssl? ?-key privateKey? ?-cert Certificate? ?-myaddr addr? port",
                (char *) NULL);
        return TCL_ERROR;
    }

    if (a == argc-1) {
	if (TclSockGetPort(interp, argv[a], "tcp", &port) != TCL_OK) {
	    return TCL_ERROR;
	}
    } else {
	goto wrongNumArgs;
    }

    if (server) {

	if ( ssl && !cert ) {
		Tcl_AppendResult(interp, "Certificate is required for an SSL server\n",
			"Use (-cert filename) to set!",NULL);
		return TCL_ERROR;
	}
        acceptCallbackPtr = (AcceptCallback *) ckalloc((unsigned)
                sizeof(AcceptCallback));
        copyScript = ckalloc((unsigned) strlen(script) + 1);
        strcpy(copyScript, script);
        acceptCallbackPtr->script = copyScript;
        acceptCallbackPtr->interp = interp;
        chan = Tcl_OpenSSLServer(interp, port, host, AcceptCallbackProc,
                (ClientData) acceptCallbackPtr,ssl,key,cert);
        if (chan == (Tcl_Channel) NULL) {
            ckfree(copyScript);
            ckfree((char *) acceptCallbackPtr);
            return TCL_ERROR;
        }

        /*
         * Register with the interpreter to let us know when the
         * interpreter is deleted (by having the callback set the
         * acceptCallbackPtr->interp field to NULL). This is to
         * avoid trying to eval the script in a deleted interpreter.
         */

        RegisterTcpServerInterpCleanup(interp, acceptCallbackPtr);
        
        /*
         * Register a close callback. This callback will inform the
         * interpreter (if it still exists) that this channel does not
         * need to be informed when the interpreter is deleted.
         */
        
        Tcl_CreateCloseHandler(chan, TcpServerCloseProc,
                (ClientData) acceptCallbackPtr);
    } else {
        chan = Tcl_OpenSSLClient(interp, port, host, myaddr, myport, async,ssl);
        if (chan == (Tcl_Channel) NULL) {
            return TCL_ERROR;
        }
    }
    Tcl_RegisterChannel(interp, chan);            
    Tcl_AppendResult(interp, Tcl_GetChannelName(chan), (char *) NULL);
    
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TclSSL_Init --
 *
 *	This is a package initialization procedure, which is called
 *	by Tcl when this package is to be added to an interpreter.
 *
 * Results:  TclSSL configured and loaded
 *
 * Side effects:
 *	 rename some of the standard tcl functions dealing with sockets.  This
 *       will let existing code run correctly with ssl.
 *
 *----------------------------------------------------------------------
 */

int
Tclssl_Init(interp)
    Tcl_Interp *interp;		/* Interpreter in which the package is
				 * to be made available. */
{
    int code;

	ERR_load_SSL_strings();
    code = Tcl_PkgProvide(interp, "TclSSL", "1.0");
    if (code != TCL_OK) {
	return code;
    }
    Tcl_CreateCommand(interp, "socket", Tcl_SSLSocketCmd , (ClientData) 0,
	    (Tcl_CmdDeleteProc *) NULL);
    return TCL_OK;
}
