/*-
 * Copyright (c) 1993, 1994, 1995 Michael B. Durian.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Michael B. Durian.
 * 4. The name of the the Author may be used to endorse or promote 
 *    products derived from this software without specific prior written 
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#ifdef USE_MPU401
#include <assert.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <strstream.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/ioctl.h>

#ifdef SVR4
#include <sys/filio.h>
#endif

#ifdef linux
#include <linux/termios.h>
#endif

#include <sys/midiioctl.h>

#include "MPU401.h"
#include "EvntUtil.h"
#include "Note.h"

// does it take a void or an int?
typedef void Sigfunc(int);

static MPU401 *CurrentMPU401 = 0;
#if 0
static void DebugCallback(const Event *e);
#endif
static Sigfunc *posixsignal(int signo, Sigfunc *func);

MPU401::MPU401() : last_play_time(0), last_rec_time(0), fd(-1), finished(0),
    last_record_rs(0)
{

	assert(CurrentMPU401 == 0);
	CurrentMPU401 = this;
	/* no one agrees on what default signals should look like */
	posixsignal(SIGIO, SIG_IGN);
}

MPU401::MPU401(const char *dev) : MidiDevice(dev), last_play_time(0),
    last_rec_time(0), fd(-1), finished(0), last_record_rs(0)
{

	assert(CurrentMPU401 == 0);
	CurrentMPU401 = this;
	posixsignal(SIGIO, SIG_IGN);
}

MPU401::~MPU401()
{

	if (fd != -1)
		Stop();
	if (curr_event != 0)
		delete curr_event;
	CurrentMPU401 = 0;
}

int
MPU401::Play(Song *s, int r)
{
	ostrstream err;
	const char *name;
	char *str;
	int arg, i;

//	SetPlayCallback(DebugCallback);
	// open the device
	if (fd != -1) {
		SetError("Device already open");
		return (0);
	}
	if ((name = GetName()) == 0) {
		SetError("No device set");
		return (0);
	}

	// set repeat
	SetRepeat(r);

	play_song = s;

	// initialize track/event info
	last_play_time = 0;
	finished = 0;
	curr_event = new Event *[play_song->GetNumTracks()];
	if (curr_event == 0) {
		SetError("No more memory");
		return (0);
	}
	for (i = 0; i < play_song->GetNumTracks(); i++)
		curr_event[i] = play_song->GetTrack(i).GetEventsNoMod(0);

	// set up async stuff
	posixsignal(SIGIO, MPU401SigIO);
	/*
	 * Ick!  Linux feels they can define SIGURG to be SIGIO.
	 * Obviously that won't work here, so the Linux driver uses 
	 * SIGUSR1 instead.
	 *
	 * This is fixed as of 1.1.82
	 */
	posixsignal(SIGURG, MPU401SigUrg);

	// open device
	if ((fd = open(name, O_WRONLY)) == -1) {
		err << "open failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

	// set division
	arg = play_song->GetDivision();
	if (ioctl(fd, MSDIVISION, &arg) == -1) {
		err << "ioctl MSDIVISION failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

	arg = GetMidiThru();
	if (ioctl(fd, MTHRU, &arg) == -1) {
		err << "ioctl MTHRU failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

	// the following should be a fcntl FIOASYNC call, but the
	// FIOASYNC ioctl doesn't always get passed down to the driver
	// properly under all OSes (read Linux), thus we use the MASYNC
	// entry point.
	arg = 1;
	if (ioctl(fd, MASYNC, &arg) == -1) {
		err << "ioctl MASYNC failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

	return (1);
}

int
MPU401::Record(Song *rs, Song *ps, int r)
{
	ostrstream err;
	const char *name;
	char *str;
	int arg, i;

	if (fd != -1)
		return (0);
	if ((name = GetName()) == 0)
		return (0);

	rec_song = rs;
	last_play_time = 0;
	last_rec_time = 0;
	last_record_rs = 0;
	if (ps == 0) {
		play_song = 0;
		if ((fd = open(name, O_RDONLY)) == -1) {
			err << "open failed: " << strerror(errno) << ends;
			str = err.str();
			SetError(str);
			delete str;
			return (0);
		}
		arg = rec_song->GetDivision();
		if (ioctl(fd, MSDIVISION, &arg) == -1) {
			err << "ioctl MSDIVISION failed: " << strerror(errno)
			    << ends;
			str = err.str();
			SetError(str);
			delete str;
			return (0);
		}
	} else {
		play_song = ps;

		// initialize track/event info
		curr_event = new Event *[play_song->GetNumTracks()];
		if (curr_event == 0) {
			SetError("No more memory");
			return (0);
		}
		for (i = 0; i < play_song->GetNumTracks(); i++)
			curr_event[i] =
			    play_song->GetTrack(i).GetEventsNoMod(0);

		if ((fd = open(name, O_RDWR)) == -1) {
			err << "open failed: " << strerror(errno) << ends;
			str = err.str();
			SetError(str);
			delete str;
			return (0);
		}

		// set division
		arg = play_song->GetDivision();
		if (ioctl(fd, MSDIVISION, &arg) == -1) {
			err << "ioctl MSDIVISION failed: " << strerror(errno)
			    << ends;
			str = err.str();
			SetError(str);
			delete str;
			return (0);
		}

		// start record timer when first event is scheduled to play
		arg = 1;
		if (ioctl(fd, MRECONPLAY, &arg) == -1) {
			err << "ioctl MRECONPLAY failed: " << strerror(errno)
			    << ends;
			str = err.str();
			SetError(str);
			delete str;
			return (0);
		}

		finished = 0;
	}

	// set repeat
	SetRepeat(r);

	// set up async stuff
	posixsignal(SIGIO, MPU401SigIO);
	/*
	 * Ick!  Linux feels they can define SIGURG to be SIGIO.
	 * Obviously that won't work here, so the Linux driver uses 
	 * SIGUSR1 instead.
	 *
	 * This is fixed as of 1.1.82
	 */
	posixsignal(SIGURG, MPU401SigUrg);

	arg = GetMidiThru();
	if (ioctl(fd, MTHRU, &arg) == -1) {
		err << "ioctl MTHRU failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

	// the following should be a fcntl FIOASYNC call, but the
	// FIOASYNC ioctl doesn't always get passed down to the driver
	// properly under all OSes (read Linux), thus we use the MASYNC
	// entry point.
	arg = 1;
	if (ioctl(fd, MASYNC, &arg) == -1) {
		err << "ioctl MASYNC failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

	return (1);
}

int
MPU401::Stop(void)
{
	ostrstream err;
	char *str;
	int arg;

	finished = 1;
	if (fd == -1)
		return (1);

	posixsignal(SIGIO, SIG_IGN);

	arg = 0;
	if (ioctl(fd, MASYNC, &arg) == -1) {
		err << "ioctl MASYNC failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

	if (ioctl(fd, MGPLAYQ, &arg) == -1) {
		err << "ioctl MGPLAYQ failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}
	// flush queues
	if (arg != 0)
		if (ioctl(fd, MDRAIN, NULL) == -1) {
			err << "ioctl MDRAIN failed: " << strerror(errno)
			    << ends;
			str = err.str();
			SetError(str);
			delete str;
			return (0);
		}

	close (fd);
	fd = -1;

	curr_event = 0;
	play_song = 0;
	rec_song = 0;
	return (1);
}

int
MPU401::Wait(void)
{
	sigset_t sigset;

	finished = 0;
	sigemptyset(&sigset);
	do
		sigsuspend(&sigset);
	while (!finished);
	return (1);
}

/*
 * get the current time.  Either as SMPTE or SMF.
 */
int
MPU401::GetTime(TimeType tt, void *time)
{
	ostrstream err;
	char *str;

	if (fd == -1) {
		err << "device must be open." << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

	switch (tt) {
	case SMFTIME:
		if (ioctl(fd, MGSMFTIME, (unsigned long *)time) == -1) {
			err << "ioctl MGSMFTIME failed: " << strerror(errno)
			    << ends;
			str = err.str();
			SetError(str);
			delete str;
			return (0);
		}
		break;
	case SMPTETIME:
		SMPTE_frame t;
		SMPTETime *smpte_time;
		unsigned short flags;

		if (ioctl(fd, MGSMPTE, &t) == -1) {
			err << "ioctl MGSMPTE failed: " << strerror(errno)
			    << ends;
			str = err.str();
			SetError(str);
			delete str;
			return (0);
		}
		smpte_time = (SMPTETime *)time;
		smpte_time->SetHour(t.hour);
		smpte_time->SetMinute(t.minute);
		smpte_time->SetSecond(t.second);
		smpte_time->SetFrame(t.frame);
		smpte_time->SetFraction(t.fraction);
		flags = 0;
		if (t.status & SMPTE_SYNC)
			flags |= SMPTE_INSYNC;
		if (t.status & SMPTE_FRAMEDROP)
			flags |= SMPTE_DROPFRAME;
		smpte_time->SetFlags(flags);
		break;
	}
	return (1);
}

int
MPU401::GetMidiThru(void)
{
	ostrstream err;
	const char *name;
	char *str;
	int d, mt;

	// open the device
	if (fd != -1)
		d = fd;
	else {
		if ((name = GetName()) == 0) {
			SetError("No device set");
			return (-1);
		}
		// open device
		if ((d = open(name, O_RDONLY)) == -1) {
			err << "open failed: " << strerror(errno) << ends;
			str = err.str();
			SetError(str);
			delete str;
			return (-1);
		}
	}
	if (ioctl(d, MGTHRU, &mt) == -1) {
		err << "ioctl failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (-1);
	}
	if (fd == -1)
		close(d);
	return (mt);
}

int
MPU401::SetMidiThru(int mt)
{
	ostrstream err;
	const char *name;
	char *str;
	int d;

	// open the device
	if (fd != -1)
		d = fd;
	else {
		if ((name = GetName()) == 0) {
			SetError("No device set");
			return (0);
		}
		// open device
		if ((d = open(name, O_RDONLY)) == -1) {
			err << "open failed: " << strerror(errno) << ends;
			str = err.str();
			SetError(str);
			delete str;
			return (0);
		}
	}
	if (ioctl(d, MTHRU, &mt) == -1) {
		err << "ioctl failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}
	if (fd == -1)
		close(d);
	return (1);
}

/*
 * Since this routine is called ASYNC, we just print out errors
 * instead of putting them in SetError
 */
void
MPU401SigIO(int notused)
{
	struct timeval t;
	fd_set rset, wset;
	MPU401 *mpu;

	mpu = CurrentMPU401;
	if (mpu == 0 || mpu->fd == -1)
		return;

	FD_ZERO(&rset);
	FD_ZERO(&wset);
	FD_SET(mpu->fd, &rset);
	FD_SET(mpu->fd, &wset);
	t.tv_sec = 0;
	t.tv_usec = 0;

	// poll to find out what caused the SIGIO
	if (select(mpu->fd + 1, &rset, &wset, 0, &t) == -1) {
		cerr << "error selecting MPU401: " << strerror(errno) << "\n";
		return;
	}

	// check for record event
	if (FD_ISSET(mpu->fd, &rset)) {
		if (mpu->rec_song != 0) {
			mpu->ReadEvents();
		}
	}

	if (FD_ISSET(mpu->fd, &wset)) {
		if (mpu->play_song != 0 && !mpu->finished) {
			if (mpu->WriteEvents() == 0)
				mpu->finished = 1;
		}
	}
}

/*
 * We get SIGURGs when we are using a SMPTE device and the time
 * has changed so much that we need to reposition ourselves
 */
void
MPU401SigUrg(int notused)
{
	unsigned long new_time;
	int i;
	MPU401 *mpu;

	mpu = CurrentMPU401;
	if (mpu == 0 || mpu->fd == -1)
		return;

	if (ioctl(mpu->fd, MGSMFTIME, &new_time) == -1) {
		/* we can't update, but we still need to send events */
		mpu->WriteEvents();
		return;
	}

	/* redo curr_event pointers based on new time */
	for (i = 0; i < mpu->play_song->GetNumTracks(); i++)
		mpu->curr_event[i] =
		    mpu->play_song->GetTrack(i).GetEventsNoMod(new_time);
	mpu->last_play_time = new_time;
	mpu->last_rec_time = new_time;
	/* now write the new events */
	mpu->WriteEvents();
}

int
MPU401::WriteEvents(void)
{
	SMFTrack smf;
	unsigned long low_time;
	long len;
	int bytes_written, i, low_index, numtowrite, numwritten, pq;
	const char *errstr;

	if (ioctl(fd, MGQAVAIL, &numtowrite) == -1) {
		cerr << "MPU401Thread: ioctl MGQAVAIL failed: "
		    << strerror(errno) << "\n";
		return (-1);
	}
	for (numwritten = 0; numwritten < numtowrite;) {
		// find track with lowest timed event
		low_index = -1;
		low_time = 0;
		for (i = 0; i < play_song->GetNumTracks(); i++) {
			if (curr_event[i] == 0)
				continue;
			if (low_index == -1) {
				low_time = curr_event[i]->GetTime();
				low_index = i;
				continue;
			} else {
				if (curr_event[i]->GetTime() < low_time) {
					low_time = curr_event[i]->GetTime();
					low_index = i;
				}
			}
		}
		// Check to see if we're done
		if (low_index == -1) {
			if (!GetRepeat()) {
				// write any remaining events
				if (numwritten != 0) {
					len = smf.GetLength();
					bytes_written = write(fd,
					    smf.GetData(len), len);
					if (bytes_written != len) {
						cerr << "MPU401: write wrote "
						    "wrong amount";
						if (bytes_written == -1)
							cerr << ": " <<
							    strerror(errno) <<
							    "\n";
						else {
							cerr << " " <<
							    bytes_written
							    << " of " <<
							    len << "\n";
						}
					}
					return (numwritten);
				} else {
					/*
					 * if play queue is empty, we're
					 * done, otherwise do nothing
					 */
					if (ioctl(fd, MGPLAYQ, &pq) == -1) {
						cerr << "MPU401: MGPLAYQ "
						    "failed: " <<
						    strerror(errno) << "\n";
						return (-1);
					}
					if (pq == 0)
						return (0);
					else
						return (-1);
				}
			}
			// otherwise reset event pointers
			for (i = 0; i < play_song->GetNumTracks(); i++)
				curr_event[i] = play_song->GetTrack(i).
				    GetEventsNoMod(0);
			last_play_time = 0;
			continue;
		}
		// Write events until time changes
		while (curr_event[low_index] != 0 &&
		    curr_event[low_index]->GetTime() == low_time) {
			if (!WriteEventToSMFTrack(smf, last_play_time,
			    curr_event[low_index], errstr)) {
				cerr << "error playing: " << errstr << "\n";
				return (-1);
			}
			numwritten++;
			curr_event[low_index] = play_song->GetTrack(low_index).
			    NextEvent(curr_event[low_index]);
		}
	}
	if (numwritten != 0) {
		len = smf.GetLength();
		bytes_written = write(fd, smf.GetData(len), len);
		if (bytes_written != len) {
			cerr << "MPU401: write wrote wrong amount";
			if (bytes_written == -1)
				cerr << ": " << strerror(errno) << "\n";
			else
				cerr << " " << bytes_written <<
				    " of " << len << "\n";
		}
	}
	return (numwritten);
}

void
MPU401::ReadEvents(void)
{
	MidiDeviceCallback callback;
	SMFTrack track;
	int num_read;
	EventType etype;
	unsigned char event[MaxEventSize];
	const char *errstr;
	Event *e, *first_e;

	do {
		if ((num_read = read(fd, event, MaxEventSize)) == -1) {
			cerr << "error reading MPU401 event: " <<
			    strerror(errno) << "\n";
			return;
		}
		track.PutData(event, num_read);
	} while (num_read == MaxEventSize);

	track.SetRunningState(last_record_rs);
	first_e = 0;
	while ((e = ReadEventFromSMFTrack(track, last_rec_time, errstr)) != 0) {
		Event *new_e;

		if (first_e == 0)
			first_e = e;
		new_e = rec_song->PutEvent(0, *e);
		delete e;
		if (new_e == 0)
			continue;
		etype = new_e->GetType();
		// put links on noteoffs
		if ((etype == NOTEON && ((NoteEvent *)new_e)->GetVelocity()
		    == 0) || etype == NOTEOFF)
			rec_song->SetNotePair(0, new_e);
	}
	if (errstr != 0)
		cerr << "Error while recording: " << errstr << "\n";
	last_record_rs = track.GetRunningState();
	if ((callback = GetRecordCallback()) != 0)
		callback(first_e);
}

ostream &
operator<<(ostream &os, const MPU401 &mpu)
{

	os << mpu.GetName();
	if (mpu.play_song != 0)
		os << " Playing";
	if (mpu.rec_song != 0)
		os << " Recording";
	return (os);
}

#if 0
void
DebugCallback(const Event *e)
{

	for (; e != 0; e = e->GetNextEvent())
		cout << *e << endl;
}
#endif

Sigfunc *
posixsignal(int signo, Sigfunc *func)
{
	struct sigaction act, oact;

	sigemptyset(&oact.sa_mask);
	sigemptyset(&act.sa_mask);
	act.sa_handler = func;
	act.sa_flags = 0;

	if (signo == SIGALRM) {
#ifdef SA_INTERRUPT
		act.sa_flags |= SA_INTERRUPT;	/* SunOS */
#endif
	} else {
#ifdef SA_RESTART
		act.sa_flags |= SA_RESTART;	/* SVR4, 4.4BSD */
#endif
	}

	if (sigaction(signo, &act, &oact) < 0)
		return (SIG_ERR);
	return (oact.sa_handler);
}
#endif
