# NOTE: This is a documentation demo, the code has been 
# shortened so it may no longer work!

# Copyright (c) 1996 by SoftWorks.  All Rights Reserved.
# File:   main.tcl
# Author: Richard Schwaninger
# Date:   20/02/1996
# System: chiron.standalone
# RCS:    $Id: font.tcl,v 1.1 1996/08/30 08:53:21 /SW/app Exp /SW/app $


# XLFD
# registry
# foundry
# family
# weight
# slant
# setwidth
# style
# pixel
# size
# xres
# yres
# spacing
# width
# charset
# encoding

# =head1 package initialisation
# =head2 Description
# this routine has to be called before you can use any of the
# predefined font schemes. This is used to define some default schemes
# that are often used.

# Of course you can define your own fonts
# anywhere in your code but this is cumbersome and nonportable.

# NOTE: softWorks packages call this automatically.

proc font_Init { args } {
# standard template for outgoing real font defs
   global Font+Tmpl+Default
   global Font+Prop+Default
   global Font+Prop+*-*
   
   # conversion table for different attributes like italic and oblique, 
   # also defines things that do not exist at all (like light fonts).
   
   array set Font+Prop+Default {
      -slant.italic i
      -slant.normal r
      -weight.light normal
      -width.wide normal
      -width.condensed normal
      -size {[expr \$val*10]}
      -spacing.fixed c
      -spacing.proportional p
   }
   
   Font def Times \
	 {-foundry adobe -family times -charset iso8859 \
	 -encoding 1 -spacing p  } \
	 {weight.normal medium} \
	 {-slant normal -width normal -weight normal -size 14}
   
}


# =head1 font management
# =head2 Description
# This is the generic toplevel font management routine. It follows
# standard tcl call syntax like eg the C<info> command. 

# Input is a B<logical font name>, output is a standard 
# B<X  font description>.
# 
# =head2 Background
# We define a B<logical font name> (eg I<Times>, or I<LabelFont>) with
# the basic properties of the font (eg registry, foundry, charset, but also
# default size and slant). The definition itself is not very portable but at
# least it can be kept in a single place.

# A definition looks like this:

#         Font def LabelFont \
#	    {-foundry adobe -family helvetica -charset iso8859 \
#	     -encoding 1 -spacing p -pixel0 } \
#	    {-slant normal -width normal -weight normal  -size 14}

# As you can see I used the standard XLFD conventions (don't know
# enough about Windows or Mac). The first list gives things that
# should be constant for this font (that is you can no longer change
# eg. the encoding for the logical font you create). The second list
# states the default values used if you don't override them when you
# get the font. All values not mentioned in one of the lists will be
# replaced by '*' in the output.

# =head2 Usage
# Logical fontnames make it easy to change fonts througout the source
# code as the use looks always something like this:

#         button .b -text "huhu?" -font [Font get LabelFont]
# 
# To give the button a bold appearance you can define a new logical
# font or (if this is just a onetime change) do this

#         button .b config -font [Font get LabelFont weight=bold]

# Of course you can do more complicated things:

#         button .b -font [Font get LabelFont -weight bold \
#                  -slant italic -size 24]

# C<Font set> may be used to incrementally define a logical font (for the
# really complicated things...)

# C<Font exists> I<font> returns 1 if a logical font with this name
# exists, 0 otherwise.

# C<Font names> returns a list of all defined logical fonts.

# C<Font copy> I<old> I<new> copies a logical font definition and makes
# that font availabe under a new name. You may now use C<Font set> to
# change the attributes of the new font. 

# B<NOTE:> Attributes defined statically with C<Font def> (the first
# list - see above) cannot be changed!

# =head2 Arguments
# here is a description of the arguments:
# =over 3
# =item cmd
# the subcommand to execute, one of
# =over 3
# =item get
# to get a font definition from a logical name
# =item set
# to set values of a logical font
# =item def
# to define a new logical font
# =item info
# to get info about a font
# =item exists
# to check if a logical font name exists
# =item names
# to get a list of font names
# =item copy
# to copy a font description to a new name
# =back
# =item name
# generic fontname
# =item args
# arguments, depending on subcommand
# =back

# =head2 Return Value
# returns a standard I<X font name> most of the time (depending on
# subcommand). 

proc Font { cmd {name Default} args } {
   switch $cmd {
      get {
	 return [FontGen $name $args]
      }
      default {
	 error "Font <cmd> <args>"
      }
   }
   return {}
}


# =head1 generate X font
# =head2 Description
# this routine is local

# takes a logical font name and a list of attributes and constructs a
# proper XLFD.

# =head2
# =over 3
# =item name
# logical font name
# =item par
# a list of attributes in the form C<-attr value>
# =back

# =head2 Return Value
# returns the X font name

proc FontGen { name par } {
   upvar \#0 Font+Tmpl+$name orig
   upvar \#0 Font+Def+$name def

   # do not change the original, use a copy!
   if { ![info exists orig] } {
      error "Font '$name' does not exist"
   }
   regsub -all -- {--[^-]+} $template -* template
   return $template
}


# =head1 substitute attributes
# =head2 Description
# this routine is local

# substitute real attribute names for logical ones. This copes with
# the fact that we do want a layer in between our own attribute names
# and the names of X. (I<italic> is mapped to either I<i> or I<o> for
# example). This is a two stage process - if we don't find a mapping
# in the current font we look into the 'Default' font. 

# The substitutions can even handle executable tcl code (see the size
# definition where we multiply the point value by 10 to get the X value.

# =head2 Arguments
# =over 3
# =item name
# logical font name
# =item spec
# attribute name (eg slant)
# =item val
# attribute value (eg italic)
# =back

# head2 Return VAlue
# returns the definition that can be inserted into the XLFD

proc FontSub { name spec val } {
   upvar \#0 Font+Tmpl+$name tmpl
   
#   puts "tmpl=$tmpl"
   regexp -- {-+([^-]+-+[^-]+)} $tmpl dummy propname
   upvar \#0 Font+Prop+$propname prop
   set form \$val
   eval set ret $form
   return $ret
}


# =head1 make font template
# =head2 Description
# this is a local routine

# create a XLFD template for the new font with the constant items
# substituted. Start with the default template.

# =head2 Arguments
# =over 3
# =item name
# new logical name
# =item par
# a list of default parameters
# =back

# =head2 Return Value
# returns nothing, sets the corresponding global vars

proc FontTemplate { name par } {
   upvar \#0 Font+Tmpl+Default orig
   upvar \#0 Font+Tmpl+$name template
   # don't use the original here
   set template $orig

   # alas: we need foundry and family set in template to allow for
   # FontSub to be called
   set arr(-foundry) *
   foreach {i j } $par {
      set val [FontSub $name $i $arr($i)]
      regsub -- $i $template $val template

   }
}

# EOF
