//	Copyright (C) 1998 Equi4 Software.  All rights reserved.
//
//	This is the Tcl-specific code to turn MetaKit into a Tcl extension.
//
//	NOTE: this is prerelease software, it's neither well-behaved nor robust!
///////////////////////////////////////////////////////////////////////////////

#include "m4kit.h"
#include "scripted.h"

#include <tcl.h>

#include <stdlib.h>
#include <string.h>
#include <ctype.h>

///////////////////////////////////////////////////////////////////////////////

	static MkWorkspace work;		// there is one common workspace

///////////////////////////////////////////////////////////////////////////////

#if defined (_WIN32)
	#define EXPORT(t) __declspec(dllexport) t
#elif defined (_WIN)
	#define EXPORT(t) t _export
#else
	#define EXPORT(t) t
#endif

///////////////////////////////////////////////////////////////////////////////

static c4_String TclToKitDesc(const char* desc_)
{
	c4_Bytes temp;
	char* p = (char*) temp.SetBuffer(3 * strlen(desc_) + 100);

	*p++ = '[';

	int level = 0;
	bool prop = false;

	while (*desc_)
	{
		if (isalnum(*desc_))
		{
			if (prop)
				*p++ = ',';

			do
				*p++ = *desc_;
			while (isalnum(*++desc_));
			
			while (isspace(*desc_))
				++desc_;

			if (*desc_ == ':')
			{
				*p++ = *desc_++;
				*p++ = *desc_++;
			}
			else if (*desc_ != '{')
			{
				*p++ = ':';
				*p++ = 'S';
			}

			prop = true;
		}
		else
		{
			switch (*desc_++)
			{
				case '{':	if (++level % 2 == 0)
							{
								*p++ = '[';
								prop = false;
							}
							break;
				case '}':	if (level-- % 2 == 0)
							{
								*p++ = ']';
								prop = true;
							}
							break;
			}
		}
	}

	*p++ = ']';
	*p = 0;

	return (const char*) temp.Contents();
}

///////////////////////////////////////////////////////////////////////////////
//
//	Interface to Tcl 8.0 type mechanism, defines a new "mkProperty" datatype

	static void FreePropertyInternalRep(Tcl_Obj* propPtr);
	static void DupPropertyInternalRep(Tcl_Obj* srcPtr, Tcl_Obj* copyPtr);
	static int SetPropertyFromAny(Tcl_Interp* interp, Tcl_Obj* objPtr);
	static void UpdateStringOfProperty(Tcl_Obj* propPtr);

static Tcl_ObjType mkPropertyType = {
	"mkProperty",				// name
	FreePropertyInternalRep,	// freeIntRepProc
	DupPropertyInternalRep,		// dupIntRepProc
	UpdateStringOfProperty,		// updateStringProc
	SetPropertyFromAny			// setFromAnyProc
};

///////////////////////////////////////////////////////////////////////////////

	static int& AsPropId(Tcl_Obj* obj_)
	{
		// assert(obj_->typePtr = &mkPropertyType);

		return (int&) obj_->internalRep.longValue;
	}

static void 
FreePropertyInternalRep(Tcl_Obj* propPtr)
{
	c4_Property prop (AsPropId(propPtr));
	prop.Refs(-1);
}

static void
DupPropertyInternalRep(Tcl_Obj* srcPtr, Tcl_Obj* copyPtr)
{
	c4_Property prop (AsPropId(srcPtr));
	prop.Refs(+1);

	AsPropId(copyPtr) = AsPropId(srcPtr);
	copyPtr->typePtr = &mkPropertyType;
}

static int
SetPropertyFromAny(Tcl_Interp* interp, Tcl_Obj* objPtr) 
{
	Tcl_ObjType* oldTypePtr = objPtr->typePtr;

	char type = 'S';

	int length;
	char* string = Tcl_GetStringFromObj(objPtr, &length);

	if (length > 2 && string[length-2] == ':')
	{
		type = string[length-1];
		length -= 2;
	}

	c4_Property prop (type, c4_String (string, length));
	prop.Refs(+1);

	if (oldTypePtr && oldTypePtr->freeIntRepProc)
		oldTypePtr->freeIntRepProc(objPtr);
	
	objPtr->typePtr = &mkPropertyType;
	AsPropId(objPtr) = prop.GetId();
	return TCL_OK;
}

static void
UpdateStringOfProperty(Tcl_Obj* propPtr)
{
	c4_Property prop (AsPropId(propPtr));
	
	c4_String s = prop.Name();
	if (prop.Type() != 'S')
	{
		s += ':';
		s += prop.Type();
	}

	propPtr->length = s.GetLength();
	propPtr->bytes = strcpy(Tcl_Alloc(propPtr->length + 1), s);
}

///////////////////////////////////////////////////////////////////////////////
//
//	Interface to Tcl 8.0 type mechanism, defines a new "mkCursor" datatype

	static void FreeCursorInternalRep(Tcl_Obj* propPtr);
	static void DupCursorInternalRep(Tcl_Obj* srcPtr, Tcl_Obj* copyPtr);
	static int SetCursorFromAny(Tcl_Interp* interp, Tcl_Obj* objPtr);
	static void UpdateStringOfCursor(Tcl_Obj* propPtr);

static Tcl_ObjType mkCursorType = {
	"mkCursor", 				// name
	FreeCursorInternalRep,		// freeIntRepProc
	DupCursorInternalRep,		// dupIntRepProc
	UpdateStringOfCursor,		// updateStringProc
	SetCursorFromAny			// setFromAnyProc
};

///////////////////////////////////////////////////////////////////////////////
//
//	Cursors in Tcl are implemented as a pointer to an MkPath plus an index.
//	The MkPath reference count in incremented, they

	static MkPath& AsPath(Tcl_Obj* obj_)
	{
		// assert(obj_->typePtr = &mkCursorType);
		// assert(obj_->internalRep.twoPtrValue.ptr2 != 0);

		return *(MkPath*) obj_->internalRep.twoPtrValue.ptr2;
	}

	static int& AsIndex(Tcl_Obj* obj_)
	{
		// assert(obj_->typePtr = &mkCursorType);
		// assert(obj_->internalRep.twoPtrValue.ptr2 != 0);

		return (int&) obj_->internalRep.twoPtrValue.ptr1;
	}

static void 
FreeCursorInternalRep(Tcl_Obj* cursorPtr)
{
	AsPath(cursorPtr).Refs(-1);
}

static void
DupCursorInternalRep(Tcl_Obj* srcPtr, Tcl_Obj* copyPtr)
{
	AsPath(srcPtr).Refs(+1);

	copyPtr->internalRep = srcPtr->internalRep;
	copyPtr->typePtr = &mkCursorType;
}

static int
SetCursorFromAny(Tcl_Interp* interp, Tcl_Obj* objPtr) 
{
	Tcl_ObjType* oldTypePtr = objPtr->typePtr;

	const char* string = Tcl_GetStringFromObj(objPtr, 0);

	if (oldTypePtr && oldTypePtr->freeIntRepProc)
		oldTypePtr->freeIntRepProc(objPtr);

	objPtr->typePtr = &mkCursorType;
	objPtr->internalRep.twoPtrValue.ptr2 = work.AddPath(string);
	AsIndex(objPtr) = *string ? atoi(string) : -1;

	return TCL_OK;
}

static void
UpdateStringOfCursor(Tcl_Obj* cursorPtr)
{
	c4_String s = AsPath(cursorPtr)._path;

	int index = AsIndex(cursorPtr);
	if (index >= 0)
	{
		char buf [20];
		sprintf(buf, "%s%d", s.IsEmpty() ? "" : "!", index);
		s += buf;
	}

	cursorPtr->length = s.GetLength();
	cursorPtr->bytes = strcpy(Tcl_Alloc(cursorPtr->length + 1), s);
}

///////////////////////////////////////////////////////////////////////////////
// The Tcl class is an generic interface to Tcl, providing some C++ wrapping.

class Tcl
{
protected:
	Tcl_Interp* interp;

public:
	Tcl (Tcl_Interp* ip_)
		: interp (ip_)
	{
	}

	Tcl_Obj* tcl_GetObjResult()
	{
		return Tcl_GetObjResult(interp);
	}

	void tcl_SetObjResult(Tcl_Obj* obj_)
	{
		Tcl_SetObjResult(interp, obj_);
	}

	int tcl_ListObjLength(Tcl_Obj* obj_)
	{
		int result;
		if (Tcl_ListObjLength(interp, obj_, &result) != TCL_OK)
			throw "";
		return result;
	}

	Tcl_Obj* tcl_ListObjIndex(Tcl_Obj* obj_, int index_)
	{
		Tcl_Obj* result;
		if (Tcl_ListObjIndex(interp, obj_, index_, &result) != TCL_OK)
			throw "";
		return result;
	}

	void tcl_ListObjAppendElement(Tcl_Obj* obj_, Tcl_Obj* value_)
	{
		if (Tcl_ListObjAppendElement(interp, obj_, value_) != TCL_OK)
			throw "";
	}

	int tcl_GetIntFromObj(Tcl_Obj* obj_)
	{
		int value;
		if (Tcl_GetIntFromObj(interp, obj_, &value) != TCL_OK)
			throw "";
		return value;
	}

	long tcl_GetLongFromObj(Tcl_Obj* obj_)
	{
		long value;
		if (Tcl_GetLongFromObj(interp, obj_, &value) != TCL_OK)
			throw "";
		return value;
	}

	double tcl_GetDoubleFromObj(Tcl_Obj* obj_)
	{
		double value;
		if (Tcl_GetDoubleFromObj(interp, obj_, &value) != TCL_OK)
			throw "";
		return value;
	}
	
	int tcl_GetIndexFromObj(Tcl_Obj *obj_, char **table_, char *msg_ ="option")
	{
		int index;
		if (Tcl_GetIndexFromObj(interp, obj_, table_, msg_, 0, &index) != TCL_OK)
			index = -1;
		return index;
	}

	long tcl_ExprLongObj(Tcl_Obj *obj_)
	{
		long result;
		if (Tcl_ExprLongObj(interp, obj_, &result) != TCL_OK)
			throw "";
		return result;
	}

	Tcl_Obj* tcl_NewStringObj(const char* str_, int len_ =-1)
	{
		return Tcl_NewStringObj((char*) str_, len_);
	}
};

///////////////////////////////////////////////////////////////////////////////
// The MkTcl class adds MetaKit-specific utilities and all the command procs.

class MkTcl : public Tcl
{
	int id;
	int objc;
	Tcl_Obj* const* objv;
	c4_String msg;

	static int Dispatcher(ClientData cd, Tcl_Interp* ip, int oc, Tcl_Obj* const* ov)
	{
		MkTcl* self = (MkTcl*) cd;

		try
		{
			if (self == 0 || self->interp != ip)
				throw "Initialization error in dispatcher";

			self->Execute(oc, ov);
		}
		catch (int result)
		{
			return result;
		}
		catch (const char* msg)
		{
			if (*msg)
				Tcl_SetResult(ip, (char*) msg, TCL_VOLATILE);

			return TCL_ERROR;
		}

		return TCL_OK;
	}

	static void Cleaner(ClientData clientData)
	{
		delete (MkTcl*) clientData;
	}

public:

	MkTcl (Tcl_Interp* ip_, int id_, const char* cmd_)
		: Tcl (ip_), id (id_)
	{
		Tcl_RegisterObjType (&mkPropertyType);
		Tcl_RegisterObjType (&mkCursorType);

		Tcl_CreateObjCommand(ip_, (char*) cmd_, Dispatcher, this, Cleaner);
	}

	int asProperty(Tcl_Obj* obj_)
	{
		if (SetPropertyFromAny(interp, obj_) != TCL_OK)
			throw "";

		return AsPropId(obj_);
	}

	enum { kAnyRow, kExistingRow, kLimitRow, kExtendRow };

	c4_View asView(Tcl_Obj* obj_)
	{
		if (SetCursorFromAny(interp, obj_) != TCL_OK)
			throw "";
		
		return AsPath(obj_)._view;
	}

	c4_RowRef asRowRef(Tcl_Obj* obj_, int type_ =kExistingRow)
	{
		c4_View view = asView(obj_);
		int index = AsIndex(obj_);
		int size = view.GetSize();

		switch (type_)
		{
			case kExtendRow:	if (index >= size)
									view.SetSize(size = index + 1);
			case kLimitRow:		if (index > size)
									throw "view index is too large";
			case kExistingRow:	if (index < 0 || index >= size)
									throw "view index is out of range";
			case kAnyRow:		;
		}

		return view[index];
	}

	Tcl_Obj* GetValue(const c4_RowRef& row_, const c4_Property& prop_)
	{
		Tcl_Obj* value = Tcl_NewObj();

		switch (prop_.Type())
		{
			case 'S':
			case 'B':
			case 'M':
				{
					c4_Bytes temp;
					if (prop_ (row_).GetData(temp))
						Tcl_SetStringObj(value, (char*) temp.Contents(), temp.Size());
				}
				break;

			case 'F':
				Tcl_SetDoubleObj(value, ((c4_FloatProp&) prop_) (row_));
				break;

			case 'D':
				Tcl_SetDoubleObj(value, ((c4_DoubleProp&) prop_) (row_));
				break;

			case 'I':
				Tcl_SetLongObj(value, ((c4_IntProp&) prop_) (row_));
				break;

			case 'V':
				{
					c4_View view = ((c4_ViewProp&) prop_) (row_);
					Tcl_SetIntObj(value, view.GetSize());
				}
				break;

			default:	
				throw "unsupported property type";	//##// clean up value!
		}

		return value;
	}

	void GetCmd()
	{
		c4_RowRef row = asRowRef(objv[1], kExistingRow);

		Tcl_Obj* result = tcl_GetObjResult();

		if (objc < 3)
		{
			c4_View view = row.Container();
			for (int i = 0; i < view.NumProperties(); ++i)
			{
				c4_Property prop = view.NthProperty(i);
				c4_String name = prop.Name();

				switch (prop.Type())
				{
					case 'S':	break;
					case 'V':	continue; // omit subviews
					default:	name += ':';
								name += prop.Type();
				}

				tcl_ListObjAppendElement(result, tcl_NewStringObj(name));
				tcl_ListObjAppendElement(result, GetValue(row, prop));
			}
		}
		else
		{
			for (int i = 2; i < objc; ++i)
				tcl_ListObjAppendElement(result, GetValue(row, asProperty(objv[i])));
		}
	}

	void SetValues(const c4_RowRef& row_, int objc, Tcl_Obj* const* objv)
	{
		while (objc >= 2)
		{
			c4_Property prop (asProperty(objv[0]));

			switch (prop.Type())
			{
				case 'S':
				case 'B':
				case 'M':
					{
						int len;
						const char* ptr = Tcl_GetStringFromObj(objv[1], &len);
						prop (row_).SetData(c4_Bytes (ptr, len));
					}
					break;

				case 'F':
					((c4_FloatProp&) prop) (row_) = (float) tcl_GetDoubleFromObj(objv[1]);
					break;

				case 'D':
					((c4_DoubleProp&) prop) (row_) = tcl_GetDoubleFromObj(objv[1]);
					break;

				case 'I':
					((c4_IntProp&) prop) (row_) = tcl_GetLongFromObj(objv[1]);
					break;

				default:
					throw "unsupported property type";
			}

			objc -= 2;
			objv += 2;
		}
	}

	void SetCmd()
	{
		if (objc < 4)
		{
			GetCmd();
		}
		else
		{
			c4_RowRef row = asRowRef(objv[1], kExtendRow);
			SetValues(row, objc - 2, objv + 2);

			tcl_SetObjResult(objv[1]);
		}
	}

	void RowCmd()
	{
		static char* cmds [] =
		{
			"create",
			"append",
			"delete",
			"insert",
			"replace",
			0
		};

		int id = tcl_GetIndexFromObj(objv[1], cmds);
		if (id < 0)
		{
			id = 0;	// defaults to "create"
			--objv;
			++objc;
		}

		switch (id)
		{
			case 0:
				{
					c4_View view = asView(tcl_GetObjResult());
					// assert(AsIndex(var) == 0);

					SetValues(view[0], objc - 2, objv + 2);
				}
				break;

			case 1:
				{
					Tcl_Obj* var = Tcl_DuplicateObj(objv[2]);
					tcl_SetObjResult(var);
					
					c4_View view = asView(var);

					Tcl_InvalidateStringRep(var);
					AsIndex(var) = view.GetSize();

					SetValues(asRowRef(var, kExtendRow), objc - 3, objv + 3);
				}
				break;

			case 2:
				{
					c4_RowRef row = asRowRef(objv[2]);
					c4_View view = row.Container();
					int index = AsIndex(objv[2]);

					int count = objc > 3 ? tcl_GetIntFromObj(objv[3]) : 1;
					if (count > view.GetSize() - index)
						count = view.GetSize() - index;

					if (count >= 1)
						view.RemoveAt(index, count);
				}
				break;

			case 3:
				{
					c4_RowRef toRow = asRowRef(objv[2], kLimitRow);
					c4_View view = toRow.Container();
					int n = AsIndex(objv[2]);

					int count = objc > 3 ? tcl_GetIntFromObj(objv[3]) : 1;
					if (count >= 1)
					{
						c4_Row temp;
						view.InsertAt(n, temp, count);

						if (objc > 4)
						{
							c4_RowRef fromRow = asRowRef(objv[4]);
							while (--count >= 0)
								view[n++] = fromRow;
						}
					}
				}
				break;

			case 4:
				{
					c4_RowRef row = asRowRef(objv[2]);

					if (objc > 3)
						row = asRowRef(objv[3]);
					else
						row = c4_Row ();
				}
				break;
		}

		if (id > 1)
			tcl_SetObjResult(objv[2]);
	}

	void FileCmd()
	{
		static char* cmds [] =
		{
			"open",
			"close",
			"commit",
			"rollback",
			"load",
			"save",
			0
		};

		int id = tcl_GetIndexFromObj(objv[1], cmds);
		if (id < 0)
			throw "";

		const char* string = Tcl_GetStringFromObj(objv[2], 0);

		int n = work.Find(f4_GetToken(string));
		if (n < 0 && id > 0)
			throw "no storage with this name";

		switch (id)
		{
			case 0:
				{
					if (n >= 0)
						throw "file already open";

					const char* name = Tcl_GetStringFromObj(objv[2], 0);
					const char* file = Tcl_GetStringFromObj(objv[3], 0);
					bool readonly = objc > 4;

					n = work.Open(name, file, !readonly);
					if (n < 0)
						throw "file open failed";

					if (!readonly)
						work.Nth(n)->_storage.AutoCommit(); //##// for testing
				}
				break;
			
			case 1:
				{
					work.Close(n);
				}
				break;
			
			case 2:
				{
					work.Nth(n)->_storage.Commit();
				}
				break;

			case 3:
				{
					work.Nth(n)->_storage.Rollback();
				}
				break;

			case 4:
				{
					const char* file = Tcl_GetStringFromObj(objv[3], 0);

					FILE* fp = fopen(file, "rb");
					if (fp == 0)
						throw "file open failed";

					work.Nth(n)->_storage.LoadFromStream(fp);

					fclose(fp);
				}
				break;
			
			case 5:
				{
					const char* file = Tcl_GetStringFromObj(objv[3], 0);

					FILE* fp = fopen(file, "wb");
					if (fp == 0)
						throw "file open failed";

					work.Nth(n)->_storage.SaveToStream(fp);

					fclose(fp);
				}
				break;
		}

		tcl_SetObjResult(objv[2]);
	}

	void ViewCmd()
	{
		static char* cmds [] =
		{
			"layout",
			"delete",
			"size",
			"info",
			0
		};

		int id = tcl_GetIndexFromObj(objv[1], cmds);
		if (id < 0)
			throw "";

		switch (id)
		{
			case 0:
			case 1:
				{
					const char* string = Tcl_GetStringFromObj(objv[2], 0);

					int n = work.Find(f4_GetToken(string));
					if (n < 0 && id != 4)
						throw "no storage with this name";

					c4_String s = f4_GetToken(string);
					if (s.IsEmpty() || *string != 0)
						throw "unrecognized view name";

					if (id == 0)
					{
						const char* desc = Tcl_GetStringFromObj(objv[3], 0);
						if (desc && *desc)
							work.Nth(n)->_storage.GetAs(s + TclToKitDesc(desc));
					}
					else
					{
						work.Nth(n)->_storage.GetAs(s);
					}
				}
				break;

			case 2:
				{
					c4_View view = asView(objv[2]);

					if (objc > 3)
						view.SetSize(tcl_GetIntFromObj(objv[3]));

					Tcl_SetIntObj(tcl_GetObjResult(), view.GetSize());
				}
				break;

			case 3:
				{
					c4_View view = asView(objv[2]);
					Tcl_Obj* result = tcl_GetObjResult();

					for (int i = 0; i < view.NumProperties(); ++i)
					{
						c4_Property prop (view.NthProperty(i));

						c4_String s = prop.Name();
						if (prop.Type() != 'S')
						{
							s += ":";
							s += prop.Type();
						}

						tcl_ListObjAppendElement(result, tcl_NewStringObj(s));
					}
				}
				break;
		}

		if (id != 2)
			tcl_SetObjResult(objv[2]);
	}

	void LoopCmd()
	{
		if (objc >= 4 && Tcl_ObjSetVar2(interp, objv [1], 0, objv [2],
										TCL_LEAVE_ERR_MSG | TCL_PARSE_PART1) == 0)
				throw "";

		Tcl_Obj* var = Tcl_ObjGetVar2(interp, objv[1], 0,
										TCL_LEAVE_ERR_MSG | TCL_PARSE_PART1);
		if (var == 0)
			throw ""; // it has to exist, since it can't be valid otherwise

		c4_View view = asView(var);

		long first = 0;
		long limit = view.GetSize();
		long incr = 1;

		if (objc >= 5)
			first = tcl_ExprLongObj (objv [3]);

		if (objc >= 6)
			limit = tcl_ExprLongObj (objv [4]);

		if (objc >= 7)
			incr = tcl_ExprLongObj (objv [5]);

		Tcl_Obj* cmd = objv[objc-1];

			// about to modify, so make sure we are sole owners
		Tcl_Obj* original = 0;
		if (Tcl_IsShared(var))
		{
			original = var;
			var = Tcl_DuplicateObj(var);
		}

//		Tcl_IncrRefCount(var);

		int result = TCL_OK;

		for (int i = first; (i < limit && incr >= 0) || (i >  limit && incr < 0); i += incr)
		{
			Tcl_InvalidateStringRep(var);
			AsIndex(var) = i;

			var = Tcl_ObjSetVar2(interp, objv [1], 0, var,
									TCL_LEAVE_ERR_MSG | TCL_PARSE_PART1);
			if (var == 0)
			{
				if (original != NULL)
					Tcl_DecrRefCount(original);
				throw "";
			}

			result = Tcl_EvalObj(interp, cmd);

			if (result == TCL_BREAK || result == TCL_RETURN)
				break;

			if (result == TCL_ERROR)
			{
				char buf [64];
				sprintf (buf, "\n    (\"mk loop\" body line %d)", interp->errorLine);
				Tcl_AppendStringsToObj(tcl_GetObjResult(), buf, 0);
				break;
			}
		}

		if (Tcl_ObjSetVar2(interp, objv [1], 0, var,
								TCL_LEAVE_ERR_MSG | TCL_PARSE_PART1) == 0)
		{
			if (original != NULL)
				Tcl_DecrRefCount(original);
			throw "";
		}

//		Tcl_DecrRefCount(var);

		if (result == TCL_ERROR)
			throw "";

		if (result == TCL_RETURN)
			throw result;
	}

	void CursorCmd()
	{
		static char* cmds [] =
		{
			"create",
			"position",
			"incr",
			0
		};

		int id = tcl_GetIndexFromObj(objv[1], cmds);
		if (id < 0)
		{
			id = 0;	// defaults to "create"
			--objv;
			++objc;
		}

		Tcl_Obj* var = 0;

		if (id == 0)
		{
			var = Tcl_NewObj();
			asView(var).SetSize(1);
		}
		else	// alter an existing cursor
		{	
			var = Tcl_ObjGetVar2(interp, objv[2], 0,
										TCL_LEAVE_ERR_MSG | TCL_PARSE_PART1);
			if (var == 0)
				throw ""; // it has to exist, since it can't be valid otherwise
		}

			// about to modify, so make sure we are sole owners
		Tcl_Obj* original = 0;
		if (Tcl_IsShared(var))
		{
			original = var;
			var = Tcl_DuplicateObj(var);
		}

		c4_View view = asView(var);

		int value;
		if (objc <= 3)
		{
			if (id == 1) // position without value returns current value
			{
				Tcl_SetIntObj(tcl_GetObjResult(), AsIndex(var));
				return;
			}

			value = id == 0 ? 0 : 1; // create defaults to 0, incr defaults to 1
		}
		else if (Tcl_GetIntFromObj(interp, objv[3], &value) != TCL_OK)
		{
			const char* step = Tcl_GetStringFromObj(objv[3], 0);
			if (strcmp(step, "end") == 0)
				value = view.GetSize() - 1;
			else
			{
				if (original)
					Tcl_DecrRefCount(original);
				throw "";
			}
			
			Tcl_ResetResult(interp);
		}

		if (id < 2)
			AsIndex(var) = value;
		else
			AsIndex(var) += value;

		Tcl_InvalidateStringRep(var);
		Tcl_IncrRefCount(var);

		Tcl_Obj* result = Tcl_ObjSetVar2(interp, objv[2], 0, var,
										TCL_LEAVE_ERR_MSG | TCL_PARSE_PART1);
		if (result == 0)
			throw "";

		tcl_SetObjResult(result);

		Tcl_DecrRefCount(var);
	}

	void Execute(int oc, Tcl_Obj* const* ov)
	{
		struct CmdDef
		{
			void (MkTcl::*proc)();
			int min;
			int max;
			const char* desc;
		};

		static CmdDef defTab [] =
		{
			{ GetCmd,		2,	0,	"get cursor ?prop ...?" },
			{ SetCmd,		3,	0,	"set cursor prop ?value prop value ...?" },
			{ CursorCmd,	2,	4,	"cursor option cursorname ?value?" },
			{ RowCmd,		3,	0,	"row option cursor ?...?" },
			{ ViewCmd,		3,	4,	"view option view ?arg?" },
			{ FileCmd,		3,	5,	"file option tag ?...?" },
			{ LoopCmd,		3,	7,	"loop cursorname ?path first limit incr? {command}" },
			{ 0,			0,	0,	0 },
		};

		CmdDef& cd = defTab[id];

		objc = oc;
		objv = ov;

		if (oc < cd.min || (cd.max > 0 && oc > cd.max))
		{
			msg = "wrong # args: should be \"mk::";
			msg += cd.desc;
			msg += "\"";

			throw (const char*) msg;
		}

		(this->*cd.proc)();
	}
};

///////////////////////////////////////////////////////////////////////////////

static int
Mktcl_Cmds(Tcl_Interp* interp, bool safe)
{
		// this list must match the "CmdDef defTab []" above.
	static char* cmds [] =
	{
		"get",
		"set",
		"cursor",
		"row",
		"view",
		"file",
		"loop",
		0
	};

	c4_String prefix = "mk::";

	for (int i = 0; cmds[i]; ++i)
		new MkTcl (interp, i, prefix + cmds[i]);	// registers itself

	return Tcl_PkgProvide(interp, "mk", "0.2");
}

///////////////////////////////////////////////////////////////////////////////

extern "C" EXPORT(int) Mk_Init(Tcl_Interp* interp)
{
	return Mktcl_Cmds(interp, false);
}

extern "C" EXPORT(int) Mk_SafeInit(Tcl_Interp* interp)
{
	return Mktcl_Cmds(interp, true);
}

///////////////////////////////////////////////////////////////////////////////
