################################################################################
# asedmain.tcl -
#
# Copyright 1999-2000 by Andreas Sievers
#
# andreas.sievers@t-online.de
################################################################################

################################################################################
#proc delete_id
#called when a editorwindow is closed to delete the correspondend undo_id
################################################################################
proc delete_id {} {
    global undo_id
    delete textUndoer $Editor::current(undo_id)
    return
}

namespace eval Editor {
    variable initDone 0
    variable _wfont
    variable notebook
    variable list_notebook
    variable con_notebook
    variable pw1
    variable pw2
    variable procWindow
    variable markWindow
    variable mainframe
    variable status
    variable prgtext
    variable prgindic
    variable font
    variable font_name
    variable Font_var
    variable FontSize_var
    variable toolbar1  1
    variable toolbar2  0
    variable showConsoleWindow 1
    variable sortProcs 1
    variable showProc 1
    variable checkNestedProc 1
    variable showProcWindow 1
    variable search_var
    variable search_combo
    variable current
    variable last
    variable text_win
    variable index_counter 0
    variable index
    variable slaves
    variable startTime [clock seconds]
    variable options
    variable lineNo
    variable lineEntryCombo
    variable toolbarButtons
    variable searchResults
    variable procMarks
    variable treeMenu
    variable textMenu
    variable serverUp 0
}

proc Editor::tick {} {
    global clock_var
    variable mainframe
    variable startTime
    set seconds [expr [clock seconds]  - $startTime]
    set clock_var [clock format $seconds -format %H:%M:%S -gmt 1]
    after 1000 Editor::tick
}

proc Editor::aboutBox {} {\
    tk_messageBox -message \
            "      ASED Tcl/Tk-IDE 2.0.0
    (c) Andreas Sievers, 1999-2000\n\n\
            Parts of this Editor are based upon the\n\
            BWidget Toolkit Demo from UNIFIX and
    Tcldev from Alexey Kakunin\n\
            Special Thanks to Carsten Zerbst who provided
    the basics of the Code Browser Window and iTcl-support!
    email:andreas.sievers@t-online.de" \
            -type ok \
            -title About \
            -icon info
}

proc Editor::getWindowPositions {} {
    global EditorData
    variable pw1
    variable pw2
    variable notebook
    variable list_notebook
    variable con_notebook
    variable current
    
    update idletasks
    
    
    set EditorData(options,notebookWidth) [winfo width $notebook]
    set EditorData(options,notebookHeight) [winfo height $notebook]
    set EditorData(options,list_notebookWidth) [winfo width $list_notebook]
    set EditorData(options,list_notebookHeight) [winfo height $list_notebook]
    set EditorData(options,con_notebookWidth) [winfo width $con_notebook]
    set EditorData(options,con_notebookHeight) [winfo height $con_notebook]
    #    get position of mainWindow
    set EditorData(options,mainWinSize) [wm geom .]
}

proc Editor::restoreWindowPositions {} {
    global EditorData
    variable pw1
    variable pw2
    variable notebook
    variable list_notebook
    variable con_notebook
    
    
    
    $notebook configure -width $EditorData(options,notebookWidth)
    $notebook configure -height $EditorData(options,notebookHeight)
    $list_notebook configure -width $EditorData(options,list_notebookWidth)
    $list_notebook configure -height $EditorData(options,list_notebookHeight)
    $con_notebook configure -width $EditorData(options,con_notebookWidth)
    $con_notebook configure -height $EditorData(options,con_notebookHeight)
    showConsole $EditorData(options,showConsole)
    showProcWin $EditorData(options,showProcs)
}



################################################################################
#proc Editor::saveOptions
#called when ASED exits
#saves only "EditorData(options,...)"
#but might be easily enhanced with other options
################################################################################
proc Editor::saveOptions {} {
    global EditorData
    global ASEDsRootDir
    
    Editor::getWindowPositions
    
    set configData "#ASED Configuration File\n\n"
    set configFile [file join "~" ased.cfg]
    set optionlist [array names EditorData]
    foreach option $optionlist {
        set optiontext EditorData($option)
        if {![string match "*(options*" $optiontext]} {
            continue
        }
        set value \"[subst \$[subst {$optiontext}]]\"
        append configData "set EditorData($option) $value\n"
        
    }
    set result [Editor::_saveFile $configFile $configData]
}

proc Editor::CreateFonts {} {
    global EditorData
    
    variable configError
    variable Font_var
    variable FontSize_var
    
    # set editor font
    if {$configError} {
        
        set font [list -family Courier -size 10 -weight normal -slant roman -underline 0 -overstrike 0]
        
        set EditorData(options,fonts,editorFont) $font
    }
    eval font create editorFont $EditorData(options,fonts,editorFont)
    # set comment font
    if {$configError} {
        
        set font [list -family Courier -size 10 -weight normal -slant italic -underline 0 -overstrike 0]
        
        set EditorData(options,fonts,commentFont) $font
    }
    eval font create commentFont $EditorData(options,fonts,commentFont)
    
    # set keyword font
    if {$configError} {
        
        set font [list -family Courier -size 10 -weight bold -slant roman -underline 0 -overstrike 0]
        set EditorData(options,fonts,keywordFont) $font
    }
    eval font create keywordFont $EditorData(options,fonts,keywordFont)
    set Font_var [font configure editorFont -family]
    return
}

proc Editor::setDefault {} {
    global tcl_platform
    global EditorData
    global ASEDsRootDir
    variable current
    variable configError
    variable toolbar1
    variable toolbar2
    variable showConsoleWindow
    variable showProcWindow
    variable sortProcs
    variable showProc
    variable checkNestedProc
    variable options
    variable lineNo
    
    
    set current(is_procline) 0; #is needed for checkProcs
    set current(procLine) 0
    set current(autoUpdate) 1
    set current(procSelectionChanged) 0
    set current(procListHistoryPos) 0
    set current(procListHistory) [list "mark1"]
    set current(procList_hist_maxLength) 20
    set current(lastPos) "1.0"
    set current(isNode) 0
    set current(checkRootNode) 0
    set current(isUpdate) 0
    
    
    # set keywords
    set fd [open [file join $ASEDsRootDir keywords.txt ] r]
    set keyList ""
    while {![eof $fd]} {
        gets $fd word
        lappend keyList $word
    }
    close $fd
    set EditorData(keywords) $keyList
    set EditorData(files) {}
    set EditorData(curFile) ""
    set EditorData(find) {}
    set EditorData(find,lastOptions) ""
    set EditorData(replace) {}
    set EditorData(projectFile) ""
    set EditorData(cursor,line) 1
    set EditorData(cursor,pos) 0
    set EditorData(cursorPos) "Line: 1   Pos: 0"
    set Editor::lineNo $EditorData(cursor,line)
    
    if $configError {
        if {$tcl_platform(platform) == "windows"} {
            set EditorData(options,fontSize) 15
            set EditorData(options,fontEncoding) ""
        } else {
            set EditorData(options,fontSize) 12
            set EditorData(options,fontEncoding) "koi8-r"
        }
        
        
        font create nbFrameFont -size 8
        
        set EditorData(options,useEvalServer) 0
        set options(useEvalServer) 0
        set EditorData(options,serverPort) 9001
        set options(serverPort) 9001
        set EditorData(options,serverWish) [info nameofexecutable]
        set options(serverPort) [info nameofexecutable]
        set EditorData(options,useIndent) 1
        set options(useIndent) 1
        set EditorData(options,useSintaxIndent) 1
        set options(useSI) 1
        set EditorData(options,indentSize) 4
        set EditorData(options,changeTabs) 1
        set options(changeTabs) 1
        set EditorData(options,tabSize) 4
        set EditorData(options,useHL) 1
        set options(useHL) 1
        set EditorData(options,useTemplates) 1
        set options(useTemplates) 1
        set EditorData(options,useTemplatesForKeywords) 1
        set options(useKeywordTemplates) 1
        set EditorData(options,useTemplatesForBrace) 1
        set options(useBracesTemplates) 1
        set EditorData(options,useTemplatesForParen) 1
        set options(useParenTemplates) 1
        set EditorData(options,useTemplatesForBracket) 1
        set options(useBracketTemplates) 1
        set EditorData(options,useTemplatesForQuoteDbl) 1
        set options(useQuotesTemplates) 1
        set EditorData(options,showToolbar1) $toolbar1
        set EditorData(options,showToolbar2) $toolbar2
        set options(showConsole) 1
        set EditorData(options,showConsole) $showConsoleWindow
        set options(showProcs) 1
        set EditorData(options,showProcs) $showProcWindow
        set options(sortProcs) 1
        set EditorData(options,sortProcs) $sortProcs
        set options(autoUpdate) 1
        set EditorData(options,autoUpdate) 1
        set options(showProc) 1
        set EditorData(options,showProc) $showProc
        set options(defaultProjectFile) "none"
        set EditorData(options,defaultProjectFile) "none"
        set current(project) "none"
        set options(workingDir) "~"
        set EditorData(options,workingDir) "~"
        set options(showSolelyConsole) 0
        set EditorData(options,showSolelyConsole) 0
    } else  {
        set toolbar1 $EditorData(options,showToolbar1)
        set toolbar2 $EditorData(options,showToolbar2)
        set options(useEvalServer) $EditorData(options,useEvalServer)
        set options(serverPort) $EditorData(options,serverPort)
        set options(useIndent) $EditorData(options,useIndent)
        set options(useSI) $EditorData(options,useSintaxIndent)
        set options(useHL) $EditorData(options,useHL)
        set options(useTemplates) $EditorData(options,useTemplates)
        set options(useKeywordTemplates) $EditorData(options,useTemplatesForKeywords)
        set options(useBracesTemplates) $EditorData(options,useTemplatesForBrace)
        set options(useParenTemplates) $EditorData(options,useTemplatesForParen)
        set options(useBracketTemplates) $EditorData(options,useTemplatesForBracket)
        set options(useQuotesTemplates) $EditorData(options,useTemplatesForQuoteDbl)
        set options(changeTabs) $EditorData(options,changeTabs)
        set options(showConsole) $EditorData(options,showConsole)
        set options(showProcs) $EditorData(options,showProcs)
        set options(showProc) $EditorData(options,showProc)
        set options(autoUpdate) $EditorData(options,autoUpdate)
        set options(sortProcs) $EditorData(options,sortProcs)
        set options(defaultProjectFile) $EditorData(options,defaultProjectFile)
        set current(project) $EditorData(options,defaultProjectFile)
        set options(workingDir) $EditorData(options,workingDir)
        set options(showSolelyConsole) $EditorData(options,showSolelyConsole)
        set options(serverWish) $EditorData(options,serverWish)
    }
    set EditorData(indentString) "    "
    Editor::CreateFonts
    return
}

proc Editor::newFile {{force 0}} {
    variable notebook
    variable current
    global EditorData
    set pages [NoteBook::pages $notebook]
    if {([llength $pages] > 0) && ($force == 0)} {\
        if {[info exists current(text)]} {\
            set f0 [NoteBook::raise $notebook]
            set text [NoteBook::itemcget $notebook $f0 -text]
            set data [$current(text) get 1.0 end-1c]
            if {($data == "") && ($text == "Untitled")} {return}
        }
    }
    set temp $current(hasChanged)
    set f0 [EditManager::create_text $notebook "Untitled"]
    set Editor::text_win($Editor::index_counter,undo_id) [new textUndoer [lindex $f0 2]]
    set current(hasChanged) $temp
    NoteBook::raise $notebook [lindex $f0 1]
    set current(hasChanged) 0
    $Editor::mainframe setmenustate noFile normal
    updateObjects
}


proc Editor::scanLine {} {
    variable current
    
    # is current line a proc-line?
    set result [$current(text) search "proc " "insert linestart" "insert lineend"]
    if {$result == ""} {
        # this is not a proc-line
        # was it a proc-line?
        if {$current(is_procline)} {
            set current(is_procline) 0
            set current(procSelectionChanged) 1
        } else {
            set current(is_procline) 0
            set current(procSelectionChanged) 0
        }
    } else  {
        # is current line really a proc-line?
        set line [$current(text) get "$result linestart" "$result lineend"]
        set temp [string trim $line \ \t\;]
        set proc ""
        set procName ""
        # is it really a proc-line?
        if {[scan $temp %\[proc\]%s proc procName] != 2} {
            set result ""
        } elseif {$proc != "proc"} {
            set result ""
        }
        if {$result != ""} {
            if {$current(procName) != $procName} {
                set current(procName) $procName
                set current(procSelectionChanged) 1
                set current(is_procline) 1
            } else  {
                set current(procSelectionChanged) 0
            }
        } else  {
            if {$current(is_procline)} {
                set current(is_procline) 0
                set current(procSelectionChanged) 1
            } else {
                set current(is_procline) 0
                set current(procSelectionChanged) 0
            }
        }
    }
    return $result
}

proc Editor::updateOnIdle {range} {
    variable current
    # if theres a pending update only store new range
    if {$current(isUpdate)} {
        if {[$current(text) compare $current(updateStart) > [lindex $range 0]]} {
            set current(updateStart) [$current(text) index [lindex $range 0]]
        }
        if {[$current(text) compare $current(updateEnd) < [lindex $range 1]]} {
            set current(updateEnd) [$current(text) index [lindex $range 1]]
        }
    } else  {
        set current(isUpdate) 1
        set current(updateStart) [$current(text) index [lindex $range 0]]
        set current(updateEnd) [$current(text) index [lindex $range 1]]
        after idle {
            # wait for a longer idle period
            for {set i 0} {$i <= 10000} {incr i} {
                update
                set Editor::current(idleID) [after idle {
                    update
                    after idle {set Editor::current(idleID) ""}
                }]
                vwait Editor::current(idleID)
                if {$i == 100} {
                    set range [editorWindows::deleteMarks $Editor::current(updateStart) $Editor::current(updateEnd) ]
                    Editor::updateObjects $range
                    Editor::selectObject 0
                    set Editor::current(isUpdate) 0
                    break
                }
            }
        }
    }
}


################################################################################
#
#  proc Editor::updateObjects
#
#  reparse the complete file and rebuild object tree
#
# zerbst@tu-harburg.d
#
# adopted for ASED by A.Sievers, dated 02/06/00
#
################################################################################

proc Editor::updateObjects {{range {}}} {
    global EditorData
    variable current
    variable treeWindow
    
    if {!$EditorData(options,autoUpdate) || !$EditorData(options,showProcs)} {
        return
    }
    while {[llength $range] == 1} {
        eval set range $range
    }
    
    if {$range == {}} {
        # switch on progressbar
        set Editor::prgindic 0
        set current(checkRootNode) 0
        set start 1.0
        set end "end-1c"
        catch {
            editorWindows::deleteMarks "1.0" "end -1c"
        }
        Editor::tdelNode $current(file)
    } else  {
        set current(checkRootNode) 1
    }
    set code {
        set temp [expr int($nend / $end * 100)]
        if {!$recursion && $temp > $Editor::prgindic && [expr $temp % 10] == 0 } {
            set Editor::prgindic [expr int($nend / $end * 100)]
            set Editor::status "Parsing: $Editor::prgindic % "
            update idletasks
        }
    }
    
    # call parser
    set nodeList [Parser::parseCode $current(file) $current(text) $range $code]
    # switch off progressbar
    set Editor::prgindic -1
    set Editor::status ""
    
    foreach node $nodeList {
        catch {Editor::tnewNode $node}
    }
    update
    if {$Editor::options(sortProcs)} {catch {Editor::torder $current(file)}}
}

################################################################################
#
#  proc Editor::selectObject
#
#  selects an object by a given position in the text
#
# zerbst@tu-harburg.d
#
# adopted for ASED by A.Sievers, dated 02/06/00
#
################################################################################

proc Editor::selectObject {{update 1} {Idx insert}} {
    global EditorData
    variable current
    variable treeWindow
    variable procMarks

    if {!$EditorData(options,showProcs) || !$EditorData(options,showProc)} {
        set current(node) ""
        return ""
    }
    if {$update != 0} {
        set rexp {^(( |\t|\;)*((namespace )|(class )|(proc )|(body )|(configbody )))|((( |\t|\;)*[^\#]*)((method )|(constructor )|(destructor )))}
        if {[regexp $rexp [$current(text) get "$Idx linestart" "$Idx lineend"]]} {
            set start [$current(text) index "$Idx"]
            set end [$current(text) index "$Idx"]
            set range [editorWindows::deleteMarks $start $end]
            updateObjects $range
            set current(isNode) 1
        } else  {
            set current(isNode) 0
        }
    }
    set index [$current(text) index $Idx]
    # marknames equal nodenames
    set node $Idx
    set markList [array names procMarks]
    #get the right mark
    while {[lsearch $markList $node] == -1 || $procMarks($node) == "dummy"} {
        set index [$current(text) index $node]
        set result -1
        foreach { type node idx} [$current(text) dump -mark $index] {
            set result [lsearch $markList $node]
            if {$result != -1} {
                if {$procMarks([lindex $markList $result]) != "dummy"} {
                    break
                } else  {
                    set result -1
                }
            }
        }
        if {$result == -1 && $index != 1.0} {
            set node [$current(text) mark previous $index]
            if {$node == ""} {
                break
            }
        } elseif {$result == -1} {
            set node ""
            break
        }
    }
    if {$node == ""} {
        $treeWindow selection clear
        set current(node) $node
        return $node
    }
    #if it is an end_of_proc mark skip this proc
    if {[string match "*_end_of_proc" $node]} {
        set count -2
        while {$count != 0} {
            set node [$current(text) index $node]
            set node [$current(text) mark previous "$node -1c"]
            if {$node == ""} {
                break
            }
            while {[lsearch $markList $node] == -1 || $procMarks($node) == "dummy"} {
                set index [$current(text) index $node]
                foreach { type node idx} [$current(text) dump -mark $index] {
                    set result [lsearch $markList $node]
                    if {$result != -1} {
                        if {$procMarks($node) != "dummy"} {
                            break
                        } else  {
                            set result -1
                        }
                    }
                }
                if {$result == -1 && $index != 1.0} {
                    set node [$current(text) mark previous $index]
                    if {$node == ""} {
                        break
                    }
                } elseif {$result == -1}  {
                    set node ""
                    break
                }
            }
            if {$node == ""} {
                break
            }
            if {[string match "*_end_of_proc" $node]} {
                incr count -1
            } else {
                incr count
            }
        }
    }
    $treeWindow selection clear
    if {$node != ""} {
        $treeWindow selection set $node
        $treeWindow see $node
    }
    set current(node) $node
    return $node
}

################################################################################
#
# Gui components of the treewidget
#
#
#
################################################################################

################################################################################
#
#  proc Editor::tdelNode
#
#  deletes a node and its children from the tree
#
# zerbst@tu-harburg.d
################################################################################
proc Editor::tdelNode {node} {
    variable treeWindow
    
    regsub -all " " $node \306 node
    regsub ":$" $node \327 node
    regsub -all "\\\$" $node "" node
    $treeWindow delete $node
}


################################################################################
#
#  proc Editor::tnewNode
#
#  inserts a new node into the tree. Gets a string representation of
#  the namspace/class/method/proc name and the type of object
#
# zerbst@tu-harburg.d
#
# adopted for ASED by A.Sievers, dated 02/06/00
#
################################################################################
proc Editor::tnewNode {nodedata} {
    variable current
    variable treeWindow
    set instanceNo 0
    
    set node [lindex $nodedata 0]
    set type [lindex $nodedata 1]
    set startIndex [lindex $nodedata 2]
    set endIndex [lindex $nodedata 3]
    
    # mask spaces in the node name
    regsub -all " " $node \306 node
    # mask ending single : in node name
    regsub ":$" $node \327 node
    # mask "$" in nodename
    regsub -all "\\\$" $node "" node
    # mask instance number
    regsub "\367.+\376$" $node "" node
    
    if {[string index $node [expr [string length $node] -1]] == "#"} {
        append node "{}"
    }
    #check current namespace in normal editing mode
    if {$current(checkRootNode) != 0} {
        # if node doesnt present a qualified name,
        # which presents its rootnode by itself (e.g. test::test)
        # try to set its rootnode
        # use regsub to count qualifiers (# in nodes instead of ::)
        if {[regsub -all -- {#} $node "" dummy] > 1} {
            # do nothing
        } else  {
            set rootNode [selectObject 0 "insert linestart -1c"]
            if {$rootNode != ""} {
                set name [string range $node [expr [string last \# $node]+1] end]
                if {$name == ""} {
                    set name $node
                }
                set node "$rootNode\#$name"
            }
        }
    }
    
    set rootnode [string range $node 0 [expr [string last \# $node] -1]]
    set name [string range $node [expr [string last \# $node]+1] end]
    
    # get rid of the  in the node
    regsub -all \306 $name " " name
    regsub \327 $name ":" name
    regsub -all "" $name "\$" name
    if {$name == ""} {
        set name $node
    }
    
    #Does the rootnode exist ? Otherwise call tnewNode recursively
    if {![string match $type file]} {
        if {![$treeWindow exists $rootnode]} {
            tnewNode [list [list $rootnode] dummy $startIndex $endIndex]
        }
    }
    # Does node exist ? Then append an instance counter
    while {[$treeWindow exists $node]} {
        regsub "\367.+\376$" $node "" node
        # append instance number"
        incr instanceNo
        append node \367$instanceNo\367
    }
    switch $type {
        "file" {
            $treeWindow insert end root $node -text $name \
                    -open 1 -data $type -image [Bitmap::get openfold]
        }
        
        "code" {
            $treeWindow insert end $rootnode $node -text $name \
                    -open 1 -data $type  -image [Bitmap::get oplink]
            if {$name == "<Top>"} {
                $treeWindow itemconfigure $node -image [Bitmap::get top]
            } elseif {$name == "<Bottom>"}  {
                $treeWindow itemconfigure $node -image [Bitmap::get bottom]
            } else  {
                $treeWindow itemconfigure $node -image [Bitmap::get qmark]
            }
        }
        
        "namespace" -
        "class" {
            $treeWindow insert end  $rootnode $node  -text "$type: $name" \
                    -open 1 -data $type -image [Bitmap::get openfold]
        }
        "dummy" {
            $treeWindow insert end  $rootnode $node  -text "namespace: $name" \
                    -open 1 -data $type -image [Bitmap::get openfold]
        }
        "proc" {
            $treeWindow insert end  $rootnode $node  -text "$type: $name" \
                    -open 1 -data $type -image [Bitmap::get file]
        }
        "method" {
            $treeWindow insert end  $rootnode $node  -text "$type: $name" \
                    -open 1 -data $type -image [Bitmap::get new]
        }
        "forward" {
            $treeWindow insert end  $rootnode $node  -text "$name" \
                    -open 1 -data $type -image [Bitmap::get oplink]
        }
        "body" {
            $treeWindow insert end  $rootnode $node  -text "$name" \
                    -open 1 -data $type -image [Bitmap::get file]
        }
        "configbody" {
            $treeWindow insert end  $rootnode $node  -text "$type: $name" \
                    -open 1 -data $type -image [Bitmap::get file]
        }
        
        "constructor" -
        "destructor" {
            $treeWindow insert end  $rootnode $node  -text "$type" \
                    -open 1 -data $type -image  [Bitmap::get new]
        }
        
        
        default {puts "Oops $nodedata"}
    }
    switch -- $name {
        "<Top>" -
        "<Bottom>" {
            set end_of_proc_name $node
            append end_of_proc_name "_end_of_proc"
            $Editor::current(text) mark set $node $startIndex
            if {$name == "<Top>"} {
                $Editor::current(text) mark gravity $node left
                $Editor::current(text) mark gravity $end_of_proc_name left
            }
            $Editor::current(text) mark set $end_of_proc_name $endIndex
            return $node
        }
        "file" {return ""}
        
        default {
            set Editor::procMarks($node) $type
            set end_of_proc_name $node
            append end_of_proc_name "_end_of_proc"
            set Editor::procMarks($end_of_proc_name) $type
            $Editor::current(text) mark set $node $startIndex
            $Editor::current(text) mark set $end_of_proc_name $endIndex
            $Editor::current(text) mark gravity $end_of_proc_name left
            return $node
        }
    }
}

################################################################################
#
#  proc Editor::tgetData
#
#  gets the data for a given node
#
# zerbst@tu-harburg.d
################################################################################
proc Editor::tgetData {node} {
    variable treeWindow
    
    
    if {[catch {$treeWindow itemcget $node -data} data]} {
        set data ""
    }
    return $data
}


################################################################################
#
#  proc Editor::tmoddir
#
#  needed to open / close a node in the tree. Gets open/close in $idx and
#  the name of the node in $node
#
# zerbst@tu-harburg.d
################################################################################

proc Editor::tmoddir { idx node } {
    variable treeWindow
    
    set data [$treeWindow itemcget $node -data]
    set type [lindex $data 0]
    if $idx {
        #Opening
        switch $type {
            "namespace" -
            "class" {
                if { [llength [$treeWindow nodes $node]] } {
                    set img openfold
                } else  {
                    set img folder
                }
            }
            "code" {
                set name [lindex $data 1]
                if {$name == "<Top>"} {
                    set img top
                } elseif {$name == "<Bottom>"} {
                    set img bottom
                } else  {
                    set img qmark
                }
            }
            "forward" {set img oplink}
            "proc" -
            "method" { set img file}
            "configbody" -
            "body" {set img file}
            "constructor" -
            "destructor" {set img new}
            default {set img openfold}
        }
    } else  {
        #Closing
        switch $type {
            "Code" -
            "namespace" -
            "class" { set img folder}
            default { set img folder}
        }
    }
    
    $treeWindow itemconfigure $node -image [Bitmap::get $img]
    
}


################################################################################
#
#  proc Editor::topen
#
#  opens the complete tree
#
# zerbst@tu-harburg.d
################################################################################
proc Editor::topen {} {
    variable treeWindow
    variable current
    
    regsub -all " " $current(file) \306 node
    regsub ":$" $node \327 node
    regsub -all "\\\$" $node "" node
    $treeWindow opentree $node
}

################################################################################
#
#  proc Editor::tclose
#
#  closes the complete tree
#
# zerbst@tu-harburg.d
################################################################################
proc Editor::tclose {} {
    variable treeWindow
    variable current
    
    regsub -all -- " " $current(file) /306 node
    regsub ":$" $current(file) \327 node
    regsub -all "\\\$" $node "" node
    $treeWindow closetree $node
}
################################################################################
#
#  proc Editor::tselectObject
#
#  selects the objects choosen from the tree
#
# zerbst@tu-harburg.d
################################################################################

proc Editor::tselectObject {node} {
    variable current
    variable treeWindow
    variable notebook
    variable procMarks
    
    Editor::procList_history_add
    $treeWindow selection clear
    $treeWindow selection set $node
    
    #Switch to the right notebook
    set filename [lindex [split $node \#] 0]
    catch {$notebook raise [set ::Editor::text_win($n)]}
    if {[catch {$current(text) mark set insert $node}]} {
        return
    }
    $current(text) see insert
    focus $current(text)
    editorWindows::flashLine
    Editor::selectObject 0
    editorWindows::ReadCursor
    Editor::procList_history_add
    set current(lastPos) [$current(text) index insert]
    return
}


################################################################################
#
#  proc Editor::tsee
#
#  show the now
#
# zerbst@tu-harburg.d
################################################################################
proc Editor::tsee {node} {
    variable treeWindow
    
    $treeWindow see $node
}


################################################################################
#
#  proc Editor::torder
#
#  order the tree by types and alphabet
#
#  zerbst@tu-harburg.de
#
#  rewritten by A.Sievers 02/06/00
# 
################################################################################
proc Editor::torder {node} {
    variable treeWindow
    variable current
    
    regsub -all " " $node \306 node
    regsub ":$" $node \327 node
    regsub -all "\\\$" $node "" node
    
    proc sortTree {node} {
        variable treeWindow
        set children [$treeWindow nodes $node]
        if {[llength $children] == 0} {
            return
        }
        set tempList ""
        foreach child $children {
            sortTree $child
            set childText [$treeWindow itemcget $child -text]
            if {$childText == "<Top>" || $childText == "<Bottom>"} {
                continue
            }
            lappend tempList "$childText\#$child"
        }
        set sortedList ""
        set tempList [lsort -dictionary $tempList]
        
        foreach childNode $tempList {
            set nodeName [string range $childNode [expr [string first \# $childNode]+1] end]
            lappend sortedList $nodeName
        }
        $treeWindow reorder $node $sortedList
        return
    }
    
    proc realorderTree {node} {
        variable treeWindow
        variable current
        
        set children [$treeWindow nodes $node]
        if {[llength $children] == 0} {
            return
        }
        set indexList {}
        foreach child $children {
            set childText [$treeWindow itemcget $child -text]
            if {$childText == "<Top>" || $childText == "<Bottom>"} {
                continue
            }
            realorderTree $child
            set index [$current(text) index $child]
            set newItem $index
            lappend newItem $child
            lappend indexList $newItem
        }
        #now we have a list of children with "index name"
        set itemList [lsort -dictionary $indexList]
        set realorderList {}
        foreach item $itemList {
            lappend realorderList [lindex $item 1]
        }
        #now we have a realorderList for child
        $treeWindow reorder $node $realorderList
        return
    }
    
    if {$Editor::options(sortProcs)} {
        sortTree $node
    } else {
        realorderTree $node
    }
}

################################################################################
#proc Editor::evalMain
#will be called from within a slave-interpreter to do something within the main-interpreter
################################################################################
proc Editor::evalMain {args} {
    
    uplevel #0 eval $args
}

proc Editor::setTestTermBinding {sock terminal} {
    variable current
    set current(sock) $sock
    bind $terminal <KeyPress-Return> {%W mark set insert "prompt lineend"}
    bind $terminal <KeyRelease-Return> {
        set command [getCommand %W]
        %W tag configure output -foreground blue
        interp eval $Editor::current(slave) set command [list $command]
        interp eval $Editor::current(slave) {
            eval puts $sock [list $command]
        }
        break
    }
    
}

proc Editor::deleteTestTerminal {pagename} {
    variable con_notebook
    
    $con_notebook delete $pagename
    $con_notebook raise asedCon
}

################################################################################
#proc Editor::execFile
#runs current editor-data without saving to file,
#or associated or default projectfile with data of the current window
################################################################################
proc Editor::execFile {} {
    global tk_library
    global tcl_library
    global tcl_platform
    global auto_path
    global conWindow
    global code
    global EditorData
    variable current
    
    if {$EditorData(options,useEvalServer)} {
        serverExecFile
        return
    }
    
    Editor::argument_history_add
    #aleady running ?
    if {[interp exists $current(slave)]} {
        switch -- [tk_messageBox -message "$current(file) is already running!\nRestart ?" -type yesnocancel -icon question -title "Question"] {
            yes {
                Editor::exitSlave $current(slave)
                set tempFile "[file rootname $current(file)].tmp"
                if {[Editor::file_copy $tempFile $current(file)] == 0} {
                    file delete $tempFile
                }
                after idle Editor::execFile
                return
            }
            no {}
            cancel {}
        }
        return
    }
    set cursor [. cget -cursor]
    . configure -cursor watch
    
    set hasChanged $current(hasChanged)
    if {$current(file) != "Untitled"} {
        if {[file_copy $current(file) "[file rootname $current(file)].tmp"]} {
            Editor::saveFile
            file_copy $current(file) "[file rootname $current(file)].tmp"
        } else  {
            Editor::saveFile
        }
    }
    
    update
    set current(slave) [interp create]
    set Editor::slaves($current(slave)) $Editor::current(pagename)
    interp eval $current(slave) set page $current(pagename)
    $current(slave) alias _exitSlave Editor::exitSlave
    if {"$tcl_platform(platform)" == "windows"} {
        $current(slave) alias consolePuts consolePuts
        interp eval $current(slave) {
            rename puts Puts
            proc puts {args} {
                switch -- [llength $args] {
                    0 {return}
                    1 {eval consolePuts $args}
                    2 {if {[lindex $args 0] == "-nonewline"} {
                            eval consolePuts $args
                        } else  {
                            eval Puts $args
                        }}
                    default {eval Puts $args}
                }
            }
        }
    }
    $current(slave) alias evalMain Editor::evalMain
    if {($current(project) == "none") || ($current(file) == "Untitled" || $current(file) == $current(project))} {
        set current(data) [$current(text) get 1.0 end-1c]
        interp eval $current(slave) set data [list $current(data)]
    } else  {
        if {[file exists $current(project)]} {
            set fd [open $current(project) r]
            interp eval $current(slave) set data [list [read $fd]]
            close $fd
        } else  {
            tk_messageBox -message "ProjectFile <$current(project)> not found !" -title Error -icon error
            after idle Editor::exitSlave $current(slave)
            return
        }
    }
    # ToDo:
    # setup for interpreter environment via dialog
    interp eval $current(slave) set slave $current(slave)
    interp eval $current(slave) set conWindow $conWindow
    interp eval $current(slave) set argv [list $Editor::argument_var]
    interp eval $current(slave) set argc [llength [list $Editor::argument_var]]
    interp eval $current(slave) set argv0 [list $current(file)]
    interp eval $current(slave) set tcl_library [list $tcl_library]
    interp eval $current(slave) set tk_library [list $tk_library]
    interp eval $current(slave) set auto_path [list $auto_path]
    interp eval $current(slave) {
        proc _exitProc {{exitcode 0}} {
            global slave
            catch {_exitSlave $slave}
        }
        load {} Tk
        interp alias {} exit {} _exitProc
        wm protocol . WM_DELETE_WINDOW {_exitProc}
        set code [catch {eval $data} info]
        catch {
            if {$code} {
                tk_messageBox -message $errorInfo -title Error -icon error
                after idle _exitProc
            }
        }
    }
    if {$current(file) != "Untitled"} {
        set tempFile "[file rootname $current(file)].tmp"
        if {![Editor::file_copy $tempFile $current(file)]} {
            file delete $tempFile
        }
    }
    set current(hasChanged) $hasChanged
    if {$current(hasChanged)} {
        $Editor::notebook itemconfigure $current(pagename) -image [Bitmap::get redball]
    }
    update idletasks
    . configure -cursor $cursor
    catch {
        interp eval $current(slave) {
            if {[wm title .] != ""} {
                wm title . "ASED is running: >>[wm title .]<<"
            } else  {
                if {$current(project) != "none" && $current(project) != $current(file)} {
                    wm title . "ASED is running \"$current(project)\" testing >>$current(file)<<"
                } else  {
                    wm title . "ASED is running: >>$current(file)<<"
                }
            }
        }
    }
    
}

proc Editor::chooseWish {} {
    global tcl_platform
    global EditorData
    global ASEDsRootDir
    variable serverUp
    
    if {$serverUp} {
        switch [tk_messageBox \
                -message "Restart Server ?\nThis will shutdown currently running applications!" \
                -icon warning \
                -title "Restart Server ?" \
                -type yesnocancel] {
                    yes {
                        foreach slaveInterp [interp slaves] {
                            # dont delete console interpreter
                            if {$slaveInterp != "asedCon"} {
                                Editor::exitSlave $slaveInterp
                            }
                }
                set slave [interp create exitInterp]
                interp eval $slave set ASEDsRootDir [list $ASEDsRootDir]
                interp eval $slave {set argv0 "shutdown Server"}
                interp eval $slave {load {} Tk}
                interp eval $slave source [file join $ASEDsRootDir evalClient.tcl]
                $slave alias _exitSlave Editor::exitSlave
                interp eval $slave set slave $slave
                interp eval $slave {
                    proc _exitProc {} {
                        global slave
                        after 500 {_exitSlave $slave}
                    }
                    interp alias {} exit {} _exitProc
                    wm protocol . WM_DELETE_WINDOW _exitProc
                }
                interp eval $slave Client::exitExecutionServer $EditorData(options,serverPort)
                interp delete $slave
            }
            default {
                return
            }
        }
    }
    
    if {$tcl_platform(platform) == "windows"} {
        set filePatternList [list "Executables {*.exe}" "All-Files {*.*}"]
    } else  {
        set filePatternList [list "All-Files {*}"]
    }
    set initialFile $EditorData(options,serverWish)
    set initialDir [file dirname [info nameofexecutable]]
    set serverWish [tk_getOpenFile \
            -filetypes $filePatternList \
            -initialdir $initialDir \
            -initialfile $initialFile \
            -title "Choose Server Wish"]
    if {$serverWish != ""} {
        set EditorData(options,serverWish) [list $serverWish]
    }
    return
}

################################################################################
# proc Editor::serverExecFile
# runs current editor-data via the evalServer without saving to file,
# or associated or default projectfile with data of the current window
################################################################################
proc Editor::serverExecFile {} {
    
    global tk_library
    global tcl_library
    global tcl_platform
    global auto_path
    global conWindow
    global ASEDsRootDir
    global EditorData
    variable current
    variable con_notebook
    
    Editor::argument_history_add
    #aleady running ?
    if {[interp exists $current(slave)]} {
        switch -- [tk_messageBox -message "$current(file) is already running!\nRestart ?" -type yesnocancel -icon question -title "Question"] {
            yes {
                Editor::exitSlave $current(slave)
                set tempFile "[file rootname $current(file)].tmp"
                if {![Editor::file_copy $tempFile $current(file)]} {
                    file delete $tempFile
                }
                after idle Editor::serverExecFile
                return
            }
            no {}
            cancel {}
        }
        return
    }
    set cursor [. cget -cursor]
    . configure -cursor watch
    
    set hasChanged $current(hasChanged)
    if {$current(file) != "Untitled"} {
        # make safety copy to tmp file
        if {[file_copy $current(file) "[file rootname $current(file)].tmp"]} {
            Editor::saveFile
            file_copy $current(file) "[file rootname $current(file)].tmp"
        } else  {
            Editor::saveFile
        }
    }
    
    update
    set current(slave) [interp create]
    set Editor::slaves($current(slave)) $Editor::current(pagename)
    interp eval $current(slave) set page $current(pagename)
    $current(slave) alias _exitSlave Editor::exitSlave
    $current(slave) alias ConPuts conPuts
    $current(slave) alias EvalMain Editor::evalMain
    $current(slave) alias NoteBookDelete Editor::deleteTestTerminal
    $current(slave) alias SetTestTermBinding Editor::setTestTermBinding
    if {($current(project) == "none") || ($current(file) == "Untitled" || $current(file) == $current(project))} {
        set current(data) [$current(text) get 1.0 end-1c]
        interp eval $current(slave) set data [list $current(data)]
    } else  {
        if {[file exists $current(project)]} {
            set fd [open $current(project) r]
            interp eval $current(slave) set data [list [read $fd]]
            close $fd
        } else  {
            tk_messageBox -message "ProjectFile <$current(project)> not found !" -title Error -icon error
            after idle Editor::exitSlave $current(slave)
            return
        }
    }
    #create testTerminal
    set testTerminal [EditManager::create_testTerminal $con_notebook $current(pagename) [file tail $current(file)]]
    $con_notebook raise $current(pagename)
    
    # ToDo:
    # setup for interpreter environment via dialog
    interp eval $current(slave) set slave $current(slave)
    interp eval $current(slave) set conWindow $conWindow
    interp eval $current(slave) set argv [list $Editor::argument_var]
    interp eval $current(slave) set argc [llength [list $Editor::argument_var]]
    interp eval $current(slave) set argv0 [list $current(file)]
    interp eval $current(slave) set tcl_library [list $tcl_library]
    interp eval $current(slave) set tk_library [list $tk_library]
    interp eval $current(slave) set auto_path [list $auto_path]
    interp eval $current(slave) set ASEDsRootDir [list $ASEDsRootDir]
    interp eval $current(slave) set title [file tail $current(file)]
    interp eval $current(slave) set testTerminal $testTerminal
    interp eval $current(slave) set con_notebook $con_notebook
    interp eval $current(slave) set pagename $current(pagename)
    interp eval $current(slave) set port $EditorData(options,serverPort)
    interp eval $current(slave) set serverWish $EditorData(options,serverWish)
    interp eval $current(slave) {
        proc _exitProc {{exitcode 0}} {
            global slave
            global pagename
            NoteBookDelete $pagename
            catch {_exitSlave $slave}
            return
        }
        interp alias {} exit {} _exitProc
        load {} Tk
        
        wm protocol . WM_DELETE_WINDOW {_exitProc}
        wm withdraw .
        source [file join $ASEDsRootDir evalClient.tcl]
        
        # new Client handler, overwrites default handler
        proc Client::newSockHandler {testTerminal sock} {
            variable serverResult
            
            if [eof $sock] {
                catch {close $sock}
                ConPuts "Socket closed $sock" error $testTerminal
                exit
                return
            }
            while {[gets $sock serverResult] > -1 } {
                if {$serverResult != ""} {
                    if {[string first "#echo:" $serverResult] == 0} {
                        ConPuts $serverResult prompt $testTerminal
                    } else  {
                        ConPuts $serverResult output $testTerminal
                    }
                }
            }
            return
        }
        
        eval [list set sock [Client::initExecutionClient \
                localhost \
                $port \
                "Client::newSockHandler $testTerminal" \
                [file join $ASEDsRootDir evalServer.tcl] \
                $serverWish \
                ]]
        if {$sock == {}} {
            exit
            return
        }
        EvalMain {set Editor::serverUp 1}
        SetTestTermBinding $sock $testTerminal
        
        puts $sock [list set argv $argv]
        puts $sock [list set argc $argc]
        puts $sock [list set argv0 $argv0]
        # concat lines if they are splitted
        regsub -all -- "{\\\n}" $data "" data
        puts $sock "eval $data"
        puts $sock "wm deiconify ."
        puts $sock [list wm title . "ASED Test Server ([file tail $serverWish]): $title"]
        puts $sock "focus ."
        eval [list wm title . "ASED Test Terminal: Output of $title"]
    }
    # restore original file from tmp file
    if {$current(file) != "Untitled"} {
        set tempFile "[file rootname $current(file)].tmp"
        if {![Editor::file_copy $tempFile $current(file)]} {
            file delete $tempFile
        }
    }
    set current(hasChanged) $hasChanged
    if {$current(hasChanged)} {
        $Editor::notebook itemconfigure $current(pagename) -image [Bitmap::get redball]
    }
    update idletasks
    . configure -cursor $cursor
    return
}

################################################################################
#proc Editor::terminate
#terminates execution of current editor-file or associated projectfile
################################################################################
proc Editor::terminate {} {
    variable current
    Editor::exitSlave $current(slave)
    set tempFile "[file rootname $current(file)].tmp"
    if {[file exists $tempFile] && [file mtime $tempFile] > [file mtime $current(file)]} {
        if {![Editor::file_copy $tempFile $current(file)]} {
            file delete $tempFile
        }
    } elseif {[file exists $tempFile]} {
        file delete $tempFile
    }
}

proc Editor::exitSlave {slave} {
    if {[interp exists $slave]} {
        interp eval $slave {
            set taskList [after info]
            foreach id $taskList {
                after cancel $id
            }
        }
        catch {$Editor::notebook raise $Editor::slaves($slave)}
        catch {Editor::deleteTestTerminal $Editor::current(pagename)}
        catch {interp delete $slave}
        update
        catch {$Editor::con_notebook delete $current(pagename)}
    }
    . configure -cursor {}
    return
}


proc Editor::file_copy {in out} {\
    
    if {[file exists $in]} {
        file copy -force $in $out
        return 0
    } else {
        return 1
    }
}
################################################################################
# proc Editor::getFile
# openfile dialog
# returns filename and content of the file
################################################################################
proc Editor::getFile {{filename {}}} {
    global EditorData
    variable treeWindow
    
    # workaround to avoid button1 events to the procWindow while
    # double clicking a file
    bind $treeWindow <Button-1> {}
    
    if {$filename == {}} {
        set filePatternList [list "Tcl-Files {*.tcl *.tk *.itcl *.itk}" "All-Files {*.* *}"]
        set defaultExt .tcl
        set initialFile ""
        set filename [tk_getOpenFile -filetypes $filePatternList -initialdir $EditorData(options,workingDir) -title "Open File"]
    }
    
    if {[file exists $filename]} {
        set cursor [. cget -cursor]
        . configure -cursor watch
        update
        set fd [open $filename r+]
        set data [read $fd]
        close $fd
        . configure -cursor $cursor
        set EditorData(options,workingDir) [file dirname $filename]
        return [list $filename $data]
    }
}


proc Editor::openNewPage {{file {}}} {\
    global EditorData
    variable notebook
    variable current
    
    set temp [Editor::getFile $file];#returns filename and textdata
    set filename [lindex $temp 0]
    if {$filename == ""} {return 1}
    if {$filename == $current(file)} {
        tk_messageBox -message "File already opened !" -title Warning -icon warning
        return 1
    }
    set EditorData(options,workingDir) [file dirname $filename]
    set f0 [EditManager::create_text $notebook $filename ]
    set data [lindex $temp 1]
    
    set temp $current(hasChanged)
    set editorWindows::TxtWidget [lindex $f0 2]
    $editorWindows::TxtWidget insert 1.0 $data
    $editorWindows::TxtWidget mark set insert 1.0
    set current(hasChanged) $temp
    editorWindows::colorize; #needs TxtWidget !
    set Editor::text_win($Editor::index_counter,undo_id) [new textUndoer $editorWindows::TxtWidget]
    NoteBook::raise $notebook [lindex $f0 1]
    $Editor::mainframe setmenustate noFile normal
    #Now the new textwindow is the current
    set current(hasChanged) 0
    set current(lastPos) [$current(text) index insert]
    return 0
}

################################################################################
#proc Editor::setDefaultProject
#if default project file is set then this will be run from any window by
#pressing the Test button instead of the current file, except for the current
#file is associated to another projectfile
################################################################################
proc Editor::setDefaultProject {{filename {}}} {
    global EditorData
    variable current
    
    if {$filename == "none"} {
        switch -- [tk_messageBox -message "Do you want to unset current default project ?" \
                -type yesnocancel -icon question -title Question] {
                    yes {
                        set EditorData(options,defaultProjectFile) "none"
                        set current(project) "none"
                    }
            default {}
        }
        return
    }
    
    if {$filename == {}} {
        set filePatternList [list "Tcl-Files {*.tcl *.tk *.itcl *.itk}" "All-Files {*.* *}"]
        set defaultExt .tcl
        set initialFile ""
        set filename [tk_getOpenFile -filetypes $filePatternList -initialdir $EditorData(options,workingDir) -title "Select Default Project File"]
    }
    if {$filename != ""} {
        set oldfile $EditorData(options,defaultProjectFile)
        set EditorData(options,defaultProjectFile) $filename
        # only set current(project) if it is not set by projectassociaion
        if {$current(project) == "$oldfile"} {
            set current(project) $filename
        }
    }
}

################################################################################
#proc Editor::associateProject
#if there is a projectfile associated to the current file,
#this file will be started by pressing the test button.
#This overrides the option for the default project file
################################################################################
proc Editor::associateProject {} {
    global EditorData
    variable current
    
    set filePatternList [list "Tcl-Files {*.tcl *.tk *.itcl *.itk}" "All-Files {*.* *}"]
    set defaultExt .tcl
    set initialFile ""
    set filename [tk_getOpenFile -filetypes $filePatternList -initialdir $EditorData(options,workingDir) -title "Select Project File"]
    if {$filename != ""} {
        set current(project) $filename
        set prjFile [file rootname $current(file)].prj
        set result [Editor::_saveFile $prjFile $current(project)]
    }
}

proc Editor::unsetProjectAssociation {} {
    global EditorData
    variable current
    
    set prjFile [file rootname $current(file)].prj
    if {[file exists $prjFile]} {
        file delete $prjFile
    }
    set current(project) $EditorData(options,defaultProjectFile)
}

proc Editor::openFile {{file {}}} {
    
    variable notebook
    variable current
    variable index
    variable last
    
    set deleted 0
    # test if there is a page opened
    set pages [NoteBook::pages $notebook]
    if {[llength $pages] == 0} { \
        Editor::openNewPage
        return
    } else {
        # test if current page is empty
        if {[info exists current(text)]} {\
            set f0 $current(pagename)
            set text [NoteBook::itemcget $notebook $f0 -text]
            
            set data [$current(text) get 1.0 end-1c]
            if {($data == "") && ($text == "Untitled")} {\
                # page is empty
                delete_id
                NoteBook::delete $notebook $current(pagename)
                tdelNode $current(file)
                set idx $Editor::index($current(text))
                foreach entry [array names Editor::text_win $idx,*] {
                    unset Editor::text_win($entry)
                }
                unset index($current(text))
                set deleted 1
            }
        }
        set result [Editor::openNewPage $file]
        if {$deleted && $result} {
            set force 1
            Editor::newFile force
        }
    }
}

proc Editor::saveAll {} {
    foreach textWin [array names ::Editor::index] {
        set idx $Editor::index($textWin)
        set data [$textWin get 1.0 "end -1c"]
        set filename $Editor::text_win($idx,file)
        Editor::file_copy $filename [file rootname $filename].bak
        Editor::_saveFile $filename $data
        set Editor::text_win($idx,hasChanged) 0
        $Editor::notebook itemconfigure $Editor::text_win($idx,pagename) -image ""
    }
    set Editor::current(hasChanged) 0
}

proc Editor::_saveFile {filename data} {
    
    if {$filename == "Untitled"} {
        Editor::_saveFileas $filename $data
        return
    }
    set cursor [. cget -cursor]
    . configure -cursor watch
    update
    set fd [open $filename w+]
    puts -nonewline $fd $data
    close $fd
    . configure -cursor $cursor
    return
}

proc Editor::saveFile {} {
    variable notebook
    variable current
    variable index
    
    if {[$notebook pages] == {}} {
        return ;# No open file
    }
    
    set filename $current(file)
    if {$filename == "Untitled"} {
        Editor::saveFileas
        return
    }
    set data [$current(text) get 1.0 end-1c]
    catch {Editor::file_copy $filename "[file rootname $filename].bak"}
    set result [Editor::_saveFile $current(file) $data]
    set current(hasChanged) 0
    $Editor::notebook itemconfigure $current(pagename) -image ""
    set idx $index($current(text))
    set Editor::text_win($idx,hasChanged) $current(hasChanged)
    if {$current(project) != "none"} {
        set prjFile [file rootname $current(file)].prj
        set result [Editor::_saveFile $prjFile $current(project)]
    }
}

proc Editor::_saveFileas {filename data} {
    global EditorData
    
    set filePatternList [list "Tcl-Files {*.tcl *.tk *.itcl *.itk}" "Alle {*.* *}"]
    set defaultExt .tcl
    set initialFile $filename
    set file [tk_getSaveFile -filetypes $filePatternList -initialdir $EditorData(options,workingDir) \
            -initialfile $filename -defaultextension $defaultExt -title "Save File"]
    if {$file != ""} {
        set cursor [. cget -cursor]
        . configure -cursor watch
        update
        set fd [open $file w+]
        puts -nonewline $fd $data
        close $fd
        . configure -cursor $cursor
    }
    set EditorData(options,workingDir) [file dirname $file]
    return $file
}


proc Editor::saveFileas {} {
    variable notebook
    variable current
    
    if {[$notebook pages] == {}} {
        return ;# no open file
    }
    
    set filename $current(file)
    set data [$current(text) get 1.0 end-1c]
    set result [Editor::_saveFileas $current(file) $data]
    if {$result == ""} {return 1}
    set insertCursor [$current(text) index insert]
    editorWindows::deleteMarks 1.0 "end -1c"
    tdelNode $current(file)
    set current(hasChanged) 0
    $Editor::notebook itemconfigure $current(pagename) -image ""
    
    set current(file) $result
    set idx $Editor::index($current(text))
    set Editor::text_win($idx,file) $current(file)
    $notebook itemconfigure $current(pagename) -text [file tail $result]
    $current(text) mark set insert $insertCursor
    Editor::updateObjects
    Editor::selectObject 0
    
    #if there was already a .prj-File then copy to new name too
    if {[file exists [file rootname $filename].prj]} {
        set prjFile [file rootname $current(file)].prj
        set result [Editor::_saveFile $prjFile $current(project)]
    }
    Editor::file_copy $current(file) "[file rootname $current(file)].bak"
}

proc Editor::showConsole {show} {
    variable con_notebook
    
    set win [winfo parent $con_notebook]
    set win [winfo parent $win]
    set panedWin [winfo parent $win]
    update idletasks
    if {$show} {
        grid configure $panedWin.f0 -rowspan 1
        grid $panedWin.sash1
        grid $win
        grid rowconfigure $panedWin 2 -minsize 100
    } else  {
        grid remove $win
        grid remove $panedWin.sash1
        grid configure $panedWin.f0 -rowspan 3
    }
}


proc Editor::showTreeWin {show} {
    variable list_notebook
    
    set win [winfo parent $list_notebook]
    set win [winfo parent $win]
    set panedWin [winfo parent $win]
    update idletasks
    
    if {$show} {
        grid configure $panedWin.f1 -column 2 -columnspan 1
        grid $panedWin.sash1
        grid $win
        grid columnconfigure $panedWin 0 -minsize 200
        Editor::updateObjects
        Editor::selectObject
    } else  {
        grid remove $win
        grid remove $panedWin.sash1
        grid configure $panedWin.f1 -column 0 -columnspan 3
    }
}



proc Editor::showSolelyConsole {show} {
    variable notebook
    
    set win [winfo parent $notebook]
    set win [winfo parent $win]
    set panedWin [winfo parent $win]
    set frame [winfo parent $panedWin]
    set frame [winfo parent $frame]
    set panedWin [winfo parent $frame]
    update idletasks
    
    if {$show} {
        grid remove $frame
        grid remove $panedWin.sash1
        grid configure $panedWin.f1 -rowspan 3
        grid rowconfigure $panedWin 2 -weight 100
        grid rowconfigure $panedWin 2 -minsize 100
    } else  {
        grid configure $panedWin.f1 -rowspan 1
        grid $panedWin.sash1
        grid $frame
        grid rowconfigure $panedWin 2 -minsize 100
    }
}
################################################################################
#proc Editor::close_dialog
#called whenever the user wants to exit ASED and there are files
#that have changed and are still not saved yet
################################################################################
proc Editor::close_dialog {} {\
    variable notebook
    variable current
    variable index
    variable text_win
    
    set result [tk_messageBox -message "File <$current(file)> has changed!\n Save it ?" -type yesnocancel -icon warning -title "Question"]
    switch -- $result {
        yes {Editor::saveFile ; set result 0; return $result}
        no  {set result 0 ; return $result}
        cancel {set result 1 ; return $result}
    }
}


proc Editor::closeFile {{exit 0}} {
    
    variable notebook
    variable current
    variable index
    variable last
    variable text_win
    variable procWindow
    variable mainframe
    
    # Is there an open window
    if {[$notebook pages] == {}} {
        return
    }
    if {$current(hasChanged)} {\
        set result [Editor::close_dialog]
        if {$result} {return $result}
    }
    if {[info exists current(undo_id)]} {delete_id}
    # if current file is running terminate execution
    Editor::terminate
    NoteBook::delete $notebook $current(pagename)
    set idx $Editor::index($current(text))
    foreach entry [array names Editor::text_win "$idx,*"] {
        unset Editor::text_win($entry)
    }
    unset index($current(text))
    #delete node
    tdelNode $current(file)
    set indexList [NoteBook::pages $notebook]
    if {[llength $indexList] != 0} {
        NoteBook::raise $notebook [lindex $indexList end]
    } else {
        if {$exit == 0} {Editor::newFile}
    }
    return 0
}


proc Editor::exit_app {} {
    global EditorData
    global ASEDsRootDir
    variable notebook
    variable current
    variable index
    variable text_win
    variable serverUp
    
    set taskList [after info]
    foreach id $taskList {
        after cancel $id
    }
    if {$current(hasChanged)} {\
        if {[catch {set idx $index($current(text))}]} {exit}
        set text_win($idx,hasChanged) $current(hasChanged)
    }
    set newlist ""
    set index_list [array names index]
    foreach idx $index_list {\
        set newlist [concat $newlist  $index($idx)]
    }
    
    Editor::getWindowPositions
    Editor::saveOptions
    
    #    if no window is open, we can exit at once
    if {[llength $newlist] == ""} {
        exit
    }
    
    foreach idx $newlist {\
        set current(text) $text_win($idx,path)
        set current(page) $text_win($idx,page)
        set current(pagename) $text_win($idx,pagename)
        set current(hasChanged) $text_win($idx,hasChanged)
        set current(undo_id) $text_win($idx,undo_id)
        set current(file) $text_win($idx,file)
        set current(slave) $text_win($idx,slave)
        set result [Editor::closeFile exit]
        if {$result} {\
            NoteBook::raise $notebook $current(pagename)
            return
        }
    }
    if {$serverUp} {
        set slave [interp create]
        interp eval $slave set ASEDsRootDir [list $ASEDsRootDir]
        interp eval $slave set argv0 shutdown_Server
        interp eval $slave {load {} Tk}
        interp eval $slave source [file join $ASEDsRootDir evalClient.tcl]
        interp eval $slave set Client::port $EditorData(options,serverPort)
        interp eval $slave Client::exitExecutionServer
        interp delete $slave
    }
    exit
}

proc Editor::gotoLineDlg {} {
    variable current
    
    set gotoLineDlg [toplevel .dlg]
    set entryLabel [label .dlg.label -text "Please enter line number"]
    set lineEntry [entry .dlg.entry -textvariable lineNo]
    set buttonFrame [frame .dlg.frame]
    set okButton [button $buttonFrame.ok -width 8 -text Ok -command {catch {destroy .dlg};Editor::gotoLine $lineNo}]
    set cancelButton [button $buttonFrame.cancel -width 8 -text Cancel -command {catch {destroy .dlg}}]
    wm title .dlg "Goto Line"
    bind $lineEntry <KeyRelease-Return> {.dlg.frame.ok invoke}
    pack $entryLabel
    pack $lineEntry
    pack $okButton $cancelButton -padx 10 -pady 10 -side left -fill both -expand yes
    pack $buttonFrame
    BWidget::place $gotoLineDlg 0 0 center
    focus $lineEntry
}

proc Editor::gotoLine {lineNo} {
    variable current
    
    switch -- [string index $lineNo 0] {
        "-" -
        "+" {set curLine [lindex [split [$current(text) index insert] "."] 0]
            set lineNo [expr $curLine $lineNo]}
    }
    if {[catch {$current(text) mark set insert $lineNo.0}]} {
        tk_messageBox -message "Line number out of range!" -icon warning -title "Warning"
    }
    $current(text) see insert
    editorWindows::flashLine
    editorWindows::ReadCursor
    selectObject 0
    focus $current(text)
}

proc Editor::cut {} {
    variable current
    
    editorWindows::cut
    set current(lastPos) [$current(text) index insert]
}

proc Editor::copy {} {
    variable current
    editorWindows::copy
}

proc Editor::paste {} {
    
    variable current
    editorWindows::paste
    set current(lastPos) [$current(text) index insert]
}

proc Editor::delete {} {
    variable current
    editorWindows::delete
    set current(lastPos) [$current(text) index insert]
}

proc Editor::delLine {} {
    global EditorData
    variable current
    if {[$current(text) tag range sel] != ""} {
        Editor::delete
    }
    set curPos [$current(text) index insert]
    if {$EditorData(options,autoUpdate)} {
        set range [editorWindows::deleteMarks "$curPos linestart" "$curPos lineend"]
        $current(text) delete "$curPos linestart" "$curPos lineend +1c"
        Editor::updateOnIdle $range
    } else  {
        $current(text) delete "$curPos linestart" "$curPos lineend +1c"
    }
    set current(lastPos) [$current(text) index insert]
}

proc Editor::SelectAll {} {
    variable current
    
    $current(text) tag remove sel 1.0 end
    $current(text) tag add sel 1.0 end
}

proc Editor::insertFile {} {
    variable current
    
    set temp [Editor::getFile];#returns filename and textdata
    set filename [lindex $temp 0]
    if {$filename == ""} {return 1}
    if {$filename == $current(file)} {
        tk_messageBox -message "File already opened !" -title Warning -icon warning
        return 1
    }
    set EditorData(options,workingDir) [file dirname $filename]
    set data [lindex $temp 1]
    set index [$current(text) index insert]
    $current(text) insert insert $data
    $current(text) see insert
    set end [$current(text) index insert]
    editorWindows::ColorizeLines [lindex [split $index "."] 0] [lindex [split [$editorWindows::TxtWidget index insert] "."] 0 ]
    
    updateObjects "$index $end"
    torder $current(file)
    selectObject 0
    set current(lastPos) [$current(text) index insert]
    return 0
}

proc Editor::getFirstChar { index } {
    
    set w $Editor::current(text)
    set curLine [lindex [split [$w index $index] "."] 0]
    set pos $curLine.0
    set char [$w get $pos]
    for {set i 0} {$char == " " || $char == "\t"} {incr i} {
        set pos $curLine.$i
        set char [$w get $pos]
    }
    return [list $char $pos]
}

proc Editor::make_comment_block {} {
    variable current
    
    set commentLineString \
            "################################################################################\n"
    
    if {[$current(text) tag ranges sel] == ""} {
        #no selection
        set curPos [$current(text) index insert]
        set curLine [lindex [split [$current(text) index $curPos] "."] 0]
        $current(text) insert $curLine.0 $commentLineString
        editorWindows::ColorizeLine $curLine
    } else {
        set firstLine [lindex [split [$current(text) index sel.first] "."] 0]
        set lastLine [lindex [split [$current(text) index sel.last] "."] 0]
        for {set line $firstLine} {$line <= $lastLine} {incr line} {
            $current(text) insert $line.0 "# "
        }
        $current(text) insert $firstLine.0 $commentLineString
        set lastLine [expr $lastLine+2]
        $current(text) insert $lastLine.0 $commentLineString
        for {set line $firstLine} {$line <= $lastLine} {incr line} {
            editorWindows::ColorizeLine $line
        }
        $current(text) tag remove sel sel.first sel.last
        $current(text) mark set insert "insert+2l linestart"
    }
    selectObject
}

################################################################################
# proc Editor::toggle_comment
# toggles the comment status of the current line or selection
################################################################################
proc Editor::toggle_comment {} {
    variable current
    
    if {[$current(text) tag ranges sel] == ""} {
        #no selection
        set curPos [$current(text) index insert]
        set result [Editor::getFirstChar $curPos]
        if {[lindex $result 0]  == "#"} {
            $current(text) delete [lindex $result 1]
            while {[$current(text) get [lindex $result 1]] == " " \
                        || [$current(text) get [lindex $result 1]] == "\t"} {
                $current(text) delete [lindex $result 1]
            }
            set curLine [lindex [split [$current(text) index $curPos] "."] 0]
            editorWindows::ColorizeLine $curLine
        } else  {
            set curLine [lindex [split [$current(text) index $curPos] "."] 0]
            $current(text) insert [lindex $result 1] "# "
            editorWindows::ColorizeLine $curLine
        }
        updateOnIdle [list $curLine.0 "$curLine.0 lineend"]
    } else {
        set firstLine [lindex [split [$current(text) index sel.first] "."] 0]
        set lastLine [lindex [split [$current(text) index sel.last] "."] 0]
        set result [Editor::getFirstChar $firstLine.0]
        set char [lindex $result 0]
        if {$char == "#"} {
            #if first char of first line is # then uncomment selection complete
            for {set line $firstLine} {$line <= $lastLine} {incr line} {
                set result [Editor::getFirstChar $line.0]
                if {[lindex $result 0] == "#"} {
                    $current(text) delete [lindex $result 1]
                    while {[$current(text) get [lindex $result 1]] == " " \
                                || [$current(text) get [lindex $result 1]] == "\t"} {
                        $current(text) delete [lindex $result 1]
                    }
                    editorWindows::ColorizeLine $line
                }
            }
        } else  {
            #if first char of first line is not # then comment selection complete
            for {set line $firstLine} {$line <= $lastLine} {incr line} {
                set insertPos [lindex [getFirstChar $line.0] 1]
                $current(text) insert $insertPos "# "
                editorWindows::ColorizeLine $line
            }
        }
        set start [$current(text) index sel.first]
        set end [$current(text) index sel.last]
        updateOnIdle [list $start $end]
        $current(text) tag remove sel sel.first sel.last
    }
    selectObject
}

proc Editor::procList_history_get_prev {} {
    variable current
    
    if  {$current(procListHistoryPos) == 0} {
        set index [$Editor::current(text) index insert]
        set Editor::current(lastPos) $index
        Editor::procList_history_add $index
    } elseif {$current(procListHistoryPos) == -1} {
        incr current(procListHistoryPos)
        set index [$Editor::current(text) index insert]
        set Editor::current(lastPos) $index
        Editor::procList_history_add $index
    }
    if {$current(procListHistoryPos) < [expr [llength $current(procListHistory)]-1]} {
        incr current(procListHistoryPos)
    } else  {
        selectObject 0
        return
    }
    
    $current(text) mark set insert [lindex $current(procListHistory) $current(procListHistoryPos)]
    $current(text) see insert
    focus $current(text)
    editorWindows::ReadCursor 0
    editorWindows::flashLine
    selectObject 0
}

proc Editor::procList_history_get_next {} {
    variable current
    
    if {$current(procListHistoryPos) > 0} {
        incr current(procListHistoryPos) -1
    } else  {
        set current(procListHistoryPos) "-1"
        selectObject 0
        return
    }
    catch {$current(text) mark set insert [lindex $current(procListHistory) $current(procListHistoryPos)]}
    if {$current(procListHistoryPos) == 0} {
        set current(procListHistoryPos) "-1"
    }
    
    $current(text) see insert
    focus $current(text)
    editorWindows::ReadCursor 0
    editorWindows::flashLine
    
    selectObject 0
}


proc Editor::procList_history_add {{pos {}}} {
    variable current
    
    if {$pos == {}} {
        set index [$Editor::current(text) index insert]
        
    } else  {
        set index [$Editor::current(text) index $pos]
    }
    set lineNum [lindex [split $index "."] 0]
    set mark "mark$lineNum"
    
    if {![info exists Editor::current(procListHistory)]} {
        set Editor::current(procListHistory) [list "mark1"]
    } elseif {[lsearch $Editor::current(procListHistory) "$mark"] == 0} {
        $Editor::current(text) mark set $mark $index
        set Editor::current(procListHistoryPos) 0
        return
    }
    
    catch [$Editor::current(text) mark set $mark $index]
    
    set Editor::current(procListHistory) [linsert $Editor::current(procListHistory) 0 $mark]
    if {[llength $Editor::current(procListHistory)] > $Editor::current(procList_hist_maxLength)} {
        $Editor::current(text) mark unset [lindex $Editor::current(procListHistory) end]
        set Editor::current(procListHistory) [lreplace $Editor::current(procListHistory) end end]
    }
    set Editor::current(procListHistoryPos) 0
}

proc Editor::lineNo_history_add {} {\
    variable lineEntryCombo
    variable lineNo
    set newlist [ComboBox::cget $lineEntryCombo -values]
    if {[lsearch -exact $newlist $lineNo] != -1} {return}
    set newlist [linsert $newlist 0 $lineNo]
    ComboBox::configure $lineEntryCombo -values $newlist
}

proc Editor::argument_history_add {} {\
    variable argument_combo
    variable argument_var
    set newlist [ComboBox::cget $argument_combo -values]
    if {[lsearch -exact $newlist $argument_var] != -1} {return}
    set newlist [linsert $newlist 0 $argument_var]
    ComboBox::configure $argument_combo -values $newlist
}

proc Editor::search_history_add {} {\
    variable search_combo
    variable search_var
    set newlist [ComboBox::cget $search_combo -values]
    if {[lsearch -exact $newlist $search_var] != -1} {return}
    set newlist [linsert $newlist 0 $search_var]
    ComboBox::configure $search_combo -values $newlist
}

proc Editor::search_forward {} {\
    global search_option_icase search_option_match search_option_blink
    variable search_combo
    variable current
    Editor::search_history_add
    set search_string $Editor::search_var
    set result [Editor::search $current(text) $search_string search $search_option_icase\
            forwards $search_option_match $search_option_blink]
    focus $current(text)
    selectObject 0
}


proc Editor::search_backward {{searchText {}}} {\
    global search_option_icase search_option_match search_option_blink
    variable search_combo
    variable current
    
    if {$searchText == {}} {
        Editor::search_history_add
        set search_string $Editor::search_var
        set result [Editor::search $current(text) $search_string search $search_option_icase\
                backwards $search_option_match $search_option_blink]
        
    } else  {
        set result [Editor::search $current(text) $searchText search 0\
                backwards $search_option_match 0]
    }
    
    if {$result != ""} {$current(text) mark set insert [lindex $result 0]}
    focus $current(text)
    selectObject 0
}

proc Editor::load_search_defaults {} {\
    search_default_options
}

proc Editor::search {textWindow search_string tagname icase where match blink} {\
    variable current
    set result [search_proc $textWindow $search_string $tagname $icase $where $match $blink]
    editorWindows::ReadCursor 0
    set current(lastPos) [$current(text) index insert]
    return $result
}

proc Editor::search_dialog {} {\
    
    variable current
    
    search_dbox $current(text)
    Editor::search_history_add
    focus $current(text)
}

proc Editor::replace_dialog {} {\
    variable current
    
    replace_dbox $current(text)
    focus $current(text)
}

proc Editor::findInFiles {} {
    global EditorData
    
    set resultList [fif::openFifDialog $EditorData(options,workingDir)]
}

proc Editor::showResults {resultList} {
    variable resultWindow
    variable TxtWidget
    variable con_notebook
    variable searchResults
    
    catch {
        NoteBook::delete $Editor::con_notebook resultWin
        NoteBook::raise $Editor::con_notebook asedCon
    }
    set resultWindow [EditManager::createResultWindow $con_notebook]
    foreach entry [array names searchResults] {
        unset searchResults($entry)
    }
    foreach entry $resultList {
        set line "File: [lindex $entry 0] --> \"[lindex $entry 3]\""
        set searchResults($line) $entry
        $resultWindow insert 0 $line
    }
    $resultWindow see 0
    $con_notebook raise resultWin
    bind $resultWindow <Button-1> {
        $Editor::resultWindow selection clear 0 end
        $Editor::resultWindow selection set @%x,%y
        set index [$Editor::resultWindow curselection]
        set result [$Editor::resultWindow get $index]
        set result $Editor::searchResults($result)
        Editor::openFile [lindex $result 0]
        set index "[lindex $result 1].[lindex $result 2]"
        $editorWindows::TxtWidget mark set insert $index
        $editorWindows::TxtWidget see insert
        editorWindows::flashLine
        Editor::selectObject 0
    }
}


proc Editor::undo {} {\
    variable notebook
    variable current
    
    set cursor [. cget -cursor]
    . configure -cursor watch
    update
    set range [textUndoer:undo $Editor::current(undo_id)]
    . configure -cursor $cursor
    if {$range != {}} {
        set curPos [lindex $range 1]
        if {[$current(text) compare [lindex $range 0] == [lindex $range 1]]} {
            #delete all marks at insert
            set range [editorWindows::deleteMarks [$current(text) index insert] [$current(text) index insert]]
        } else  {
            set range [editorWindows::deleteMarks [lindex $range 0] [lindex $range 1]]
        }
        set startLine [lindex [split [$current(text) index [lindex $range 0]] "."] 0]
        set endLine [lindex [split [$current(text) index [lindex $range 1]] "."] 0]
        update
        after idle "editorWindows::ColorizeLines $startLine $endLine"
        updateOnIdle $range
        $Editor::current(text) mark set insert $curPos
    }
    
    set current(lastPos) [$current(text) index insert]
    focus $current(text)
}

proc Editor::redo {} {\
    variable notebook
    variable current
    
    set cursor [. cget -cursor]
    . configure -cursor watch
    update
    set range [textRedoer:redo $Editor::current(undo_id)]
    . configure -cursor $cursor
    
    if {$range != {}} {
        set curPos [lindex $range 1]
        if {[$current(text) compare [lindex $range 0] == [lindex $range 1]]} {
            #delete all marks at insert
            set range [editorWindows::deleteMarks [$current(text) index insert] [$current(text) index insert]]
        } else  {
            set range [editorWindows::deleteMarks [lindex $range 0] [lindex $range 1]]
        }
        set startLine [lindex [split [$current(text) index [lindex $range 0]] "."] 0]
        set endLine [lindex [split [$current(text) index [lindex $range 1]] "."] 0]
        updateOnIdle $range
        after idle "editorWindows::ColorizeLines $startLine $endLine"
        $Editor::current(text) mark set insert $curPos
    }
    
    set current(lastPos) [$current(text) index insert]
    focus $current(text)
}


proc Editor::toggleTreeOrder {} {
    global EditorData
    if {$EditorData(options,sortProcs)} {
        set Editor::options(sortProcs) 0
        Editor::torder $Editor::current(file)
    } else  {
        set Editor::options(sortProcs) 1
        Editor::torder $Editor::current(file)
    }
    set EditorData(options,sortProcs) $Editor::options(sortProcs)
    Editor::selectObject 0
}

proc Editor::create { } {
    global tcl_platform
    global clock_var
    global EditorData
    global ASEDsRootDir
    
    variable _wfont
    variable notebook
    variable list_notebook
    variable con_notebook
    variable pw2
    variable pw1
    variable procWindow
    variable treeWindow
    variable markWindow
    variable mainframe
    variable font
    variable prgtext
    variable prgindic
    variable status
    variable search_combo
    variable argument_combo
    variable current
    variable clock_label
    variable defaultFile
    variable defaultProjectFile
    variable Font_var
    variable FontSize_var
    variable options
    
    variable toolbarButtons
    variable treeMenu
    variable textMenu
    
    
    set result [catch {source [file join "~" ased.cfg]} info]
    variable configError $result
    
    _create_intro
    update
    
    # if an update was installed with a different configuration of ased.cfg
    # use default values
    if {[catch {Editor::setDefault}]} {
        set configError 1
        Editor::setDefault
    }
    Editor::load_search_defaults
    Editor::tick
    
    # Menu description
    set descmenu {
        "&File" {} {} 0 {
            {command "&New" {} "New Text" {} -command Editor::newFile}
            {command "&Open" {}  "Open File" {Ctrl o} -command Editor::openFile}
            {command "&Save" {noFile}  "Save active File" {Ctrl s} -command Editor::saveFile}
            {command "Save &as ..." {noFile}  "Save active File as .." {} -command Editor::saveFileas}
            {command "Save all" {noFile}  "Save all File" {} -command Editor::saveAll}
            {command "&Close" {noFile} "Close active File" {} -command Editor::closeFile }
            {separator}
            {command "E&xit" {}  "Exit ASED" {Alt x} -command Editor::exit_app}
        }
        "&Edit" {noFile} {} 0 {
            {command "Copy" {} "Copy to Clipboard" {Ctrl c} -command Editor::copy }
            {command "Cut" {} "Cut to Clipboard" {Ctrl x} -command Editor::cut }
            {command "Paste" {} "Paste from Clipboard" {Ctrl v} -command Editor::paste }
            {command "Delete" {} "Delete Selection" {} -command Editor::delete }
            {command "Delete Line" {} "Delete current line" {} -command {Editor::delLine ; break} }
            {separator}
            {command "Select all" {} "Select All" {} -command Editor::SelectAll }
            {separator}
            {command "Insert File ..." {} "Insert file at current cursor position" {} -command Editor::insertFile }
            {separator}
            {command "Goto Line ..." {} "Goto Line" {} -command Editor::gotoLineDlg }
            {separator}
            {command "Search ..." {} "Search dialog" {} -command Editor::search_dialog }
            {command "Search in files ..." {} "Search in files" {} -command Editor::findInFiles}
            {command "Replace ..." {} "Replace dialog" {} -command Editor::replace_dialog }
            {separator}
            {command "Undo" {} "Undo" {CtrlAlt u} -command Editor::undo }
            {command "Redo" {} "Redo" {} -command Editor::redo }
            {separator}
            {command "AutoIndent File" {} "AutoIndent current file" {} -command editorWindows::autoIndent}
        }
        "&Test" {} {} 0 {
            {command "Run" {noFile} "Test current File" {CtrlAlt r} -command Editor::execFile }
            {command "Stop" {noFile} "Terminate Execution of current File" {CtrlAlt s} -command Editor::terminate  }
            {separator}
            {command "Set Default Startfile..." {} "Set Default Startfile" {} -command Editor::setDefaultProject }
            {command "Unset Default Startfile" {} "Unset Default Startfile" {} -command {Editor::setDefaultProject "none"} }
            {separator}
            {command "Set Startfile for current file..." {noFile} "Associate a Startfile to the current File" {} -command Editor::associateProject }
            {command "Unset Startfile for current file..." {noFile} "Clear current Startfile Association" {} -command Editor::unsetProjectAssociation }
            {separator}
            {checkbutton "Use EvalServer" {all option} "Use EvalServer for Testing Apps" {}
                -variable Editor::options(useEvalServer)
                -command  {set EditorData(options,useEvalServer) $Editor::options(useEvalServer)
                }
            }
            {command "Choose Wish ..." {} "Choose Wish for EvalServer" {} -command {Editor::chooseWish} }
        }
        "&Options" all options 0 {
            {checkbutton "Main Toolbar" {all option} "Show/hide Main Toolbar" {}
                -variable Editor::toolbar1
                -command  {$Editor::mainframe showtoolbar 0 $Editor::toolbar1
                    set EditorData(options,showToolbar1) $Editor::toolbar1}
            }
            {checkbutton "&Font Toolbar" {all option} "Show/hide Font Toolbar" {}
                -variable Editor::toolbar2
                -command  {$Editor::mainframe showtoolbar 1 $Editor::toolbar2
                    set EditorData(options,showToolbar2) $Editor::toolbar2}
            }
            {separator}
            {command "Increase Fontsize" {} "Increase Fontsize" {} -command {Editor::increaseFontSize up} }
            {command "Decrease Fontsize" {} "Decrease Fontsize" {} -command {Editor::increaseFontSize down} }
            {separator}
            {checkbutton "Sort Code Browser Tree" {all option} "Sort Treenodes in Code Browser" {}
                -variable Editor::options(sortProcs)
                -command  {set editorData(options,sortProcs) $Editor::options(sortProcs)
                    Editor::toggleTreeOrder}
            }
            {checkbutton "Highlight current node" {all option} "Highlight current node in Code Browser" {}
                -variable Editor::options(showProc)
                -command  {set EditorData(options,showProc) $Editor::options(showProc)
                    catch {Editor::selectObject 0}}
            }
            {checkbutton "Auto Update" {all option} "Automatic Code Browser Update" {}
                -variable Editor::options(autoUpdate)
                -command  {set EditorData(options,autoUpdate) $Editor::options(autoUpdate)
                    catch {Editor::updateObjects}}
            }
            {separator}
            {checkbutton "Syntax &Highlighting" {all option} "Use Keyword Highlighting" {}
                -variable Editor::options(useHL)
                -command  {set EditorData(options,useHL) $Editor::options(useHL)
                    catch editorWindows::colorize}
            }
            {checkbutton "&Indent" {all option} "Use Indent" {}
                -variable Editor::options(useIndent)
                -command  {set EditorData(options,useIndent) $Editor::options(useIndent)
                }
            }
            {checkbutton "&SyntaxIndent" {all option} "Use Syntax Indent" {}
                -variable Editor::options(useSI)
                -command  {set EditorData(options,useSintaxIndent) $Editor::options(useSI)
                }
            }
            
            {checkbutton "Auto Completion for Keywords" {all option} "Use Templates for Keywords" {}
                -variable Editor::options(useKeywordTemplates)
                -command  {set EditorData(options,useTemplatesForKeywords) $Editor::options(useKeywordTemplates)
                }
            }
            {checkbutton "Auto Completion for \{" {all option} "Use Templates for Braces" {}
                -variable Editor::options(useBracesTemplates)
                -command  {set EditorData(options,useTemplatesForBrace) $Editor::options(useBracesTemplates)
                }
            }
            {checkbutton "Auto Completion for \(" {all option} "Use Templates for Paren" {}
                -variable Editor::options(useParenTemplates)
                -command  {set EditorData(options,useTemplatesForParen) $Editor::options(useParenTemplates)
                }
            }
            {checkbutton "Auto Completion for \[" {all option} "Use Templates for Bracket" {}
                -variable Editor::options(useBracketTemplates)
                -command  {set EditorData(options,useTemplatesForBracket) $Editor::options(useBracketTemplates)
                }
            }
            {checkbutton "Auto Completion for \"" {all option} "Use Templates for DblQuotes" {}
                -variable Editor::options(useQuotesTemplates)
                -command  {set EditorData(options,useTemplatesForQuoteDbl) $Editor::options(useQuotesTemplates)
                }
            }
            {separator}
            {checkbutton "Change tabs into spaces" {all option} "Change tabs into spaces" {}
                -variable Editor::options(changeTabs)
                -command  {set EditorData(options,changeTabs) $Editor::options(changeTabs)
                }
            }
        }
        "&View" all options 0 {
            {checkbutton "Show Console" {all option} "Show Console Window" {}
                -variable Editor::options(showConsole)
                -command  {set EditorData(options,showConsole) $Editor::options(showConsole)
                    Editor::showConsole $EditorData(options,showConsole)
                    update idletasks
                    catch {$Editor::current(text) see insert}
                }
            }
            {checkbutton "Show Code Browser" {all option} "Show Code Browser" {}
                -variable Editor::options(showProcs)
                -command  {set EditorData(options,showProcs) $Editor::options(showProcs)
                    Editor::showTreeWin $EditorData(options,showProcs)
                    update idletasks
                    catch {$Editor::current(text) see insert}
                }
            }
            {checkbutton "Solely Console" {all option} "Only Console Window" {}
                -variable Editor::options(solelyConsole)
                -command  {set EditorData(options,solelyConsole) $Editor::options(solelyConsole)
                    Editor::showSolelyConsole $EditorData(options,solelyConsole)
                    update idletasks
                }
            }
        }
        "&Help" {} {} 0 {
            {command "About" {} "About" {F1} -command Editor::aboutBox }
        }
    }
    
    set Editor::treeMenu [menu .treemenu]
    $Editor::treeMenu add command -label "Top" \
            -command {
                $Editor::current(text) mark set insert 1.0
                $Editor::current(text) see insert
            }
    $Editor::treeMenu add command -label "Bottom" \
            -command {
                $Editor::current(text) mark set insert end
                $Editor::current(text) see insert
            }
    $Editor::treeMenu add command -label "Last Line" \
            -command {
                if {$Editor::current(node) != ""} {
                    set markName $Editor::current(node)
                    append markName "_end_of_proc"
                    $Editor::current(text) mark set insert $markName
                    $Editor::current(text) see insert
                }
    }
    $Editor::treeMenu add separator
    $Editor::treeMenu add checkbutton -label "sort tree" \
            -variable Editor::options(sortProcs) \
            -command  {set editorData(options,sortProcs) $Editor::options(sortProcs)
                Editor::toggleTreeOrder}
    $Editor::treeMenu add separator
    $Editor::treeMenu add command -label "update tree" \
            -command {set cursor [. cget -cursor]
                . configure -cursor watch
                update
                set updateState $EditorData(options,autoUpdate)
                set EditorData(options,autoUpdate) 1
                Editor::updateObjects
                set EditorData(options,autoUpdate) $updateState
                . configure -cursor $cursor
                Editor::selectObject 0
                update
            }
    
    set Editor::textMenu [menu .textmenu]
    $Editor::textMenu add command -label "cut" -command Editor::cut
    $Editor::textMenu add command -label "copy" -command Editor::copy
    $Editor::textMenu add command -label "paste" -command Editor::paste
    $Editor::textMenu add separator
    $Editor::textMenu add command -label "undo" -command Editor::undo
    $Editor::textMenu add command -label "redo" -command Editor::redo
    $Editor::textMenu add separator
    $Editor::textMenu add command -label "Auto Indent Selection" -command editorWindows::autoIndent
    $Editor::textMenu add separator
    $Editor::textMenu add separator
    $Editor::textMenu add checkbutton -label "Auto Update" \
            -variable Editor::options(autoUpdate) \
            -command  {
                set EditorData(options,autoUpdate) $Editor::options(autoUpdate)
                set EditorData(options,showProc) $Editor::options(autoUpdate)
                set Editor::options(showProc) $Editor::options(autoUpdate)
                catch {
                    if {$Editor::options(autoUpdate)} {
                        Editor::updateObjects
                    }
        }
    }
    $Editor::textMenu add checkbutton -label "Show Console" \
            -variable Editor::options(showConsole) \
            -command  {
                set EditorData(options,showConsole) $Editor::options(showConsole)
                Editor::showConsole $EditorData(options,showConsole)
                update idletasks
                catch {$Editor::current(text) see insert}
            }
    $Editor::textMenu add checkbutton -label "Show Code Browser" \
            -variable Editor::options(showProcs) \
            -command  {
                set EditorData(options,showProcs) $Editor::options(showProcs)
                Editor::showTreeWin $EditorData(options,showProcs)
                update idletasks
                catch {$Editor::current(text) see insert}
            }
    $Editor::textMenu add checkbutton -label "Solely Console" \
            -variable Editor::options(solelyConsole) \
            -command  {
                set EditorData(options,solelyConsole) $Editor::options(solelyConsole)
                Editor::showSolelyConsole $EditorData(options,solelyConsole)
                update idletasks
            }
    
    set Editor::prgindic -1
    set Editor::status ""
    set mainframe [MainFrame::create .mainframe \
            -menu $descmenu \
            -textvariable Editor::status \
            -progressvar  Editor::prgindic \
            -progressmax 100 \
            -progresstype normal \
            -progressfg blue ]
    $mainframe showstatusbar progression
    # toolbar 1 creation
    set tb1  [MainFrame::addtoolbar $mainframe]
    set bbox [ButtonBox::create $tb1.bbox1 -spacing 0 -padx 1 -pady 1]
    set toolbarButtons(new) [ButtonBox::add $bbox -image [Bitmap::get new] \
            -highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 1 -pady 1 \
            -helptext "Create a new file" -command Editor::newFile ]
    set toolbarButtons(open) [ButtonBox::add $bbox -image [Bitmap::get open] \
            -highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 1 -pady 1 \
            -helptext "Open an existing file" -command Editor::openFile ]
    set toolbarButtons(save) [ButtonBox::add $bbox -image [Bitmap::get save] \
            -highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 1 -pady 1 \
            -helptext "Save file" -command Editor::saveFile]
    set toolbarButtons(saveAll) [ButtonBox::add $bbox -image [Bitmap::get saveAll] \
            -highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 1 -pady 1 \
            -helptext "Save all files" -command Editor::saveAll]
    set toolbarButtons(close) [ButtonBox::add $bbox -image [Bitmap::get delete] \
            -highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 1 -pady 1 \
            -helptext "Close File" -command Editor::closeFile]
    
    pack $bbox -side left -anchor w
    
    set sep0 [Separator::create $tb1.sep0 -orient vertical]
    pack $sep0 -side left -fill y -padx 4 -anchor w
    
    set bbox [ButtonBox::create $tb1.bbox2 -spacing 0 -padx 1 -pady 1]
    set toolbarButtons(cut) [ButtonBox::add $bbox -image [Bitmap::get cut] \
            -highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 1 -pady 1 \
            -helptext "Cut selection" -command Editor::cut]
    set toolbarButtons(copy) [ButtonBox::add $bbox -image [Bitmap::get copy] \
            -highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 1 -pady 1 \
            -helptext "Copy selection" -command Editor::copy]
    set toolbarButtons(paste) [ButtonBox::add $bbox -image [Bitmap::get paste] \
            -highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 1 -pady 1 \
            -helptext "Paste selection" -command Editor::paste]
    
    pack $bbox -side left -anchor w
    set sep2 [Separator::create $tb1.sep2 -orient vertical]
    pack $sep2 -side left -fill y -padx 4 -anchor w
    
    set bbox [ButtonBox::create $tb1.bbox2b -spacing 0 -padx 1 -pady 1]
    set toolbarButtons(toglcom) [ButtonBox::add $bbox -image [Bitmap::get toglcom] \
            -highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 1 -pady 1 \
            -helptext "Toggle Comment" -command Editor::toggle_comment]
    set toolbarButtons(comblock) [ButtonBox::add $bbox -image [Bitmap::get comblock] \
            -highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 1 -pady 1 \
            -helptext "Make Comment Block" -command Editor::make_comment_block]
    set toolbarButtons(unindent) [ButtonBox::add $bbox -image [Bitmap::get unindent] \
            -highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 1 -pady 1 \
            -helptext "Unindent Selection" -command editorWindows::unindentSelection]
    set toolbarButtons(indent) [ButtonBox::add $bbox -image [Bitmap::get indent] \
            -highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 1 -pady 1 \
            -helptext "Indent Selection" -command editorWindows::indentSelection]
    pack $bbox -side left -anchor w
    
    set sep1c [Separator::create $tb1.sep1c -orient vertical]
    pack $sep1c -side left -fill y -padx 4 -anchor w
    
    set bbox [ButtonBox::create $tb1.bbox3 -spacing 0 -padx 1 -pady 1]
    
    set toolbarButtons(undo) [ButtonBox::add $bbox -image [Bitmap::get undo] \
            -highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 1 -pady 1 \
            -helptext "Undo" -command Editor::undo ]
    
    set toolbarButtons(redo) [ButtonBox::add $bbox -image [Bitmap::get redo] \
            -highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 1 -pady 1 \
            -helptext "Redo" -command Editor::redo ]
    pack $bbox -side left -anchor w
    
    set sep3 [Separator::create $tb1.sep3 -orient vertical]
    pack $sep3 -side left -fill y -padx 4 -anchor w
    
    set bbox [ButtonBox::create $tb1.bbox4 -spacing 0 -padx 1 -pady 1]
    
    set toolbarButtons(find) [ButtonBox::add $bbox -image [Bitmap::get find] \
            -highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 1 -pady 1 \
            -helptext "Find Dialog" -command Editor::search_dialog ]
    
    set toolbarButtons(replace) [ButtonBox::add $bbox -image [Bitmap::get replace] \
            -highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 1 -pady 1 \
            -helptext "Replace Dialog" -command Editor::replace_dialog ]
    
    pack $bbox -side left -anchor w
    
    set search_combo [ComboBox::create $tb1.combo -label "" -labelwidth 0 -labelanchor w \
            -textvariable Editor::search_var\
            -values {""} \
            -helptext "Enter Searchtext" \
            -entrybg white\
            -width 15]
    pack $search_combo -side left
    
    set bbox [ButtonBox::create $tb1.bbox5 -spacing 1 -padx 1 -pady 1]
    
    set down_arrow [ArrowButton::create $bbox.da -dir bottom \
            -height 21\
            -width 21\
            -helptype balloon\
            -helptext "Search forwards"\
            -command Editor::search_forward]
    set up_arrow [ArrowButton::create $bbox.ua -dir top\
            -height 21\
            -width 21\
            -helptype balloon\
            -helptext "Search backwards"\
            -command Editor::search_backward]
    
    pack $down_arrow $up_arrow -side left
    pack $bbox -side left -anchor w
    
    set sep [Separator::create $tb1.sep -orient vertical]
    pack $sep -side left -fill y -padx 4 -anchor w
    
    set bbox [ButtonBox::create $tb1.bbox1b -spacing 0 -padx 1 -pady 1]
    ButtonBox::add $bbox -image [Bitmap::get stop] \
            -highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 1 -pady 1 \
            -helptext "Terminate Execution" -command Editor::terminate
    
    pack $bbox -side left -anchor w -padx 2
    
    set bbox [ButtonBox::create $tb1.bbox1c -spacing 1 -padx 1 -pady 1]
    
    set right_arrow [ArrowButton::create $bbox.ua -dir right\
            -height 21\
            -width 21\
            -helptype balloon\
            -helptext "Test Application"\
            -command {
                set code [catch Editor::execFile cmd]
                if $code {
                    tk_messageBox -message $errorInfo -title Error -icon error
                }
    }]
    
    pack $right_arrow -side left -padx 4
    pack $bbox -side left -anchor w
    
    set argument_combo [ComboBox::create $tb1.combo2 -label "" -labelwidth 0 -labelanchor w \
            -textvariable Editor::argument_var\
            -values {""} \
            -helptext "Enter optional argument" \
            -entrybg white\
            -width 15]
    pack $argument_combo -side left
    
    set sep4 [Separator::create $tb1.sep4 -orient vertical]
    pack $sep4 -side left -fill y -padx 4 -anchor w
    
    set bbox [ButtonBox::create $tb1.bbox6 -spacing 1 -padx 1 -pady 1]
    ButtonBox::add $bbox -image [Bitmap::get exitdoor] \
            -highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 1 -pady 1 \
            -helptext "Exit Ased" -command Editor::exit_app
    pack $bbox -side right -anchor w
    
    #get Entry path out of Combo Widget
    set childList [winfo children $search_combo]
    foreach w $childList {if {[winfo class $w] == "Entry"} { set entry $w ; break}}
    bind $entry <KeyRelease-Return> {Editor::search_forward ; break}
    set childList [winfo children $argument_combo]
    foreach w $childList {if {[winfo class $w] == "Entry"} { set entry2 $w ; break}}
    bind $entry2 <KeyRelease-Return> {
        set code [catch Editor::execFile cmd]
        if $code {
            tk_messageBox -message $errorInfo -title Error -icon error
        }
        break
    }
    
    # toolbar 2 creation
    set tb2  [MainFrame::addtoolbar $mainframe]
    
    set bbox [ButtonBox::create $tb2.bbox2 -spacing 0 -padx 1 -pady 1]
    
    ButtonBox::add $bbox -image [Bitmap::get incfont] \
            -highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 1 -pady 1 \
            -helptext "Increase Fontsize" -command {Editor::increaseFontSize up}
    pack $bbox -side left -anchor w
    
    ButtonBox::add $bbox -image [Bitmap::get decrfont] \
            -highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 1 -pady 1 \
            -helptext "Decrease Fontsize" -command {Editor::increaseFontSize down}
    pack $bbox -side left -anchor w
    set sep3 [Separator::create $tb2.sep3 -orient vertical]
    pack $sep3 -side left -fill y -padx 4 -anchor w
    set Editor::Font_var [font configure editorFont -family]
    set Font_combo [ComboBox::create $tb2.combo -label "" -labelwidth 0 -labelanchor w \
            -textvariable Editor::Font_var\
            -values [font families] \
            -helptext "Choose Font" \
            -entrybg white\
            -modifycmd {Editor::changeFont}\
            ]
    pack $Font_combo -side left
    set Editor::FontSize_var [font configure editorFont -size]
    set FontSize_combo [ComboBox::create $tb2.combo2 -width 2 -label "" -labelwidth 0 -labelanchor w \
            -textvariable Editor::FontSize_var\
            -values {8 9 10 11 12 14 16 20 24 30} \
            -helptext "Choose Fontsize" \
            -entrybg white\
            -modifycmd {Editor::changeFont}\
            ]
    pack $FontSize_combo -side left
    
    $Editor::mainframe showtoolbar 0 $Editor::toolbar1
    $Editor::mainframe showtoolbar 1 $Editor::toolbar2
    
    # set statusbar indicator for file-directory clock and Line/Pos
    set temp [MainFrame::addindicator $mainframe -text "Current Project: " ]
    set temp [MainFrame::addindicator $mainframe -textvariable Editor::current(project) ]
    set temp [MainFrame::addindicator $mainframe -text " File: " ]
    set temp [MainFrame::addindicator $mainframe -textvariable Editor::current(file) ]
    set temp [MainFrame::addindicator $mainframe -textvariable EditorData(cursorPos)]
    set temp [MainFrame::addindicator $mainframe -textvariable clock_var]
    
    # NoteBook creation
    
    set frame    [$mainframe getframe]
    
    set pw1 [PanedWindow::create $frame.pw -side left]
    set pane [PanedWindow::add $pw1 -minsize 200]
    set pw2 [PanedWindow::create $pane.pw -side top]
    
    
    set pane1 [PanedWindow::add $pw2 -minsize 100]
    set pane2 [PanedWindow::add $pw2 -minsize 100]
    set pane3 [PanedWindow::add $pw1 -minsize 100]
    
    set list_notebook [NoteBook::create $pane1.nb]
    set notebook [NoteBook::create $pane2.nb]
    set con_notebook [NoteBook::create $pane3.nb]
    
    set pf1 [EditManager::create_treeWindow $list_notebook]
    set treeWindow $pf1.sw.objTree
    
    # Binding on tree widget
    $treeWindow bindText <ButtonPress-1> Editor::tselectObject
    $treeWindow bindImage <ButtonPress-1> Editor::tselectObject
    
    bind $treeWindow <Button-3> {
        tk_popup $Editor::treeMenu %X %Y
    }
    
    
    set f0 [EditManager::create_text $notebook "Untitled"]
    set Editor::text_win($Editor::index_counter,undo_id) [new textUndoer [lindex $f0 2]]
    
    NoteBook::compute_size $list_notebook
    pack $list_notebook -side left -fill both -expand yes -padx 4 -pady 4
    
    NoteBook::compute_size $notebook
    pack $notebook -side left -fill both -expand yes -padx 4 -pady 4
    
    pack $pw2 -fill both -expand yes
    
    set cf0 [EditManager::create_conWindow $con_notebook]
    NoteBook::compute_size $con_notebook
    pack $con_notebook -side bottom -fill both -expand yes -padx 4 -pady 4
    
    pack $pw1 -fill both -expand yes
    
    $list_notebook raise objtree
    $con_notebook raise asedCon
    $notebook raise [lindex $f0 1]
    
    pack $mainframe -fill both -expand yes
    
    update idletasks
    destroy .intro
    wm protocol . WM_DELETE_WINDOW Editor::exit_app
    
    if {!$configError} {catch Editor::restoreWindowPositions}
}

proc Editor::changeFont {} {
    global EditorData
    
    variable index
    variable text_win
    variable Font_var
    variable FontSize_var
    
    
    font configure editorFont -family $Font_var \
            -size $FontSize_var
    set EditorData(options,fonts,editorFont) [font configure editorFont]
    
    font configure commentFont -family $Font_var \
            -size $FontSize_var
    set EditorData(options,fonts,commentFont) [font configure commentFont]
    
    font configure keywordFont -family $Font_var \
            -size $FontSize_var
    set EditorData(options,fonts,keywordFont) [font configure keywordFont]
    
    set newlist ""
    set index_list [array names index]
    foreach idx $index_list {\
        set newlist [concat $newlist  $index($idx)]
    }
    
    foreach idx $newlist {\
        editorWindows::onChangeFontSize $text_win($idx,path)
    }
    $conWindow configure -font $EditorData(options,fonts,editorFont)
    
}

proc Editor::increaseFontSize {direction} {
    global EditorData
    global conWindow
    
    variable notebook
    variable current
    variable index
    variable text_win
    variable list_notebook
    variable pw2
    variable con_notebook
    variable pw1
    variable mainframe
    variable FontSize_var
    
    
    set minSize 8
    set maxSize 30
    
    set newlist ""
    set index_list [array names index]
    foreach idx $index_list {\
        set newlist [concat $newlist  $index($idx)]
    }
    
    if {$direction == "up"} {
        if {[font configure editorFont -size] == $maxSize} {
            return
        }
        font configure editorFont -size [expr [font configure editorFont -size]+1]
        set EditorData(options,fonts,editorFont) [font configure editorFont]
        font configure commentFont -size [expr [font configure commentFont -size]+1]
        set EditorData(options,fonts,commentFont) [font configure commentFont]
        font configure keywordFont -size [expr [font configure keywordFont -size]+1]
        set EditorData(options,fonts,keywordFont) [font configure keywordFont]
    } else  {
        if {[font configure editorFont -size] == $minSize} {
            return
        }
        font configure editorFont -size [expr [font configure editorFont -size]-1]
        set EditorData(options,fonts,editorFont) [font configure editorFont]
        font configure commentFont -size [expr [font configure commentFont -size]-1]
        set EditorData(options,fonts,commentFont) [font configure commentFont]
        font configure keywordFont -size [expr [font configure keywordFont -size]-1]
        set EditorData(options,fonts,keywordFont) [font configure keywordFont]
    }
    
    
    foreach idx $newlist {\
        editorWindows::onChangeFontSize $text_win($idx,path)
    }
    set FontSize_var [font configure editorFont -size]
    $conWindow configure -font $EditorData(options,fonts,editorFont)
    
}

proc Editor::update_font { newfont } {
    variable _wfont
    variable notebook
    variable font
    variable font_name
    variable current
    variable con_notebook
    
    . configure -cursor watch
    if { $font != $newfont } {
        SelectFont::configure $_wfont -font $newfont
        set raised [NoteBook::raise $notebook]
        $current(text) configure -font $newfont
        NoteBook::raise $con_notebook
        $con_notebook configure -font $newfont
        set font $newfont
    }
    . configure -cursor ""
}


proc Editor::_create_intro { } {
    global tcl_platform
    global ASEDsRootDir
    
    set top [toplevel .intro -relief raised -borderwidth 2]
    
    wm withdraw $top
    wm overrideredirect $top 1
    
    set image [image create photo -file [file join $ASEDsRootDir ased.gif]]
    set splashscreen  [label $top.x -image $image]
    pack $splashscreen
    BWidget::place $top 0 0 center
    wm deiconify $top
}

