# -*- tcl -*-
# (C) 1997 Andreas Kupries <a.kupries@westend.com>
#
# CVS: $Id: classParse.tcl,v 1.2 1998/06/03 20:19:29 aku Exp $
#
# @c Helper procedures used by <c classDescription>-objects to extract class
# @c level documentation and defined entities (methods, options, members).
# @s <c classDescription> helper procedures.
# @i helper procedures, documentation parsing, definition extraction
#-----------------------------------------------------------------------
package require Pool_Base

proc class_util_init {} {
    # @c Noop. Just an entrypoint for autoloading.
}


# shortcuts
interp alias {} cd_c {} cd_comment
interp alias {} cd_d {} cd_danger
interp alias {} cd_n {} cd_note
interp alias {} cd_i {} cd_index
interp alias {} cd_o {} cd_option
interp alias {} cd_v {} cd_var
interp alias {} cd_m {} cd_var
interp alias {} cd_member {} cd_var


proc cd_comment {line} {
    # @c Process @comment, @c commands
    # @a line: tail of line containing the command (= the embedded
    # @a line: documentation)

    upvar comment    comment
    upvar ignoreDesc ignoreDesc

    if {$ignoreDesc} {return}

    append comment " $line"
    return
}


proc cd_danger {line} {
    # @c Process @danger, @d commands
    # @a line: tail of line containing the command (= the embedded
    # @a line: documentation)

    upvar danger     danger
    upvar ignoreDesc ignoreDesc

    if {$ignoreDesc} {return}

    append danger " $line"
    return
}


proc cd_note {line} {
    # @c Process @note, @n commands
    # @a line: tail of line containing the command (= the embedded
    # @a line: documentation)

    upvar note       note
    upvar ignoreDesc ignoreDesc

    if {$ignoreDesc} {return}

    append note " $line"
    return
}


proc cd_see {line} {
    # @c Process @see commands
    # @a line: tail of line containing the command (= the embedded
    # @a line: documentation)

    upvar seeAlso    seeAlso
    upvar ignoreDesc ignoreDesc

    if {$ignoreDesc} {return}

    append seeAlso " $line"
    return
}


proc cd_index {line} {
    # @c Process @index, @i commands
    # @a line: tail of line containing the command (= the embedded
    # @a line: documentation)

    upvar keywords   keywords
    upvar ignoreDesc ignoreDesc

    if {$ignoreDesc} {return}

    append keywords ", $line"
    return
}


proc cd_author {line} {
    # @c Process @author commands
    # @a line: tail of line containing the command (= the embedded
    # @a line: documentation)

    upvar authorName authorName
    upvar ignoreDesc ignoreDesc

    if {$ignoreDesc} {return}

    set authorName $line
    return
}


proc cd_option {line} {
    # @c Process @option, @o commands
    # @a line: tail of line containing the command (= the embedded
    # @a line: documentation)

    upvar ovComment ovComment

    regexp -indices "^(\[^ \t\]*)" $line dummy word
    set a [lindex $word 0]
    set e [lindex $word 1]

    set o    [string trimright [string range $line $a $e] :]
    set line [string range $line [incr e] end]

    append ovComment(o,$o) " $line"
    return
}


proc cd_var {line} {
    # @c Process @var, @v, @member, @m commands. The latter 2 are old-style
    # @c commands and will be obsolete in future releases.
    # @a line: tail of line containing the command (= the embedded
    # @a line: documentation)

    upvar ovComment ovComment

    regexp -indices "^(\[^ \t\]*)" $line dummy word
    set a [lindex $word 0]
    set e [lindex $word 1]

    set v    [string trimright [string range $line $a $e] :]
    set line [string range $line [incr e] end]

    append ovComment(v,$v) " $line"
    return
}


proc cd_classvar {line} {
    # @c Process @classvar, @cv commands
    # @a line: tail of line containing the command (= the embedded
    # @a line: documentation)

    upvar ovComment ovComment

    regexp -indices "^(\[^ \t\]*)" $line dummy word
    set a [lindex $word 0]
    set e [lindex $word 1]

    set v    [string trimright [string range $line $a $e] :]
    set line [string range $line [incr e] end]

    append ovComment(cv,$v) " $line"
    return
}


proc cd_extract_definitions {name log spec} {
    # @c extracts the method, member and option definitions
    # @c contained in the class-<a spec>.
    # @a spec: tcl code containing the specification to scan.
    # @a name: name of class to analyse
    # @a log: reference to logger object
    # @r A 5-element list. First element is a list of procedure-definitions,
    # @r followed by the lists of options, members and superclasses. At last
    # @r the list of packages the class depends on.
    # @r See <p fd_extract_definitions> for the format of procedure
    # @r definitions.

    set sList [::pool::list::prev $spec]
    set spec  [::pool::list::last $spec]

    if {[string compare [lindex $sList 0] ":"] == 0} {
	::pool::list::shift sList
    } else {
	set sList {}
    }

    interp create _e -safe
    interp share  {} stdout _e
    interp expose _e source

    _e eval {
	set __vars__  ""
	set __opts__  ""
	set __deps__  ""


	proc var {args} {
	    global  __vars__

	    if {[string compare [lindex $args 0] -array] == 0} {
		lappend __vars__ [lindex $args 1]
	    } else {
		lappend __vars__ [lindex $args 0]
	    }
	    return
	}

	proc option {name args} {
	    global  __opts__
	    lappend __opts__ $name
	    return
	}

	proc method {name arglist body} {
	    #	    puts "found proc $name"
	    global __procedures__ __deps__
	    set    __procedures__($name) [list $arglist $body]

	    # check for required packages inside of methods.
	    set body [::pool::string::removeComments $body]

	    while {[regexp {package[ 	]+require[ 	]+([a-zA-Z0-9_]*)} \
		    $body dummy matchVar]} {
		if {[string length $matchVar] > 0} {
		    lappend __deps__ $matchVar
		}
		#::puts "regsub $dummy"
		regsub $dummy $body {} body
	    }
	    return
	}

	proc unknown {args} {}
	proc package {cmd args} {
	    global  __deps__
	    if {[string compare $cmd require] == 0} {
		lappend __deps__ [lindex $args 0]
	    }
	    return
	}
    }

    set fail [catch {_e eval $spec} msg]
    if {$fail} {
	$log log warning $name: $msg
    }

    set result [list \
	    [_e eval array get __procedures__] \
	    [_e eval set       __opts__] \
	    [_e eval set       __vars__] \
	    $sList \
	    [_e eval set       __deps__]]

    interp delete _e

    return $result
}

