# -*- tcl -*-
# (C) 1997 Andreas Kupries <a.kupries@westend.com>
#
# CVS: $Id: genericFormatter.cls,v 1.2 1999/02/15 14:17:31 aku Exp $
#
# @c The base class of all formatters
# @s Base class of all formatters
# @i Output formatter, formatting output, interface to formatter
# ----------------------------------------------------------------------


class genericFormatter : distInterface {

    # @c This class defines the interface used by the autodoc engine to produce
    # @c the documentation from data embedded into tcl code.


    # @o outputdir: Path to the directory to store the generated files into.

    option outputdir

    # @o replyaddr: Address of person to reply to in case of problems with
    # @o replyaddr: the documented distribution.

    option replyaddr

    # @o adlocation: Location of the Autodoc documentation.  Required for
    # @o adlocation: generation of backreferences for any generated page.

    option adlocation -d ""


    # @o ip: Requests the formatter to use the specified interpreter for
    # @o ip: script evaluation instead of uplevel. Only if not empty.

    option ip


    # @v extension: The extension to add to a page name to convert it into the
    # @v extension: name of the associated file (if there is any). Has to
    # @v extension: contain the separating dot or whatever is used on the
    # @v extension: particular platform to separate file names adn their
    # @v extension: extensions.

    var extension "."

    # @v currentPage: A token allowing the high level formatting routines to
    # @v currentPage: use the current page as target of hyperlinks.

    var currentPage ""



    # -------------------------------------
    # handle pages

    method pageFile {pageName} {
	# @c Converts the name of page into the name of the file containing it.
	# @a pageName: Guess what :-)
	# @r the name of the file containing the specified page.

	return ${pageName}${extension}
    }


    method clear {} {
	# @c Clear internal state, prepare for new scan.
	error "Abstract method called, was not overidden by subclass"
    }


    method newPage {file title {firstheading {}}} {
	# @c Start a new page, implicitly completes the current page.
	# @a file:  name of file to contain the generated page
	# @a title: string to be used as title of the page
	# @a firstheading: string to use in toplevel heading. Defaults
	# @a firstheading: to <a title>. Required to allow hyperlinks
	# @a firstheading: in toplevel headings without violating
	# @a firstheading: HTML syntax in the title.

	error "Abstract method called, was not overidden by subclass"
    }



    method closePage {} {
	# @c Completes the generation of the current page

	error "Abstract method called, was not overidden by subclass"
    }



    method prepareForOutput {} {
	# @c Called just before the generation of output starts. Checks for the
	# @c existence of a target directory and creates it, if necessary.

	if {[string length $opt(-outputdir)] == 0} {
	    error "no target directory given"
	}

	if {! [file exists $opt(-outputdir)]} {
	    file mkdir $opt(-outputdir)
	}
    }


    method getString {script} {
	# @c Executes the specified <a script> in the calling context and
	# @c captures any generated output in a string, which is then returned
	# @c as the result of the call.
	#
	# @a script: The tcl code to execute in the calling context.
	# @r a string containing all output produced by the <a script>

	error "Abstract method called, was not overidden by subclass"
    }



    method write {text} {
	# @c Has to write the specified <a text> into the current page.
	# @a text: The string to place into the current page.

	error "Abstract method called, was not overidden by subclass"
    }




    method par {args} {
	# @c Writes a paragraph into the current page, uses all arguments as
	# @c one string.
	#
	# @a args: The text to format and write as paragraph. Actually a list
	# @a args: of arguments put together into one string

	error "Abstract method called, was not overidden by subclass"
    }


    method parbreak {} {
	# @c Writes a paragraph break into the current page.

	error "Abstract method called, was not overidden by subclass"
    }


    method ampersand {} {
	# @c Writes an ampersand ([ampersand]) into the current page.

	error "Abstract method called, was not overidden by subclass"
    }


    method hrule {} {
	# @c Writes a horizontal rule into the current page.

	error "Abstract method called, was not overidden by subclass"
    }


    method strong {text} {
	# @c Adds the appropriate formatting to the given <a text> to emphasize
	# @c it as strong, then returns the result.
	#
	# @a text: The string to mark as strong.
	# @r The emphasized <a text>.

	error "Abstract method called, was not overidden by subclass"
    }


    method section {title} {
	# @c Adds a section <a title> to the current page.
	# @a title: The text of the title.

	error "Abstract method called, was not overidden by subclass"
    }


    method chapter {title} {
	# @c Adds a chapter <a title> to the current page.
	# @a title: The text of the title.

	error "Abstract method called, was not overidden by subclass"
    }


    method link {text url} {
	# @c Combines its arguments into a hyperlink having the <a text> and
	# @c pointing to the location specified via <a url>
	#
	# @a text: The string to use as textual part of the hyperlink.
	# @a url:  The location to point at.
	# @r the formatted hyperlink.

	error "Abstract method called, was not overidden by subclass"
    }


    method linkDef {code text url} {
	# @c The same as in <m link>, but the result is stored internally
	# @c instead, using <a code> as reference.
	#
	# @a code: The identifier for storage and retrieval of the hyperlink.
	# @a text: The string to use as textual part of the hyperlink.
	# @a url:  The location to point at.

	error "Abstract method called, was not overidden by subclass"
    }


    method linkRef {code} {
	# @r the hyperlink generated by <m linkDef> and then stored under
	# @r <a code>.
	#
	# @a code: The identifier for storage and retrieval of the hyperlink.

	error "Abstract method called, was not overidden by subclass"
    }


    method imgDef {code text geometry imgfile} {
	# @c Stores an hyperlink to an image under <a code>, allowing later
	# @c retrieval via <m imgRef>.
	#
	# @a code:     The identifier for storage and retrieval of the image
	# @a code:     link.
	# @a geometry: A list containing the width and height of the image, in
	# @a geometry: this order. Can be empty. Used to insert geometry
	# @a geometry: information into the link, for better display even if
	# @a geometry: the image is not loaded.
	# @a text:     Alternative text describing the contents of the picture.
	# @a imgfile:  The location to point at, i.e. the image file.

	error "Abstract method called, was not overidden by subclass"
    }


    method imgRef {code} {
	# @r the image link generated by <m imgDef> and then stored under
	# @r <a code>.
	#
	# @a code: The identifier for storage and retrieval of the image link.

	error "Abstract method called, was not overidden by subclass"
    }


    method table {args} {
	# @c Executes the specified script (last argument) in the calling
	# @c context, captures the produced formatted text and organizes it
	# @c into a table. The arguments before the scripts are interpreted
	# @c as 'name=value'-style parameterization.
	#
	# @a args: A list of 'name=value' parameters and a script to evaluate
	# @a args: in the calling context (last element).

	error "Abstract method called, was not overidden by subclass"
    }


    method table_row {args} {
	# @c Executes the specified script (last argument) in the calling
	# @c context, captures the produced formatted text and organizes it
	# @c into a table row. The arguments before the script are interpreted
	# @c as 'name=value'-style parameterization.
	#
	# @a args: A list of 'name=value' parameters and a script to evaluate
	# @a args: in the calling context (last element).

	error "Abstract method called, was not overidden by subclass"
    }


    method table_data {args} {
	# @c Executes the specified script (last argument) in the calling
	# @c context, captures the produced formatted text and organizes it
	# @c into a table cell. The arguments before the script are interpreted
	# @c as 'name=value'-style parameterization.
	#
	# @a args: A list of 'name=value' parameters and a script to evaluate
	# @a args: in the calling context (last element).

	error "Abstract method called, was not overidden by subclass"
    }


    method table_head {args} {
	# @c Executes the specified script (last argument) in the calling
	# @c context, captures the produced formatted text and organizes it
	# @c into a table cell formatted as heading. The arguments before the
	# @c script are interpreted as 'name=value'-style parameterization.
	#
	# @a args: A list of 'name=value' parameters and a script to evaluate
	# @a args: in the calling context (last element).

	error "Abstract method called, was not overidden by subclass"
    }


    method caption {args} {
	# @c Executes the specified script (last argument) in the calling
	# @c context, captures the produced formatted text and organizes it
	# @c into a table caption. The arguments before the script are
	# @c interpreted as 'name=value'-style parameterization.
	#
	# @a args: A list of 'name=value' parameters and a script to evaluate
	# @a args: in the calling context (last element).

	error "Abstract method called, was not overidden by subclass"
    }


    method getAnchor {name} {
	# @c Generates a <a name>d anchor and returns the HTML to the caller.
	# @a name: The name of the generated anchor.
	# @r the HTML string defining the <a name>d anchor.

	error "Abstract method called, was not overidden by subclass"
    }


    method setAnchor {name} {
	# @c Generates a <a name>d anchor at the current location in the
	# @c current page.
	# @a name: The name of the generated anchor.

	error "Abstract method called, was not overidden by subclass"
    }


    method definitionList {script} {
	# @c Executes the specified <a script> in the calling context and
	# @c captures any generated output in a string, which is then formatted
	# @c as definition list.
	#
	# @a script: The tcl code to execute in the calling context.

	error "Abstract method called, was not overidden by subclass"
    }


    method defterm {term text} {
	# @c Generates an item in a definition list.
	# @a term: The name of the thing to explain.
	# @a text: The text explaining the <a term>.

	error "Abstract method called, was not overidden by subclass"
    }


    method defterm2 {term} {
	# @c Generates an item in a definition list.
	# @a term: The name of the thing to explain. But without explanation.

	error "Abstract method called, was not overidden by subclass"
    }


    method itemize {script} {
	# @c Executes the specified <a script> in the calling context and
	# @c captures any generated output in a string, which is then formatted
	# @c as itemized list.
	#
	# @a script: The tcl code to execute in the calling context.

	error "Abstract method called, was not overidden by subclass"
    }


    method enumerate {script} {
	# @c Executes the specified <a script> in the calling context and
	# @c captures any generated output in a string, which is then formatted
	# @c as enumerated list.
	#
	# @a script: The tcl code to execute in the calling context.

	error "Abstract method called, was not overidden by subclass"
    }


    method item {text} {
	# @c Generates an item in an itemized list.
	# @a text: The paragraph to format as item in the list.

	error "Abstract method called, was not overidden by subclass"
    }


    method quote {string} {
	# @c Takes the specified <a string>, add protective signs to all
	# @c character (sequences) having special meaning for the formatter
	# @c and returns the so enhanced text.
	#
	# @a string: The string to protect against interpretation by the
	# @a string: formatter.
	# @r a string containing no unprotected special character (sequences).

	error "Abstract method called, was not overidden by subclass"
    }


    method markError {text} {
	# @c Formats the incoming <a text> as error and returns the modified
	# @c information.
	#
	# @a text: The text to reformat.
	# @r a string containing the given <a text> formatted as error.

	error "Abstract method called, was not overidden by subclass"
    }


    # -------------------------------------
    # higher level procedures providing complexer formatting of whole text
    # blocks. Built upon the low-level method declared before, and therefore
    # independent of the particular system.

    method MakeError {category output error} {
	# @c Internal method to generate a problem report. The report is added
	# @c to the object currently writing its documentation linking it to
	# @c the place of the problem.
	#
	# @a category: Category of the problem, either [strong desc] or
	# @a category: [strong crossref]
	# @a output: The text to print at the place of the problem.
	# @a error: The text to use in the problem report to describe it.
	#
	# @r a string containing the <a output> formatted as problem and having
	# @r an anchor the report can link to.

	set anchor xr[::pool::serial::new]

	[$dist theContext] addProblem desc $error "${currentPage}#${anchor}"
	$dist log error "${currentPage}#${anchor}: $error"

	# fake out the -ip mechanism, enforce evaluation here !

	set ipSave   $opt(-ip)
	set opt(-ip) {}

	set     result [$this getAnchor $anchor]
	append  result [$this markError $output]

	set opt(-ip) $ipSave

	return $result
    }


    method missingDescError {errorText} {
	# @c Generates a documention missing problem at the place of its
	# @c calling. Uses <m MakeError> as workhorse.
	#
	# @a errorText: The text to use in the problem report to describe it.

	return [MakeError desc {not documented} $errorText]
    }


    method crError {outputText errorText} {
	# @c Generates a crossreference problem at the place of its calling.
	# @c Uses <m MakeError> as workhorse.
	#
	# @a outputText: The text to print at the place of the problem.
	# @a errorText:  The text to use in the problem report to describe it.

	return [MakeError crossref $outputText $errorText]
    }



    # definition list blocks

    method formattedTerm {term text} {
	# @c Like <m defterm>, but crossreferences in <a text> are resolved.
	# @a term: The name of the thing to explain.
	# @a text: The text explaining the <a term>.

	$this defterm $term [string trim [$dist crResolve $text]]
	return
    }



    method formattedTermVar {term var} {
	# @c Same as <m formattedTerm>, but the explanation is specified as
	# @c name of a variable. An empty explanation causes the system to
	# @c ignore the call.
	#
	# @a term: The name of the thing to explain.
	# @a var:  The name of the variable containing the epxlanatory text.

	upvar $var v

	if {[string length $v] != 0} {
	    formattedTerm $term $v
	}

	return
    }



    method formattedRow {term text} {
	# @c See <m formattedTerm>, but writes out a table row, not a term of
	# @c a definition list.
	# @a term: The name of the thing to explain.
	# @a text: The text explaining the <a term>.

	$this table_row {
	    $this table_data colspan=3 {
		$this write "${term}: $text"
	    }
	}
	return
    }



    method formattedRowVar {term var} {
	# @c Same as <m formattedRow>, but the explanation is specified as
	# @c name of a variable. An empty explanation causes the system to
	# @c ignore the call. Crossreferences are resolved too.
	#
	# @a term: The name of the thing to explain.
	# @a var:  The name of the variable containing the epxlanatory text.

	upvar $var v

	if {[string length $v] != 0} {
	    formattedRow $term [$dist crResolve $v]
	}

	return
    }



    method termVar {term var} {
	# @c Same as <m formattedTerm>, but the explanation is specified as
	# @c name of a variable. An empty explanation causes the system to
	# @c ignore the call. There is no crossreference resolution.
	#
	# @a term: The name of the thing to explain.
	# @a var:  The name of the variable containing the epxlanatory text.

	# var = member variable

	upvar $var v

	if {[string length $v] != 0} {
	    $this defterm $term $v
	}

	return
    }



    method sortByName {objList} {
	# @c Sorts the specified list of objects alphabetically in ascending
	# @c order by the name of the objects.
	# @r the sorted list.
	# @a objList: List of object handles. The objects in it have to
	# @a objList: understand the [strong name] method.

	set olist ""
	foreach o $objList {
	    lappend olist [list [$o name] $o]
	}

	return [::pool::list::projection [lsort -index 0 $olist] 1]
    }



    method linkCommaList {objList} {
	# @c Takes the specified list of objects and converts it into a comma
	# @c separated list of hyperlinks to the pages describing them.
	# @a objList: List of object handles. The objects in it have to
	# @a objList: understand the [strong link] method.
	# @r a string containing several hyperlinks.

	set text ""

	foreach o $objList {
	    append text "[$o link], "
	}

	return [string trimright $text ", "]
    }



    method commaList {textList} {
	# @c Takes the specified list of strings and converts it into a comma
	# @c separated list.
	# @a textList: List of strings.
	# @r a string separating the incoming texts by commas.

	return [join $textList ", "]
    }


    method sortedObjectList {objlist} {
	# @c Takes the specified list of objects, sorts them by name, then
	# @c generates a definition list containing hyperlinks to the object
	# @c pages as terms and their short descriptions as explanation.
	#
	# @a objlist: List of object handles. The objects in it have to
	# @a objlist: understand the methods [strong link] and [strong short].

	$this definitionList {
	    foreach obj [sortByName $objlist] {
		# Get short description from object
		# Use name and page to construct the hyperlink

		set shortDescription [$obj short]

		if {[string length $shortDescription] != 0} {
		    $this defterm [$obj link] [$dist crResolve $shortDescription]
		} else {
		    $this defterm2 [$obj link]
		}
	    }
	}

	return
    }



    method mailToDefterm {prefix nameVar addrVar} {
	# @c Writes hyperlink for a person and its mail address.
	#
	# @a prefix: String to write before the actual hyperlink
	# @a nameVar: Name of the variable containing the name to write.
	# @a addrVar: Name of the variable containing the address.

	upvar $nameVar nvar
	upvar $addrVar avar

	set name $nvar
	set addr $avar

	# remove address from name, no need for duplication
	regsub -- "$addr" $name {} name
	regsub -- {<>}    $name {} name
	set name [string trim $name]

	if {[string length $addr] != 0} {
	    $this defterm $prefix [$this link $name "mailto:$addr"]
	} else {
	    # no address available, therefore no hyperlink

	    $this defterm $prefix [$this strong $name]
	}

	return
    }



    method linkMail {prefix nameVar addrVar} {
	# @c Generates hyperlink for name (in attr)
	#
	# @a prefix: String to write before the actual hyperlink
	# @a nameVar: Name of the variable containing the name to write.
	# @a addrVar: Name of the variable containing the address.
	# @r A string containing a hyperlink to the given mail address.


	upvar $nameVar nvar
	upvar $addrVar avar

	set name $nvar
	set addr $avar

	# remove address from name, no need for duplication
	regsub -- "$addr" $name {} name
	regsub -- {<>}    $name {} name
	set name [string trim $name]

	if {[string length $addr] != 0} {
	    return "$prefix [$this link $name "mailto:$addr"]"
	} else {
	    # no address available, therefore no hyperlink
	    return "$prefix [$this strong $name]"
	}

	return
    }
}

