# -*- tcl -*-
# (C) 1997 Andreas Kupries <a.kupries@westend.com>
#
# CVS: $Id: htmlFormatter.cls,v 1.3 1999/02/15 14:17:31 aku Exp $
#
# @c The strategy class for production of HTML formatted output.
# @s Strategy class, produces HTML output
# @i 
# ----------------------------------------------------------------------


class htmlFormatter : genericFormatter {
    init {
	# Hack. Create a dummy 'option' command to suppress the '::option'
	# alias of '::cgi_option', as '::option' is used by the OO kernel as
	# well, and the alias will interfere with that.

	if {[llength [info commands option]] == 0} {
	    proc option {} {}

	    package require cgi

	    # force cgi commands into the interpreter
	    catch {cgi_error_occurred}

	    rename option {}
	} else {
	    # Tk loaded, no need to fake out the cgi library

	    package require cgi
	    # force cgi commands into the interpreter
	    catch {cgi_error_occurred}
	}
    }


    # @v pages:       Array containing the names of all HTML files generated
    # @v pages:       so far.

    var -array pages

    # @v footer:      Contains the HTML commands to generate the footer line.

    var footer


    # -------------------------------------


    method htmlFormatter {} {
	# @c Constructor, initializes the file extension.

	set extension .htm
    }


    method RecordPage {pagefile} {
	# @c Internal method used by <m newPage> to keep track of all generated
	# @c files. Trying to write a file twice causes an exception aborting
	# @c the engine.
	#
	# @a pagefile: The name of the to write.

	if {[::info exists pages($pagefile)]} {
	    error "page $pagefile already written"
	}

	set pages($pagefile) .
	return
    }


    method clear {} {
	# @c Clear internal state, prepare for new scan.
	::pool::array::clear pages
    }


    method newPage {file title {firstheading {}}} {
	# @c Start a new page, implicitly completes the current page.
	# @a file:  name of file to contain the generated page
	# @a title: string to be used as title of the page
	# @a firstheading: string to use in toplevel heading. Defaults
	# @a firstheading: to <a title>. Required to allow hyperlinks
	# @a firstheading: in toplevel headings without violating
	# @a firstheading: HTML syntax in the title.

	RecordPage $file
	set currentPage $file

	if {[string length $firstheading] == 0} {
	    set firstheading $title
	}


	if {[string compare [file extension $file] ".htm"] != 0} {
	    set file ${file}.htm
	}

	::pool::cgi::openPage [file join $opt(-outputdir) $file]

	cgi_html_start
	cgi_title $title
	cgi_body_start

	cgi_h1 $firstheading
	return
    }



    method closePage {} {
	# @c Completes the generation of the current page

	if {[string length $footer] == 0} {
	    set app  "AutoDoc 2.1"
	    set now  [::pool::date::now]
	    set user [cgi_link _ [::pool::misc::currentUser] \
		    "mailto:$opt(-replyaddr)"]

	    if {[string length $opt(-adlocation)] != 0} {
		set app [cgi_link _ $app $opt(-adlocation)]
	    }

	    set footer "Generated by $app at $now, invoked by $user"
	}

	# auto footer (page generated by whom, where, when ?)
	cgi_hr
	cgi_p align=right $footer

	cgi_body_end
	cgi_html_end

	::pool::cgi::closePage

	set currentPage {}
	return
    }


    method write {text} {
	# @c Has to write the specified <a text> into the current page.
	# @a text: The string to place into the current page.

	cgi_puts $text
	return
    }


    method getString {script} {
	# @c Executes the specified <a script> in the calling context and
	# @c captures any generated output in a string, which is then returned
	# @c as the result of the call.
	#
	# @a script: The tcl code to execute in the calling context.
	# @r a string containing all output produced by the <a script>

	if {[string length $opt(-ip)] > 0} {
	    return [::pool::cgi::getString {
		$opt(-ip) eval $script
	    }] ;#{}
	} else {
	    return [::pool::cgi::getString {
		uplevel $script
	    }] ;#{}
	}
    }


    method par {args} {
	# @c Writes a paragraph into the current page, uses all arguments as
	# @c one string.
	#
	# @a args: The text to format and write as paragraph. Actually a list
	# @a args: of arguments put together into one string

	eval cgi_p $args
	return
    }


    method parbreak {} {
	# @c Writes a paragraph break into the current page.

	cgi_br
	return
    }


    method ampersand {} {
	# @c Writes an ampersand ([ampersand]) into the current page.

	return [cgi_amp]
    }


    method hrule {} {
	# @c Writes a horizontal rule into the current page.

	cgi_hr
	return
    }


    method strong {text} {
	# @c Adds the appropriate formatting to the given <a text> to emphasize
	# @c it as strong, then returns the result.
	#
	# @a text: The string to mark as strong.
	# @r The emphasized <a text>.

	return [cgi_bold $text]
    }


    method section {title} {
	# @c Adds a section <a title> to the current page.
	# @a title: The text of the title.

	cgi_h2 $title
	return
    }


    method chapter {title} {
	# @c Adds a chapter <a title> to the current page.
	# @a title: The text of the title.

	cgi_h1 $title
	return
    }


    method link {text url} {
	# @c Combines its arguments into a hyperlink having the <a text> and
	# @c pointing to the location specified via <a url>
	#
	# @a text: The string to use as textual part of the hyperlink.
	# @a url:  The location to point at.
	# @r the formatted hyperlink.

	return [cgi_link _ $text $url]
    }


    method linkDef {code text url} {
	# @c The same as in <m link>, but the result is stored internally
	# @c instead, using <a code> as reference.
	#
	# @a code: The identifier for storage and retrieval of the hyperlink.
	# @a text: The string to use as textual part of the hyperlink.
	# @a url:  The location to point at.

	cgi_link $code $text $url
	return
    }


    method linkRef {code} {
	# @r the hyperlink generated by <m linkDef> and then stored under
	# @r <a code>.
	#
	# @a code: The identifier for storage and retrieval of the hyperlink.

	return [cgi_link $code]
    }


    method imgDef {code text geometry imgfile} {
	# @c Stores an hyperlink to an image under <a code>, allowing later
	# @c retrieval via <m imgRef>.
	#
	# @a code:     The identifier for storage and retrieval of the image
	# @a code:     link.
	# @a text:     Alternative text describing the contents of the picture.
	# @a geometry: A list containing the width and height of the image, in
	# @a geometry: this order. Can be empty. Used to insert geometry
	# @a geometry: information into the link, for better display even if
	# @a geometry: the image is not loaded.
	# @a imgfile:  The location to point at, i.e. the image file.

	set w [lindex $geometry 0]
	set h [lindex $geometry 1]

	if {([string length $w] == 0) || ([string length $h] == 0)} {
	    cgi_imglink $code $imgfile "alt=$text"
	} else {
	    cgi_imglink $code $imgfile "alt=$text" "width=$w" "height=$h"
	}
	return
    }


    method imgRef {code} {
	# @r the image link generated by <m imgDef> and then stored under
	# @r <a code>.
	#
	# @a code: The identifier for storage and retrieval of the image link.

	return [cgi_imglink $code]
    }


    method table {args} {
	# @c Executes the specified script (last argument) in the calling
	# @c context, captures the produced formatted text and organizes it
	# @c into a table. The arguments before the scripts are interpreted
	# @c as 'name=value'-style parameterization.
	#
	# @a args: A list of 'name=value' parameters and a script to evaluate
	# @a args: in the calling context (last element).

	set script [::pool::list::pop args]

	if {[string length $opt(-ip)] > 0} {
	    eval cgi_table $args [list [list $opt(-ip) eval $script]]
	} else {
	    uplevel cgi_table $args [list $script]
	}

	return
    }


    method table_row {args} {
	# @c Executes the specified script (last argument) in the calling
	# @c context, captures the produced formatted text and organizes it
	# @c into a table row. The arguments before the script are interpreted
	# @c as 'name=value'-style parameterization.
	#
	# @a args: A list of 'name=value' parameters and a script to evaluate
	# @a args: in the calling context (last element).

	set script [::pool::list::pop args]
	if {[string length $opt(-ip)] > 0} {
	    eval cgi_table_row $args [list [list $opt(-ip) eval $script]]
	} else {
	    uplevel cgi_table_row $args [list $script]
	}
	return
    }


    method table_data {args} {
	# @c Executes the specified script (last argument) in the calling
	# @c context, captures the produced formatted text and organizes it
	# @c into a table cell. The arguments before the script are interpreted
	# @c as 'name=value'-style parameterization.
	#
	# @a args: A list of 'name=value' parameters and a script to evaluate
	# @a args: in the calling context (last element).

	set script [::pool::list::pop args]
	if {[string length $opt(-ip)] > 0} {
	    eval cgi_table_data $args [list [list $opt(-ip) eval $script]]
	} else {
	    uplevel cgi_table_data $args [list $script]
	}
	return
    }


    method table_head {args} {
	# @c Executes the specified script (last argument) in the calling
	# @c context, captures the produced formatted text and organizes it
	# @c into a table cell formatted as heading. The arguments before the
	# @c script are interpreted as 'name=value'-style parameterization.
	#
	# @a args: A list of 'name=value' parameters and a script to evaluate
	# @a args: in the calling context (last element).

	set script [::pool::list::pop args]
	if {[string length $opt(-ip)] > 0} {
	    eval cgi_table_head $args [list [list $opt(-ip) eval $script]]
	} else {
	    uplevel cgi_table_head $args [list $script]
	}
	return
    }


    method caption {args} {
	# @c Executes the specified script (last argument) in the calling
	# @c context, captures the produced formatted text and organizes it
	# @c into a table caption. The arguments before the script are
	# @c interpreted as 'name=value'-style parameterization.
	#
	# @a args: A list of 'name=value' parameters and a script to evaluate
	# @a args: in the calling context (last element).

	set script [::pool::list::pop args]
	if {[string length $opt(-ip)] > 0} {
	    eval cgi_caption $args [list [list $opt(-ip) eval $script]]
	} else {
	    uplevel cgi_caption $args [list $script]
	}
	return
    }


    method getAnchor {name} {
	# @c Generates a <a name>d anchor and returns the HTML to the caller.
	# @a name: The name of the generated anchor.
	# @r the HTML string defining the <a name>d anchor.

	return [cgi_anchor_name $name]
    }


    method setAnchor {name} {
	# @c Generates a <a name>d anchor at the current location in the
	# @c current page.
	# @a name: The name of the generated anchor.

	# @d Usable only in conjunction with cgi 0.7,
	# @d and not cgi 0.4, as for earlier versions of autodoc.

	cgi_puts [cgi_anchor_name $name]
	return
    }


    method definitionList {script} {
	# @c Executes the specified <a script> in the calling context and
	# @c captures any generated output in a string, which is then formatted
	# @c as definition list.
	#
	# @a script: The tcl code to execute in the calling context.

	if {[string length $opt(-ip)] > 0} {
	    cgi_definition_list {
		$opt(-ip) eval $script
	    }
	} else {
	    uplevel cgi_definition_list [list $script]
	}
	return
    }


    method defterm {term text} {
	# @c Generates an item in a definition list.
	# @a term: The name of the thing to explain.
	# @a text: The text explaining the <a term>.

	cgi_term            $term
	cgi_term_definition $text
	return
    }


    method defterm2 {term} {
	# @c Generates an item in a definition list.
	# @a term: The name of the thing to explain. But without explanation.

	cgi_term $term
	return
    }


    method itemize {script} {
	# @c Executes the specified <a script> in the calling context and
	# @c captures any generated output in a string, which is then formatted
	# @c as itemized list.
	#
	# @a script: The tcl code to execute in the calling context.

	if {[string length $opt(-ip)] > 0} {
	    cgi_bullet_list {
		::$opt(-ip) eval $script
	    }
	} else {
	    uplevel cgi_bullet_list [list $script]
	}
	return
    }


    method enumerate {script} {
	# @c Executes the specified <a script> in the calling context and
	# @c captures any generated output in a string, which is then formatted
	# @c as itemized list.
	#
	# @a script: The tcl code to execute in the calling context.

	if {[string length $opt(-ip)] > 0} {
	    cgi_number_list {
		$opt(-ip) eval $script
	    }
	} else {
	    uplevel cgi_number_list [list $script]
	}
	return
    }


    method item {text} {
	# @c Generates an item in an itemized list.
	# @a text: The paragraph to format as item in the list.

	cgi_li $text
	return
    }


    method quote {string} {
	# @c Takes the specified <a string>, add protective signs to all
	# @c character (sequences) having special meaning for the formatter
	# @c and returns the so enhanced text.
	#
	# @a string: The string to protect against interpretation by the
	# @a string: formatter.
	# @r a string containing no unprotected special character (sequences).

	return [cgi_quote_html $string]
    }


    method markError {text} {
	# @c Formats the incoming <a text> as error and returns the modified
	# @c information.
	#
	# @a text: The text to reformat.
	# @r a string containing the given <a text> formatted as error.

	return [cgi_font color=red $text]
    }
}

