# -*- tcl -*-
# (C) 1997 Andreas Kupries <a.kupries@westend.com>
#
# CVS: $Id: proc.cls,v 1.8 1998/05/26 20:34:38 aku Exp $
#
# @c Management of procedures and class methods.
# @s Procedures, methods
# @i procedure, method
# ----------------------------------------------


class procDescription : azIndexEntry problems {

    # @o parent: Handle of object containing the procedure.
    # @o parent: Can be a file or class (<c fileDescription>,
    # @o parent: <c classDescription>).
    # @o args:   List containing the names of all procedure arguments.
    # @o body:   Script containing the body of the procedure definition.
    # @o table:  Boolean flag. If set procedure documentation is written
    # @o table   as table.

    option parent
    option args
    option body
    option table -d 0

    # @v keywords: List of keywords associated to the procedure (by @index).
    # @v comment:  Description of procedure, as given by @comment.
    # @v danger:   Description of dangers in usage of the procedure,
    # @v danger:   as given by @danger.
    # @v note:     Notes about usage of the procedure, as given by @note.
    # @v result:   Description of procedure result if any.
    # @v arglist:  List containing the names of all arguments to the procedure.
    # @v anchor:   Name of the anchor associated to the procedure.

    var keywords
    var comment
    var danger
    var note
    var result
    var arglist
    var anchor ""

    # @v authorName:    Name of author, may contain a mail-address too.
    # @v authorName:    Specified via @author.
    # @v authorAddress: Mail address of author, as extracted
    # @v authorAddress: from <v authorName>.

    var authorName
    var authorAddress

    # @v argDef: Maps the arguments of the procedure/method to information
    # @v argDef: describing them (default value, textual description)

    var -array argDef




    method procDescription {} {
	# @c Constructor. Initializes the parser functionality in
	# @c <f lib/procParse.tcl>.

	pd_util_init
	return
    }



    method firstLetter {} {
	# @c Overides base classs functionality (<m azIndexEntry:firstLetter>)
	# @c to use the first letter of the actual procedure as indexing
	# @c criterium, [strong not] the first letter of the namespace path.

	set pname [lindex [split $opt(-name) :] end]

	regexp -- {^[^a-zA-Z]*([a-zA-Z])} $pname dummy letter
	return $letter
    }



    method keywords {} {
	# @r The list of keywords this procedure is indexed under.
	return $keywords
    }



    method completeKwIndex {phrases} {
	# @c Called by the containing file (or class) to propagate its
	# @c indexing information.
	#
	# @a phrases: List of index phrases used by the containing file
	# @a phrases: or class.

	# no propogation if we have our own keywords.

	if {[llength $keywords] != 0} {
	    return
	}

	set keywords $phrases
	return
    }



    method authorSet {author} {
	# @c Called by the containing file or class to propagate its author
	# @c information.
	#
	# @a author:  Author of containing file.

	if {[string length $authorName] == 0} {
	    set authorName $author
	}

	set currentAddr   [::pool::misc::currentAddress]
	set authorAddress [::pool::mail::addressB $authorName $currentAddr]
	return
    }



    method scan {} {
	# @c Scans the procedure body for embedded documentation.

	# i. argument list
	foreach a $opt(-args) {
	    if {[llength $a] == 2} {
		set aname [lindex $a 0]
		set argDef($aname,default) 1
		set argDef($aname,defaultValue) [lindex $a 1]
	    } else {
		set aname $a
		set argDef($aname,default) 0
	    }

	    lappend arglist $aname
	    set argDef($aname,comment) ""
	}

	# ii. now get the information contained in comments
	#     inside the procedure body, just like for files

	set opt(-body) [split $opt(-body) \n]
	while {[llength $opt(-body)] > 0} {
	    set line [::pool::list::shift opt(-body)]
	    set line [string trim $line]
	    if {![regexp ^# $line]} {continue}
	    set line [string trimleft $line "#\t "]
	    if {![regexp ^@ $line]} {continue}
	    set line [string trimleft $line "@"]

	    regexp -indices "^(\[^ \t\]*)" $line dummy word
	    set start [lindex $word 0]
	    set end   [lindex $word 1]

	    set cmd  [string range $line $start $end]
	    set line [string range $line [incr end] end]

	    if {[llength [::info command pd_$cmd]] == 0} {
		continue
	    }

	    pd_$cmd [string trim $line]
	}

	PostProcessInformation
	return
    }



    method write {} {
	# @c Generates the formatted text describing this procedure.
	# @c In contrast to files, packages and classes no separate page is
	# @c written. As this method is called from inside
	# @c <m fileDescription:write> (or <m classDescription:write>) the
	# @c description will go into this page. This is reflected by
	# @c <m pPage> too!

	# @n Dependent on the configuration either <m WriteStandard> or
	# @n <m WriteAsTable> is called to do the real work.

	#puts "$this writing html"

	$dist pushContext $this
	$fmt setAnchor [Anchor]

	if {$opt(-table)} {
	    WriteAsTable
	} else {
	    WriteStandard
	}

	ReportSpamArguments

	$dist popContext
	return
    }



    method WriteAsTable {} {
	# @c Generates the formatted text describing the procedure. This
	# @c method generates a tabular formatting.

	$fmt par [$fmt getString {
	    $fmt table border width=100% {
		# name
		$fmt table_row {
		    $fmt table_data colspan=3 {
			$fmt section "[name] ($arglist)"
		    }
		}

		# author reference
		if {[string compare $authorName [$opt(-parent) author]] != 0} {
		    $fmt table_row {
			$fmt table_data colspan=3 {
			    $fmt write \
				    [$fmt linkMail [$fmt strong by] \
				    authorName authorAddress]
			}
		    }
		}

		# description sections
		if {
		    ([string length $comment]  != 0) ||
		    (([string length $comment] == 0) &&
		     ([string length $result]  == 0))
		} {
		    # write comment section if text present or neither
		    # description nor description of result in existence.
		    # Result: A missing description is tolerated if a
		    # return value was described. It is assumed that this
		    # description is sufficient.
		    
		    $fmt table_row {
			$fmt table_data colspan=3 {
			    $fmt write [ProcDescription]
			}
		    }
		}

		$fmt formattedRowVar [$fmt strong Dangers] danger
		$fmt formattedRowVar [$fmt strong Notes]   note

		WriteArgumentsAsTable

		$fmt formattedRowVar [$fmt strong Returns] result
	    }
	}] ;#{}

	return
    }



    method ArgDescription {a} {
	# @c Internal method used by all procedures generating a formatted
	# @c procedure description.
	# @r the description of the given <a a>rgument or an error text.
	#
	# @a a: The name of the procedure argument to look at.

	if {[string length $argDef($a,comment)] == 0} {

	    return [$fmt missingDescError \
		    "Argument [$fmt markError $a] not documented"]
	} else {
	    return [$dist crResolve $argDef($a,comment)]
	}
    }



    method ArgName {a} {
	# @c Internal method used by all procedures generating a formatted
	# @c procedure description.
	# @r the given <a a>rgument itself, or the argument with its
	# @r defaultvalue attached to it.
	#
	# @a a: The name of the procedure argument to look at.

	if {$argDef($a,default)} {
	    # Argument has default value, add this to the name

	    if {[string length $argDef($a,defaultValue)] == 0} {
		return "$a (= \{\})"
	    } else {
		return "$a (= $argDef($a,defaultValue))"
	    }
	} else {
	    # Simple argument without default

	    return $a
	}
    }



    method ReportSpamArguments {} {
	# @c Report all arguments which were documented, but are
	# @c no arguments of the procedure.

	foreach key [array names argDef *,comment] {
	    set arg [lindex [split $key ,] 0]

	    if {[lsearch -exact $arglist $arg] < 0} {
		addProblem crossref \
			"Documented undefined argument [$fmt markError $arg]" \
			"[page]"
	    }
	}
	return
    }


    method hasArgument {a} {
	# @c Checks wether the procedure has an argument named <a a>.
	# @a a: The name of the possible argument to this procedure.
	# @r A boolean value. True if <a a> is an argument of this procedure,
	# @r or else False.

	return [::info exists argDef($a,default)]
    }



    method WriteArgumentsAsList {} {
	# @c Generates a definition list describing the arguments of this
	# @c procedure.

	if {[llength $arglist] != 0} {
	    $fmt defterm Arguments [$fmt getString {
		$fmt definitionList {
		    foreach a $arglist {
			$fmt defterm [ArgName $a] [ArgDescription $a]
		    }
		}
	    }] ;#{}
	}
	return
    }



    method WriteArgumentsAsTable {} {
	# @c Generates a table describing the arguments of this procedure.

	if {[llength $arglist] != 0} {
	    foreach a $arglist {
		$fmt table_row {
		    $fmt table_data {
			$fmt write "[$fmt strong Argument]: [ArgName $a]"
		    }
		    $fmt table_data colspan=2 {
			$fmt write [ArgDescription $a]
		    }
		}
	    }
	}
	return
    }



    method ProcDescription {} {
	# @r a string containing the documentation of this procedure or an
	# @r error text.

	if {[string length $comment] == 0} {
	    return [$fmt missingDescError "Documentation missing"]
	} else {
	    return [$dist crResolve $comment]
	}
    }



    method WriteStandard {} {
	# @c Generates the formatted text describing the procedure. This
	# @c method generates a definition list.

	$fmt definitionList {
	    $fmt defterm [$fmt strong "[name] ($arglist)"] [$fmt getString {
		$fmt definitionList {
		    if {
			([string length $comment]  != 0) ||
			(([string length $comment] == 0) &&
			 ([string length $result]  == 0))
		    } {
			# write comment section if text present or neither
			# description nor description of result in existence.
			# Result: A missing description is tolerated if a
			# return value was described. It is assumed that this
			# description is sufficient.

			$fmt defterm Description [ProcDescription]
		    }

		    $fmt formattedTermVar Dangers danger
		    $fmt formattedTermVar Notes   note
		    
		    WriteArgumentsAsList

		    $fmt formattedTermVar Result  result

		    if {[string compare $authorName [$opt(-parent) author]] != 0} {
			$fmt mailToDefterm "Written by" \
				authorName authorAddress
		    }
		}
	    }] ;#{}
	}
	return
    }



    method getPage {} {
	# @r The url of the page containing the procedure description.

	return [$opt(-parent) page]#[Anchor]
    }



    method Anchor {} {
	# @r The name of the anchor at the start of the procedure description.

	if {[string length $anchor] == 0} {
	    set anchor p[::pool::serial::new]
	}

	return $anchor
    }



    method PostProcessInformation {} {
	# @c Called to streamline the extracted embedded documentation.
	# @c This mainly consists of removing superfluous whitespace.
	# @c Additionally converts a comma separated list of index
	# @c phrases into a real tcl list.

	foreach i {comment danger note result} {
	    set text [::pool::string::removeWhitespace [set $i]]
	    set $i   [string trim [::pool::string::oneLine $text]]
	}

	foreach aName $arglist {
	    set argComment \
		    [::pool::string::removeWhitespace $argDef($aName,comment)]

	    set argDef($aName,comment) \
		    [string trim [::pool::string::oneLine $argComment]]
	}

	set keywords [::pool::string::removeWhitespace $keywords]
	set keywords [split [string trim $keywords ", \t"] ,]

	while {[::pool::list::delete keywords {}]} {}
	return
    }



    method addProblem {category errortext url} {
	# @c Overides the baseclass functionality (<m problems:addProblem>) to
	# @c allow the procedure to signal itself as problematic to its parent
	# @c (file or class).
	#
	# @a category:  Either [strong crossref] or [strong desc].
	# @a errortext: A description of the problem.
	# @a url:       Reference to the place of the problem.

	if {[numProblems] == 0} {
	    $opt(-parent) addProblemObject $this
	}

	problems_addProblem $category $errortext $url
	return
    }



    method pPage {} {
	# @c Special part of the problem reporting facility. Used by the
	# @c procedures parent to get a handle on the exact location of the
	# @c detailed problem description.
	# @r a reference to the exact location of the detailed problem
	# @r description for this procedure.

	return [$opt(-parent) pPage]#$this
    }
}

