# -*- tcl -*-
#		Makedist @mFullVersion@, as of @mDate@
#
# CVS: $Id: plugin.tcl,v 1.2 1998/06/04 20:10:08 aku Exp $
#
# @c Plugin management of makedist. Builds a database containing the
# @c information about all available plugins during startup.
#
# @s Plugin management.
# @i plugin management
# ---------------------------

package require Tcl 8.0
package require Pool_Base

# Create the required namespaces before adding information to them.
# Initialize some info variables.

namespace eval ::makedist {
    variable version @mFullVersion@
    variable asOf    @mDate@

    namespace export *

    variable retrievers {}
    variable retrNames  {}
    variable retrMap
    ::pool::array::def retrMap

    variable packers    {}
    variable packNames  {}
    variable packMap
    ::pool::array::def packMap

    variable transports {}
    variable transNames {}
    variable transMap
    ::pool::array::def transMap

    namespace eval plugin {
	# The namespaces containing the registered plugin procedures.
	namespace export *

	namespace eval retriever {
	    namespace export *
	}
	namespace eval packer {
	    namespace export *
	}
	namespace eval transport {
	    namespace export *
	}
    }

}



proc ::makedist::retrievers {} {
    # @r The list of registered retrieval procedures.

    variable retrievers
    return  $retrievers
}



proc ::makedist::packers {} {
    # @r The list of registered packer procedures.

    variable packers
    return  $packers
}



proc ::makedist::transports {} {
    # @r The list of registered transport procedures.

    variable transports
    return  $transports
}



proc ::makedist::retrieverNames {} {
    # @r The list of registered retrieval procedures, their external names,
    # @r that is.

    variable retrNames
    return  $retrNames
}



proc ::makedist::packerNames {} {
    # @r The list of registered packer procedures, their external names, that
    # @r is.

    variable packNames
    return  $packNames
}



proc ::makedist::transportNames {} {
    # @r The list of registered transport procedures, their external names,
    # @r that is.

    variable transNames
    return  $transNames
}



proc ::makedist::isRetriever {o v} {
    # @c Type checker procedure. Tests wether argument <a v> contains the
    # @c internal code of a registered retriever procedure, or not.
    #
    # @a o: The name of the option to check
    # @a v: The value to check.
    # @r a boolean value, 1 if <a v> contains a registered retriever, 0 else.

    variable retrMap
    return [::info exists retrMap(nameOf,$v)]
}



proc ::makedist::isPacker {o v} {
    # @c Type checker procedure. Tests wether argument <a v> contains the
    # @c internal code of a registered packer procedure, or not.
    #
    # @a o: The name of the option to check
    # @a v: The value to check.
    # @r a boolean value, 1 if <a v> contains a registered packer, 0 else.

    variable packMap
    return [::info exists packMap(nameOf,$v)]
}



proc ::makedist::isPackerList {o v} {
    # @c Type checker procedure. Tests wether argument <a v> contains a list
    # @c of internal codes of registered packer procedures, or not.
    #
    # @a o: The name of the option to check
    # @a v: The value to check.
    # @r a boolean value, 1 if <a v> contains a list of registered packers,
    # @r 0 else.

    variable packMap

    foreach item $v {
	if {! [::info exists packMap(nameOf,$item)]} {
	    return 0
	}
    }

    return 1
}



proc ::makedist::isTransport {o v} {
    # @c Type checker procedure. Tests wether argument <a v> contains the
    # @c internal code of a registered transport procedure, or not.
    #
    # @a o: The name of the option to check
    # @a v: The value to check.
    # @r a boolean value, 1 if <a v> contains a registered transport, 0 else.

    variable transMap
    return [::info exists transMap(nameOf,$v)]
}



proc ::makedist::retrieverName {code} {
    # @r The external name of the retriever refered by <a code>
    # @a code: symbolic name of the retriever to look for.

    variable retrMap
    return  $retrMap(nameOf,$code)
}



proc ::makedist::packerName {code} {
    # @r The external name of the packer refered by <a code>
    # @a code: symbolic name of the packer to look for.

    variable packMap
    return  $packMap(nameOf,$code)
}



proc ::makedist::packerExtension {code} {
    # @r The extension used by the packer refered by <a code>
    # @a code: symbolic name of the packer to look for.

    variable packMap
    return  $packMap(extOf,$code)
}



proc ::makedist::transportName {code} {
    # @r The external name of the retriever refered by <a code>
    # @a code: symbolic name of the transport to look for.

    variable transMap
    return  $transMap(nameOf,$code)
}



proc ::makedist::retrieverCode {name} {
    # @r The internal code of the retriever refered by <a name>
    # @a name: external name of the retriever to look for.

    variable retrMap
    return  $retrMap(codeOf,$name)
}



proc ::makedist::packerCode {name} {
    # @r The internal code of the retriever refered by <a name>
    # @a name: external name of the packer to look for.

    variable packMap
    return  $packMap(codeOf,$name)
}



proc ::makedist::transportCode {name} {
    # @r The internal code of the retriever refered by <a name>
    # @a name: external name of the transport to look for.

    variable transMap
    return  $transMap(codeOf,$name)
}



# --------------------------------------------------
# procedures to define the various types of external vectors
# essentially wrappers to 'proc'.

proc ::makedist::retriever {code name body} {
    # @c Defines a retriever procedure.
    # @a code: Internal code of the retriever
    # @a name: Name to use externally
    # @a body: script to execute.

    variable retrievers
    variable retrNames
    variable retrMap

    lappend retrievers $code
    lappend retrNames  $name

    set retrMap(nameOf,$code) $name
    set retrMap(codeOf,$name) $code

    # define the retrieval procedure
    proc ::makedist::plugin::retriever::$code {module version} $body
    return
}



proc ::makedist::packer {code name extension body} {
    # @c Defines a packer procedure.
    # @a code: Internal code of the packer
    # @a name: Name to use externally
    # @a body: script to execute.
    # @a extension: Extension of archivefiles
    # @a extension: generated by the packer defined here.

    variable packers
    variable packNames
    variable packMap

    lappend packers   $code
    lappend packNames $name

    set packMap(nameOf,$code) $name
    set packMap(codeOf,$name) $code
    set packMap(extOf,$code)  $extension

    # define the retrieval procedure
    proc ::makedist::plugin::packer::$code {module} $body
    return
}



proc ::makedist::transporter {code name body} {
    # @c Defines a transporter procedure.
    # @a code: Internal code of the transporter
    # @a name: Name to use externally
    # @a body: script to execute.

    variable transports
    variable transNames
    variable transMap

    lappend tranports  $code
    lappend transNames $name

    set transMap(nameOf,$code) $name
    set transMap(codeOf,$name) $code

    # define the retrieval procedure
    proc ::makedist::plugin::transport::$code {module builddir} $body
    return
}



# --------------------------------------------------
# initialization code, sources everything in subdirectory
# 'plug', a sibling to this file.

set _plugdir     [file join [::pool::file::here] plug]
set _plugpattern [file join $_plugdir *]

foreach _ [glob -nocomplain $_plugpattern] {
    if {[file isdirectory $_]} {continue}
    source $_
}

catch {unset _}
catch {unset _plugdir}
catch {unset _plugpattern}

# remove the definition procedures

catch {rename ::makedist::retriever   {}}
catch {rename ::makedist::packer      {}}
catch {rename ::makedist::transporter {}}
