# -*- tcl -*-
# --------------------------------------------------
# (C) 1997	Andreas Kupries <a.kupries@westend.com>
#
# CVS:	$Id: prepare.tcl,v 1.12 1998/06/04 20:10:08 aku Exp $
#
# @c Preparation of a retrieved module.
# @s Preparation phase.
# @i preparation
# --------------------------------------------------

package require Tcl        8.0
package require Pool_Base

# Create the required namespaces before adding information to them.
# Initialize some info variables.

namespace eval ::makedist {
    variable version @mFullVersion@
    variable asOf    @mDate@

    namespace export *
}


proc ::makedist::prepare {opt} {
    # @c Interface to the preparation phase. In contrast to the phases
    # @c <p ::makedist::retrieve>, <p ::makedist::pack> and
    # @c <p ::makedist::transport>, which only forward the request made to the
    # @c appropriate plugin, this one does considerable action by itself before
    # @c and after execution of the module specific preparation script.
    #
    # @n Assumes that the working directory is set to be the build directory.
    # @n The called preparation script should not change the working
    # @n directory, but this code will work even if that is done.
    #
    # @a opt: Name of the array variable containing the configuration to be
    # @a opt: used. Required entry is the module. This entry will be changed
    # @a opt: from the bare name to one containing the version as suffix. The
    # @a opt: option array is communicated to the external script too
    # @a opt: (as 'mOpt'), to allow its customization.

    upvar $opt o

    set m $o(-module)
    Log info preparing $m

    ::pool::array::def mInfo
    ::pool::array::def mDesc

    SetupInfo mInfo mDesc $m

    # Incorporate version information into the module name

    set mv $mInfo(@mDirectory@)
    file rename $m $mv
    set o(-module) $mv
    set m          $mv

    # Provide an area for supporting files and take care of the special files,
    # if present.  Must be done before module specific preparation as this code
    # is free to delete them!

    file mkdir sup.$m
    CareForSpecials $m

    # We play it safe and use a separate interpreter to execute the module
    # specific preparation code.  No messing around in the current global
    # namespace.  in future we may even try to make it completely safe (no
    # file access outside the module, ...)

    Log info running module specific code

    cd $m
    set here [pwd]

    SetupInterpreter _mkdp
    set fail [catch {_mkdp eval source PREPARE} msg]

    if {$fail} {
	# write error stack to the log

	global errorInfo

	foreach line [split $errorInfo \n] {
	    Log warning $line
	}
    }


    # force correct directory
    if {[string compare $here [pwd]] != 0} {
	cd $here
    }


    # read back option array, to get at any changes made by the external script
    array set o [_mkdp eval array get mOpt]

    interp delete _mkdp
    cd ..

    Log info module specific preparation complete

    # force writability of module directory, at least for user
    # @d Usage of 'chmod' creates unix dependency
    exec chmod u+w $m

    # post preparation actions:
    # <> don't distribute the prepration code (if not deleted already)
    # <> Copy out the remaining special files again.
    #    This is done to have the right version in
    #    case of them being changed by the module
    #    specific code.
    # <> Generate and place a manifest into the distribution.

    catch {file delete -force [file join $m PREPARE]}

    CareForSpecials $m
    Manifest $m
    return
}



proc ::makedist::SetupInfo {infovar descvar module} {
    # @c Initializes the array variable serving as interface to the module
    # @c specific preparation code. 
    #
    # @a infovar: Name of the array variable to fill in.
    # @a descvar: Name of the array variable to fill with DESCRIPTION
    # @a descvar: information.
    # @a module:  Name of the module we are preparing.

    upvar $infovar mInfo
    upvar $descvar mDesc

    GetDescription $module mDesc

    set v     $mDesc(version)
    set vlist [AnalyseVersion $v]

    # foreach pattern -W-
    set major [lindex $vlist 0]
    set minor [lindex $vlist 1]
    set plev  [lindex $vlist 2]

    set mInfo(@mFullVersion@)	  $v
    set mInfo(@mVersion@)	  ${major}.${minor}
    set mInfo(@mMajor@)	          $major
    set mInfo(@mMinor@)	          $minor
    set mInfo(@mDosVersion@)      ${major}${minor}${plev}
    set mInfo(@mShortDosVersion@) ${major}${minor}
    set mInfo(@mName@)	          $module
    set mInfo(@mDirectory@)       ${module}${major}.${minor}${plev}

    set date   [::pool::date::now]
    set short  [clock format [clock scan $date] -format {%d-%b-%Y}]
    set normal [clock format [clock scan $date] -format {%B %d, %Y}]

    set mInfo(@mDate@)	    $normal
    set mInfo(@mShortDate@) [string toupper $short]
    return
}



proc ::makedist::GetDescription {module descvar} {
    # @c Retrieves the description of the distributed <a module>, as specified
    # @c in the file DESCRIPTION.
    #
    # @a module:  Name of the module we are preparing. This is equal to the
    # @a module:  name of the directory to search for DESCRIPTION.
    # @a descvar: Name of the array variable to fill in.

    upvar $descvar mDesc

    set desc [file join $module DESCRIPTION]

    # sanity checks
    if {! [file exists $desc]} {
	error "no file DESCRIPTION in this package"
    }

    if {[file isdirectory $desc]} {
	error "no file DESCRIPTION in this package, its a directory"
    }

    if {! [file readable $desc]} {
	error "DESCRIPTION exists, but not readable"
    }

    # now get the information contained
    proc extension {name spec} {
	upvar mDesc attr
	array set attr $spec
    }

    source $desc
    rename extension ""
    return
}



proc ::makedist::AnalyseVersion {versionnumber} {
    # @c Splits the <a versionnumber> into its parts (major, minor, patchlevel)
    # @a versionnumber: Number to split.
    # @r a 3-element list containing major, minor and patchlevel, in this
    # @r order.

    regexp {([0-9]*)\.([0-9]*)([abp][0-9]*(unoff)?)?$} \
	    $versionnumber dummy major minor patchlevel

    return [list $major $minor $patchlevel]
}



proc ::makedist::CareForSpecials {module} {
    # @c Copies special files of <a module> into the area of supporting files.
    #
    # @a module: Name of the module we are preparing. This is equal to the
    # @a module: name of the directory to search in.

    CareFor $module README
    CareFor $module LSM
    CareFor $module ANNOUNCE
    CareFor $module DESCRIPTION
    return
}



proc ::makedist::CareFor {module file} {
    # @c Searches for <a file> in the <a module>-directory. Copies it to the
    # @c area of supporting files, if found.
    #
    # @a module: Name of the module we are preparing. This is equal to the
    # @a module: name of the directory to search in.
    # @a file:   Name of the file to search and copy.

    foreach p {{} doc documentation} {
	set f [file join $module $p $file]

	if {[file exists $f]} {
	    file copy -force $f [file join sup.$module $file]
	    return
	}
    }
    return
}



proc ::makedist::SetupInterpreter {name} {
    # @c Generates and prepares the interpreter the module specific
    # @c preparation code shall be run in.
    #
    # @a name: Name of the interpreter to create and prepare.

    upvar mInfo mInfo
    upvar mDesc mDesc
    upvar o     opt

    interp create $name

    $name alias mLog             ::makedist::Log
    $name alias mPackerExtension ::makedist::packerExtension

    $name eval package require Pool_Base
    $name eval package require Makedist_Support
    $name eval array set mInfo [list [array get mInfo]]
    $name eval array set mDesc [list [array get mDesc]]
    $name eval array set mOpt  [list [array get opt]]
    # Redirect syslog of safe interpreter to ourselves
    $name eval ::pool::syslog::def mLog

    foreach m [lsort [array names mInfo]] {
	regsub -all {@} $m {} _

	Log      debug $_ =     $mInfo($m)
	$name eval set $_ [list $mInfo($m)]
    }
    return
}



proc ::makedist::Manifest {module} {
    # @c Generates a manifest for <a module> and places it into it.
    #
    # @a module: Name of the module we are preparing. This is equal to the
    # @a module: name of the directory to scan.

    Log info manifest...

    cd $module
    ::pool::file::descendFiles f . {
	set _($f) .
    }
    cd ..

    set   fh [open [file join $module MANIFEST] w]
    puts $fh "# -*- tcl -*-"
    puts $fh "# ----------- MANIFEST of files contained in $module -----------"

    foreach f [lsort [array names _]] {
	puts $fh "file $f"
    }
    close $fh

    # should not be writable
    # @d Usage of 'chmod' generates unix dependency.
    exec chmod -wx [file join $module MANIFEST]
    return
}
