# -*- tcl -*-
# --------------------------------------------------
# (C) 1997	Andreas Kupries <a.kupries@westend.com>
#
# CVS:	$Id: latex2html.tcl,v 1.7 1998/06/07 16:22:38 aku Exp $
#
# @c Uses the <xref latex2html> translator to generate
# @c hyperized documentation from latex sources.
# @s Generating documentation with <xref latex2html>.
# @i prepare support, latex2html, documentation
# --------------------------------------------------

package require Tcl        8.0
package require Pool_Base

# Create the required namespaces before adding information to them.
# Initialize some info variables.

namespace eval ::makedist {
    variable version @mFullVersion@
    variable asOf    @mDate@

    namespace eval support {
	namespace export *

	# determine the directory containing the different sets of
	# navigation buttons

	variable navIconDir [file join [::pool::file::here] img]
    }
}


proc ::makedist::support::SetupNavigationImages {navset htmldir} {
    # @c Utility to copy the chosen navigational
    # @c button images to a local place.
    # @a navset: name of the subdirectory the images are found in.
    # @a htmldir: directory containing the subdirectories
    # @a htmldir with the hyperized texts. The images will be
    # @a htmldir: placed into a sibling directory of these, of name 'img'.

    variable navIconDir

    if {[string length $htmldir] == 0} {
	set htmldir [pwd]
    }

    set tmp [file join $navIconDir $navset]

    if {
	! ([file exists $tmp] && [file isdirectory $tmp])
    } {
	mLog warning $navset does not exist, falling back to 'std'.
	set navset std
    }

    set imgsrc  [file join $navIconDir $navset]
    set ballsrc [file join $navIconDir balls]
    set target  [file join $htmldir img]

    file mkdir $target
    eval file copy [glob [file join $imgsrc  *]] $target
    eval file copy [glob [file join $ballsrc *]] $target
    return
}



proc mLatex2html {args} {
    # @c Generates documentation with <xref latex2html>. Configuration is done
    # @c via hardwired defaults and the options given in <a args>. Every file
    # @c specified causes the creation of a subdirectory below -htmldir, or
    # @c the current directory.
    #
    # @a args: A list of option/value pairs, followed by the list of files to
    # @a args: convert. Recognized options are:&p -htmldir, -title, -nav,
    # @a args: -up, -up-title.&p -htmldir specifies the base directory of
    # @a args: the generated documentation directories (one per file).&p -nav
    # @a args: specifies the images to use in the navigation bar of the
    # @a args: generated pages.&p -title specifies the title to use for
    # @a args: file XX (array set notation).&p -up specifies the url to use as
    # @a args: upward link for file XX (array set notation).&p -up-title
    # @a args: specifies the title to use at the upward link for file XX (array
    # @a args: set notation).
    #
    # @n Requires <d latex2html>, must be found in the path.

    global env

    ::pool::array::def opt
    ::pool::getopt::defOption opt htmldir  -d ""
    ::pool::getopt::defOption opt nav      -d std
    ::pool::getopt::defOption opt title    -d {}
    ::pool::getopt::defOption opt up       -d {}
    ::pool::getopt::defOption opt up-title -d {}

    ::pool::array::def oVal
    set args [::pool::getopt::process opt $args oVal]
    unset opt

    if {[llength $args] == 0} {
	return
    }

    # convert array set notation for -title, -up, -up-title
    # into the notation used later, and undefineable by the
    # new option processor.

    foreach o {title up up-title} {
	if {[info exist oVal(-$o)]} {
	    foreach {key val} $oVal(-$o) {
		set oVal(-$o-$key) $val
	    }
	    unset oVal(-$o)
	}
    }

    mLog info hyperize latex sources...

    if {[string length $oVal(-htmldir)] > 0} {
	mLog info "-htmldir   = $oVal(-htmldir)"
    }

    ::makedist::support::SetupNavigationImages $oVal(-nav) $oVal(-htmldir)

    set tmpfile [file join / tmp mltoh.[pid]]
    set    tfh  [open $tmpfile w]
    puts  $tfh  "\$ICONSERVER='../img'"
    #            ~~~~~~~~~~~~~~~~~~~~ force usage of distributed images
    close $tfh

    foreach file $args {
	set stem [file rootname [file tail $file]]
	set cmdline "latex2html -init_file $tmpfile"
	#                       ~~~~~~~~~~~~~~~~~~~
	#                       enforce usage of distributed images

	if {[string length $oVal(-htmldir)] > 0} {
	    set basedir [file join $oVal(-htmldir) $stem]
	    append cmdline " -dir $basedir"
	} else {
	    set basedir $stem
	}

	if {
	    [info exists oVal(-title-$stem)] &&
	    ([string length $oVal(-title-$stem)] > 0)
	} {
	    append cmdline " -t [list $oVal(-title-$stem)]"
	}

	if {
	    [info exists oVal(-up-$stem)] &&
	    ([string length $oVal(-up-$stem)] > 0)
	} {
	    append cmdline " -up_url [list $oVal(-up-$stem)]"
	}

	if {
	    [info exists oVal(-up-title-$stem)] &&
	    ([string length $oVal(-up-title-$stem)] > 0)
	} {
	    append cmdline " -up_title [list $oVal(-up-title-$stem)]"
	}

	append cmdline " $file >/dev/null 2>/dev/null"

	mLog info hyperize $file...

	set fail [catch {
	    eval exec $cmdline
	    eval file delete [glob  \
		    $basedir/.*.dir \
		    $basedir/.*.pag \
		    $basedir/*.pl   \
		    $basedir/images.*]
	} msg]	;#{}

	if {$fail} {
	    mLog warning $msg
	}
    }

    file delete $tmpfile
    return
}
