# -*- tcl -*-
# --------------------------------------------------
# (C) 1997	Andreas Kupries <a.kupries@westend.com>
#
# CVS:	$Id: urls.tcl,v 1.4 1999/01/20 19:31:51 aku Exp $
#
# @c Handling of *-urls definitions.
# @s Handling of *-urls definitions.
# @i prepare support, location munging
# --------------------------------------------------

package require Tcl        8.0
package require Pool_Base
package require Pool_Net

# Create the required namespaces before adding information to them.
# Initialize some info variables.

namespace eval ::makedist {
    variable version @mFullVersion@
    variable asOf    @mDate@

    namespace eval support {
	namespace export *
    }
}


proc ::makedist::support::GetUrl {pu su tu} {
    # @n Internal procedure
    #
    # @c Retrieves the urls-definitions from DESCRIPTION, removes any gaps
    # @c between primary, secondary and tertiary definitions.
    #
    # @a pu: Name of the variable to write the list of primary   urls into.
    # @a su: Name of the variable to write the list of secondary urls into.
    # @a tu: Name of the variable to write the list of tertiary  urls into.

    global mDesc
    upvar $pu p $su s $tu t

    set p {}; catch {set p $mDesc(primary-urls)}
    set s {}; catch {set s $mDesc(secondary-urls)}
    set t {}; catch {set t $mDesc(tertiary-urls)}

    if {
	([llength $p] == 0) &&
	([llength $s] == 0) &&
	([llength $t] == 0)
    } {
	error "nothing defined"
    }

    # shift down non-empty strings to fill possible gaps.
    if {[llength $p] == 0} {
	set p $s
	set s $t
	set t {}
    }

    if {[llength $p] == 0} {
	set p $s
	set s $t
	set t {}
    }

    if {[llength $s] == 0} {
	set s $t
	set t {}
    }

    return
}


proc mExpandUrls {} {
    # @c Rewrites all definition of the form 'path/<ARCH>' found in DESCRIPTION
    # @c (*-urls) into a set of complete urls, depending on the chosen packers.
    # @c 'tgz, zip' for example yields the result: path/fullName.tar.gz,
    # @c path/fullName.zip and path/fullName.README. The found paths are
    # @c stored in 'mOpt(-locations)', for possible usage by the transport
    # @c system. Rereads the changed DESCRIPTION, to give later commands
    # @c access to the newly generated urls.

    global mDesc mOpt mDirectory

    mLog info expanding partial urls in description...

    # storage of strings to rewrite.
    ::pool::array::def pat

    set schemes     "(file:)|(http:)|(ftp:)|(gopher:)"
    set notfilename "\] \t\n`!@$^&\\*()_+=\\\[{}\\|'\\\"<>,?"
    set pattern     "($schemes)(\[^$notfilename\]*/<ARCH>)"

    # locate paths waiting for expansion
    foreach level {primary secondary tertiary} {
	if {[info exists mDesc($level-urls)]} {
	    set text $mDesc($level-urls)
	    while {[regexp $pattern $text match]} {
		set pat($match) ""
		regsub -- $match $text "" text
	    }
	}
    }

    # generate filenames to attach to the paths
    set files [list "$mDirectory.README"]
    foreach p $mOpt(-packer) {
	lappend files [list "$mDirectory.[mPackerExtension $p]"]
    }

    # merge both informations, insert into DESCRIPTION, read it back
    foreach m [array names pat] {
	foreach f $files {
	    regsub "<ARCH>" $m $f mnew
	    append pat($m) "$mnew\n\t"
	}
	set pat($m) [string trimright $pat($m)]
    }

    ::pool::string::multipleReplaceInFile DESCRIPTION pat
    mGetDescription

    # remember the pure paths for possible use by transport system
    foreach m [array names pat] {
	regsub "/<ARCH>" $m {} m
	lappend mOpt(-locations) $m
    }
    return
}



proc mUrls2Text {} {
    # @c Transforms the *-urls definitions into text
    # @c useable for an ANNOUNCEment.
    global mDesc

    if {[catch {::makedist::support::GetUrl p s t}]} {
	return "No locations providing the distribution are known."
    }

    # at least 'p' now contains some text.

    set text "The distribution can be found at\n"
    foreach url [::pool::urls::findUrls $p] {
	append text "\t<$url>\n"
    }

    if {[llength $s] == 0} {
	return [string trim $text]
    }

    append text "\nor\n"
    foreach url [::pool::urls::findUrls $s] {
	append text "\t<$url>\n"
    }

    if {[llength $t] == 0} {
	return [string trim $text]
    }

    append text "\nor\n"
    foreach url [::pool::urls::findUrls $t] {
	append text "\t<$url>\n"
    }

    return [string trim $text]
}



proc mGetDescription {} {
    # @c Reads the description file into 'mDesc'.
    # @n See <p ::makedist::GetDescription> too.

    upvar mDesc mDesc

    set desc DESCRIPTION

    # sanity checks
    if {! [file exists $desc]} {
	error "no file DESCRIPTION in this package"
    }

    if {[file isdirectory $desc]} {
	error "no file DESCRIPTION in this package, its a directory"
    }

    if {! [file readable $desc]} {
	error "DESCRIPTION exists, but not readable"
    }

    # now get the information contained
    proc extension {name spec} {
	upvar mDesc attr
	array set attr $spec
    }

    source $desc
    rename extension ""
    return
}
