# Query Users OpenVerse Plugin 
#
# Copyright (C) 1999 David Gale <cruise@openverse.org>
# For more information visit http://OpenVerse.org/
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
# USA.
# -----------------------------------------------------------------------
# This plugin will allow users to click on the names of other users
# and have them in "Query" mode. It makes use of several available traps
# within Metaverse.
#
# Written by Cruise <cruise@drunkenbastards.org>
#
# This was to be included as a feature but I decided to make it a seperate
# plugin to being working on plugins in general. This can be used as an 
# example of how to make a plugin, there will be others as I make more
# plugin support available.
#
# SEE THE Plugins.tcl FILE FOR MORE INFORMATION ON CREATING PLUGINS!
#
# It is a good idea to put all of your variables into one container.
# This will ensure that "your" variables do not conflict with variables
# from another plugin. You should have one global variable which is an 
# array of all the variables which your plugin will need. You should name
# this variable the same as your directory name.

global Query

set Query(users) {}
set Query(OnOff) 0

#
# You must register your plugin with all the traps you intend to use.
# The arguments for the register command are....
#
# The name of your plugin's Directory (Case Counts)
# Trap Name (see lib/Global.tcl for a list)
# Your function to call when this trap is executed.
#
# NOTE!! Your function must return 1 or 0 indicating to the parent
# function that it should continue(1) processing normally or that it
# should stop(0) normal processing.
#
RegisterPlugin Query MoveTo.Pre Query_CheckClick
RegisterPlugin Query MoveTo.Post Query_ColorLocal
RegisterPlugin Query MoveUser.Post Query_ColorRemote
RegisterPlugin Query NewPerson Query_NewPerson
RegisterPlugin Query PersonLeft Query_PersonLeft
RegisterPlugin Query ChangeAvatar Query_ChangeAvatar
RegisterPlugin Query ChangeUserAvatar Query_ChangeUserAvatar
RegisterPlugin Query SendText Query_SendText
RegisterPlugin Query Disconnect Query_Disconnect
RegisterPlugin Query DoNames Query_DoNames
RegisterPlugin Query ShowNames Query_ShowNames

proc Query_NewPerson {who x y image xx yy size bx by} {
	global Query
	set Query($who) 0
	return 1
}

proc Query_PersonLeft {who} {
	global Query MV

	set Query($who) 0
	set idx [lsearch -exact $Query(users) $who]
	set Query(users) [lreplace $Query(users) $idx $idx]
	if ![llength $Query(users)] {
		.top.c itemconfigure $MV($MV(nick).nameplate) -fill "black"
		set Query(OnOff) 0
	}
	return 1
}

proc Query_CheckClick {x y} {
	global Query MV

	#
	# If we are not displaying names, Don't bother checking.
	# This plugin requires names be displayed.
	# They can be shut off after you select a user for query.
	#
	if !$MV(names) {return 1}

	#
	# Check and see if the user is within the space of a name
	# tag on the screen. If they are, don't move.. query the
	# user and color their nametag blue (and your's!)
	# No, You cannot query yourself.
	#
	foreach who $MV(people) {
		if {$x >= $MV($who.name_tl_x) &&
			$x <= $MV($who.name_br_x) &&
			$y >= $MV($who.name_tl_y) &&
			$y <= $MV($who.name_br_y)} {
			if $Query($who) {
				.top.c itemconfigure $MV($who.nameplate) -fill "black"
				set Query($who) 0
				set idx [lsearch -exact $Query(users) $who]
				set Query(users) [lreplace $Query(users) $idx $idx]
				if ![llength $Query(users)] {
					.top.c itemconfigure $MV($MV(nick).nameplate) -fill "black"
					set Query(OnOff) 0
				}
			} else {
				.top.c itemconfigure $MV($who.nameplate) -fill "blue"
				.top.c itemconfigure $MV($MV(nick).nameplate) -fill "blue"
				lappend Query(users) "$who"
				set Query($who) 1
				if !$Query(OnOff) {
					set Query(OnOff) 1
				}
			}
			DoNames
			return 0
		}
	}
	return 1
}

proc Query_ChangeAvatar {notused} {
	global Query MV

	if $Query(OnOff) {
		.top.c itemconfigure $MV($MV(nick).nameplate) -fill "blue"
	}
	return 1
}

proc Query_ChangeUserAvatar {who what x y size bx by} {
	global Query MV

	if $Query($who) {
		.top.c itemconfigure $MV($who.nameplate) -fill "blue"
	}
	return 1
}

proc Query_ColorLocal {x y} {
	global Query MV

	if $Query(OnOff) {
		.top.c itemconfigure $MV($MV(nick).nameplate) -fill "blue"
	}
	return 1
}

proc Query_ColorRemote {who x y speed} {
	global Query MV

	if $Query($who) {
		.top.c itemconfigure $MV($who.nameplate) -fill "blue"
	}
	return 1
}

proc Query_ShowNames {who} {
	global Query MV
	
	if {$MV(nick) == $who} {
		Query_ColorLocal -1 -1
	} else {
		Query_ColorRemote $who -1 -1 -1
	}

	return 1
}

proc Query_SendText {stuff} {
	global Query MV

	if !$Query(OnOff) {return 1}
	if {[string range $stuff 0 0] == "/"} {return 1}

	foreach who $Query(users) {
		SendToServer "PRIVMSG $who $stuff"
	}
	ProcChat $MV(nick) "$stuff" 0 0 "light blue"
	return 0
}

proc Query_Disconnect {} {
	global Query MV

	set Query(users) {}
	set Query(OnOff) 0
	.top.c itemconfigure $MV($MV(nick).nameplate) -fill "black"
	return 1
}

proc Query_DoNames {win who idx} {
	global Query

	if $Query($who) {
		$win add command -label "UnQuery" -command "Query_ByNumber $idx"
	} else {
		$win add command -label "Query" -command "Query_ByNumber $idx"
	}
	return 1
}

proc Query_ByNumber {idx} {
	global Query MV

	if !$Query($MV(whois.$idx)) {
		if $MV(names) {
			.top.c itemconfigure $MV($MV(whois.$idx).nameplate) -fill "blue"
			if !$Query(OnOff) {
				.top.c itemconfigure $MV($MV(nick).nameplate) -fill "blue"
			}	
		}
		lappend Query(users) "$MV(whois.$idx)"
		set Query($MV(whois.$idx)) 1
		set Query(OnOff) 1
	} else {
		.top.c itemconfigure $MV($MV(whois.$idx).nameplate) -fill "black"
		set Query($MV(whois.$idx)) 0
		set id [lsearch -exact $Query(users) $MV(whois.$idx)]
		set Query(users) [lreplace $Query(users) $id $id]
		if ![llength $Query(users)] {
			.top.c itemconfigure $MV($MV(nick).nameplate) -fill "black"
			set Query(OnOff) 0
		}
	}
	DoNames
	return 1
}

if $MV(debug) {puts "Query Plugin Initalized!"}

