# -----------------------------------------------------------------------------
# classBrowser.tcl --
#
#       A Browser for incr-Tcl Classes.  This package exports a procedure for
#       creating a "classBrowser" widget, which contains a scrollable instance
#       of the "table" widget together with an update button.  On hitting the
#       button, the table is filled with information concerning each of the
#       visible classes.
#
# Restrictions:
#       Only one instance of a classBrowser widget may be created.
# -----------------------------------------------------------------------------
# Copyright 2000 Paul Welton
#       Permission to use, copy, modify, and distribute this software and its
#       documentation for any purpose and without fee is hereby granted,
#       provided that this copyright notice appears in all copies.  No
#       representations are made about the suitability of this software for any
#       purpose.  It is provided "as is" without express or implied warranty.
# -----------------------------------------------------------------------------
#  Revision:   1.1   : Initial Release
# -----------------------------------------------------------------------------

package require classIntrospection
package require Tktable

package provide classBrowser 1.0

namespace eval ::classBrowser {

    namespace export classBrowser

    variable WidgetName
    variable TableData
}

# -----------------------------------------------------------------------------
# ::classBrowser::Rowtagcommand --
#
#       This procedure gives the tag row<n> to each row <n>, so that the "tag
#       config" command can be used to configure each row directly.  See the
#       definition of the table widget to understand the significance.
#
# Arguments:
#       row - row to which a tag is to be given.
#
# Results:
#       Returns the tag to be given to that row.

proc ::classBrowser::Rowtagcommand {row} {
    return row$row
}

# -----------------------------------------------------------------------------
# ::classBrowser::classBrowser --
#
#       Widget creation procedure for the classBrowser widget.
#
# Arguments:
#       name - window name for the widget.
#
# Results:
#       Returns the window name for the widget.

proc ::classBrowser::classBrowser {name} {

    variable WidgetName
    variable TableData
    
    frame $name
    
    #  An extra column is added to overcome certain display problems with
    #  Tktable 2.6, as discussed in comp.lang.tcl,
    #  "Refresh problems (WAS Re: ANNOUNCE: Tktable v2.6...)", 04 Aug 2000.
    table $name.t                                   \
        -state          disabled                    \
        -maxwidth       2000                        \
        -titlerows      2                           \
        -cols           7                           \
        -variable       ::classBrowser::TableData   \
        -yscrollcommand [list $name.s set]          \
        -rowtagcommand  ::classBrowser::Rowtagcommand
        
    scrollbar $name.s                               \
        -command        [list $name.t yview]        \
        -orient         vertical
        
    button $name.b                                  \
        -text           update                      \
        -command        [list ::classBrowser::ConstructTable]
    
    pack $name.b -side top   -fill x
    pack $name.s -side right -fill y
    pack $name.t -side left  -fill y

    array set TableData {
        0,0 "class name"
        0,1 heritage
        0,2 public      1,2 methods
        0,3 public      1,3 procs
        0,4 public      1,4 variables
        0,5 public      1,5 commons
    }
    
    set WidgetName $name
    
    return $name
}

# -----------------------------------------------------------------------------
# PRIVATE PROCEDURES
# -----------------------------------------------------------------------------

# -----------------------------------------------------------------------------
# ::classBrowser::ConstructTable --
#
#       This procedure is called when the update button is hit.  It causes all
#       classes to be scanned and the table contents and dimensions to be
#       updated.
#
# Arguments:
#       None.
#
# Results:
#       No return value.

proc ::classBrowser::ConstructTable {} {

    variable WidgetName

    #  The entries for the first class start directly below the header rows.
    set FirstRow [$WidgetName.t cget -titlerows]
    
    #  Apply the AnalyzeClasses procedure to all classes, in all namespaces.
    ::classBrowser::WalkNamespaces :: ::classBrowser::AnalyzeClasses FirstRow
    
    resizeColumns $WidgetName.t
    
    return
}

# -----------------------------------------------------------------------------
# ::classBrowser::WalkNamespaces --
#
#       This procedure is called recursively to perform a tree-walk of all
#       namespaces, finding the classes within them and updating the table
#       accordingly.
#
# Arguments:
#       space     - namespace in which to search for subnamespaces.
#       cmd       - command to execute in each namespace.
#       pFirstRow - name of a variable which contains the first row of the
#                   display of the next class to be analyzed, which is
#                   updated as each class is analyzed.
#
# Results:
#       No return value.

proc ::classBrowser::WalkNamespaces {space cmd pFirstRow} {

    upvar $pFirstRow FirstRow
    
    #  Apply the specified command to the specified namespace.
    $cmd $space FirstRow
    
    #  Call this procedure recursively for all sub-namespaces.
    foreach subSpace [lsort [namespace children $space]] {
        WalkNamespaces $subSpace $cmd FirstRow
    }
    
    return
}

# -----------------------------------------------------------------------------
# ::classBrowser::AnalyzeClasses --
#
#       This procedure reconstructs the classBrowser view by analyzing each
#       class visible in the specified namespace.  The table is correctly
#       sized in terms of the number of rows and the width of the columns.
#
# Arguments:
#       enclosingSpace - namespace in which a search for classes is made.
#       pFirstRow      - name of a variable which contains the first row of the
#                        display of the next class to be analyzed, which is
#                        updated as each class is analyzed.
#
# Results:
#       No return value.

proc ::classBrowser::AnalyzeClasses {enclosingSpace pFirstRow} {

    upvar $pFirstRow FirstRow
    
    #  Note that some classes redeclare "find", so when finding the classes
    #  within a namespace, the "find" command is called in fully qualified
    #  form.
    foreach className [lsort                                     \
                    [namespace eval $enclosingSpace ::itcl::find classes]] {
        #  If the className is already fully qualified, then this indicates,
        #  according to the specification for "find classes", that the class
        #  is not declared locally, but is imported or global.  Therefore
        #  ignore it, and it will be processed as a local class when the
        #  namespace in which it was declared is reached.
        if {![regexp {^::} $className]} {
            AnalyzeClass [namespace eval $enclosingSpace         \
                                         namespace which $className] FirstRow
        }
    }
    
    return
}

# -----------------------------------------------------------------------------
# ::classBrowser::AnalyzeClass --
#
#       This procedure constructs the table entries for the specified class.
#       The number of rows occupied by the class is determined by the largest
#       set of enties for the class in any one column.
#
# Arguments:
#       className - the class to be analyzed.
#       pFirstRow - name of variable containing the first row in the table for
#                   this class.
#
# Results:
#       No return value.

proc ::classBrowser::AnalyzeClass {className pFirstRow} {

    variable TableData
    variable WidgetName
        
    upvar $pFirstRow FirstRow

    #  Keep a record of the starting row for this class, as the row variable
    #  will be set to the HWM of the row reached in all of the following column
    #  creating procedure calls, so as to define the base row of the next
    #  class.
    set row $FirstRow
    
    createEntries $FirstRow row 0 [list $className]
    foreach item   {heritage method proc variable common}          \
            column {1 2 3 4 5} {
        #  Attempt to obtain a list of each "item" of "className".  If this
        #  fails, then span all the item columns for className and place the
        #  error message in it.
        if {[catch [list ::itcl::class $className info $item] result]}  {
            set row $FirstRow
            createEntries $FirstRow row 1                          \
                    [list "no information available - $result"]
            $WidgetName.t configure -rows $row
            $WidgetName.t spans $FirstRow,1                        \
                    0,[expr [$WidgetName.t cget -cols] - 2]
            break
        }
        createEntries $FirstRow row $column $result
    }
    
    #  Set in preparation for the next class, leaving a blank line
    set NextRow [expr $row + 1]
    $WidgetName.t configure -rows $NextRow
    
    #  Span the blank line between classes, and make it a flat, dark grey bar
    $WidgetName.t spans         $row,0 0,[expr [$WidgetName.t cget -cols] - 1]
    $WidgetName.t tag config row$row -bg grey -relief flat
    $WidgetName.t height        $row -4
    
    
    #  Span the complete section of the "className" column that belongs to
    #  the class, excluding the blank dividing line.
    if {$row - $FirstRow - 1 > 0} {
        $WidgetName.t spans $FirstRow,0 [expr $row - $FirstRow - 1],0
    }
    
    set FirstRow $NextRow
    
    return
}

# -----------------------------------------------------------------------------
# ::classBrowser::createEntries --
#
#       This procedure constructs a specified column starting at a specified
#       row by assigning elements of the array bound to the table with elements
#       of a specified list.  A specified variable is set to the maximum of
#       its initial value and the number of the row following the last one
#       used.
#
# Arguments:
#       row      - starting row
#       pNextRow - running maximum of the next row to use after the entries
#                  for this class.
#       column   - column being constructed
#       lItems   - the items to be assigned to the array bound to the table.
#
# Results:
#       No return value.

proc ::classBrowser::createEntries {row pNextRow column lItems} {

    variable TableData
    
    upvar $pNextRow NextRow
    
    foreach item $lItems {
        set TableData($row,$column) $item
        incr row
    }
    
    #   Keep a running maximum of the row reached
    if {$NextRow < $row} {
        set NextRow $row
    }
    
    return
}

# -----------------------------------------------------------------------------
# ::classBrowser::resizeColumns --
#
#       This procedure sets the width of each column of the specified table to
#       the width of the largest element of the table.
#
# Arguments:
#       table - the window name for the table name to which this is to be
#               applied.  This procedure is potentially of more general
#               application and accesses no namespace variables so that
#               it could be moved elsewhere more easily.
#
# Results:
#       No return value.

proc ::classBrowser::resizeColumns {table} {

#  This procedure sets the width of each column of the specified table to the
#  width of the largest element of the table.

    set nRows [$table cget -rows]
    set nCols [$table cget -cols]
    for {set col 0} {$col < $nCols} {incr col} {
        set width 0
        for {set row 0} {$row < $nRows} {incr row} {
            set requiredWidth [string length [$table get $row,$col]]
            if {$requiredWidth > $width} {
                set width $requiredWidth
            }
        }
        $table width $col $width
    }
    
    return
}

# -----------------------------------------------------------------------------
# END OF PACKAGE "classBrowser"
# -----------------------------------------------------------------------------

