# -----------------------------------------------------------------------------
# classIntrospection.tcl --
#
#       This package wraps the standard procedure "class" from the Itcl package
#       to extend the set of argument formats supported to include an "info"
#       option.  The extended syntax is as follows:
#
#       class <className> <definition>
#              This is the original form. In addition to declaring the
#              specified class, information about the class is collected and
#              stored in a database.
#
#       class <className> info heritage|variable|proc|method|common
#              The first option returns a list of the classes which are
#              inherited directly by the specified class, while the
#              remainder return a list of the names of the public items in the
#              class definition of the specified type.
# -----------------------------------------------------------------------------
# Copyright 2000 Paul Welton
#       Permission to use, copy, modify, and distribute this software and its
#       documentation for any purpose and without fee is hereby granted,
#       provided that this copyright notice appears in all copies.  No
#       representations are made about the suitability of this software for any
#       purpose.  It is provided "as is" without express or implied warranty.
# -----------------------------------------------------------------------------
# Revision:   1.1   : Initial Release
# -----------------------------------------------------------------------------

package require Itcl 3.1
package provide classIntrospection 1.0

namespace eval classIntrospection {

    namespace export class

    #  two dimensional array indexed by fully qualified class name and
    #  item type, which may be one of heritage, variable, proc, method or
    #  common.
    variable ClassDatabase
    
    namespace eval ClassIntercept {

    #  This namespace exists to allow some fundamental standard procedures
    #  to be redeclared in a restricted environment.  NOTE: within this
    #  environment the standard procedures are accessed with fully
    #  qualified names; for example "::set" and "::variable".

        #  Contains the fully qualified class name during processing
        #  of the definition of that class.
        ::variable CurrentClass
        
        #  Indicates what scope should be assumed for methods, procs,
        #  commons or variables not prefixed with public, private or
        #  protected.  Normally, methods and procs would be public,
        #  while variables and commons would be private.  In the form
        #  of the public definition that is followed by a script, this
        #  variable is set to public while processing the script, to
        #  make all contained definitions public.
        ::variable DefaultScope ItemDefault
    }
}

#  The following conditional rename is needed so that it will not fail when
#  "pkg_mkIndex" parses the file.  The renaming is needed so that
#  ::itcl::class can be wrapped.
if {[llength [info commands ::itcl::class]] > 0} {
    if {[llength [info commands ::classIntrospection::ClassStandard]] <= 0} {
        rename ::itcl::class ::classIntrospection::ClassStandard
    }
}

#  The following is needed to ensure that the namespace exists prior to the
#  declaration of ::itcl::class, in the case of parsing by "pkg_mkIndex".
namespace eval ::itcl {}

# -----------------------------------------------------------------------------
# ::classIntrospection::class --
#
#       This wrapper for the standard class procedure performs one of two
#       different functions depending on which of the following argument
#       formats is used.  See the package header.
#
# Arguments: see the package header.
#
# Results:
#       For the "info" argument variant, the return value is a list containing
#       the required information, and otherwise it is the result of the call to
#       the standard procedure "class".

proc ::itcl::class {className definitionOrInfo args} {

    #  "class" remains in the namespace "itcl", but it will access
    #  a database in the "::classIntrospection" namespace.
    upvar 0 ::classIntrospection::ClassDatabase ClassDatabase
    
    #  Distinguish the original and enhanced forms of the command
    #  according to the length of $args.
    switch [llength $args] {
    
    0 {
        #  The standard version of class allows for only two arguments,
        #  so that $args will be empty for standard calls.

        #  Perform the original function of ::itcl::class
        set Result [uplevel [list ::classIntrospection::ClassStandard    \
                                  $className $definitionOrInfo]]

        #  Ensure that the className is fully qualified.
        set QualifiedName [uplevel namespace which $className]
        
        #  Clear the database, based on the fact that there can be
        #  only one class definition for each class.
        set ClassDatabase($QualifiedName,proc)     {}
        set ClassDatabase($QualifiedName,method)   {}
        set ClassDatabase($QualifiedName,common)   {}
        set ClassDatabase($QualifiedName,variable) {}
        set ClassDatabase($QualifiedName,heritage) {}
            
        #  Direct the results of the interception to the right class, and then
        #  intercept it to extract the information.
        set ::classIntrospection::ClassIntercept::CurrentClass $QualifiedName
        namespace eval ::classIntrospection::ClassIntercept $definitionOrInfo
    }
    
    1 {
        if {![string equal $definitionOrInfo "info"]} {
            error "class: too many arguments for non-\"info\" variant"
        }
        
        set classItem [lindex $args 0]
        switch -- $classItem {
        
        variable -
        proc     -
        method   -
        common   -
        heritage {
            set QualifiedName [uplevel namespace which $className]
            if {[info exists ClassDatabase($QualifiedName,$classItem)]} {
                set Result $ClassDatabase($QualifiedName,$classItem)
            } else {
                error "Unknown class \"$className\""
            }
        }
        
        default {
            error [format "Must be one of %s, not \"%s\""             \
                          {"variable", "proc", "method" or "common"}  \
                          $classItem]
        }
        
        }
    }
    
    default {
        error "Too many arguments to \"class\" ([expr [llength $args] + 2])"
    }
    
    }
    
    return $Result
}

#  If ::itcl::class has been imported into the global name space, then it must
#  be reimported to pick up the overloaded version.
namespace import -force ::itcl::class

# -----------------------------------------------------------------------------
# ::classIntrospection::ClassInfo --
#
#       This procedure is called by the class intercept procedures to load
#       information about the class into the database.
#
# Arguments:
#       The class, the item (which may be one of heritage, variable, proc,
#       method or common) and the value of the item.
#
# Results:
#       The database is updated, and there no return value.

proc ::classIntrospection::ClassInfo {className item value} {

    variable ClassDatabase
    
    lappend ClassDatabase($className,$item) $value
    
    return
}

# -----------------------------------------------------------------------------
# ::classIntrospection::ClassIntercept::<class statement>
#
#       There is a procedure mimicking each permitted statement within
#       a class definition.  The procedures extract information and
#       load it into the database of classes.
#
# Arguments:
#       Identical to the standard versions supported by the standard
#       version of "class".
#
# Results:
#       No return value.

proc ::classIntrospection::ClassIntercept::itk_option {args} {
#  No information is extracted from the presence of these statements.
    
    return
}

proc ::classIntrospection::ClassIntercept::inherit {args} {

    foreach baseClass $args {
        ::classIntrospection::ClassInfo                               \
                $::classIntrospection::ClassIntercept::CurrentClass   \
                heritage $baseClass
    }
    
    return
}

proc ::classIntrospection::ClassIntercept::constructor {args} {
#  No information is extracted from the presence of these statements.
    
    return
}

proc ::classIntrospection::ClassIntercept::destructor {body} {
#  No information is extracted from the presence of these statements.
    
    return
}

proc ::classIntrospection::ClassIntercept::public {commandOrScript args} {

    ::variable DefaultScope
    
    #  Identify the "script" form of the public command as one
    #  with not arguments except for the script.
    if {[llength $args] > 0} {
        ::classIntrospection::ClassInfo                               \
                $::classIntrospection::ClassIntercept::CurrentClass   \
                $commandOrScript [lindex $args 0]
    } else {
        ::set DefaultScope public
        eval $commandOrScript
        ::set DefaultScope ItemDefault
    }
    
    return
}

proc ::classIntrospection::ClassIntercept::protected {command args} {
#  Protected constructs are ignored.
    
    return
}

proc ::classIntrospection::ClassIntercept::private {command args} {
#  Private constructs are ignored.
    
    return
}

proc ::classIntrospection::ClassIntercept::set {varName args} {
#  No information is extracted from the presence of these statements.
    
    return
}

proc ::classIntrospection::ClassIntercept::array {option args} {
#  No information is extracted from the presence of these statements.
    
    return
}

proc ::classIntrospection::ClassIntercept::method {name args} {
#  Experimentation has shown that "method"s are public by default.

    ::classIntrospection::ClassInfo                                   \
            $::classIntrospection::ClassIntercept::CurrentClass       \
            method $name
    
    return
}

proc ::classIntrospection::ClassIntercept::proc {name args} {
#  Experimentation has shown that "proc"s are public by default.

    ::classIntrospection::ClassInfo                                   \
            $::classIntrospection::ClassIntercept::CurrentClass       \
            proc $name
    
    return
}

proc ::classIntrospection::ClassIntercept::variable {name args} {
#  Experimentation has shown that "variables"s are private by default,
#  so these statements are ignored except when processing a public script.

    ::variable DefaultScope
    
    if {[string equal "$DefaultScope" "public"]} {
        ::classIntrospection::ClassInfo                               \
                $::classIntrospection::ClassIntercept::CurrentClass   \
                variable $name
    }
    
    return
}

proc ::classIntrospection::ClassIntercept::common {name args} {
#  Experimentation has shown that "common"s are private by default,
#  so these statements are ignored except when processing a public script.

    ::variable DefaultScope
    
    if {[string equal "$DefaultScope" "public"]} {
        ::classIntrospection::ClassInfo                               \
                $::classIntrospection::ClassIntercept::CurrentClass   \
                common $name
    }
    
    return
}

# -----------------------------------------------------------------------------
# END OF PACKAGE "classIntrospection"
# -----------------------------------------------------------------------------

