#------------------------------------------------------------------------
#
#  mkEditShortcuts.tcl ---
#  
#  Part of the Whiteboard application. 
#  It creates a dialog for editing the shortcuts.
#   
#  The whiteboard application has been developed by:
#  
#	Mats Bengtsson, 
#	Hamngatan 21,
#	SE-58226, Linkoping Sweden,
#	matben@lin.foa.se,
#	phone: +46 13 136114
# 
#  It is distributed under the standard GPL.
#  See the README file for further details.
#  
#------------------------------------------------------------------------

proc mkEditShortcuts { w nameOfShortcutList }  {
    global  finEdShort shortCopy
    
    # Call by reference.
    upvar #0 $nameOfShortcutList theShortcuts
    
    # First, make a copy of shortcuts to work on.
    set shortCopy $theShortcuts
    
    catch {toplevel $w}
    wm title $w "Edit Shortcuts"
    pack [frame $w.frall -borderwidth 1 -relief raised]
    pack [frame $w.frtop -borderwidth 0] -in $w.frall
    
    # The top part.
    set wcont [LabeledFrame $w.frtop "Edit Shortcuts"]
    
    # Overall frame for whole container.
    set frtot [frame $wcont.fr]
    # Frame for listbox and scrollbar.
    set frlist [frame $frtot.lst]
    # The listbox.
    set wsb $frlist.sb
    set wlbox [listbox $frlist.lb -height 6 -width 14 -yscrollcommand "$wsb set"]
    eval "$wlbox insert 0 [lindex $shortCopy 0]"
    scrollbar $wsb -command "$wlbox yview"
    pack $wlbox  -side left -fill both
    pack $wsb -side left -fill both
    $wlbox selection set 0
    
    # Buttons at the right side.
    button $frtot.btadd -text "Add..."   \
      -command "AddOrEditShortcuts add shortCopy -1 $wlbox"
    button $frtot.btrem -text "Remove"  \
      -command "RemoveShortcuts shortCopy $wlbox"
    # Trick: postpone command substitution; only variable substitution.
    button $frtot.btedit -text "Edit..."  \
      -command "AddOrEditShortcuts edit shortCopy \[$wlbox curselection] $wlbox"
    
    grid $frlist -rowspan 3
    grid $frtot.btadd -column 1 -row 0 -sticky ew -padx 10
    grid $frtot.btrem -column 1 -row 1 -sticky ew -padx 10
    grid $frtot.btedit -column 1 -row 2 -sticky ew -padx 10
    
    pack $frtot -side left -padx 16 -pady 10    
    pack $wcont -fill x    
    
    # The bottom part.
    pack [frame $w.frbot -borderwidth 0] -in $w.frall -fill both   \
      -padx 8 -pady 6
    pack [button $w.frbot.bt1 -text "  Save  " -default active  \
      -command [list DoSaveEditedShortcuts $nameOfShortcutList shortCopy]]   \
      -side right -padx 5 -pady 5
    pack [button $w.frbot.bt2 -text Cancel -command {set finEdShort 0}]  \
      -side right -padx 5 -pady 5
    
    bind $w <Return> [list $w.frbot.bt1 invoke]
    bind $wlbox <Double-Button-1> "$frtot.btedit invoke"
    bind $w <Key-Up> "EditShortsKey $wlbox up"
    bind $w <Key-Down> "EditShortsKey $wlbox down"
    wm resizable $w 0 0
    
    # Grab and focus.
    focus $w
    catch {grab $w}
    tkwait variable finEdShort
    catch {grab release $w}
    destroy $w
    if {$finEdShort == 1}  {
	# Save shortcuts.
	return 1
    } else {
	# Cancel, keep old shortcuts.
	return 0
    }
}

proc EditShortsKey { wlbox upOrDown }  {
    set ind [$wlbox curselection]
    if {$upOrDown == "up" && $ind != 0}  {
	$wlbox selection clear 0 end
	$wlbox selection set [expr $ind - 1]
	$wlbox see [expr $ind - 1]
    } elseif {$upOrDown == "down" && $ind != [expr [$wlbox index end] - 1]}  {
	$wlbox selection clear 0 end
	$wlbox selection set [expr $ind + 1]
	$wlbox see [expr $ind + 1]
    }
}

proc DoSaveEditedShortcuts { nameOfShortcutList nameOfShortsCopy }  {
    global  finEdShort
    
    # Call by reference.
    upvar $nameOfShortcutList theShortcuts
    upvar $nameOfShortsCopy shortCopy
    set theShortcuts $shortCopy
    set finEdShort 1
}

#   AddOrEditShortcuts ---
#
#   'what': "add" or "edit".
#   'indShortcuts': the index in the shortcut list (-1 if add).
#   'wListBox': widget path of list box; if empty, do not update listbox entry.

proc AddOrEditShortcuts { what nameOfShorts indShortcuts {wListBox {}} }  {
    global  finAddShort sysFont
    
    # Call by reference.
    upvar #0 $nameOfShorts shortCopy
    #puts "AddOrEditShortcuts:: what=$what, indShortcuts=$indShortcuts"
    #puts "AddOrEditShortcuts:: shortCopy=$shortCopy"
    
    if {$what == "edit" && $indShortcuts == ""}  {
	return
    } 
    set w .taddshorts$what
    catch {toplevel $w}
    if {$what == "add"}  {
	set txt "Add Shortcut"
	set txt1 "New shortcut:"
	set txt2 "Complete ip number or name:"
	set txtbt "Add"
    } elseif {$what == "edit"}  {
	set txt "Edit Shortcut"
	set txt1 "Shortcut:"
	set txt2 "Complete ip number or name:"
	set txtbt "Save"
    }
    wm title $w $txt
    
    pack [frame $w.frall -borderwidth 1 -relief raised]
    pack [frame $w.frtop -borderwidth 0] -in $w.frall
    
    # The top part.
    set wcont [LabeledFrame $w.frtop $txt]
    
    # Overall frame for whole container.
    set frtot [frame $wcont.fr]
    label $frtot.lbl1 -text $txt1 -font $sysFont(sb)
    entry $frtot.ent1 -width 36 
    label $frtot.lbl2 -text $txt2 -font $sysFont(sb)
    entry $frtot.ent2 -width 36 
    grid $frtot.lbl1 -sticky w -padx 6 -pady 1
    grid $frtot.ent1 -sticky ew -padx 6 -pady 1
    grid $frtot.lbl2 -sticky w -padx 6 -pady 1
    grid $frtot.ent2 -sticky ew -padx 6 -pady 1
    
    pack $frtot -side left -padx 16 -pady 10
    pack $wcont -fill x    
    focus $frtot.ent1
    
    # Get the short pair to edit.
    if {[string compare $what "edit"] == 0}  {
	$frtot.ent1 insert 0 [lindex [lindex $shortCopy 0] $indShortcuts]
	$frtot.ent2 insert 0 [lindex [lindex $shortCopy 1] $indShortcuts]
    } elseif {[string compare $what "add"] == 0}  {
    }
    # The bottom part.
    pack [frame $w.frbot -borderwidth 0] -in $w.frall -fill both  \
      -padx 8 -pady 6
    # Trick: postpone command substitution; only variable substitution.
    button $w.frbot.bt1 -text "  $txtbt  " -default active  \
      -command "PushBtAddOrEditShortcut $what shortCopy $indShortcuts   \
      \[$frtot.ent1 get] \[$frtot.ent2 get] "
    pack $w.frbot.bt1 -side right -padx 5 -pady 5
    pack [button $w.frbot.bt2 -text Cancel -command {set finAddShort 0}]  \
      -side right -padx 5 -pady 5
    
    bind $w <Return> "$w.frbot.bt1 invoke"
    wm resizable $w 0 0
    
    # Grab and focus.
    focus $w
    catch {grab $w}
    tkwait variable finAddShort
    catch {grab release $w}
    destroy $w
    if {$finAddShort == 1}  {
	# Save shortcuts in listbox.
	# Remove old and insert new content in listbox.
	if {[string length $wListBox] > 0}  {
	    $wListBox delete 0 end
	    eval "$wListBox insert 0 [lindex $shortCopy 0]"
	}
	return 1
    } else {
	# Cancel, keep old shortcuts.
	return 0
    }
}

#   PushBtAddOrEditShortcut ---
#  
#   'what': "add" or "edit".
#   'nameOfShorts': name of variable that contains the complete
#   shortcut list to edit. Either replace or add shorts at the end.

proc PushBtAddOrEditShortcut { what nameOfShorts ind short ip }  {
    global  finAddShort 
    
    # Call by reference.
    upvar #0 $nameOfShorts shortCopy
    
    #puts "PushBtAddOrEditShortcut:: what=$what, ind=$ind, short=$short, ip=$ip"
    #puts "PushBtAddOrEditShortcut:: shortCopy=$shortCopy"
    if {$what == "edit" && $ind == -1}  {
	set finAddShort 0
	return
    }
    if {$short == "" || $ip == ""}  {
	set finAddShort 0
	return
    }
    set shcp [lindex $shortCopy 0]
    set lncp [lindex $shortCopy 1]
    if {$what == "add"}  {
	# Just append at the end.
	lappend shcp $short
	lappend lncp $ip
	set shortCopy [list $shcp $lncp]
    } elseif {$what == "edit"}  {
	# Replace old with new.
	set shcp [lreplace $shcp $ind $ind $short]
	set lncp [lreplace $lncp $ind $ind $ip]
	set shortCopy [list $shcp $lncp]
    }
    set finAddShort 1
}

proc RemoveShortcuts { nameOfShorts wlist }  {

    # Call by reference.
    upvar $nameOfShorts shortCopy

    set ind [$wlist curselection]
    if {$ind < 0}  {
	return
    }
    $wlist delete $ind
    set shortlist [lreplace [lindex $shortCopy 0] $ind $ind]
    set iplist [lreplace [lindex $shortCopy 1] $ind $ind]
    set shortCopy [list $shortlist $iplist]
}

#---------------------------------------------------------------------
