#
#  mkOpenMulticast.tcl ---
#  
#  Part of the Whiteboard application. 
#  It creates a dialog for connecting to a stream using SDP file
#  and QuickTime.
#   
#  The whiteboard application has been developed by:
#  
#	Mats Bengtsson, 
#	Hamngatan 21,
#	SE-58226, Linkoping Sweden,
#	matben@lin.foa.se,
#	phone: +46 13 136114
# 
#  It is distributed under the standard GPL.
#  See the README file for further details.
#  
#------------------------------------------------------------------------

proc mkOpenMulticast  {  }  {
    global  sysFont finishedQTMulticast shortsMulticastQT selMulticastName  \
      entMulticast prefs

    set w .opqtmulti
    catch {toplevel $w}
    wm title $w "Open Stream"
    # Global frame.
    pack [frame $w.frall -borderwidth 1 -relief raised] -fill both -expand 1
    
    # Labelled frame.
    set wcfr [frame $w.frall.fr -borderwidth 0]
    pack $wcfr -side top -fill both -ipadx 10 -ipady 6 -in $w.frall
    set wcont [LabeledFrame $wcfr "Open QuickTime live stream"]
    # Overall frame for whole container.
    set frtot [frame $wcont.frin]
    pack $frtot
    label $frtot.lbltop -text "Write URL or choose from shortcut:" \
      -font $sysFont(sb)
    set shorts [lindex $shortsMulticastQT 0]
    set optMenu [eval {tk_optionMenu $frtot.optm selMulticastName} $shorts]
    $frtot.optm configure -highlightthickness 0  \
      -background $prefs(bgColGeneral) -foreground black
    label $frtot.lblhttp -text "http://" -font $sysFont(sb)
    entry $frtot.entip -width 60 -font $sysFont(s) -textvariable entMulticast
    message $frtot.msg -borderwidth 0 -font $sysFont(s) -aspect 500 \
      -text "Open a URL which contains a SDP file with extension .mov\
      for a QuickTime realtime live streaming sending. Can be audio (radio)\
      or video (TV). Alternatively, use your web browser to find\
      the SDP file for a live transmission, download it on disk and\
      open it is an ordinary movie."
    grid $frtot.lbltop -column 0 -row 0 -sticky sw -padx 0 -pady 2 -columnspan 2
    grid $frtot.optm -column 2 -row 0 -sticky e -padx 2 -pady 2
    grid $frtot.lblhttp -column 0 -row 1 -sticky e -padx 0 -pady 6
    grid $frtot.entip -column 1 -row 1 -columnspan 2 -sticky w -padx 0 -pady 6
    grid $frtot.msg -column 0 -row 2 -columnspan 3 -padx 4 -pady 2 -sticky news
    set selMulticastName [lindex [lindex $shortsMulticastQT 0] 0]

    # Button part.
    set frbot [frame $w.frall.frbot -borderwidth 0]  
    pack [button $frbot.btconn -text "   Open   " -default active  \
      -command "OpenMulticastQTStream $frtot.entip"]  \
      -side right -padx 5 -pady 5
    pack [button $frbot.btcancel -text " Cancel "  \
      -command {set finishedQTMulticast 0}]  \
      -side right -padx 5 -pady 5
    pack [button $frbot.btedit -text " Edit... "  \
      -command "DoAddOrEditQTMulticastShort edit $frtot.optm"]  \
      -side right -padx 5 -pady 5
    pack [button $frbot.btadd -text " Add... "  \
      -command "DoAddOrEditQTMulticastShort add $frtot.optm"]  \
      -side right -padx 5 -pady 5
    pack $frbot -side top -fill both -expand 1 -in $w.frall  \
      -padx 8 -pady 6
    
    wm resizable $w 0 0
    # grab and focus
    focus $w
    focus $frtot.entip
    bind $w <Return> "$frbot.btconn invoke"
    trace variable selMulticastName w TraceSelMulticastName

    catch {grab $w}
    tkwait variable finishedQTMulticast

    catch {grab release $w}
    destroy $w

    return $finishedQTMulticast
}

proc DoAddOrEditQTMulticastShort { what wOptMenu }  {
    global  shortsMulticastQT selMulticastName
    
    if {[string compare $what "add"] == 0}  {
	# Use the standard edit shortcuts dialogs. (0: cancel, 1 added)
	set btAns [AddOrEditShortcuts add shortsMulticastQT -1]
    } elseif {[string compare $what "edit"] == 0}  {
	set btAns [mkEditShortcuts .edtstrm shortsMulticastQT]
    }
    # Update the option menu as a menubutton.
    # Destroying old one and make a new one was the easy way out.
    if {$btAns == 1}  {
	set shorts [lindex $shortsMulticastQT 0]
	set gridInfo [grid info $wOptMenu]
	#puts "wOptMenu=$wOptMenu, gridInfo=$gridInfo"
	destroy $wOptMenu
	set optMenu [eval {tk_optionMenu $wOptMenu selMulticastName} $shorts]
	eval {grid $wOptMenu} $gridInfo
    }
}

proc TraceSelMulticastName { name junk1 junk2 }  {
    upvar #0 $name locName
    global  shortsMulticastQT entMulticast
    
    #puts "TraceSelMulticastName:: name=$name"
    set ind [lsearch [lindex $shortsMulticastQT 0] $locName]
    set entMulticast [lindex [lindex $shortsMulticastQT 1] $ind]
}

proc OpenMulticastQTStream  { wentry }  {
    global  finishedQTMulticast chunkSize thisPath prefs

    # Patterns.
    set proto_ {[^:]+}
    set domain_ {[A-Za-z0-9\-\_\.]+}
    set port_ {[0-9]+}
    set path_ {/.*}
    set url [$wentry get]
    # Add leading http:// if not there.
    if {![regexp -nocase "^http://.+" $url]}  {
	set url "http://$url"
    }
    # Check and parse url.
    catch {unset port}
    if {![regexp -nocase "($proto_)://($domain_)(:($port_))?($path_)$"  \
      $url match protocol domain junk port path]}  {
	tk_messageBox -message   \
	  "Inconsistent url=$url." -icon error -type ok
	set finishedQTMulticast 0
	return ""
    }
    if {[string length $port] == 0}  {
	set port 80
    }
    set fileTail [string trim [file tail $path] /]
    set fullName [file join $thisPath incoming $fileTail]
    #puts "domain=$domain, port=$port, fileTail=$fileTail, path=$path, pwd=[pwd]"
    if {[string length $fileTail] == 0}  {
	tk_dialog .wrfn "No Path" "No file name in path." \
	  error 0 Cancel
	return ""
    }
    # Open a file in cache to store the content: state(body).
    if {[catch {open $fullName w} fid]}  {
	tk_messageBox -message   \
	  "Failed when trying to open $fullName." -icon error -type ok
	# Handle error.
	set finishedQTMulticast 0
	return ""
    }
    
    # Get http. The 'CleanupMulticastQTStream' is called when finished.
    # Everything takes place in the background thanks to -command.
    if {[catch {   \
      ::http::geturl $url   \
      -command [list CleanupMulticastQTStream $fid $fullName]  \
      -blocksize $chunkSize -channel $fid   \
      -progress [list ProgressMulticastQTStream $fileTail]  \
      -timeout [expr 1000*$prefs(timeout)]} token]}  {
	
	# Cancel the timout event. Not working.
	if {[info exists state(after)]}  {
	    catch {after cancel $state(after)}
	}
	tk_messageBox -message   \
	  "Failed open url=$url. Returned with: $token." -icon error -type ok
	set finishedQTMulticast 1
	return
    }
    # Wait for connection to open.
    StartStopAnimatedWave .fcomm.stat.lbl 1
    set finishedQTMulticast 1
}

proc CleanupMulticastQTStream  { fid fullName token }  { 
    global  statMess wCan
    upvar #0 $token state
    
    #puts "CleanupMulticastQTStream:: token=$token"
    set no_ {^2[0-9]+}
    catch {close $fid}
    parray state
    # Waiting is over.
    StartStopAnimatedWave .fcomm.stat.lbl 0
    
    # Access state as a Tcl array.
    # Check errors. 
    if {[info exists state(status)] &&  \
      ([string compare $state(status) "timeout"] == 0)}  {
	tk_messageBox -message   \
	  "Timout event for url=$state(url)" -icon error -type ok
	return
    } elseif {[info exists state(status)] &&  \
      ([string compare $state(status) "ok"] != 0)}  {
	tk_messageBox -message   \
	  "Not ok return code from url=$state(url); status=$state(status)" \
	  -icon error -type ok
	return
    }
    # The http return status. Must be 2**.
    set httpCode [lindex $state(http) 1]
    if {![regexp "$no_" $httpCode]}  {
	tk_messageBox -message   \
	  "Failed open url=$url. Returned with code: $httpCode." \
	  -icon error -type ok
    }
    # Check that type of data is the wanted. Check further.
    if {[info exists state(type)] &&  \
      ([string compare $state(type) "video/quicktime"] != 0)}  {
	tk_messageBox -message   \
	  "Not correct file type returned from url=$state(url); \
	  filetype=$state(type); expected video/quicktime." \
	  -icon error -type ok
	return
    }
    # This is opened as an ordinary movie.
    set anchor [NewImportAnchor]
    DoImportImageOrMovie $wCan $fullName "$anchor" "own"
    set fileTail [file tail $fullName]
    set statMess "Opened streaming live multicast: $fileTail."
    update idletasks
}

proc ProgressMulticastQTStream  { fileTail token totalBytes currentBytes }  {
    global  statMess
    upvar #0 $token state
    # Access state as a Tcl array.
    if {$totalBytes != 0}  {
	set percentLeft [expr ($totalBytes - $currentBytes)/$totalBytes]
	set txtLeft ", $percentLeft% left"
    } else  {
	set txtLeft ""
    }
    set statMess "Getting $fileTail$txtLeft"
    update idletasks
}

#---------------------------------------------------------------------
