
#  FilesAndCanvas.tcl ---
#  
#      This file is part of the whiteboard application. It implements procedures
#      for transfering the items of a canvas to and from files.
#      
#  Copyright (c) 1999-2000  Mats Bengtsson
#  
#  See the README file for license, bugs etc.
 
# DrawCanvasItemFromFile ---
#
#    Takes the canvas items in the 'filePath' and draws them in the canvas 'w'.

proc DrawCanvasItemFromFile { w filePath }  {

    # Opens the data file.
    if {[catch [list open $filePath r] fileId]} {
	tk_messageBox -message "Cannot open [file tail $filePath] for reading." \
	  -icon error -type ok
    }
    FileToCanvas $w $fileId  $filePath "all"
    close $fileId
}
	  
# FileToCanvas --
#   
#   Reads line by line from file. Each line contains an almost complete 
#   canvas command except for the widget path. 
#   Lines can also contain 'image create ...' commands.
#   The file must be opened and file id given as 'fileId'.
#   where = "all": write to this canvas and all others.
#   where = "other": write only to remote client canvases.
#   where = ip number: write only to this remote client canvas and not to own.
#   where = "own": write only to this canvas and not to any other.

proc FileToCanvas { w fileId absFilePath {where all} }  {
    global  itno myItpref allIPnumsToSend ipNum2Socket imageItno2FileTail  \
      prefs
    
    # Should file names in file be translated to native path?
    set fileNameToNative 1
    #puts "FileToCanvas:: absFilePath=$absFilePath, pwd=[pwd]"
    
    set pre_ {[^/ ]+}
    set ino_ {[0-9]+}
    set nl_ "\\n"
    
    # New freash 'itno' only if writing to own canvas as well.
    if {$where == "all" || $where == "own"}  {
	set updateItnos 1
    } else  {
	set updateItnos 0
    }
    
    # Read line by line; each line contains an almost complete canvas command.
    # Item prefix and item numbers need to be taken care of.
    
    while {[gets $fileId line] >= 0}  { 
	
	#puts "FileToCanvas:: line=$line"
	set previousImageOrMovieCmd {}
	
	# Figure out if image create command...or movie command.
	if {([string compare [lindex $line 0] "image"] == 0) ||  \
	  ([string compare [lindex $line 0] "Movie"] == 0)}  {
	    set previousImageOrMovieCmd $line
	    # Get the next line as well.
	    gets $fileId line
	} 
	
	# This must be a canvas command.
	set type [lindex $line 1]
	
	# This can be a bug if text contains "-tags"???
	set ind [lsearch -exact $line "-tags"]
	if {$ind >= 0}  {
	    set tagInd [expr $ind + 1]
	    set theTags [lindex $line $tagInd]
	} else  {
	    continue
	}
	
	# The it tags must get new numbers since 'itno' may never be reused.
	# This is only valid when writing on this canvas, see 'updateItnos'.
	# Be sure to replace any other prefix with own prefix; else 
	# a complete mess is achieved.
	
	if {![regexp "(^| )($pre_)/($ino_)" $theTags match junk prefix oldno]} {
	    puts "FileToCanvas:: Warning, didn't match tags! theTags=$theTags"
	    continue
	}
	if {$updateItnos}  {
	    regsub "(^| )($pre_)/($ino_)" $theTags  " $myItpref/$itno" newTags
	    
	    # Replace tags.
	    set line [lreplace $line $tagInd $tagInd $newTags]
	    set oldittag $prefix/$oldno
	    set ittag $myItpref/$itno
	} else {
	    set ittag $prefix/$oldno
	    set oldittag $ittag
	}
	
	# Make newline substitutions.
	set newcmd [subst -nocommands -novariables $line]
	
	# Images and movies. Handle in a different manner.
	
	if {$type == "image" || $type == "window"}  {
	    
	    #puts "FileToCanvas:: updateItnos=$updateItnos, oldittag=$oldittag, $ittag"
	    # Do not forget to update links to files through 
	    # 'imageItno2FileTail'.
	    
	    if {$updateItnos && [info exists imageItno2FileTail($oldittag)]}  {
	        set imageItno2FileTail($ittag) $imageItno2FileTail($oldittag)
	    }	    
	    
	    # First, try to localize the original file. If unknown just skip.
	    # Use the image create command on the previous line if exist.
	    # Extract the complete file path.
	    
	    if {($type == "image" || $type == "window") &&   \
	      ([llength $previousImageOrMovieCmd] > 0)}  {
		set ind [lsearch -exact $previousImageOrMovieCmd "-file"]
		if {$ind >= 0}  {
		    set filePath [lindex $previousImageOrMovieCmd [expr $ind + 1]]
		}
		
		# Translate to native file path? Useful if want to have
		# platform independent canvas file format.
		# Bug if path contains '/' on Mac, or ':' on unix?
		# The relative path in the file is relative that file
		# and not relative present directory!
		
		set filePath [AddAbsolutePathWithRelative $absFilePath $filePath]
		#puts "FileToCanvas:: AddAbsolutePathWithRelative=$filePath"

		if {$fileNameToNative}  {
		    set filePath [file nativename $filePath]
		}
		#puts "previousImageOrMovieCmd:: ind=$ind, filePath=$filePath"
	    } elseif {[info exists imageItno2FileTail($ittag)]}  {
	        set filePath [GetKnownPathFromTail $imageItno2FileTail($ittag)]
	    } else {
	        puts "FileToCanvas:: couldn't localize image/window $oldittag"
	        continue
	    }
	    # Need to know the coordinates.
	    set x [lindex $line 2]
	    set y [lindex $line 3]
	    
	    # Let the import procedure do the job; manufacture an option list.
	    DoImportImageOrMovie $w $filePath   \
	     [list coords: [list $x $y] tags: $ittag] $where
	    
	    
	} else  {
	    
	    # Draw ordinary item not image nor window (movie).
	    if {$where == "all" || $where == "own"}  {
	
		# If html font sizes, then translate these to point sizes.
		if {$prefs(useHtmlSizes) && $type == "text"}  {
		    catch {eval $w [FontHtmlSizeToPointSize $newcmd]} id
		} else  {
		    catch {eval $w $newcmd} id
		}
		
		# Speak...
		if {$prefs(TclSpeech) && $prefs(TclSpeechOn) && $type == "text"}  {
		    speak [$w itemcget $id -text]
		}
	    }
	    
	    # Encode all newlines as \n .
	    regsub -all "\n" $line $nl_ newcmd
	    
	    # Write to all other clients.
	    if {$where == "all" || $where == "other"}  {
		foreach ip $allIPnumsToSend  {
		    puts $ipNum2Socket($ip) "CANVAS: $newcmd"
		}
	    } else  {
		# Write only to specified client with ip number 'where'.
		puts $ipNum2Socket($where) "CANVAS: $newcmd"
	    }
	}
	
	# Update 'itno' only when also writing to own canvas!
	if {$updateItnos}  {
	    incr itno
	}
    }
}

# CanvasToFile --
#   
#   Writes line by line to file. Each line contains an almost complete 
#   canvas command except for the widget path. 
#   The file must be opened and file id given as 'fileId'.
#   If 'filePathsAbsolute' the -file option contains the full path name, 
#   else relative path name to this script path.
#   
# Arguments:
#   w                canvas widget path.
#   fileId           file descriptor of the save file.
#   absFilePath      absolute path to the save file.
#   'saveImageOrMovieCmds' = 0: writes canvas commands to file except for images
#   or movies.
#   'saveImageOrMovieCmds' = 1: prepends each line with an image or movie canvas
#   command with the create commands. 

proc CanvasToFile { w fileId absFilePath {saveImageOrMovieCmds 0} }  {
    global  thisPath prefs fontPoints2Size movieItemno2File
    
    # When saving images or movies, save relative or absolute path names?
    # It is perhaps best to choose a path relative the actual file path of the 
    # file?
    
    set filePathsAbsolute 0
    set nl_ {\\n}
    set ids [$w find all]
    foreach id $ids {
	
	# Do not save grid or markers.
	
	set theTags [$w gettags $id]
	if {[lsearch $theTags grid] >= 0 || [lsearch $theTags tbbox] >= 0}  {
	    continue
	}
	set theItno [CanvasGetItnoFrom $w $id]
	set theType [$w type $id]
	#puts "theType=$theType"

	# If image and if 'saveImageOrMovieCmds', then save the image creating
	# command before the item create command.
	
	if {$saveImageOrMovieCmds && ([string compare $theType "image"] == 0)} {
	    
	    # Get image name.
	    set imageName [$w itemcget $id -image]
	    # ...and its type; bitmap or photo.
	    set imageType [image type $imageName]
	    set opts [$imageName configure]
	    #puts "opts=$opts"
	    set opcmd {}
	    foreach opt $opts {
		set op [lindex $opt 0]
		set val [lindex $opt 4]
		#puts "op=$op, val=$val"
		
		# Intercept file path if not 'filePathsAbsolute', and make it 
		# relative instead.
		if {!$filePathsAbsolute && ([string compare $op "-file"] == 0)} {
		    set val [GetRelativePath $absFilePath $val]
		}
		lappend opcmd $op $val
	    }
	    set cmd [concat "image create $imageType $imageName" $opcmd]
	    puts $fileId $cmd    
	    
	} elseif {$saveImageOrMovieCmds &&   \
	  ([string compare $theType "window"] == 0)} {

	    # A movie: for QT we have a complete widget; xanim treated differently.

	    set windowName [$w itemcget $id -window]
	    set movieName ${windowName}.m
	    if {$prefs(QuickTimeTcl)}  {
		set opts [$movieName configure]
		set opcmd {}
		foreach opt $opts {
		    set op [lindex $opt 0]
		    set val [lindex $opt 4]
		    
		    # Intercept file path if not 'filePathsAbsolute', and make it 
		    # relative instead.
		    
		    if {!$filePathsAbsolute &&  \
		      ([string compare $op "-file"] == 0)} {
			set val [GetRelativePath $absFilePath $val]
		    }
		    lappend opcmd $op $val
		}
	    } elseif {$prefs(xanim)}  {
		
		# How about relative path?
		if {[info exists movieItemno2File($theItno)]} {
		    set opcmd "-file $movieItemno2File($theItno)" 
		}
	    }
	    
	    # Complete commad for QT; 'xanim' saved in same format.
	    set cmd [concat "Movie $movieName" $opcmd]
	    puts $fileId $cmd    
	}
	
	# The actual canvas item.
	set opts [$w itemconfigure $id]
	set opcmd {}
	foreach opt $opts {
	    set op [lindex $opt 0]
	    set val [lindex $opt 4]
	    #puts "op=$op, val=$val"
	    
	    if {[string compare $op "-text"] == 0}  {

		# If multine text, encode as one line with explicit "\n".
		regsub -all "\n" $val $nl_ oneliner
	        regsub -all "\r" $oneliner $nl_ oneliner
		set val $oneliner
	    } elseif {[string compare $op "-tags"] == 0}  {
		
		# Any tags "current" or "selected" must be removed before save,
		# else when writing them on canvas things become screwed up.
		
		regsub -all "current" $val "" val
		regsub -all "selected" $val "" val
	    }
	    
	    # If using 'useHtmlSizes', translate font  point size to html size.
	    # Needs perhaps some more error checking here.
	    
	    if {$prefs(useHtmlSizes) && ([string compare $op "-font"] == 0)}  {
		set val [lreplace $val 1 1 $fontPoints2Size([lindex $val 1])]
	    }
	    lappend opcmd $op $val
	}
	set co [$w coords $id]
	set cmd [concat "create" $theType $co $opcmd]
	puts $fileId $cmd
    }
}

# DoOpenCanvasFile --
#
#   Creates a standard file open dialog, opens the file, and draws to
#   canvas via 'FileToCanvas'. If 'filePath' given, dont show file
#   open dialog.

proc DoOpenCanvasFile { w {filePath {}}}  {
    
    if {[string length $filePath] == 0}  {
	set typelist {
	    {"Canvas" {".can"}}
	    {"Text" {".txt"}}
	}
	set ans [tk_messageBox -message   \
	  "Opening a canvas file erases everything in present canvas!" \
	  -icon warning -type okcancel -default ok]
	if {$ans == "cancel"}  {
	    return
	}
	set ans [tk_getOpenFile -title "Open Canvas" -filetypes $typelist  \
	  -defaultextension ".can"]
	if {$ans == ""}  {
	    return
	}
	set fileName $ans
    } else  {
	set fileName $filePath
    }  
    
    # Opens the data file.
    if [catch [list open $fileName r] fileId]  {
	tk_messageBox -message "Cannot open $fileName for reading." \
	  -icon error -type ok
    }
    DoEraseAll $w     
    FileToCanvas $w $fileId $fileName
    close $fileId
}

proc DoSaveCanvasFile { w }  {
    
    set typelist {
	{"Canvas" {".can"}}
	{"Text" {".txt"}}
    }
    set ans [tk_getSaveFile -title "Save Canvas" -filetypes $typelist  \
	    -defaultextension ".can"]
    if {$ans == ""}  {
	return
    }
    set fileName $ans
    
    # Opens the data file.
    if {[catch [list open $fileName w] fileId]}  {
	tk_messageBox -message "Cannot open $fileName for writing." \
	  -icon error -type ok
	return
    }

    CanvasToFile $w $fileId $fileName 1
    close $fileId
}

#-------------------------------------------------------------------------------