
#  ItemInspector.tcl ---
#  
#      This file is part of the whiteboard application. It lets the user 
#      inspect and configure item options in the canvas. The options are 
#      organized into a list as:   
#      listOfAllOptions = {{-option oldValue entryWidget} {...} ...}
#      
#  Copyright (c) 1999-2000  Mats Bengtsson
#  
#  See the README file for license, bugs etc.

namespace eval ::ItemInspector::  {
    
    # Only the main procedure is exported.
    namespace export ItemInspector
    
    # Maps 0, 1 to and from false, true. Strange in tcl8.3; bezier? boolean?
    variable boolFull2Short
    variable boolShort2Full
    array set boolFull2Short {false 0 true 1}
    array set boolShort2Full {0 false 1 true bezier true}
}

proc ::ItemInspector::ItemInspector { wCan }  {
    global  sysFont prefs debugLevel fontSize2Points fontPoints2Size  \
      tcl_platform
    
    # We need to create an item specific instance. Use the item id
    # for instance.
    set itemId [$wCan find withtag current]
    if {[llength $itemId] == 0}  {
	return
    }
    set w ".itinsp${itemId}"
    
    # The local namespace variables.
    variable boolFull2Short
    variable boolShort2Full
    
    # Need to have instance specific namespace for regional variables.
    namespace eval ::ItemInspector::$w  {
	variable fontOpts
	variable menuBtVar
	variable finished
    }
    # Refer to them by simpler variable names.
    upvar ::ItemInspector::${w}::fontOpts fontOpts
    upvar ::ItemInspector::${w}::menuBtVar menuBtVar
    upvar ::ItemInspector::${w}::finished finished
    
    # If window already there, just return.
    if {[winfo exists $w]}  {
	error "window name $w already exists!"
    }
    set nl_ {\\n}
    if {$debugLevel >= 2}  {  
	puts "ItemInspector (entry)::"
    }
    set finished -1
    set itPrefNo [CanvasGetItnoFrom $wCan current]
    if {[llength $itPrefNo] == 0}  {
	return
    }
    
    # Movies may not be selected this way; temporary solution?
    if {[lsearch [$wCan gettags $itPrefNo] "movie"] >= 0}  {
	return
    }	
    
    # Write this container as a simple proc with automatic sizing.
    if {[winfo exists $w]} {
	return
    }
    if {[string compare $tcl_platform(platform) "macintosh"] == 0} {
	toplevel $w
	unsupported1 style $w documentProc
    } else {
	toplevel $w
    }
    wm title $w "Item Inspector"
    
    # Global frame.
    pack [frame $w.frall -borderwidth 1 -relief raised] -fill both -expand 1
    set w1 [frame $w.frall.fr1]    
    set wcont1 [LabeledFrame $w1 "Item Options"]
    
    # Overall frame for whole container.
    set frtot [frame $wcont1.frin]
    pack $frtot -padx 10 -pady 10
    
    # List available options of the option menus.
    array set theMenuOpts [list    \
      arrow {none first last both}  \
      capstyle {butt projecting round}   \
      joinstyle {bevel miter round}  \
      smooth {false true}    \
      stipple {none gray75 gray50 gray25 gray12}  \
      outlinestipple {none gray75 gray50 gray25 gray12}   \
      style {pieslice chord arc}   \
      anchor {n ne e se s sw w nw center}   \
      {font family} $prefs(canvasFonts)    \
      {font size} {1 2 3 4 5 6}   \
      {font weight} {normal bold italic}  \
      justify {left right center}]
    set listOfAllOptions {}
    
    # Item type.
    set iLine 0
    set itemType [$wCan type $itemId]
    label $frtot.lbl$iLine -text "item type:" -font $sysFont(sb)
    entry $frtot.ent$iLine -width 30 -font $sysFont(s)
    $frtot.ent$iLine insert end $itemType
    $frtot.ent$iLine configure -state disabled
    grid $frtot.lbl$iLine -column 0 -row $iLine -sticky e -padx 2 -pady 1
    grid $frtot.ent$iLine -column 1 -row $iLine -sticky w -padx 2 -pady 1
    lappend listOfAllOptions [list type $itemType $frtot.ent$iLine]
    
    # Coordinates.
    incr iLine
    label $frtot.lbl$iLine -text "coordinates:" -font $sysFont(sb)
    entry $frtot.ent$iLine -width 30 -font $sysFont(s)
    set theCoords [$wCan coords $itemId]
    $frtot.ent$iLine insert end $theCoords
    $frtot.ent$iLine configure -state disabled
    grid $frtot.lbl$iLine -column 0 -row $iLine -sticky e -padx 2 -pady 1
    grid $frtot.ent$iLine -column 1 -row $iLine -sticky w -padx 2 -pady 1
    lappend listOfAllOptions [list coords $theCoords $frtot.ent$iLine]
    
    # Get all item options. Fonts need special treatment.
    set opts [$wCan itemconfigure $itemId]
    set ind [lsearch $opts "-font *"]
    # We have got a font option.
    if {$ind >= 0}  {
	# Find the actual values set for the text.
	set fontOpts [lindex [lindex $opts $ind] 4]
	set opts [lreplace $opts $ind $ind   \
	  [list {-font family} {} {} {} [lindex $fontOpts 0]]  \
	  [list {-font size} {} {} {} $fontPoints2Size([lindex $fontOpts 1])]  \
	  [list {-font weight} {} {} {} [lindex $fontOpts 2]]]
    }
    # Loop over all options.
    foreach opt $opts {
	incr iLine
	set op [lindex $opt 0]
	set val [lindex $opt 4]
	# If multine text, encode as one line with explicit "\n".
	if {[string compare $op "-text"] == 0}  {
	    regsub -all "\n" $val $nl_ oneliner
	    regsub -all "\r" $oneliner $nl_ oneliner
	    set val $oneliner
	}
	set opname [string trim $op -]
	label $frtot.lbl$iLine -text "$opname:" -font $sysFont(sb)
	#puts "op=$op, val=$val"
	
	# Intercept options for nontext output.
	switch -exact -- $op {
	    -fill        -
	    -outline     {
		
		frame $frtot.ent$iLine
		if {[string length $val] == 0}  {
		    set menuBtVar($opname) transparent
		} else {
		    set menuBtVar($opname) fill
		}
		set wMenu [tk_optionMenu $frtot.menu$iLine   \
		  ::ItemInspector::${w}::menuBtVar($opname) transparent fill]
		$wMenu configure -font $sysFont(sb)
		$frtot.menu$iLine configure -font $sysFont(sb)  \
		  -highlightthickness 0  \
		  -background $prefs(bgColGeneral) -foreground black
		entry $frtot.entent$iLine -font $sysFont(s) -width 4   \
		  -state disabled
		if {[string length $val] > 0}  {
		    $frtot.entent$iLine configure -background $val
		}
		pack $frtot.menu$iLine -in $frtot.ent$iLine -side left
		pack $frtot.entent$iLine -in $frtot.ent$iLine  \
		  -side left -fill x -expand 1
		bind $frtot.entent$iLine <Double-Button-1>   \
		  "[namespace current]::ChooseItemColor $frtot.entent$iLine"
	    } 
	    -tags       {
		
		entry $frtot.ent$iLine -width 30 -font $sysFont(s) 
		$frtot.ent$iLine insert end $val
		$frtot.ent$iLine configure -state disabled
		
		# Pure menu options.
	    } 
	    -arrow            -
	    -capstyle         -
	    -joinstyle        -
	    -smooth           -
	    -stipple          -
	    -outlinestipple   -
	    -style            -
	    -anchor           -
	    "-font family"    -
	    "-font size"      -
	    "-font weight"    -
	    -justify          {
		if {[string compare $op "-smooth"] == 0}  {
		    
		    # Get full menu name.
		    if {[string length $val] == 0}  {
			set menuBtVar($opname) "false"
		    } else  {
			set menuBtVar($opname) $boolShort2Full($val)
		    }
		} else  {
		    if {[string length $val] == 0}  {
			set menuBtVar($opname) "none"
		    } else  {
			set menuBtVar($opname) $val
		    }
		}
		set wMenu [eval {tk_optionMenu $frtot.ent$iLine   \
		  ::ItemInspector::${w}::menuBtVar($opname)}  \
		  $theMenuOpts($opname)]
		$wMenu configure -font $sysFont(sb) 
		$frtot.ent$iLine configure -font $sysFont(sb) -highlightthickness 0  \
		  -background $prefs(bgColGeneral) -foreground black
	    } 
	    default  {
		# Just an editable text entry widget.
		entry $frtot.ent$iLine -width 30 -font $sysFont(s) 
		$frtot.ent$iLine insert end $val
	    }
	}
	grid $frtot.lbl$iLine -column 0 -row $iLine -sticky e -padx 2 -pady 1
	if {([string compare $op "-fill"] == 0) ||   \
	  ([string compare $op "-outline"] == 0)}  {
	    grid $frtot.ent$iLine -column 1 -row $iLine -sticky ew -padx 2 -pady 1
	} else  {
	    grid $frtot.ent$iLine -column 1 -row $iLine -sticky w -padx 2 -pady 1
	}
	if {([string compare $op "-fill"] == 0) ||   \
	  ([string compare $op "-outline"] == 0)}  {
	    lappend listOfAllOptions [list $op $val $frtot.entent$iLine]
	} else  {
	    lappend listOfAllOptions [list $op $val $frtot.ent$iLine]
	}
    }
    pack $w1 -fill x
    
    # Button part.
    set frbot [frame $w.frall.frbot -borderwidth 0]
    pack [button $frbot.btconn -text "   Save   " -default active  \
      -command [list [namespace current]::CanvasConfigureItem $w $wCan  \
      $itemId $listOfAllOptions]]  \
      -side right -padx 5 -pady 5
    pack [button $frbot.btcancel -text " Cancel "  \
      -command "[namespace current]::Cancel $w"]  \
      -side right -padx 5 -pady 5
    pack $frbot -side top -fill both -expand 1 -in $w.frall  \
      -padx 8 -pady 6
    
    wm resizable $w 0 0
    bind $w <Return> "$frbot.btconn invoke"
}

proc ::ItemInspector::Cancel  { w }  {
    
    set ::ItemInspector::${w}::finished 0
    destroy $w
}
    
proc ::ItemInspector::CanvasConfigureItem  { w wCan itemId listOfAllOptions }  {
    global  allIPnumsToSend ipNum2Socket fontPoints2Size fontSize2Points
    
    #puts "::ItemInspector::CanvasConfigureItem:: w=$w"
    upvar ::ItemInspector::${w}::fontOpts fontOpts
    upvar ::ItemInspector::${w}::menuBtVar menuBtVar
    upvar ::ItemInspector::${w}::finished finished
    
    set itPrefNo [CanvasGetItnoFrom $wCan $itemId]
    
    # Loop through all options. Assemble a configure list.
    set allNewOpts {}
    foreach opt $listOfAllOptions {
	set op [lindex $opt 0]
	set val [lindex $opt 1]
	set entWid [lindex $opt 2]
	set opname [string trim $op -]
	#puts "op=$op, val=$val"

	# Intercept options for nontext output.
	switch -- $op {
	    type         -
	    coords       {
		# Do nothing
		continue
	    }
	    -fill        -
	    -outline     {
		
		set newOpt $menuBtVar($opname)
		if {[string compare $newOpt "transparent"] == 0}  {
		    set newVal {}
		} else  {
		    set newVal [$entWid cget -background]
		}
		# Pure menu options.
	    } 
	    -arrow            -
	    -capstyle         -
	    -joinstyle        -
	    -smooth           -
	    -style            -
	    -anchor           -
	    "-font family"    -
	    "-font size"      -
	    "-font weight"    -
	    -justify          {
	    
		set newVal $menuBtVar($opname)
	    }
	    "-stipple"    -
	    -outlinestipple   {
	    
		set newOpt $menuBtVar($opname)
		if {[string compare $newOpt "none"] == 0}  {
		    set newVal {}
		} else  {
		    set newVal $menuBtVar($opname)
		}
	    }
	    default           {
		set newVal [$entWid get]
	    }
	}
	#puts "newVal=$newVal"
	# If new different from old, reconfigure. Reinterpret \n"
	if {[string compare $val $newVal] != 0}  {
	    lappend allNewOpts $op $newVal
	}
    }
    # We need to collect all three artificial font options to the real one.
    # Only for the text item type.
    #puts "allNewOpts=$allNewOpts"
    if {[string compare [$wCan type $itemId] "text"] == 0}  {
	set indFam [lsearch -exact $allNewOpts "-font family"]
	set indSize [lsearch -exact $allNewOpts "-font size"]
	set indWeight [lsearch -exact $allNewOpts "-font weight"]
	#puts "indFam=$indFam, indSize=$indSize, indWeight=$indWeight"
	
	# Be sure to translate font size to points.
	set newFontOpts [lreplace $fontOpts 1 1   \
	  $fontPoints2Size([lindex $fontOpts 1])]
	#puts "newFontOpts=$newFontOpts"
	
	if {$indFam >= 0 || $indSize >= 0 || $indWeight >= 0}  {
	    set cpAllNewOpts $allNewOpts
	    
	    # Any of the font options changed, reconstruct the new one.
	    foreach {ind indCmd} [list 2 $indWeight 1 $indSize 0 $indFam]  {
		if {$indCmd >= 0}  {
		    set newFontOpts [lreplace $newFontOpts $ind $ind  \
		      [lindex $cpAllNewOpts [expr $indCmd + 1]]]
		    # Remove temporary font option.
		    set allNewOpts [lreplace $allNewOpts $indCmd [expr $indCmd + 1]]
		} 
	    }
	    # Translate font size to points again.
	    set newFontOpts [lreplace $newFontOpts 1 1   \
	      $fontSize2Points([lindex $newFontOpts 1])]
	    #puts "newFontOpts=$newFontOpts"
	    lappend allNewOpts "-font" $newFontOpts
	    #puts "allNewOpts=$allNewOpts"
	}
    }
    # Do the actual change.
    if {[llength $allNewOpts] > 0}  {
	set cmd [concat itemconfigure $itPrefNo $allNewOpts]
	eval $wCan $cmd
	#puts "CANVAS: $cmd"
	
	# If selected (it should be), redo the selection to fit.
	set id [$wCan find withtag $itPrefNo]
	set idsMarker [$wCan find withtag id$id]
	# If selected.
	if {[string length $idsMarker] > 0}  {
	    $wCan delete id$id
	    MarkBbox $wCan 1 $id
	}
	# Let other also know.
	foreach ip $allIPnumsToSend  {
	    puts $ipNum2Socket($ip) "CANVAS: $cmd"
	}
    }
    set ::ItemInspector::${w}::finished 1 
    destroy $w
}
    
proc ::ItemInspector::ChooseItemColor  { wEntry }  {
    
    set col [$wEntry cget -background]
    set col [tk_chooseColor -initialcolor $col]
    if {[string length $col] > 0}	 {
	$wEntry configure -background $col
    }
}

#--------------------------------------------------------------------
