
#  MimeTypesAndPlugins.tcl ---
#  
#      This file is part of the whiteboard application. For a number of 
#      plugins and helpers, define their features, and try to load them.
#      Defines relations between MIME types and file name suffixes,
#      suffixes and mac types etc. A number of arrays are defined at the 
#      global scope. The 'typlist' option for the File Open dialogs are 
#      designed as well.
#      
#  Copyright (c) 1999-2000  Mats Bengtsson
#  
#  See the README file for license, bugs etc.

# We need to be very systematic here to handle all possible MIME types
# and extensions supported by each package or helper application.
#
# mimeType2Packages: array that maps a MIME type to a list of one or many packages.
# 
# supSuff:           maps package name to supported suffixes for that package,
#                    and maps from MIME type (image, audio, video) to suffixes.
#                    It contains a bin (binary) element as well which is the
#                    the union of all MIME types except text.
#                    
# supportedMimeTypes:maps package name to supported MIME types for that package,
#                    and maps from MIME type (image, audio, video) to MIME type.
#                    It contains a bin (binary) element as well which is the
#                    the union of all MIME types except text.
#                    
# prefMime2Package:  The preferred package to handle a file with this MIME type.
#                    If "0" then it is saved to disk and not imported.
#
#
# NOTE:  1) file suffixes shold be phased out, and eventually MIME types
#           should be used as far as possible.
#        2) potential problem with spaces in Mac types.


# We start by defining general relations between MIME types etc.
# Mapping from MIME type to a list of suffixes.

array set mime2SuffixList {
    text/plain           {.txt  .tcl .can}                          \
    application/x-tcl    {.tcl}                                     \
    video/quicktime      {.mov  .qt}                                \
    video/x-dv           {.dif  .dv}                                \
    video/mpeg           {.mpeg .m1s  .m15  .m1a  .m1v  .m64  .m75  \
                          .mpa  .mpg  .mpm  .mpv}                   \
    video/x-mpeg         {.mpeg .m1s  .m15  .m1a  .m1v  .m64  .m75  \
                          .mpa  .mpg  .mpm  .mpv}                   \
    audio/mpeg           {.mpeg .m1s  .m15  .m1a  .m1v  .m64  .m75  \
                          .mp2  .mpa  .mpg  .mpm  .mpv  .mp3}       \
    audio/x-mpeg         {.mpeg .m1s  .m15  .m1a  .m1v  .m64  .m75  \
                          .mp2  .mpa  .mpg  .mpm  .mpv  .mp3}       \
    video/x-msvideo      {.avi}                                     \
    application/sdp      {.sdp}                                     \
    audio/aiff           {.aif  .aiff .aifc}                        \
    audio/x-aiff         {.aif  .aiff .aifc}                        \
    audio/basic          {.au   .snd  .ulw}                         \
    audio/x-sd2          {.sd2}                                     \
    audio/wav            {.wav}                                     \
    audio/x-wav          {.wav}                                     \
    image/x-bmp          {.bmp}                                     \
    image/vnd.fpx        {.fpx}                                     \
    image/gif            {.gif}                                     \
    image/jpeg           {.jpg  .jpeg}                              \
    image/x-macpaint     {.pntg .pnt  .mac}                         \
    image/x-photoshop    {.psd}                                     \
    image/png            {.png}                                     \
    image/x-png          {.png}                                     \
    image/pict           {.pict .pic  .pct}                         \
    image/x-sgi          {.sgi  .rgb}                               \
    image/x-targa        {.tga}                                     \
    image/tiff           {.tif  .tiff}                              \
    image/x-tiff         {.tif  .tiff}                              \
    application/x-world  {.3dmf .3dm  .qd3d .qd3}                   \
    application/x-3dmf   {.3dmf .3dm  .qd3d .qd3}                   \
    video/flc            {.flc  .fli}                               \
    application/x-shockwave-flash  {.swf}                           \
    application/postscript  {.ps}                                   \
    audio/midi           {.mifi .mid  .smf  .kar}                   \
    audio/x-midi         {.mifi .mid  .smf  .kar}                   \
}

# Create the inverse mapping, from a suffix to a list of MIME types.
# This is not unique either.

foreach theMime [array names mime2SuffixList] {
    set suffList $mime2SuffixList($theMime)
    foreach suff $suffList {
	lappend suffix2MimeList($suff) $theMime
    }
}

# Mapping from MIME type to a list of Mac types.

array set mime2MacTypesList {
    text/plain           {"TEXT"}                                   \
    application/x-tcl    {"APPL"}                                   \
    video/quicktime      {"MooV"}                                   \
    video/x-dv           {"dvc!"}                                   \
    video/mpeg           {"MPEG" "MPGa" "MPGv" "MPGx"}              \
    video/x-mpeg         {"MPEG" "MPGa" "MPGv" "MPGx"}              \
    audio/mpeg           {"MPEG" "MPGa" "MPGv" "MPGx" "Mp3 " "SwaT" \
                          "PLAY" "MPG3" "MP3 "}                     \
    audio/x-mpeg         {"MPEG" "MPGa" "MPGv" "MPGx" "Mp3 " "SwaT" \
                          "PLAY" "MPG3" "MP3 "}                     \
    video/x-msvideo      {"VfW "}                                   \
    application/sdp      {"TEXT"}                                   \
    audio/aiff           {"AIFF" "AIFC"}                            \
    audio/x-aiff         {"AIFF" "AIFC"}                            \
    audio/basic          {"ULAW"}                                   \
    audio/x-sd2          {"Sd2f" "SD2 "}                            \
    audio/wav            {"WAVE" "WAV "}                            \
    audio/x-wav          {"WAVE" "WAV "}                            \
    image/x-bmp          {"BMP " "BMPf"}                            \
    image/vnd.fpx        {"FPix"}                                   \
    image/gif            {"GIFf" "GIF "}                            \
    image/jpeg           {"JPEG"}                                   \
    image/x-macpaint     {"PNTG"}                                   \
    image/x-photoshop    {"8BPS"}                                   \
    image/png            {"PNGf" "PNG "}                            \
    image/x-png          {"PNGf" "PNG "}                            \
    image/pict           {"PICT"}                                   \
    image/x-sgi          {"SGI "}                                   \
    image/x-targa        {"TPIC"}                                   \
    image/tiff           {"TIFF"}                                   \
    image/x-tiff         {"TIFF"}                                   \
    application/x-world  {"3DMF"}                                   \
    application/x-3dmf   {"3DMF"}                                   \
    video/flc            {"FLI "}                                   \
    application/x-shockwave-flash  {"SWFL" "SWF "}                  \
    application/postscript  {"TEXT"}                                \
    audio/midi           {"Midi"}                                   \
    audio/x-midi         {"Midi"}                                   \
}

# Create the inverse mapping, from a mac type to a list of MIME types.
# This is not unique either. Unneccesary?

foreach theMime [array names mime2MacTypesList] {
    foreach macType $mime2MacTypesList($theMime) {
	lappend macType2MimeList($macType) $theMime
    }
}

# Mapping from Mac "TYPE" to file suffix.
# This is necessary if we open a file on a mac without a file name suffix,
# the network file *must* have a suffix.
# Some are missing...
array set macType2Suff {
    TEXT   .txt          GIFf   .gif          GIFF   .gif   \
    JPEG   .jpg          MooV   .mov          PLAY   .mp3   \
    ULAW   .au           PNGf   .png          "VfW " .avi   \
    dvc!   .dv           MPEG   .mpg          MPGa   .m1a   \
    MPGv   .m1v          MPGx   .m64          AIFF   .aif   \
    "PNG " .png          TIFF   .tif          PICT   .pct   \
    SWFL   .swf          AIFC   .aif          "Mp3 " .mp3   \
    SwaT   .swa          MPG3   .mp3          "MP3 " .mp3   \
    Sd2f   .sd2          "SD2 " .sd2          WAVE   .wav   \
    "WAV " .wav          sfil   .snd          "BMP " .bmp   \
    BMPf   .bmp          FPix   .fpx          PNTG   .pnt   \
    8BPS   .psd          qtif   .qtif         "SGI " .sgi   \
    TPIC   .tga          3DMF   .3dmf         "FLI " .fli   \
    "SWF " .swf          Midi   .mid}

# ...and the inverse mapping. Maybe there are problems with uniqueness? (.gif)
InvertArray macType2Suff suff2MacType

# Search for a set of packages and define their characteristics.
#
# List all wanted plugins and short names for on which platforms they work.
# m: macintosh, u: unix, w: windows.

# We should add the mac to snack later on!
array set packages2Platform {QuickTimeTcl mw TclSpeech m snack uw Img uw}
array set helpers2Platform {xanim u}
set plugin(allPacks) [array names packages2Platform]
set plugin(allHelper) [array names helpers2Platform]
set plugin(all) [concat $plugin(allPacks) $plugin(allHelper)]
set plugin(tk,importProc) DoImportImageOrMovie

# The descriptions of the plugins:
#--- QuickTime -----------------------------------------------------------------

set plugin(QuickTimeTcl,full) "QuickTime"
set plugin(QuickTimeTcl,type) "Tcl plugin"
set plugin(QuickTimeTcl,desc) "Displays multimedia content such as\
  video, sound, mp3 etc. It also supports a large number of\
  still image formats."
set plugin(QuickTimeTcl,importProc) DoImportImageOrMovie

# We must list supported file suffixes for each package.
# For QuickTime:
set supportedMimeTypes(QuickTimeTcl) {\
    video/quicktime     video/x-dv          video/mpeg         \
    video/x-mpeg        audio/mpeg          audio/x-mpeg       \
    video/x-msvideo     application/sdp     audio/aiff         \
    audio/x-aiff        audio/basic         audio/x-sd2        \
    audio/wav           audio/x-wav         image/x-bmp        \
    image/vnd.fpx       image/gif           image/jpeg         \
    image/x-macpaint    image/x-photoshop   image/png          \
    image/x-png         image/pict          image/x-sgi        \
    image/x-targa       image/tiff          image/x-tiff       \
    application/x-world application/x-3dmf  video/flc          \
    application/x-shockwave-flash           audio/midi         \
    audio/x-midi }

#--- TclSpeech via PlainTalk if available --------------------------------------
  
set plugin(TclSpeech,full) "PlainTalk"
set plugin(TclSpeech,type) "Tcl plugin"
set plugin(TclSpeech,desc) "When enabled, a synthetic voice speaks out\
  text that is written in the canvas as well as text received\
  from remote clients. It is triggered by a punctation character (,.;)."
set plugin(TclSpeech,importProc) {}

#--- snack ---------------------------------------------------------------------
# On Unix/Linux and Windows we try to find the Snack Sound extension.
# Only the "sound" part of the extension is actually needed.

set plugin(snack,full) "Snack"
set plugin(snack,type) "Tcl plugin"
set plugin(snack,desc) "The Snack Sound extension adds audio capabilities\
  to the application. Presently supported formats include wav, au, aiff and mp3."
set plugin(snack,importProc) DoImportImageOrMovie
set supportedMimeTypes(snack) {\
    audio/wav           audio/x-wav         audio/basic        \
    audio/aiff          audio/x-aiff        audio/mpeg         \
    audio/x-mpeg }

#--- Img -----------------------------------------------------------------------
# On Unix/Linux and Windows we try to find the Img extension for reading more
# image formats than the standard core one (gif)..

set plugin(Img,full) "Img"
set plugin(Img,type) "Tcl plugin"
set plugin(Img,desc) "Adds more image formats than the standard one (gif)."
set plugin(Img,importProc) DoImportImageOrMovie
set supportedMimeTypes(Img) {\
    image/x-bmp         image/gif           image/jpeg         \
    image/png           image/x-png         image/tiff         \
    image/x-tiff        application/postscript}

#--- xanim ---------------------------------------------------------------------
# Test the 'xanim' app on Unix/Linux for multimedia.
  
set plugin(xanim,full) "xanim"
set plugin(xanim,type) "Helper application"
set plugin(xanim,desc) "A unix/Linux only application that is used\
  for displaying multimedia content in the canvas."
set plugin(xanim,importProc) DoImportImageOrMovie

# There are many more...
set supportedMimeTypes(xanim) {\
    audio/wav           audio/x-wav         video/mpeg         \
    video/x-mpeg        audio/mpeg          audio/x-mpeg       \
    audio/basic         video/quicktime }
      
# Add more supported filetypes as additional extensions and Mac types.
# Hook for adding other packages or plugins
#
# set packages2Platform(myPackage) "muw"
# lappend plugin(allPacks) myPackage
# lappend plugin(all) myPackage
# 
# set plugin(myPackage,...) ...
# set supSuff(myPackage) ...
# set supportedMimeTypes(myPackage) ...
#
#-------------------------------------------------------------------------------
# Search for the wanted packages in a systematic way.

set platform [string tolower [string index $tcl_platform(platform) 0]]
foreach packName $plugin(allPacks) {
    
    # Check first if this package can live on this platform.
    if {[string match "*${platform}*" $packages2Platform($packName)]}  {
	
	# Search for it!
	if {[info exists ::SplashScreen::startMsg]}  {
	    set ::SplashScreen::startMsg "Looking for $packName..."
	}
	if {![catch {package require $packName} msg]}  {
	    set prefs($packName) 1
	    set plugin($packName,ver) $msg
	} else {
	    set prefs($packName) 0
	}
    } else {
	set prefs($packName) 0
    }
}

# Test the 'xanim' app on Unix/Linux for multimedia.

if {[string compare $tcl_platform(platform) "unix"] == 0 }  {
    if {![catch {exec which xanim} apath]}  {
	set prefs(xanim) 1
    } else  {
	set prefs(xanim) 0
    }
} else  {
    set prefs(xanim) 0
}

# Mappings file extension (suffixes) to transfer mode; binary or text.
# Supported binary files, that is, images movies etc.
# Start with the core Tk supported formats. Mac 'TYPE'.
set prefs(tk) 1
set supSuff(text) {.txt .tcl}
set supSuff(image) {}
set supSuff(audio) {}
set supSuff(video) {}
set supSuff(application) {}

# Map keywords and package names to the supported MIME types.
# Start by initing, MIME types added below.

set supportedMimeTypes(text) {text/plain}
set supportedMimeTypes(image) {}
set supportedMimeTypes(audio) {}
set supportedMimeTypes(video) {}
set supportedMimeTypes(application) {}
set supportedMimeTypes(all) $supportedMimeTypes(text)
set supMacTypes(text) {TEXT}
if {[info tclversion] >= 8.3}  {
    
    # more?.
    set supSuff(tk) {.gif}
    set supportedMimeTypes(tk) {image/gif}
} else {
    set supSuff(tk) {.gif}
    set supportedMimeTypes(tk) {image/gif}
}


# Now its time to systematically make the 'supSuff',
# 'supMacTypes', 'supportedMimeTypes', 'mimeType2Packages'.
  
set mime_ {[^/]+}

# We add the tk library to the other ones.
foreach packName "tk $plugin(all)" {
    if {$prefs($packName)}  {
	
	# Loop over all file MIME types supported by this specific package.
	foreach mimeType $supportedMimeTypes($packName) {
	    
	    # Collect all suffixes for this package.
	    eval lappend supSuff($packName) $mime2SuffixList($mimeType)
	    
	    # Get the MIME base: text, image, audio...
	    if {[regexp "(${mime_})/" $mimeType match mimeBase]}  {

		eval lappend supSuff($mimeBase) $mime2SuffixList($mimeType)
		
		# Add upp all "binary" files.
		if {[string compare $mimeBase "text"] != 0}  {
		    eval lappend supSuff(bin) $mime2SuffixList($mimeType)
		}
		
		# Collect the mac types.
		eval lappend supMacTypes($mimeBase) $mime2MacTypesList($mimeType)
		lappend supportedMimeTypes($mimeBase) $mimeType
		lappend mimeType2Packages($mimeType) $packName
		
		# Add upp all "binary" files.
		if {[string compare $mimeBase "text"] != 0}  {
		    lappend supportedMimeTypes(bin) $mimeType
		}
	    }
	}
	eval lappend supSuff(all) $supSuff($packName)
	eval lappend supportedMimeTypes(all) $supportedMimeTypes($packName)
    }
}

# Remove duplicates in lists.
foreach packName "tk $plugin(all)" {
    if {[info exists supSuff($packName)]} {
	set supSuff($packName) [luniq $supSuff($packName)]
    }
}
foreach mimeBase {text image audio video application} {
    if {[info exists supSuff($mimeBase)]} {
	set supSuff($mimeBase) [luniq $supSuff($mimeBase)]
    }
    if {[info exists supportedMimeTypes($mimeBase)]} {
	set supportedMimeTypes($mimeBase) [luniq $supportedMimeTypes($mimeBase)]
    }
}
foreach key [array names supMacTypes] {
    set supMacTypes($key) [luniq $supMacTypes($key)]
}
set supSuff(all) [luniq $supSuff(all)]
set supSuff(bin) [luniq $supSuff(bin)]
set supportedMimeTypes(all) [luniq $supportedMimeTypes(all)]
set supportedMimeTypes(bin) [luniq $supportedMimeTypes(bin)]

# Some kind of mechanism needed to select which package to choose when
# more than one package can support a suffix.
# Here we just takes the first one. Should find a better way!

foreach theMime $supportedMimeTypes(bin) {
    set prefMimeType2Package($theMime) [lindex $mimeType2Packages($theMime) 0]
}

# By default, no importing takes place, thus {0}.
set prefMimeType2Package(text/plain) {0}
  
# Create the 'typelist' option for the Open Image/Movie dialog and 
# standard text files.

if {$tcl_platform(platform) == "macintosh"}  {

    # On Mac either file extension or 'type' must match.
    set typelistText [list  \
      [list "Text" $supSuff(text)]  \
      [list "Text" {} $supMacTypes(text)] ]
    
    set typelistImageMovie [list   \
      [list "Image" $supSuff(image)]  \
      [list "Image" {} $supMacTypes(image)] ]
    if {[llength $supSuff(audio)] > 0}  {
	lappend typelistImageMovie  \
	  [list "Audio" $supSuff(audio)]  \
	  [list "Audio" {} $supMacTypes(audio)]
    }
    if {[llength $supSuff(video)] > 0}  {
	lappend typelistImageMovie  \
	  [list "Video" $supSuff(video)]  \
	  [list "Video" {} $supMacTypes(video)]
    }	
     if {[llength $supSuff(application)] > 0}  {
	lappend typelistImageMovie  \
	  [list "Application" $supSuff(application)]  \
	  [list "Application" {} $supMacTypes(application)]
    }	
   lappend typelistImageMovie [list "Any File" *]
    
} elseif {$tcl_platform(platform) != "macintosh"}  {

    # Make a separate entry for each file extension. Sort.
    set tlist {}
    foreach ext $supSuff(text) {
	lappend tlist [list [string toupper [string trim $ext .]] $ext]
    }
    set ilist {}
    foreach ext $supSuff(image) {
	lappend ilist [list [string toupper [string trim $ext .]] $ext]
    }
    set alist {}
    foreach ext $supSuff(audio) {
	lappend alist [list [string toupper [string trim $ext .]] $ext]
    }
    set mlist {}
    foreach ext $supSuff(video) {
	lappend mlist [list [string toupper [string trim $ext .]] $ext]
    }
    set applist {}
    foreach ext $supSuff(application) {
	lappend applist [list [string toupper [string trim $ext .]] $ext]
    }
    set sortlist [lsort -index 0 [concat $ilist $alist $mlist $applist]]
    set typelistImageMovie "$sortlist {{Any File} *}"	    
}

#   GetMimeTypeFromFileName ---
#
#   Return the file's MIME type, either from it's suffix, or on mac, it's
#   file type. Returns an empty string if MIME type unknown.

proc GetMimeTypeFromFileName { fileName }  {
    global  suffix2MimeList tcl_platform
    
    set fext [string tolower [file extension $fileName]]
    if {[string compare $tcl_platform(platform) "macintosh"] == 0} {
	set theMime [GetMimeTypeForMacFile $fileName]
    } else {
	if {[info exists suffix2MimeList($fext)]} {
	    set theMime [lindex $suffix2MimeList($fext) 0]
	} else {
	    set theMime {}
	}
    }
    return $theMime
}
    
#   GetMimeTypeForMacFile ---
#
#   If file suffix exists it determines MIME type, else use Mac TYPE.

proc GetMimeTypeForMacFile { fileName }  {
    global  macType2MimeList suffix2MimeList
    
    set fext [string tolower [file extension $fileName]]
    if {[string length $fext] > 0}  {
	return [lindex $suffix2MimeList($fext) 0]
    } else {
	set macType [file attributes $fileName -type]
	if {[string length $macType] > 0}  {
	    return [lindex $macType2MimeList($macType) 0]
	} else {
	    return {}
	}
    }
}

#-------------------------------------------------------------------------------
