
#  ProgressWindow.tcl ---
#  
#      This file is part of the whiteboard application. It makes a progress 
#      window. The 'updatePerc' is a number between 0 and 100. Init by -1.
#      The 'cancelCmd' should contain the full command for the cancel operation.
#      If we have the add on "Progressbar", 'prefs(Progressbar)' is true (1).
#      
#  Copyright (c) 1999-2000  Mats Bengtsson
#  
#  See the README file for license, bugs etc.
#  
#------------------------------------------------------------------------
#
#   It is written in a simplified object oriented style. 
#   
#   NAME
#      ProgressWindow - Creates a progress window.
#   SYNOPSIS
#      ProgressWindow toplevelWindow ?configure? ?options?
#   SPECIFIC OPTIONS
#      -cancelcmd
#      -filename
#      -name
#      -percent
#      -text
#      
#   If toplevel already exists it is just configured with the new options.

namespace eval ::ProgressWindow:: {
    
    # Main routine gets exported.
    namespace export ProgressWindow
    
    # Collect all possible options and their defaults.
    variable widgetOptions
    variable widgetCmds
    array set widgetOptions {-percent 0 -filename {} -name Progress  \
      -cancelcmd {} -text {Writing file:} -font1 {}}
    set widgetCmds {configure}
    
    # Dimensions, same for all windows.
    variable dims
    array set dims {wwidth 282 wheight 88 xnw 12 ynw 42 xse 198 yse 54 ymid 48 \
      xtxt0 11 ytxt0 23 xtxt1 11 ytxt1 76 xtxt2 0 ytxt2 76 xcanbt 212  \
      xtxt0off 0}
}

# ::ProgressWindow::ProgressWindow ---
#
#    The constructor, and all methods as well.

proc ::ProgressWindow::ProgressWindow { w args } {
    global  tcl_platform prefs debugLevel

    variable widgetOptions
    variable widgetCmds
    variable dims
        
    if {$debugLevel >= 2}  {
	puts "ProgressWindow:: w=$w, args=$args"
    }
    set theCmd 0
    
    # Find out if command or not.
    if {[lsearch $widgetCmds [lindex $args 0]] >= 0} {
	set theCmd [lindex $args 0]
	set args [lreplace $args 0 0]
	puts "new args=$args"
    }
    
    # Some error checking. Odd number, bail.
    if {[expr {[llength $args]%2}] == 1}  {
	error "Options are not consistent"
    }
    set allOptNames [array names widgetOptions]
    foreach {name value} $args  {
	if {[lsearch $allOptNames $name] == -1}  {
	    error "ProgressWindow: option $name is not an option."
	}
    }
    # Configure new options.
    if {[winfo exists $w]}  {
	if {$theCmd == 0} {
	    error "ProgressWindow: configure only command if window already exists."
	}
	
	if {[llength $args] == 0}  {
	    return
	} else {
	    
	    # Update the features of changed options.
	    eval Configure $w $args
	}
    } else {
	if {$theCmd != 0} {
	    error "ProgressWindow: window must exist before configure."
	}

	# Namespace for this specific instance.
	namespace eval ::ProgressWindow::$w  {
	    variable options
	}
	# Refer to these variables by local names.
	upvar ::ProgressWindow::${w}::options options
	
	# Set default options.
	foreach name [array names widgetOptions] {
	    set options($name) $widgetOptions($name)
	}
	# Parse options. Overwrite default options.
	array set options $args
	
	# Ready to actually make it.
	Build $w
	
	# Cleanup things when finished.
	bind $w <Destroy> [list ::ProgressWindow::Cleanup $w]
    }
    return $w
}

# ::ProgressWindow::Build ---
#
#    The progress window is created.

proc ::ProgressWindow::Build { w }  {
    global  sysFont tcl_platform prefs debugLevel
    
    upvar ::ProgressWindow::${w}::options options
    variable dims
    
    if {$debugLevel >= 2}  {
	puts "::ProgressWindow::Build:: w=$w"
    }
    if {$tcl_platform(platform) == "macintosh"}  {
	toplevel $w; unsupported1 style $w documentProc;  \
	  wm withdraw $w
    } else  {
	toplevel $w; wm withdraw $w
    }
    wm title $w $options(-name)
    wm resizable $w 0 0

    # Width of progress bar.
    set prwidth [expr $dims(xse) - $dims(xnw)]
    
    # Build.
    canvas $w.c -scrollregion "0 0 $dims(wwidth) $dims(wheight)"  \
      -width $dims(wwidth) -height $dims(wheight) -highlightthickness 0  \
      -bd 1 -relief raised
    pack $w.c -side top -fill x
    if {$options(-font1) == ""} {
	set id [$w.c create text $dims(xtxt0) $dims(ytxt0) -anchor sw  \
	  -text $options(-text) -tags {ttext topttxt}]
    } else {
	set id [$w.c create text $dims(xtxt0) $dims(ytxt0) -anchor sw  \
	  -font $options(-font1) -text $options(-text) -tags {ttext topttxt}]
    }
    set xoff [expr [lindex [$w.c bbox $id] 2] + 10]
    set dims(xtxt0off) $xoff
    if {[info exists options(-filename)]}  {
	$w.c create text $xoff $dims(ytxt0) -anchor sw -text $options(-filename) \
	  -font $sysFont(sb) -tags {ttext tfilename}
    }
    set id [$w.c create text $dims(xtxt1) $dims(ytxt1) -anchor sw  \
      -text "Remaining: " -font $sysFont(s) -tags ttext]
    set dims(xtxt2) [expr [lindex [$w.c bbox $id] 2] + 10]
    if {$options(-percent) >= 100}  {
	$w.c create text $dims(xtxt2) $dims(ytxt2) -anchor sw   \
	  -text "Document: done" -font $sysFont(s) -tags {ttext percent}  \
	  -fill black
    } else  {
	$w.c create text $dims(xtxt2) $dims(ytxt2) -anchor sw   \
	  -text "[expr 100 - int($options(-percent))]% left"  \
	  -font $sysFont(s) -tags {ttext percent} -fill black
    }    
    
    # Either use the "Progressbar" package or make our own.
    if {$prefs(Progressbar)}  {
	set wpgb [::progressbar::progressbar $w.c.pr   \
	  -variable ::ProgressWindow::${w}::percent \
	  -width $prwidth -background $prefs(bgColGeneral)  \
	  -percent $options(-percent)]
	$w.c create window $dims(xnw) [expr $dims(ymid) + 1] -anchor w \
	  -window $wpgb
    } else {
	$w.c create rectangle $dims(xnw) $dims(ynw) $dims(xse) $dims(yse)  \
	  -fill #CECEFF -outline {}
	$w.c create rectangle $dims(xnw) $dims(ynw) $dims(xnw) $dims(yse)  \
	  -outline {} -fill #424242 -tag progbar
	$w.c create line $dims(xnw) $dims(ynw) $dims(xnw) $dims(yse)   \
	  $dims(xse) $dims(yse) $dims(xse) $dims(ynw) $dims(xnw) $dims(ynw)  \
	  -width 1 -fill black
	if {$options(-percent) > 0}  {
	    $w.c coords progbar $dims(xnw) $dims(ynw)   \
	      [expr $options(-percent)*($dims(xse) - $dims(xnw))/100 + $dims(xnw)] \
	      $dims(yse)
	}
    }
    
    # Change to flat when no focus, to 3d when focus. Standard MacOS.
    if {$prefs(Progressbar)}  {
	bind $w <FocusOut> [list ::ProgressWindow::PBFocusOut $w $wpgb]
	bind $w <FocusIn> [list ::ProgressWindow::PBFocusIn $w $wpgb]
    } else {
	bind $w <FocusOut> [list ::ProgressWindow::PBFocusOut $w junk]
	bind $w <FocusIn> [list ::ProgressWindow::PBFocusIn $w junk]
    }
    focus $w
    button $w.c.bt -text "Cancel"  \
      -command [list ::ProgressWindow::CancelBt $w $options(-cancelcmd)]
    $w.c create window $dims(xcanbt) $dims(ymid) -window $w.c.bt -anchor w
    update idletasks
    wm deiconify $w
    raise $w
    update idletasks
}

# ::ProgressWindow::Configure ---
#
#    Here we just configures an existing progress window.

proc ::ProgressWindow::Configure { w args }  {
    global  sysFont prefs debugLevel
    
    # Refer to these variables by local names.
    upvar ::ProgressWindow::${w}::options options
    variable dims
    
    if {$debugLevel >= 2}  {
	puts "::ProgressWindow::Configure:: w=$w, args=$args"
    }
    # 'args' contain the new options while 'options' contain the present options.
    # Only if an option has changed it should be dealt with.
    foreach {name value} $args {
	set opts($name) $value
    }
    if {[string compare [focus] $w] == 0}  {
	set col black
    } else  {
	set col #6B6B6B
    }
    #puts "::ProgressWindow::Configure:: array names opts=[array names opts]"

    # Process all configuration options.
    foreach optName [array names opts] {
	if {[string compare $opts($optName) $options($optName)] == 0}  {
	    continue
	}
	switch -- $optName  {
	    -percent        {
		set perc $opts(-percent)
		
		# Only update progress bar
		if {$prefs(Progressbar)}  {
		    set ::ProgressWindow::${w}::percent [expr int($perc)]
		} else {
		    $w.c coords progbar $dims(xnw) $dims(ynw)   \
		      [expr $perc*($dims(xse) - $dims(xnw))/100 + $dims(xnw)]  \
		      $dims(yse)
		}
		# Percentage left.
		$w.c delete percent
		if {$perc >= 100}  {
		    $w.c create text $dims(xtxt2) $dims(ytxt2) -anchor sw   \
		      -text "Document: done" -font $sysFont(s) -tags {ttext percent}  \
		      -fill $col
		} else  {
		    $w.c create text $dims(xtxt2) $dims(ytxt2) -anchor sw   \
		      -text "[expr 100 - int($perc)]% left"  \
		      -font $sysFont(s) -tags {ttext percent} -fill $col
		}
	    }
	    -name         {
		wm title $w $opts(-name)
	    }
	    -filename     {
		$w.c delete tfilename
		$w.c create text $dims(xtxt0off) $dims(ytxt0) -anchor sw   \
		  -text $opts(-filename) -fill $col   \
		  -font $sysFont(sb) -tags {ttext tfilename}
	    }
	    -cancelcmd    {
		$w.c.bt configure  \
		  -command [list ::ProgressWindow::CancelBt $w $opts(-cancelcmd)]
	    }
	    -text         {
		error "-text should only be set when creating the progress window"
	    }
	    default       {
		
	    }
	}
    }
    # Save newly set options.
    array set options $args
    update idletasks    
}

# ::ProgressWindow::CancelBt ---
#
#    When pressing the cancel button, evaluate any -cancelcmd in the
#    correct namespace, and destroy the window.

proc ::ProgressWindow::CancelBt { w cancelCmd }  {
    
    puts "::ProgressWindow::CancelBt:: w=$w, cancelCmd=$cancelCmd"
    
    # We need to have a fully qualified command name here.
    # This command is always called from the global namespace.
    if {[llength $cancelCmd] > 0}  {
	if {![string match "::*" $cancelCmd]}  {
	    set nsup1 [uplevel 1 namespace current]
	    puts "::ProgressWindow::CancelBt:: nsup1=$nsup1"
	    set cancelCmd ::$cancelCmd
	}
	puts "::ProgressWindow::CancelBt:: cancelCmd=$cancelCmd"
	eval $cancelCmd
    }
    catch {destroy $w}
}

proc ::ProgressWindow::Cleanup { w }  {
    
    catch {namespace delete ::ProgressWindow::$w}
}

proc ::ProgressWindow::PBFocusIn { w wpgb }  {
    global  prefs
    
    if {$prefs(Progressbar)}  {
	$wpgb configure -shape 3d -color @blue0
    } else {
	$w.c itemconfigure progbar -fill #424242
    }
    $w.c itemconfigure ttext -fill black
    $w.c.bt configure -state normal
}

proc ::ProgressWindow::PBFocusOut { w wpgb }  {
    global  prefs
    
    if {$prefs(Progressbar)}  {
	$wpgb configure -shape flat -color #9C9CFF
    } else {
	$w.c itemconfigure progbar -fill #9C9CFF
    }
    $w.c itemconfigure ttext -fill #6B6B6B
    $w.c.bt configure -state disabled
}

#----------------------------------------------------------------------------