
#  PutFile.tcl ---
#  
#      This file is part of the whiteboard application. It contains a number
#      of procedures for performing a put operation over the network from
#      a disk file.
#      
#  Copyright (c) 1999-2000  Mats Bengtsson
#  
#  See the README file for license, bugs etc.

namespace eval ::PutFile:: {
    
    # Main routine gets exported.
    namespace export PutFileDlg PutFile PutFileToClient PutFileCancelAll
    
    # Internal vars only.
    variable putFileKillerId
    variable recordPutFile
}

#   PutFileDlg ---
#
#   Opens a file in a dialog and lets 'PutFile' do the job of transferring
#   the file to all other clients.

proc ::PutFile::PutFileDlg  {  }  {
    global  allIPnumsToSend typelistImageMovie typelistText debugLevel
    
    if {[llength $allIPnumsToSend] == 0}  {
	return
    }
    
    # In the dialog we need all entries from 'typelistImageMovie', but also
    # the standard text files.
    set typelist [concat $typelistText $typelistImageMovie]
    set ans [tk_getOpenFile -title "Put Image/Movie" -filetypes $typelist]
    if {$ans == ""}	 {
	return
    }
    set fileName $ans
    
    # Do the actual putting once the file is chosen. 
    PutFile $fileName "all"
}

#   PutFile ---
#   
#   Transfers a file to all remote servers. It needs some negotiation to work.
#   'fileName' is the path to the file to be put.
#   where = "other" or "all": put only to remote clients.
#   where = ip number: put only to this remote client.
#   
#   The 'optList' is a list of 'key: value' pairs, resembling the html 
#   protocol for getting files, but where most keys correspond to a valid
#   "canvas create" option.
#   
#   The calling sequence is:
#   
#   'PutFileOpenAndConnect'  opens file and socket, requests a
#   connection and sets up event handler for clients answer.
#   
#   'PutFileSocketOpen'  is called when the socket becomes readable after an
#   async socket open.
#   
#   'PutFileWaitingForResponse'  waits until the client answers and
#   act accordingly; if ok it sets up the next event handler which
#   deals with the copying itself.
#
#   'PutFileCallback'  copies the file from disk to socket
#   until end of file using event handlers, when it closes things etc.

proc ::PutFile::PutFile  { fileName where {optList {}} }  {
    global  debugLevel allIPnumsToSend macType2Suff tcl_platform suffix2MimeList
    
    if {$debugLevel >= 2}  {
	puts "+PutFile:: fileName=$fileName, optList=$optList"
    }
    if {[llength $allIPnumsToSend] == 0}  {
	return
    }
    set fileTail [file tail $fileName]
    set fext [string tolower [file extension $fileTail]]
    
    # Add MIME type if not in 'optList' already, and size. Same with size.
    set theMime [GetMimeTypeFromFileName $fileName]
    if {$theMime == ""} {
	tk_messageBox -message "Error: Couldn't find the MIME type\
	  for the file $fileName" -icon error -type ok
	return
    }
    if {[lsearch $optList "Content-Type:"] < 0} {
	lappend optList "Content-Type:" $theMime
    }
    set theFileSize [file size $fileName]
    if {[lsearch $optList "size:"] < 0} {
	lappend optList "size:" $theFileSize
    }
    
    # Get the remote file name (no path).
    if {[string compare $tcl_platform(platform) "macintosh"] == 0} {
	if {[string length $fext] > 0} {
	    set fileTailRemote $fileTail
	} else {
	    set macType [file attributes $fileName -type]
	    if {[info exists macType2Suff($macType)]} {
		set fileTailRemote ${fileTail}$macType2Suff($macType)
	    } else {
		tk_messageBox -message "Error: unknown file suffix $fileName" \
		  -icon error -type ok
		return
	    }
	}	
    } else {
	set fileTailRemote $fileTail
    }
    
    # Make a list with all ip numbers to put file to.
    if {$where == "other" || $where == "all"}  {
	set allPutIP $allIPnumsToSend
    } else  {
	set allPutIP $where
    }    
    
    # Loop over all connected servers or only the specified one.
    foreach ip $allPutIP  {
	
	# All file copying to the sockets through event handlers.
	# See above for the calling sequence.
	PutFileOpenAndConnect $ip $fileName $fileTailRemote $theMime  \
	  $theFileSize $optList
    }
}

#   PutFileOpenAndConnect ---
#
#   recordPutFile(sock): {fileId filePath}; just to keep track of things.

proc ::PutFile::PutFileOpenAndConnect   \
  { ip fileName fileTailRemote theMime theFileSize {optList {}} }  {
    global  statMess ipNum2ServPort debugLevel ipNum2Name prefs
    
    variable recordPutFile

    if {$debugLevel >= 2}  {
	puts "+  PutFileOpenAndConnect:: ip=$ip, \
	  fileTailRemote=$fileTailRemote, optList=$optList"
    }
    
    # Open the file all over for each connected client.
    if {[catch {open $fileName r} orig]}  {
	tk_messageBox -message "Failed when trying to open $fileName." \
	  -icon error -type ok
	
	# Handle error.
	catch {close $orig}
	return 
    }
        
    # Use the MIME type to hint transfer mode for *file read*.
    if {[string match "text/*" $theMime]}  {
	fconfigure $orig -translation auto
    } else {
	fconfigure $orig -translation {binary binary}
    }
    
    # Open new temporary socket only for this put operation.
    set statMess "Contacting client $ipNum2Name($ip). Waiting for response..."
    update idletasks
    if {$debugLevel >= 2}  {
	puts "+  statMess=$statMess"
    }
    
    # The actual socket is opened; either -async or not.
    if {$prefs(asyncOpen)}  {
	set res [catch {socket -async $ip $ipNum2ServPort($ip)} s]
    } else {
	set res [catch {socket $ip $ipNum2ServPort($ip)} s]
    }
    
    # In case something went wrong during socket open.
    if {$res}  {
	tk_messageBox -message   \
	  "Failed when trying to connect to $ipNum2Name($ip). $s" \
	  -icon error -type ok
	
	# Handle error.
	PutFileShutDown $orig $s
	return
    }    

    # Set in nonblocking mode and register the next event handler.
    fconfigure $s -blocking 0
    fconfigure $s -buffering line
    
    # Store a record for this pair of socket fileId.
    set recordPutFile($s) [list $orig $fileName]
        
    # Schedule timeout event.
    PutFileScheduleKiller $orig $s $ip $fileTailRemote
    
    # If open socket in async mode, need to wait for fileevent.
    if {$prefs(asyncOpen)} {
	fileevent $s writable   \
	  [list [namespace current]::PutFileSocketOpen $ip $orig $s  \
	  $fileTailRemote $theMime $theFileSize $optList]
    } else {

	# Else, it is already open.
	PutFileSocketOpen $ip $orig $s $fileTailRemote $theMime $theFileSize \
	  $optList
    }
}

# PutFileSocketOpen ---
#
#     When the "put" socket is opened, either async or not, prepare for transfer
#     by sending the actual "PUT" command to the remote server, and set up
#     file event handler in order to receive the server's answer.

proc ::PutFile::PutFileSocketOpen { ip orig s fileTailRemote theMime  \
  totBytes optList}  {
    global  ipNum2Name statMess debugLevel
    
    if {$debugLevel >= 2}  {
	puts "+    PutFileSocketOpen:: ip=$ip, orig=$orig, s=$s, totBytes=$totBytes, \
	  fileTailRemote=$fileTailRemote, theMime=$theMime"
    }

    # Be sure to switch off any fileevents from previous procedures async open.
    fileevent $s writable {}
    
    # Want to be line oriented and non-blocking.
    fconfigure $s -blocking 0
    fconfigure $s -buffering line
        
    # Schedule timeout event.
    PutFileScheduleKiller $orig $s $ip $fileTailRemote
    
    # Set up event handler to wait for server response.
    
    fileevent $s readable [list [namespace current]::PutFileWaitingForResponse \
      $ip $orig $s $fileTailRemote $theMime $totBytes]
    set statMess "Client contacted: $ipNum2Name($ip); negotiating..."

    # This is the actual instruction to the remote server what to expect
    # Be sure to concat the 'optList'.
    
    puts  $s "PUT: $fileTailRemote $optList"
    flush $s    
    if {$debugLevel >= 2}  {
	puts  "+    PUT: $fileTailRemote $optList"
    }
}

# PutFileWaitingForResponse ---
#
#     Is called when the remote server responds to the "PUT" instruction.
#     The response is a one byte non-printable char.

proc ::PutFile::PutFileWaitingForResponse { ip orig s fileTailRemote theMime totBytes }  {
    global  statMess chunkSize tclwbProtMsg ipNum2Name debugLevel
    
    set int_ {[0-9]+}
    set any_ {.+}
    
    # Get server response.
    if {[gets $s line] == -1} {
	puts "Error reading server response. line=$line"
	set statMess "Error reading server response from $ipNum2Name($ip)"
	PutFileShutDown $orig $s
	return
    }
    if {$debugLevel >= 2}  {
	puts "+      PutFileWaitingForResponse:: line=$line"
    }
	
    # Catch problems.
    
    if {![regexp "^TCLWB/(${int_}\.${int_}) +($int_) +($any_)"  $line match  \
      version respCode msg]} {
	set statMess "The server at $ipNum2Name($ip) didn't respond with a\
	  well formed protocol"
 	PutFileShutDown $orig $s
	return
    } elseif {![info exists tclwbProtMsg($respCode)]}  {
	set statMess "The server at $ipNum2Name($ip) responded with an unkown code."
	PutFileShutDown $orig $s
	return
    } elseif {$respCode != 200}  {
	set statMess "$tclwbProtMsg($respCode)"
	PutFileShutDown $orig $s
	return
    } 
    
    set statMess "Client at $ipNum2Name($ip) responded."
    update idletasks
    if {$debugLevel >= 2}  {
	puts "+      PutFileWaitingForResponse:: statMess=$statMess"
    }
    if {[eof $s]}  {

	# Shutdown!
	PutFileShutDown $orig $s
	return
    }
    
    # Do the actual transfer. fcopy registers 'PutFileCallback'.
    if {$debugLevel >= 2}  {
	puts "+      PutFileWaitingForResponse:: start transfer"
    }

    # In order for the server to read a complete line, binary mode
    # must wait until the line oriented part is completed.
    # Use the MIME type to hint transfer mode for *socket write*.

    if {[string match "text/*" $theMime]}  {
        fconfigure $s -translation auto

	# For nonlatin characters to work be sure to use Unicode/UTF-8.
	if {[info tclversion] >= 8.1}  {
	    catch {fconfigure $s -encoding utf-8}
	}
    } else {
        fconfigure $s -translation {binary binary}
    }
    
    # Be sure to switch off any fileevent before fcopy.
    fileevent $s readable {}
    fileevent $s writable {}
    
    # Schedule timeout event.
    PutFileScheduleKiller $orig $s $ip $fileTailRemote

    fcopy $orig $s -size $chunkSize -command   \
      [list [namespace current]::PutFileCallback $ip $orig $s  \
      $fileTailRemote $totBytes 0]
}

#   PutFileCallback ---
#
#   Callback for file copying from file to network socket. 
#   'bytes' is the number of bytes copied and is appended by fcopy,
#   'error' is also appended by fcopy if there is one.

proc ::PutFile::PutFileCallback { ip orig s name totBytes sumBytes  \
  bytes {error {}} }  {
    global  chunkSize ipNum2Name statMess debugLevel
    
    incr sumBytes $bytes
    set percent [format "%3.0f" [expr 100 - 100*$sumBytes/($totBytes + 1.0)]]
    set statMess "Putting file $name to $ipNum2Name($ip) (${percent}% left)"
    update idletasks
    if {$debugLevel >= 2}  {
	puts "+        PutFileCallback:: error=$error, name=$name, bytes=$bytes, err=$error"
	puts "+        statMess=$statMess"
    }
    
    # Catch an error if the socket is already closed. Buggy!!!!!
    if {[catch {eof $s}]}  {
	return
    }
    
    # If error or file end.
    if {[string length $error] != 0} {
	if {$debugLevel >= 2}  {
	    puts "+        PutFileCallback:: Error!, err=$error"
	}
	PutFileShutDown $orig $s
    } elseif {[eof $orig]} {
	if {$debugLevel >= 2}  {
	    puts "+        PutFileCallback:: end transfer eof on disk file, err=$error"
	}
	PutFileShutDown $orig $s
    } elseif {[eof $s]} {
	if {$debugLevel >= 2}  {
	    puts "+        PutFileCallback:: end transfer eof on socket, err=$error"
	}
	PutFileShutDown $orig $s
	set statMess "Network error when putting file $name on $ipNum2Name($ip)."
    } else {
	
	# Schedule timeout event.
	PutFileScheduleKiller $orig $s $ip $name
	
	# Rebind copy callback.
	fcopy $orig $s -command   \
	  [list [namespace current]::PutFileCallback $ip $orig $s $name  \
	  $totBytes $sumBytes]  \
	  -size $chunkSize
	if {$debugLevel >= 2}  {
	    puts "+        PutFileCallback:: just after new fcopy callback"
	}
    }
}

# PutFileToClient ---
# 
#    Inits a put operation on an already open socket as a response to a 
#    GET request received by the servers 'HandleClientRequest'.
#    The protocol is typically:
#        TCLWB/1.0 200 OK
#        key1: value1 key2: value2 ...
#        and the data comes here...
#    
# Arguments:
#    s                 the socket.
#    ip                its ip number.
#    relativeFilePath  the optionally relative path pointeng to the file.
# Results:
#    none.

proc ::PutFile::PutFileToClient { s ip relativeFilePath optList }  {
    global  tclwbProtMsg thisPath chunkSize debugLevel

    variable recordPutFile

    if {$debugLevel >= 2}  {
	puts "+      PutFileToClient:: s=$s, ip=$ip,\
	  relativeFilePath=$relativeFilePath, optList=$optList"
    }
    fconfigure $s -buffering line
    
    # Need to find the absolute path to 'relativeFilePath' with respect to
    # our base directory 'thisPath'.

    set filePath [AddAbsolutePathWithRelative $thisPath $relativeFilePath]
    set fileTail [file tail $filePath]
    if {![file isfile $filePath]} {
	puts $s "TCLWB/1.0 404 $tclwbProtMsg(404)"
	close $s
	return 
    }	
    
    # Open the file.
    if {[catch {open $filePath r} orig]}  {
	tk_messageBox -message "Failed when trying to open $filePath." \
	  -icon error -type ok
	
	# Handle error.
	catch {close $orig}
	puts $s "TCLWB/1.0 500 $tclwbProtMsg(500)"
	close $s
	return 
    }
    
    # Add MIME type if not in 'optList' already.
    set theMime [GetMimeTypeFromFileName $fileTail]
    if {$theMime == ""} {
	tk_messageBox -message "Error: Couldn't find the MIME type\
	  for the file $fileTail" -icon error -type ok
	return
    }
    set totBytes [file size $filePath]
    
    # Store a record for this pair of socket fileId.
    set recordPutFile($s) [list $orig $fileTail]
    
    # If we have come so far it's ok.
    puts $s "TCLWB/1.0 200 $tclwbProtMsg(200)"
    flush $s
    
    # Assemble the optList.
    lappend optList {size:} $totBytes {Content-Type:} $theMime
    puts $s $optList
    flush $s
    if {$debugLevel >= 2}  {
	puts "+      PutFileToClient:: just put optList=$optList"
    }
        
    # In order for the server to read a complete line, binary mode
    # must wait until the line oriented part is completed.
    # Use the MIME type to hint transfer mode for *socket write* and
    # for *file read*.

    if {[string match "text/*" $theMime]}  {
        fconfigure $s -translation auto
	fconfigure $orig -translation auto

	# For nonlatin characters to work be sure to use Unicode/UTF-8.
	if {[info tclversion] >= 8.1}  {
	    catch {fconfigure $s -encoding utf-8}
	}
    } else {
        fconfigure $s -translation {binary binary}
	fconfigure $orig -translation {binary binary}
    }
    
    # Be sure to switch off any fileevent before fcopy.
    fileevent $s readable {}
    fileevent $s writable {}
    
    # Schedule timeout event.
    PutFileScheduleKiller $orig $s $ip $fileTail

    fcopy $orig $s -size $chunkSize -command   \
      [list [namespace current]::PutFileCallback $ip $orig $s $fileTail   \
      $totBytes 0]
}

#   PutFileScheduleKiller, PutFileKill, PutFileShutDown ---
#
#   Utility routines to handle timeout events on get file operations.

proc ::PutFile::PutFileScheduleKiller { orig s ip fileName }  {
    global  prefs
    
    variable putFileKillerId
 
    if {[info exists putFileKillerId($s)]}  {
	after cancel $putFileKillerId($s)
    }
    set putFileKillerId($s) [after [expr 1000*$prefs(timeout)]   \
      [list [namespace current]::PutFileKill $orig $s $ip $fileName]]
}

proc ::PutFile::PutFileKill { orig s ip fileName }  {
    global  prefs ipNum2Name
    
    variable putFileKillerId

    if {![info exists putFileKillerId($s)]}  {
	return
    }
    tk_messageBox -message   \
      "Timout when waiting for data for file $fileName from $ipNum2Name($ip)" \
      -icon error -type ok
    PutFileShutDown $orig $s
}

proc ::PutFile::PutFileShutDown { orig s }  {
    global  statMess debugLevel

    variable putFileKillerId
    variable recordPutFile

    if {$debugLevel >= 2}  {
	puts "+          PutFileShutDown::"
    }
    
    #  Cleanup.
    catch {unset recordPutFile($s)}
    if {[info exists putFileKillerId($s)]}  {
	after cancel $putFileKillerId($s)
    }
    catch {unset putFileKillerId($s)}
    set statMess {}
    catch {close $orig}
    catch {close $s}
}

#   PutFileCancelAll ---
#
#   It is supposed to stop every put operation taking place.
#   This may happen when the user presses a stop button or something.
#   
#   recordPutFile(sock): {fileId filePath}; just to keep track of things.

proc ::PutFile::PutFileCancelAll  {  }  {
    global  debugLevel
    
    variable recordPutFile

    if {$debugLevel >= 2}  {
	puts "+PutFileCancelAll::"
	catch {parray recordPutFile}
    }
    
    # Close and clean up. Make temporary records since they are unset in
    # the shutdown routines.
    # The Put part:
    if {[info exists recordPutFile]}  {
	array set tmpRecordPutFile [array get recordPutFile]
	foreach s [array names tmpRecordPutFile]  {
	    PutFileShutDown [lindex $tmpRecordPutFile($s) 0] $s
	}
    }
}

#-------------------------------------------------------------------------------
